<?php
// Start session
session_start();

// Include database connection
require_once "../config/connection.php";

// Redirect if not IT user - using functions from connection.php
if (!is_logged_in() || !is_user_type('it')) {
    header("Location: ../login.php");
    exit();
}

// Get IT user data
$it_id = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM it_users WHERE id = ?");
$stmt->bind_param("i", $it_id);
$stmt->execute();
$it_data = $stmt->get_result()->fetch_assoc();
$stmt->close();

// Set page title
$page_title = "Dashboard IT";

// Log activity using audit_logs
create_audit_log($conn, 'Dashboard Access', $it_id, 'it', 'IT user accessed dashboard');

// Get dashboard statistics
// 1. Total Color Code dan Stok
$total_color_stmt = $conn->prepare("SELECT COUNT(*) as total_colors FROM master_color_panel");
$total_color_stmt->execute();
$total_colors = $total_color_stmt->get_result()->fetch_assoc()['total_colors'];
$total_color_stmt->close();

$total_stock_stmt = $conn->prepare("SELECT COUNT(*) as total_stock FROM color_stock_items");
$total_stock_stmt->execute();
$total_stock = $total_stock_stmt->get_result()->fetch_assoc()['total_stock'];
$total_stock_stmt->close();

// 2. Total On Going (status peminjaman 'Dipinjam')
$ongoing_stmt = $conn->prepare("
    SELECT COUNT(*) as total_ongoing 
    FROM stock_transactions 
    WHERE status_peminjaman = 'Dipinjam'
");
$ongoing_stmt->execute();
$total_ongoing = $ongoing_stmt->get_result()->fetch_assoc()['total_ongoing'];
$ongoing_stmt->close();

// 3. Total Dipinjam (code barang dengan status 'Dipinjam')
$dipinjam_stmt = $conn->prepare("
    SELECT COUNT(*) as total_dipinjam 
    FROM color_stock_items 
    WHERE status = 'Dipinjam'
");
$dipinjam_stmt->execute();
$total_dipinjam = $dipinjam_stmt->get_result()->fetch_assoc()['total_dipinjam'];
$dipinjam_stmt->close();

// 4. Total Overdue (transaksi yang masih dipinjam dan melewati batas waktu + 15 menit)
$overdue_stmt = $conn->prepare("
    SELECT COUNT(*) as total_overdue 
    FROM stock_transactions st
    INNER JOIN transaction_items ti ON st.id = ti.transaction_id
    WHERE st.status_peminjaman = 'Dipinjam' 
    AND ti.dikembalikan = 0
    AND TIMESTAMPDIFF(MINUTE, st.tanggal_pengembalian, NOW()) > 15
");
$overdue_stmt->execute();
$total_overdue = $overdue_stmt->get_result()->fetch_assoc()['total_overdue'];
$overdue_stmt->close();

// 5. Total Renew (code barang dengan status 'Renew')
$renew_stmt = $conn->prepare("
    SELECT COUNT(*) as total_renew 
    FROM color_stock_items 
    WHERE status = 'Renew'
");
$renew_stmt->execute();
$total_renew = $renew_stmt->get_result()->fetch_assoc()['total_renew'];
$renew_stmt->close();

// 6. Total Tersedia (code barang dengan status 'Tersedia')
$tersedia_stmt = $conn->prepare("
    SELECT COUNT(*) as total_tersedia 
    FROM color_stock_items 
    WHERE status = 'Tersedia'
");
$tersedia_stmt->execute();
$total_tersedia = $tersedia_stmt->get_result()->fetch_assoc()['total_tersedia'];
$tersedia_stmt->close();

// Include header
include '../template_it/header.php';
?>

<!-- Main Content Area -->
<main class="flex-1 p-6 overflow-auto bg-gradient-to-br from-blue-100 via-teal-100 to-green-100">
    <!-- Welcome Section -->
    <div class="bg-white rounded-lg shadow-md p-6 mb-6">
        <div class="flex items-center justify-between">
            <div>
                <h2 class="text-2xl font-bold text-gray-800">Dashboard IT Support</h2>
                <p class="text-gray-600 mt-1">Selamat datang, <?php echo htmlspecialchars($it_data['username']); ?>!</p>
            </div>
            <div class="text-right">
                <p class="text-sm text-gray-500">Akses terakhir</p>
                <p class="text-lg font-medium text-gray-800"><?php echo date('d M Y, H:i'); ?></p>
            </div>
        </div>
    </div>

    <!-- Dashboard Cards -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-6">
        <!-- Total Color Code dan Stok - Abu -->
        <div class="bg-gradient-to-br from-gray-500 to-gray-600 rounded-lg shadow-md p-6 text-white">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-white bg-opacity-20 mr-4">
                    <i class="fas fa-palette text-2xl text-white"></i>
                </div>
                <div>
                    <h3 class="text-lg font-semibold text-white">Total Color & Stok</h3>
                    <p class="text-2xl font-bold text-white"><?php echo $total_colors; ?> Colors</p>
                    <p class="text-sm text-gray-200"><?php echo $total_stock; ?> Total Stok Items</p>
                </div>
            </div>
        </div>

        <!-- Total On Going - Biru -->
        <div class="bg-gradient-to-br from-blue-500 to-blue-600 rounded-lg shadow-md p-6 text-white cursor-pointer hover:shadow-lg hover:from-blue-600 hover:to-blue-700 transition-all duration-200" 
             onclick="showOngoingModal()">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-white bg-opacity-20 mr-4">
                    <i class="fas fa-clock text-2xl text-white"></i>
                </div>
                <div>
                    <h3 class="text-lg font-semibold text-white">On Going</h3>
                    <p class="text-2xl font-bold text-white"><?php echo $total_ongoing; ?></p>
                    <p class="text-sm text-blue-100">Transaksi Berlangsung</p>
                </div>
            </div>
        </div>

        <!-- Total Dipinjam - Ungu -->
        <div class="bg-gradient-to-br from-purple-500 to-purple-600 rounded-lg shadow-md p-6 text-white cursor-pointer hover:shadow-lg hover:from-purple-600 hover:to-purple-700 transition-all duration-200" 
             onclick="showDipinjamModal()">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-white bg-opacity-20 mr-4">
                    <i class="fas fa-hand-holding text-2xl text-white"></i>
                </div>
                <div>
                    <h3 class="text-lg font-semibold text-white">Dipinjam</h3>
                    <p class="text-2xl font-bold text-white"><?php echo $total_dipinjam; ?></p>
                    <p class="text-sm text-purple-100">Items Dipinjam</p>
                </div>
            </div>
        </div>

        <!-- Total Overdue - Merah -->
        <div class="bg-gradient-to-br from-red-500 to-red-600 rounded-lg shadow-md p-6 text-white cursor-pointer hover:shadow-lg hover:from-red-600 hover:to-red-700 transition-all duration-200" 
             onclick="showOverdueModal()">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-white bg-opacity-20 mr-4">
                    <i class="fas fa-exclamation-triangle text-2xl text-white"></i>
                </div>
                <div>
                    <h3 class="text-lg font-semibold text-white">Overdue</h3>
                    <p class="text-2xl font-bold text-white"><?php echo $total_overdue; ?></p>
                    <p class="text-sm text-red-100">Transaksi Terlambat</p>
                </div>
            </div>
        </div>

        <!-- Total Renew - Kuning -->
        <div class="bg-gradient-to-br from-yellow-500 to-yellow-600 rounded-lg shadow-md p-6 text-white cursor-pointer hover:shadow-lg hover:from-yellow-600 hover:to-yellow-700 transition-all duration-200" 
             onclick="showRenewModal()">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-white bg-opacity-20 mr-4">
                    <i class="fas fa-redo text-2xl text-white"></i>
                </div>
                <div>
                    <h3 class="text-lg font-semibold text-white">Renew</h3>
                    <p class="text-2xl font-bold text-white"><?php echo $total_renew; ?></p>
                    <p class="text-sm text-yellow-100">Items Perlu Renew</p>
                </div>
            </div>
        </div>

        <!-- Total Tersedia - Hijau -->
        <div class="bg-gradient-to-br from-green-500 to-green-600 rounded-lg shadow-md p-6 text-white">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-white bg-opacity-20 mr-4">
                    <i class="fas fa-check-circle text-2xl text-white"></i>
                </div>
                <div>
                    <h3 class="text-lg font-semibold text-white">Tersedia</h3>
                    <p class="text-2xl font-bold text-white"><?php echo $total_tersedia; ?></p>
                    <p class="text-sm text-green-100">Items Tersedia</p>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- Modals will be included here -->
<?php include 'dashboard_modals.php'; ?>

<!-- JavaScript for modals -->
<script>
function showOngoingModal() {
    document.getElementById('ongoingModal').classList.remove('hidden');
    loadOngoingData();
}

function showDipinjamModal() {
    document.getElementById('dipinjamModal').classList.remove('hidden');
    loadDipinjamData();
}

function showOverdueModal() {
    document.getElementById('overdueModal').classList.remove('hidden');
    loadOverdueData();
}

function showRenewModal() {
    document.getElementById('renewModal').classList.remove('hidden');
    loadRenewData();
}

function closeModal(modalId) {
    document.getElementById(modalId).classList.add('hidden');
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modals = ['ongoingModal', 'dipinjamModal', 'overdueModal', 'renewModal'];
    modals.forEach(modalId => {
        const modal = document.getElementById(modalId);
        if (event.target === modal) {
            modal.classList.add('hidden');
        }
    });
}
</script>

<?php include '../template_it/footer.php'; ?>