<?php

namespace App\Http\Controllers\Invoice;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InvoiceHardware;
use App\Models\InvoiceBom;
use App\Models\Hardware;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class InvoiceHardwareController extends Controller
{
    /**
     * Display a listing of invoice hardware
     */
    public function index()
    {
        $invoiceHardwares = InvoiceHardware::with('invoiceBom')
            ->latest()
            ->paginate(10);
        
        return view('invoice.hardware.index', compact('invoiceHardwares'));
    }

    /**
     * Show the form for creating a new invoice hardware
     */
    public function create()
    {
        // Get all fixed invoice BOMs (yang sudah fix)
        $invoiceBoms = InvoiceBom::where('is_fixed', true)
            ->whereDoesntHave('invoiceHardware')
            ->latest()
            ->get();
        
        // Get all hardware grouped by type
        $hardwareTypes = Hardware::getTypes();
        $hardwareData = [];
        
        foreach ($hardwareTypes as $type) {
            $hardwareData[$type] = Hardware::byType($type)
                ->orderBy('material_description')
                ->get();
        }
        
        return view('invoice.hardware.create', compact('invoiceBoms', 'hardwareTypes', 'hardwareData'));
    }

    /**
     * Store a newly created invoice hardware
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'invoice_bom_id' => 'required|exists:invoice_boms,id',
            'hardware_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            // Check if invoice already has hardware
            $existing = InvoiceHardware::where('invoice_bom_id', $request->invoice_bom_id)->first();
            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invoice BOM ini sudah memiliki data hardware!'
                ], 422);
            }

            // Create invoice hardware
            $invoiceHardware = InvoiceHardware::create([
                'invoice_bom_id' => $request->invoice_bom_id,
                'hardware_data' => $request->hardware_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Hardware berhasil dibuat!',
                'invoiceHardware' => $invoiceHardware
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Hardware creation error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified invoice hardware
     */
    public function show($id)
    {
        $invoiceHardware = InvoiceHardware::with('invoiceBom')->findOrFail($id);
        return view('invoice.hardware.show', compact('invoiceHardware'));
    }

    /**
     * Show the form for editing the specified invoice hardware
     */
    public function edit($id)
    {
        $invoiceHardware = InvoiceHardware::with('invoiceBom')->findOrFail($id);
        
        // Get all hardware grouped by type
        $hardwareTypes = Hardware::getTypes();
        $hardwareData = [];
        
        foreach ($hardwareTypes as $type) {
            $hardwareData[$type] = Hardware::byType($type)
                ->orderBy('material_description')
                ->get();
        }
        
        return view('invoice.hardware.edit', compact('invoiceHardware', 'hardwareTypes', 'hardwareData'));
    }

    /**
     * Update the specified invoice hardware
     */
    public function update(Request $request, $id)
    {
        $invoiceHardware = InvoiceHardware::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'hardware_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            $invoiceHardware->update([
                'hardware_data' => $request->hardware_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Hardware berhasil diupdate!',
                'invoiceHardware' => $invoiceHardware
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Hardware update error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified invoice hardware
     */
    public function destroy($id)
    {
        $invoiceHardware = InvoiceHardware::findOrFail($id);
        $invoiceHardware->delete();

        return redirect()->route('invoice.hardware.index')
            ->with('success', 'Invoice Hardware berhasil dihapus!');
    }

    /**
     * Get hardware materials by type (AJAX)
     */
    public function getHardwareByType(Request $request)
    {
        $type = $request->input('type');
        
        $hardware = Hardware::byType($type)
            ->orderBy('material_description')
            ->get();
        
        return response()->json([
            'success' => true,
            'hardware' => $hardware
        ]);
    }
}