<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class InvoiceBom extends Model
{
    use HasFactory;

    protected $fillable = [
        'product_name',
        'product_category',
        'product_type',
        'product_code',
        'size',
        'invoice_date',
        'order_qty',
        'panel_type',
        'solid_type',
        'veneer_face_type',
        'veneer_back_type',
        'box_drawer_type',
        'image_path',
        'calculation_data',
        'painting_data',
        'grand_total',
        'total_m3',        // ✅ BARU
        'harga_per_m3',    // ✅ BARU
        'biaya_produksi',  // ✅ BARU
        'is_fixed'
    ];

    protected $casts = [
        'invoice_date' => 'date',
        'order_qty' => 'integer',
        'calculation_data' => 'array',
        'painting_data' => 'array',
        'grand_total' => 'decimal:2',
        'total_m3' => 'decimal:6',       // ✅ BARU
        'harga_per_m3' => 'decimal:2',   // ✅ BARU
        'biaya_produksi' => 'decimal:2', // ✅ BARU
        'is_fixed' => 'boolean'
    ];

    /**
     * Scope untuk filter invoice yang belum fix
     */
    public function scopeEditable($query)
    {
        return $query->where('is_fixed', false);
    }

    /**
     * Scope untuk filter invoice yang sudah fix
     */
    public function scopeFixed($query)
    {
        return $query->where('is_fixed', true);
    }

    /**
     * Scope untuk filter berdasarkan product category
     */
    public function scopeByCategory($query, $category)
    {
        return $query->where('product_category', $category);
    }

    /**
     * Relationship: Has one InvoiceHardware
     */
    public function invoiceHardware()
    {
        return $this->hasOne(InvoiceHardware::class, 'invoice_bom_id');
    }

    /**
     * Relationship: Has one InvoiceFitting
     */
    public function invoiceFitting()
    {
        return $this->hasOne(InvoiceFitting::class, 'invoice_bom_id');
    }


    /**
     * Relationship: Has one InvoiceHandle
     */
    public function invoiceHandle()
    {
        return $this->hasOne(InvoiceHandle::class, 'invoice_bom_id');
    }

    /**
     * Relationship: Has one InvoiceStone
     */
    public function invoiceStone()
    {
        return $this->hasOne(InvoiceStone::class, 'invoice_bom_id');
    }

    /**
     * Relationship: Has one InvoiceMetal
     */
    public function invoiceMetal()
    {
        return $this->hasOne(InvoiceMetal::class, 'invoice_bom_id');
    }

    /**
     * Relationship: Has one InvoiceUkir
     */
    public function invoiceUkir()
    {
        return $this->hasOne(InvoiceUkir::class, 'invoice_bom_id');
    }

    /**
     * Relationship: Has one InvoiceBox
     */
    public function invoiceBox()
    {
        return $this->hasOne(InvoiceBox::class, 'invoice_bom_id');
    }

    /**
     * Check if this invoice has box
     */
    public function hasBox()
    {
        return $this->invoiceBox()->exists();
    }

    /**
     * Parse size to get P, L, T
     * Format: 620mm x 486mm x 669mm
     */
    public function parseSizeDimensions()
    {
        if (!$this->size) {
            return ['p' => 0, 'l' => 0, 't' => 0];
        }

        // Remove 'mm' and split by 'x'
        $dimensions = explode('x', str_replace(['mm', ' '], '', $this->size));
        
        return [
            'p' => (int) ($dimensions[0] ?? 0), // Panjang
            'l' => (int) ($dimensions[1] ?? 0), // Lebar
            't' => (int) ($dimensions[2] ?? 0), // Tinggi
        ];
    }

    /**
     * Check if this invoice has handle
     */
    public function hasHandle()
    {
        return $this->invoiceHandle()->exists();
    }

    /**
     * Check if this invoice has stone
     */
    public function hasStone()
    {
        return $this->invoiceStone()->exists();
    }

    /**
     * Check if this invoice has metal
     */
    public function hasMetal()
    {
        return $this->invoiceMetal()->exists();
    }

    /**
     * Check if this invoice has ukir
     */
    public function hasUkir()
    {
        return $this->invoiceUkir()->exists();
    }
    /**
     * Check if this invoice has hardware
     */
    public function hasHardware()
    {
        return $this->invoiceHardware()->exists();
    }

    /**
     * Check if this invoice has fitting
     */
    public function hasFitting()
    {
        return $this->invoiceFitting()->exists();
    }

    
    /**
 * ✅ Get total Material Painting
 */
public function getTotalMaterialPainting()
{
    if (!$this->painting_data) {
        return 0;
    }

    $total = 0;
    foreach ($this->painting_data as $component => $data) {
        $total += $data['material_painting'] ?? 0;
    }
    
    return $total;
}

    /**
 * ✅ Get total Biaya Produksi Painting
 */
public function getTotalBiayaProduksiPainting()
{
    if (!$this->painting_data) {
        return 0;
    }

    $total = 0;
    foreach ($this->painting_data as $component => $data) {
        $total += $data['biaya_produksi_painting'] ?? 0;
    }
    
    return $total;
}
}