@extends('layouts.app')

@section('title', 'Edit Invoice BOM - Cost Furniture')

@section('page-title', 'Edit Invoice BOM')

@section('content')

<!-- Step Indicator -->
<div class="step-indicator mb-4">
    <div class="step" id="step-indicator-1">
        <div class="step-number">1</div>
        <div class="step-text">Initial Form</div>
    </div>
    <div class="step-line"></div>
    <div class="step active" id="step-indicator-2">
        <div class="step-number">2</div>
        <div class="step-text">Calculator</div>
    </div>
</div>

<!-- STEP 1: Initial Form (READONLY - Cannot Edit) -->
<div class="card modern-card" id="step-1-card" style="display: none;">
    <div class="card-header">
        <h5><i class="fas fa-file-alt me-2"></i>Step 1: Informasi Product & Material Locking (READ ONLY)</h5>
    </div>
    <div class="card-body">
        <div class="alert alert-warning">
            <i class="fas fa-lock me-2"></i>
            <strong>Informasi ini tidak bisa diubah saat edit.</strong> Data ini sudah terkunci dari saat pembuatan invoice.
        </div>

        <div class="row">
            <!-- Left Column -->
            <div class="col-md-6">
                <h6 class="section-title">INFORMASI PRODUCT</h6>
                
                <div class="mb-3">
                    <label class="form-label">Product Name (Buyer)</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->product_name }}" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">Product</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->product_category }}" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">Product Type</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->product_type }}" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">Product Code</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->product_code }}" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">Size</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->size }}" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">Date</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->invoice_date->format('Y-m-d') }}" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">Order (Pcs)</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->order_qty }}" readonly>
                </div>

                <!-- ✅ IMAGE DISPLAY (If exists) -->
                @if($invoice->image_path)
                <div class="mb-3">
                    <label class="form-label">Gambar Product</label>
                    <div class="border rounded p-3">
                        <img src="{{ asset($invoice->image_path) }}" alt="Product Image" 
                             style="max-width: 100%; max-height: 300px; border-radius: 8px;">
                    </div>
                </div>
                @else
                <div class="mb-3">
                    <label class="form-label">Gambar Product</label>
                    <div class="alert alert-secondary mb-0">
                        <i class="fas fa-image me-2"></i>Tidak ada gambar
                    </div>
                </div>
                @endif
            </div>

            <!-- Right Column -->
            <div class="col-md-6">
                <h6 class="section-title">MATERIAL LOCKING</h6>
                <p class="text-muted small">Material yang dipilih sudah terkunci dan tidak bisa diubah</p>

                <div class="mb-3">
                    <label class="form-label">Panel</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->panel_type }}" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">Solid</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->solid_type }}" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">Veneer Face</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->veneer_face_type }}" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">Veneer Back</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->veneer_back_type }}" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">Box Drawer</label>
                    <input type="text" class="form-control bg-light" value="{{ $invoice->box_drawer_type }}" readonly>
                </div>

                <div class="alert alert-info">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Info:</strong> Material ini sudah terkunci saat invoice dibuat. Anda hanya bisa mengubah perhitungan di Step 2.
                </div>
            </div>
        </div>

        <hr>

        <div class="d-flex gap-2 justify-content-end">
            <button type="button" class="btn btn-primary" onclick="goToCalculator()">
                Lanjut ke Kalkulator <i class="fas fa-arrow-right ms-2"></i>
            </button>
        </div>
    </div>
</div>

<!-- STEP 2: Calculator -->
<div class="card modern-card" id="step-2-card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Step 2: Calculator Components</h5>
        <div>
            <button type="button" class="btn btn-sm btn-secondary" onclick="backToStep1()">
                <i class="fas fa-arrow-left me-2"></i>Lihat Info Product
            </button>
        </div>
    </div>
    <div class="card-body">
        <!-- Summary Info Bar -->
        <div class="summary-info-bar mb-4">
            <div class="summary-item">
                <div class="summary-label">Product</div>
                <div class="summary-value" id="summary-product">{{ $invoice->product_name }}</div>
            </div>
            <div class="summary-item">
                <div class="summary-label">Type</div>
                <div class="summary-value" id="summary-type">{{ $invoice->product_type }}</div>
            </div>
            <div class="summary-item">
                <div class="summary-label">Code</div>
                <div class="summary-value" id="summary-code">{{ $invoice->product_code }}</div>
            </div>
            <div class="summary-item">
                <div class="summary-label">Grand Total</div>
                <div class="summary-value grand-total" id="summary-grand-total">Rp {{ number_format($invoice->grand_total, 0, ',', '.') }}</div>
            </div>
        </div>

        <!-- ✅ Components Grid Layout (Card-based) -->
        <div class="components-grid">
            @php
                $components = [
                    ['index' => 0, 'name' => 'Top Panel', 'icon' => 'fa-layer-group', 'color' => '#3498db'],
                    ['index' => 1, 'name' => 'Side Panel L', 'icon' => 'fa-layer-group', 'color' => '#9b59b6'],
                    ['index' => 2, 'name' => 'Side Panel R', 'icon' => 'fa-layer-group', 'color' => '#9b59b6'],
                    ['index' => 3, 'name' => 'Bottom Panel', 'icon' => 'fa-layer-group', 'color' => '#e74c3c'],
                    ['index' => 4, 'name' => 'Back Panel', 'icon' => 'fa-layer-group', 'color' => '#f39c12'],
                    ['index' => 5, 'name' => 'Front Frame', 'icon' => 'fa-border-all', 'color' => '#1abc9c'],
                    ['index' => 6, 'name' => 'Back Frame', 'icon' => 'fa-border-all', 'color' => '#16a085'],
                    ['index' => 7, 'name' => 'Face Drawer', 'icon' => 'fa-box-open', 'color' => '#2ecc71'],
                    ['index' => 8, 'name' => 'Front Drawer', 'icon' => 'fa-box', 'color' => '#27ae60'],
                    ['index' => 9, 'name' => 'Side Drawer L', 'icon' => 'fa-box', 'color' => '#8e44ad'],
                    ['index' => 10, 'name' => 'Side Drawer R', 'icon' => 'fa-box', 'color' => '#8e44ad'],
                    ['index' => 11, 'name' => 'Back Drawer', 'icon' => 'fa-box', 'color' => '#c0392b'],
                    ['index' => 12, 'name' => 'Base Drawer', 'icon' => 'fa-box', 'color' => '#d35400'],
                    ['index' => 13, 'name' => 'Support Drawer', 'icon' => 'fa-tools', 'color' => '#7f8c8d'],
                    ['index' => 14, 'name' => 'Cleat Drawer', 'icon' => 'fa-tools', 'color' => '#95a5a6'],
                ];
            @endphp

            @foreach($components as $component)
            <!-- ✅ Component Card (Clickable) -->
            <div class="component-grid-card" onclick="openComponentModal({{ $component['index'] }})" style="border-left-color: {{ $component['color'] }}">
                <div class="component-card-icon" style="background: {{ $component['color'] }}20; color: {{ $component['color'] }}">
                    <i class="fas {{ $component['icon'] }} fa-2x"></i>
                </div>
                <div class="component-card-content">
                    <h6 class="component-card-title">{{ $component['name'] }}</h6>
                    <span class="component-card-status" id="card-status-{{ $component['index'] }}">
                        <i class="fas fa-circle text-secondary"></i> Belum diisi
                    </span>
                    <div class="component-card-total" id="card-total-{{ $component['index'] }}">Rp 0</div>
                </div>
                <div class="component-card-arrow">
                    <i class="fas fa-chevron-right"></i>
                </div>
            </div>
            @endforeach
        </div>

        <hr class="my-4">

        <!-- Action Buttons -->
        <div class="d-flex gap-2 justify-content-end">
            <a href="{{ route('invoice.bom.show', $invoice->id) }}" class="btn btn-secondary">
                <i class="fas fa-times me-2"></i>Batal
            </a>
            <button type="button" class="btn btn-success" onclick="updateInvoice()">
                <i class="fas fa-check me-2"></i>Update Invoice
            </button>
        </div>
    </div>
</div>

<!-- ✅ MODAL untuk Input Component (Reusable) -->
<div class="modal fade" id="componentModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
            <div class="modal-header" id="modalHeader">
                <h5 class="modal-title" id="modalTitle">
                    <i class="fas fa-layer-group me-2"></i>Component Name
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="modalBody">
                <!-- Content will be loaded dynamically -->
                <div class="text-center p-5">
                    <i class="fas fa-spinner fa-spin fa-3x text-primary mb-3"></i>
                    <p>Loading component...</p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-2"></i>Tutup
                </button>
                <button type="button" class="btn btn-warning" id="saveDraftModalBtn">
                    <i class="fas fa-save me-2"></i>Simpan Draft
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Loading Overlay -->
<div class="loading-overlay" id="loadingOverlay" style="display: none;">
    <div class="spinner-border text-light" role="status">
        <span class="visually-hidden">Loading...</span>
    </div>
    <p class="text-light mt-3">Updating invoice...</p>
</div>

@endsection
@push('styles')
<style>
/* ============================================
   MODERN CARD STYLES
   ============================================ */
.modern-card {
    border: none;
    border-radius: 15px;
    box-shadow: 0 5px 20px rgba(0,0,0,0.05);
    overflow: hidden;
}

.modern-card .card-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 15px 15px 0 0;
    padding: 20px 25px;
    border: none;
}

.modern-card .card-header h5 {
    margin: 0;
    font-weight: 700;
}

.modern-card .card-body {
    padding: 30px;
}

/* Section Title */
.section-title {
    font-weight: 700;
    color: #667eea;
    margin-bottom: 20px;
    padding-bottom: 10px;
    border-bottom: 2px solid #f0f0f0;
    font-size: 16px;
}

/* Form Controls */
.form-label {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 8px;
    font-size: 14px;
}

.form-control, .form-select {
    padding: 12px 15px;
    border: 2px solid #e0e0e0;
    border-radius: 10px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.form-control:focus, .form-select:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.1);
    outline: none;
}

/* Step Indicator */
.step-indicator {
    display: flex;
    align-items: center;
    justify-content: center;
    background: white;
    padding: 25px;
    border-radius: 15px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
}

.step {
    display: flex;
    flex-direction: column;
    align-items: center;
    position: relative;
}

.step-number {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: #e0e0e0;
    color: #999;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 20px;
    margin-bottom: 10px;
    transition: all 0.3s ease;
}

.step.active .step-number {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.step-text {
    font-weight: 600;
    color: #999;
    transition: all 0.3s ease;
    font-size: 14px;
}

.step.active .step-text {
    color: #667eea;
}

.step-line {
    width: 100px;
    height: 3px;
    background: #e0e0e0;
    margin: 0 20px 30px 20px;
}

/* Summary Info Bar */
.summary-info-bar {
    display: flex;
    gap: 20px;
    padding: 20px;
    background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%);
    border-radius: 12px;
    border: 2px solid rgba(102, 126, 234, 0.2);
}

.summary-item {
    flex: 1;
}

.summary-label {
    font-size: 12px;
    color: #666;
    margin-bottom: 5px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.summary-value {
    font-size: 16px;
    font-weight: bold;
    color: #2c3e50;
}

.summary-value.grand-total {
    font-size: 24px;
    color: #667eea;
}

/* ============================================
   COMPONENT GRID CARDS
   ============================================ */
.components-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.component-grid-card {
    background: white;
    border: 2px solid #e9ecef;
    border-left: 5px solid #667eea;
    border-radius: 12px;
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
}

.component-grid-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    border-color: #667eea;
}

.component-card-icon {
    width: 60px;
    height: 60px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
}

.component-card-content {
    flex: 1;
}

.component-card-title {
    font-size: 15px;
    font-weight: 700;
    color: #2c3e50;
    margin: 0 0 5px 0;
}

.component-card-status {
    font-size: 12px;
    color: #666;
    display: block;
    margin-bottom: 8px;
}

.component-card-status i {
    font-size: 8px;
    margin-right: 5px;
}

.component-card-total {
    font-size: 16px;
    font-weight: bold;
    color: #27ae60;
}

.component-card-arrow {
    color: #ccc;
    font-size: 20px;
    transition: all 0.3s ease;
}

.component-grid-card:hover .component-card-arrow {
    color: #667eea;
    transform: translateX(5px);
}

/* ============================================
   MODAL STYLES
   ============================================ */
.modal-xl {
    max-width: 1200px;
}

.modal-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border: none;
    padding: 20px 25px;
}

.modal-header .modal-title {
    font-weight: 700;
    color: white;
}

.modal-header .btn-close {
    filter: brightness(0) invert(1);
}

.modal-body {
    padding: 30px;
    max-height: 70vh;
    overflow-y: auto;
}

.modal-footer {
    border-top: 2px solid #f0f0f0;
    padding: 20px 25px;
}

/* Sub Component Card (Inside Modal) */
.sub-component-card {
    background: white;
    border: 2px solid #e9ecef;
    border-left: 4px solid #667eea;
    border-radius: 8px;
    padding: 15px;
    margin-bottom: 15px;
    transition: all 0.3s ease;
}

.sub-component-card:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.05);
}

.sub-component-title {
    font-size: 14px;
    font-weight: 700;
    color: #495057;
    margin-bottom: 15px;
}

.calculated-info {
    padding: 10px 15px;
    background: #f8f9fa;
    border-radius: 6px;
    font-size: 13px;
    border-left: 3px solid #667eea;
}

.total-component {
    background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%);
    padding: 15px 20px;
    border-radius: 8px;
    border: 2px solid rgba(102, 126, 234, 0.2);
}

/* Alert Styles */
.alert-sm {
    padding: 8px 12px;
    font-size: 13px;
    margin-bottom: 15px;
}

/* Loading Overlay */
.loading-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.7);
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    z-index: 9999;
}

/* Buttons */
.btn {
    padding: 12px 24px;
    border-radius: 8px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border: none;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #764ba2 0%, #667eea 100%);
}

.btn-sm {
    padding: 6px 12px;
    font-size: 13px;
}

/* ============================================
   RESPONSIVE
   ============================================ */
@media (max-width: 768px) {
    .components-grid {
        grid-template-columns: 1fr;
    }
    
    .summary-info-bar {
        flex-direction: column;
    }
    
    .modern-card .card-body {
        padding: 20px;
    }
}

/* ============================================
   SCROLLBAR CUSTOMIZATION
   ============================================ */
.modal-body::-webkit-scrollbar {
    width: 8px;
}

.modal-body::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 10px;
}

.modal-body::-webkit-scrollbar-thumb {
    background: #667eea;
    border-radius: 10px;
}

.modal-body::-webkit-scrollbar-thumb:hover {
    background: #764ba2;
}
</style>
@endpush
@push('scripts')
<script>
// ============================================
// GLOBAL VARIABLES
// ============================================

const productData = {
    panels: @json($panels),
    solids: @json($solids),
    veneers: @json($veneers),
    drawers: @json($drawers),
    materials: @json($materials ?? [])
};

console.log('Product Data:', productData);

// ✅ INITIAL FORM DATA (FROM DATABASE - LOCKED)
let initialFormData = {
    product_name: '{{ $invoice->product_name }}',
    product_category: '{{ $invoice->product_category }}',
    product_type: '{{ $invoice->product_type }}',
    product_code: '{{ $invoice->product_code }}',
    size: '{{ $invoice->size }}',
    invoice_date: '{{ $invoice->invoice_date->format('Y-m-d') }}',
    order_qty: {{ $invoice->order_qty }},
    panel_type: '{{ $invoice->panel_type }}',
    solid_type: '{{ $invoice->solid_type }}',
    veneer_face_type: '{{ $invoice->veneer_face_type }}',
    veneer_back_type: '{{ $invoice->veneer_back_type }}',
    box_drawer_type: '{{ $invoice->box_drawer_type }}'
};

// ✅ EXISTING CALCULATION DATA (FROM DATABASE)
let existingCalculationData = @json($calculationData ?? []);

let calculationData = {};
let grandTotal = {{ $invoice->grand_total }};
let componentDrafts = {};
let faceDrawerQty = 1;
let sidePanelLData = null;
let sideDrawerLData = null;
let currentStep = 2; // Always start at Step 2 for edit
let currentComponentIndex = null;

console.log('✅ Edit Mode - Initial Data Loaded:', {
    initialFormData,
    existingCalculationData,
    grandTotal
});

// ============================================
// HELPER FUNCTIONS (SAMA PERSIS DENGAN CREATE)
// ============================================

function calculateM2(width, length, qty = 1) {
    return ((width * length) / 1000000) * qty;
}

function formatRupiah(amount) {
    return 'Rp ' + Math.round(amount).toLocaleString('id-ID');
}

function getPanelsByType(type) {
    return productData.panels.filter(p => p.panel_type === type);
}

function getSolidsByType(type) {
    return productData.solids.filter(s => s.solid_type === type);
}

function getVeneersByType(type) {
    return productData.veneers.filter(v => v.veneer_type === type);
}

function getAllVeneers() {
    return productData.veneers;
}

function getFrontBackDrawers(material) {
    return productData.drawers.filter(d => 
        d.material_type === material && 
        d.drawer_type === 'front drawer'
    );
}

function getSideDrawers(material) {
    return productData.drawers.filter(d => 
        d.material_type === material && 
        d.drawer_type === 'side drawer'
    );
}

function getBaseDrawers(material) {
    return productData.drawers.filter(d => 
        d.material_type === material && 
        d.drawer_type === 'base drawer'
    );
}

function getCoreThickness(totalThickness) {
    const thicknessMap = {
        18: 15, 21: 18, 28: 25,
        16: 15, 19: 18, 26: 25,
        6: 5
    };
    return thicknessMap[totalThickness] || 0;
}

function getComponentName(index) {
    const names = [
        'Top Panel', 'Side Panel L', 'Side Panel R', 'Bottom Panel', 'Back Panel',
        'Front Frame', 'Back Frame', 'Face Drawer', 'Front Drawer',
        'Side Drawer L', 'Side Drawer R', 'Back Drawer', 'Base Drawer',
        'Support Drawer', 'Cleat Drawer'
    ];
    return names[index] || 'Unknown Component';
}

function getCalculationDataKey(index) {
    const keys = [
        'top-panel-0', 'sidel-panel-1', 'sider-panel-2', 'bottom-panel-3', 'back-panel-4',
        'front-frame-5', 'back-frame-6', 'face-drawer-7', 'frontdrawer-8',
        'sidedrawerl-9', 'sidedrawerr-10', 'backdrawer-11', 'basedrawer-12',
        'support-drawer-13', 'cleat-drawer-14'
    ];
    return keys[index];
}

function updateGrandTotal() {
    grandTotal = 0;
    Object.keys(calculationData).forEach(key => {
        if (calculationData[key].componentTotal) {
            grandTotal += calculationData[key].componentTotal;
        }
    });
    
    const grandTotalElement = document.getElementById('summary-grand-total');
    if (grandTotalElement) {
        grandTotalElement.textContent = formatRupiah(grandTotal);
    }
    
    console.log('Grand Total Updated:', grandTotal);
}

function updateComponentCard(index, status, total) {
    const statusElement = document.getElementById(`card-status-${index}`);
    const totalElement = document.getElementById(`card-total-${index}`);
    
    if (statusElement) {
        let icon = '<i class="fas fa-circle text-secondary"></i>';
        let text = 'Belum diisi';
        
        if (status === 'filled') {
            icon = '<i class="fas fa-check-circle text-success"></i>';
            text = 'Sudah diisi';
        } else if (status === 'draft') {
            icon = '<i class="fas fa-save text-warning"></i>';
            text = 'Draft tersimpan';
        } else if (status === 'copied') {
            icon = '<i class="fas fa-copy text-info"></i>';
            text = 'Copy dari komponen lain';
        }
        
        statusElement.innerHTML = icon + ' ' + text;
    }
    
    if (totalElement) {
        totalElement.textContent = formatRupiah(total || 0);
    }
}

// ============================================
// ✅ LOAD EXISTING DATA FROM DATABASE
// ============================================

function loadExistingDataFromDatabase() {
    console.log('=== LOADING EXISTING DATA FROM DATABASE ===');
    console.log('Existing Calculation Data:', existingCalculationData);
    
    if (!existingCalculationData || Object.keys(existingCalculationData).length === 0) {
        console.log('No existing data found');
        return;
    }
    
    // ✅ Load calculation data
    calculationData = existingCalculationData;
    
    // ✅ Extract Face Drawer Qty (if exists)
    if (calculationData['face-drawer-7']) {
        faceDrawerQty = calculationData['face-drawer-7'].details?.qty || 1;
        console.log('Face Drawer Qty loaded:', faceDrawerQty);
    }
    
    // ✅ Extract Side Panel L Data (if exists)
    if (calculationData['sidel-panel-1']) {
        sidePanelLData = calculationData['sidel-panel-1'];
        console.log('Side Panel L Data loaded:', sidePanelLData);
    }
    
    // ✅ Extract Side Drawer L Data (if exists)
    if (calculationData['sidedrawerl-9']) {
        sideDrawerLData = calculationData['sidedrawerl-9'];
        console.log('Side Drawer L Data loaded:', sideDrawerLData);
    }
    
    // ✅ Update component cards
    Object.keys(calculationData).forEach((key) => {
        const componentData = calculationData[key];
        
        // Find component index from key
        const keyToIndex = {
            'top-panel-0': 0,
            'sidel-panel-1': 1,
            'sider-panel-2': 2,
            'bottom-panel-3': 3,
            'back-panel-4': 4,
            'front-frame-5': 5,
            'back-frame-6': 6,
            'face-drawer-7': 7,
            'frontdrawer-8': 8,
            'sidedrawerl-9': 9,
            'sidedrawerr-10': 10,
            'backdrawer-11': 11,
            'basedrawer-12': 12,
            'support-drawer-13': 13,
            'cleat-drawer-14': 14
        };
        
        const index = keyToIndex[key];
        if (index !== undefined && componentData.componentTotal) {
            const status = componentData.copiedFrom ? 'copied' : 'filled';
            updateComponentCard(index, status, componentData.componentTotal);
        }
    });
    
    // ✅ Update grand total display
    updateGrandTotal();
    
    console.log('✅ Existing data loaded successfully!');
    console.log('Calculation Data:', calculationData);
    console.log('Grand Total:', grandTotal);
}

// ============================================
// ✅ AUTO SAVE & RESTORE COMPONENT VALUES
// ============================================

function saveComponentValuesToDraft(index, componentType) {
    const draftKey = `bom_component_values_edit_${index}`;
    const values = {};
    
    const modalBody = document.getElementById('modalBody');
    if (!modalBody) return;
    
    const inputs = modalBody.querySelectorAll('input, select');
    inputs.forEach(input => {
        if (input.id) {
            if (input.type === 'checkbox') {
                values[input.id] = input.checked;
            } else if (input.type === 'radio') {
                if (input.checked) {
                    values[input.id] = input.value;
                }
            } else {
                values[input.id] = input.value;
            }
        }
    });
    
    sessionStorage.setItem(draftKey, JSON.stringify(values));
    console.log(`✅ Component ${index} values auto-saved:`, values);
}

function restoreComponentValuesFromDraft(index) {
    const draftKey = `bom_component_values_edit_${index}`;
    const savedValues = sessionStorage.getItem(draftKey);
    
    if (!savedValues) {
        console.log(`No draft found for component ${index}, loading from database...`);
        restoreComponentValuesFromDatabase(index);
        return;
    }
    
    const values = JSON.parse(savedValues);
    
    setTimeout(() => {
        Object.keys(values).forEach(inputId => {
            const element = document.getElementById(inputId);
            if (element) {
                if (element.type === 'checkbox') {
                    element.checked = values[inputId];
                } else if (element.type === 'radio') {
                    if (element.value === values[inputId]) {
                        element.checked = true;
                    }
                } else {
                    element.value = values[inputId];
                }
            }
        });
        
        console.log(`✅ Component ${index} values restored from draft`);
        
        setTimeout(() => {
            triggerAutoCalculate(index);
        }, 300);
        
    }, 500);
}

// ✅ NEW: Restore values from database (existing calculation data)
function restoreComponentValuesFromDatabase(index) {
    const key = getCalculationDataKey(index);
    const componentData = calculationData[key];
    
    if (!componentData) {
        console.log(`No existing data for component ${index}`);
        return;
    }
    
    console.log(`✅ Restoring component ${index} from database:`, componentData);
    
    setTimeout(() => {
        // Restore based on component type
        const details = componentData.details || {};
        const subComponents = componentData.subComponents || {};
        
        // Common fields
        if (details.width) {
            const widthInput = document.querySelector(`input[id*="-width-${index}"]`);
            if (widthInput) widthInput.value = details.width;
        }
        
        if (details.length) {
            const lengthInput = document.querySelector(`input[id*="-length-${index}"]`);
            if (lengthInput) lengthInput.value = details.length;
        }
        
        if (details.qty) {
            const qtyInput = document.querySelector(`input[id*="-qty-${index}"]`);
            if (qtyInput) qtyInput.value = details.qty;
        }
        
        if (details.thicknessTotal) {
            const thicknessSelect = document.querySelector(`select[id*="-thickness-${index}"]`);
            if (thicknessSelect) thicknessSelect.value = details.thicknessTotal;
        }
        
        // Restore sub-components
        Object.keys(subComponents).forEach(subKey => {
            const subData = subComponents[subKey];
            
            // Restore veneer selections
            if (subKey === 'veneerFace' || subKey === 'veneerBack') {
                const prefix = subKey === 'veneerFace' ? 'face' : 'back';
                const veneerTypeSelect = document.querySelector(`select[id*="-veneer-${prefix}-type-${index}"]`);
                
                if (veneerTypeSelect && subData.veneerType) {
                    // Find veneer option by type
                    const options = veneerTypeSelect.options;
                    for (let i = 0; i < options.length; i++) {
                        if (options[i].dataset.type === subData.veneerType) {
                            veneerTypeSelect.selectedIndex = i;
                            break;
                        }
                    }
                }
                
                // Restore pattern
                if (subData.pattern) {
                    const patternSelect = document.querySelector(`select[id*="-veneer-${prefix}-pattern-${index}"]`);
                    if (patternSelect) patternSelect.value = subData.pattern;
                }
            }
            
            // Restore solid selections
            if (subKey.includes('Solid') || subKey === 'coreSolid') {
                const solidSelect = document.querySelector(`select[id*="-solid"][id*="-thickness-${index}"], select[id*="-patching-solid"][id*="-thickness-${index}"]`);
                if (solidSelect && subData.thickness) {
                    const options = solidSelect.options;
                    for (let i = 0; i < options.length; i++) {
                        if (parseFloat(options[i].dataset.thickness) === parseFloat(subData.thickness)) {
                            solidSelect.selectedIndex = i;
                            break;
                        }
                    }
                }
            }
        });
        
        // Trigger calculation after restore
        setTimeout(() => {
            triggerAutoCalculate(index);
        }, 300);
        
    }, 500);
}

function triggerAutoCalculate(index) {
    console.log(`🔄 Auto-calculating component ${index}...`);
    
    switch(index) {
        case 0:
            calculateTopPanel(index);
            break;
        case 1:
            calculateSidePanel(index, 'sidel');
            break;
        case 2:
            const siderMode = document.getElementById(`sider-mode-${index}`)?.value;
            if (siderMode === 'manual') {
                calculateSidePanel(index, 'sider');
            }
            break;
        case 3:
            calculateBottomPanel(index);
            break;
        case 4:
            calculateBackPanel(index);
            break;
        case 5:
            calculateFrame(index, 'front');
            break;
        case 6:
            calculateFrame(index, 'back');
            break;
        case 7:
            calculateFaceDrawer(index);
            break;
        case 8:
            calculateDrawer(index, 'frontdrawer');
            break;
        case 9:
            calculateDrawer(index, 'sidedrawerl');
            break;
        case 10:
            const sidedrawerrMode = document.getElementById(`sidedrawerr-mode-${index}`)?.value;
            if (sidedrawerrMode === 'manual') {
                calculateDrawer(index, 'sidedrawerr');
            }
            break;
        case 11:
            calculateDrawer(index, 'backdrawer');
            break;
        case 12:
            calculateDrawer(index, 'basedrawer');
            break;
        case 13:
            calculateSupportDrawer(index);
            break;
        case 14:
            calculateCleatDrawer(index);
            break;
    }
}

// ============================================
// MODAL SYSTEM
// ============================================

let componentModal = null;

document.addEventListener('DOMContentLoaded', function() {
    const modalElement = document.getElementById('componentModal');
    if (modalElement) {
        componentModal = new bootstrap.Modal(modalElement);
    }
    
    // ✅ Load existing data on page load
    loadExistingDataFromDatabase();
});

function openComponentModal(index) {
    currentComponentIndex = index;
    const componentName = getComponentName(index);
    
    console.log(`Opening modal for: ${componentName} (index: ${index})`);
    
    const modalHeader = document.getElementById('modalHeader');
    const modalTitle = document.getElementById('modalTitle');
    
    const colors = ['#3498db', '#9b59b6', '#9b59b6', '#e74c3c', '#f39c12', 
                   '#1abc9c', '#16a085', '#2ecc71', '#27ae60', '#8e44ad', 
                   '#8e44ad', '#c0392b', '#d35400', '#7f8c8d', '#95a5a6'];
    
    const icons = ['fa-layer-group', 'fa-layer-group', 'fa-layer-group', 'fa-layer-group', 
                  'fa-layer-group', 'fa-border-all', 'fa-border-all', 'fa-box-open', 
                  'fa-box', 'fa-box', 'fa-box', 'fa-box', 'fa-box', 'fa-tools', 'fa-tools'];
    
    modalHeader.style.background = `linear-gradient(135deg, ${colors[index]} 0%, ${colors[index]}dd 100%)`;
    modalTitle.innerHTML = `<i class="fas ${icons[index]} me-2"></i>${componentName}`;
    
    loadComponentContentToModal(index);
    
    const saveDraftBtn = document.getElementById('saveDraftModalBtn');
    saveDraftBtn.onclick = function() {
        saveComponentDraft(index, componentName);
    };
    
    if (componentModal) {
        componentModal.show();
    }
}

function loadComponentContentToModal(index) {
    const modalBody = document.getElementById('modalBody');
    
    modalBody.innerHTML = `
        <div class="text-center p-5">
            <i class="fas fa-spinner fa-spin fa-3x text-primary mb-3"></i>
            <p>Loading component...</p>
        </div>
    `;
    
    setTimeout(() => {
        let content = '';
        
        switch(index) {
            case 0:
                content = getTopPanelContent(index);
                break;
            case 1:
                content = getSidePanelContent(index, 'Side Panel L');
                break;
            case 2:
                content = getSidePanelRContent(index);
                break;
            case 3:
                content = getBottomPanelContent(index);
                break;
            case 4:
                content = getBackPanelContent(index);
                break;
            case 5:
                content = getFrameContent(index, 'Front Frame');
                break;
            case 6:
                content = getFrameContent(index, 'Back Frame');
                break;
            case 7:
                content = getFaceDrawerContent(index);
                break;
            case 8:
                content = getDrawerContent(index, 'Front Drawer');
                break;
            case 9:
                content = getDrawerContent(index, 'Side Drawer L');
                break;
            case 10:
                content = getSideDrawerRContent(index);
                break;
            case 11:
                content = getDrawerContent(index, 'Back Drawer');
                break;
            case 12:
                content = getDrawerContent(index, 'Base Drawer');
                break;
            case 13:
                content = getSupportDrawerContent(index);
                break;
            case 14:
                content = getCleatDrawerContent(index);
                break;
            default:
                content = '<div class="p-4"><p class="text-danger">Component not found</p></div>';
        }
        
        modalBody.innerHTML = content;
        
        // ✅ Restore values (from draft or database)
        restoreComponentValuesFromDraft(index);
        
    }, 300);
}

// ============================================
// SAVE COMPONENT DRAFT
// ============================================

function saveComponentDraft(index, componentName) {
    console.log('=== SAVE COMPONENT DRAFT ===');
    console.log('Index:', index);
    console.log('Component Name:', componentName);
    
    saveComponentValuesToDraft(index, componentName);
    
    const key = getCalculationDataKey(index);
    
    console.log('Key:', key);
    console.log('Calculation Data:', calculationData[key]);
    
    if (!calculationData[key]) {
        alert('❌ Tidak ada data untuk disimpan. Pastikan komponen sudah dihitung terlebih dahulu.');
        return;
    }
    
    componentDrafts[index] = calculationData[key];
    
    if (componentName === 'Side Panel L') {
        sidePanelLData = calculationData[key];
        console.log('Side Panel L Data Saved:', sidePanelLData);
    }
    
    if (componentName === 'Side Drawer L') {
        sideDrawerLData = calculationData[key];
        console.log('Side Drawer L Data Saved:', sideDrawerLData);
    }
    
    updateComponentCard(index, 'draft', calculationData[key].componentTotal);
    
    alert(`✅ Draft ${componentName} berhasil disimpan!`);
    console.log('Component Draft Saved:', componentName, componentDrafts[index]);
}

// ============================================
// STEP NAVIGATION
// ============================================

function goToCalculator() {
    document.getElementById('step-indicator-1').classList.remove('active');
    document.getElementById('step-indicator-2').classList.add('active');
    
    document.getElementById('step-1-card').style.display = 'none';
    document.getElementById('step-2-card').style.display = 'block';
    
    currentStep = 2;
    
    window.scrollTo({ top: 0, behavior: 'smooth' });
}

function backToStep1() {
    document.getElementById('step-indicator-2').classList.remove('active');
    document.getElementById('step-indicator-1').classList.add('active');
    
    document.getElementById('step-2-card').style.display = 'none';
    document.getElementById('step-1-card').style.display = 'block';
    
    currentStep = 1;
    
    window.scrollTo({ top: 0, behavior: 'smooth' });
}

console.log('✅ Invoice BOM Edit Script Loaded Successfully');
// ============================================
// ✅ CALCULATE TOP PANEL - COMPLETE & AUTO SAVE
// ============================================

function calculateTopPanel(index) {
    console.log('=== CALCULATE TOP PANEL ===');
    
    const width = parseFloat(document.getElementById(`top-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`top-length-${index}`).value) || 0;
    const thicknessTotal = parseFloat(document.getElementById(`top-thickness-${index}`).value) || 0;
    const qty = parseFloat(document.getElementById(`top-qty-${index}`).value) || 1;
    
    if (!width || !length || !thicknessTotal) {
        console.log('Width, Length, or Thickness not provided');
        return;
    }
    
    if (![18, 21, 28].includes(thicknessTotal)) {
        return;
    }
    
    const m2 = calculateM2(width, length, qty);
    document.getElementById(`top-m2-${index}`).value = m2.toFixed(4);
    
    const coreThickness = getCoreThickness(thicknessTotal);
    
    document.getElementById(`top-core-thickness-${index}`).value = coreThickness + 'mm';
    document.getElementById(`top-core-type-${index}`).value = initialFormData.panel_type;
    
    let componentTotal = 0;
    const subComponents = {};
    
    // 1. Patching Panel 1 (Core Panel)
    const corePanel = getPanelsByType(initialFormData.panel_type)
        .find(p => parseFloat(p.thickness) === parseFloat(coreThickness));
    
    if (!corePanel) {
        console.log(`Panel ${initialFormData.panel_type} ${coreThickness}mm tidak ditemukan`);
        return;
    }
    
    const patchingPanelQty = parseFloat(document.getElementById(`top-patching-panel-qty-${index}`).value) || 1;
    const patchingM2 = calculateM2(width, length, qty * patchingPanelQty);
    const patchingTotal = patchingM2 * corePanel.price;
    
    document.getElementById(`top-patching-panel-m2-${index}`).value = patchingM2.toFixed(4);
    document.getElementById(`top-patching-panel-price-${index}`).textContent = formatRupiah(corePanel.price);
    document.getElementById(`top-patching-panel-total-${index}`).textContent = formatRupiah(patchingTotal);
    
    componentTotal += patchingTotal;
    
    subComponents.patchingPanel = {
        name: 'Patching Panel 1',
        thickness: coreThickness,
        width: width,
        length: length,
        qty: qty * patchingPanelQty,
        m2: patchingM2,
        price: corePanel.price,
        total: patchingTotal,
        material: initialFormData.panel_type
    };
    
    // 2. Patching Solid 2 (PANJANG)
    const solid2Width = parseFloat(document.getElementById(`top-patching-solid2-width-${index}`).value) || 0;
    const solid2Select = document.getElementById(`top-patching-solid2-thickness-${index}`);
    const solid2Qty = parseFloat(document.getElementById(`top-patching-solid2-qty-${index}`).value) || 2;
    
    if (solid2Select && solid2Select.value && solid2Width) {
        const selectedOption = solid2Select.options[solid2Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid2M2 = calculateM2(solid2Width, length, qty * solid2Qty);
        const total = solid2M2 * price;
        
        document.getElementById(`top-patching-solid2-m2-${index}`).value = solid2M2.toFixed(4);
        document.getElementById(`top-patching-solid2-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`top-patching-solid2-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid2 = {
            name: 'Patching Solid 2 (PANJANG)',
            thickness: thickness,
            width: solid2Width,
            length: length,
            qty: qty * solid2Qty,
            m2: solid2M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 3. Patching Solid 3 (LEBAR)
    const solid3Width = parseFloat(document.getElementById(`top-patching-solid3-width-${index}`).value) || 0;
    const solid3Select = document.getElementById(`top-patching-solid3-thickness-${index}`);
    const solid3Qty = parseFloat(document.getElementById(`top-patching-solid3-qty-${index}`).value) || 2;
    
    if (solid3Select && solid3Select.value && solid3Width) {
        const selectedOption = solid3Select.options[solid3Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid3M2 = calculateM2(solid3Width, width, qty * solid3Qty);
        const total = solid3M2 * price;
        
        document.getElementById(`top-patching-solid3-m2-${index}`).value = solid3M2.toFixed(4);
        document.getElementById(`top-patching-solid3-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`top-patching-solid3-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid3 = {
            name: 'Patching Solid 3 (LEBAR)',
            thickness: thickness,
            width: solid3Width,
            length: width,
            qty: qty * solid3Qty,
            m2: solid3M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 4. Crossgrain (MDF)
    const crossgrainSelect = document.getElementById(`top-crossgrain-thickness-${index}`);
    const crossgrainQty = parseFloat(document.getElementById(`top-crossgrain-qty-${index}`).value) || 2;
    
    if (crossgrainSelect && crossgrainSelect.value) {
        const selectedOption = crossgrainSelect.options[crossgrainSelect.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const crossgrainM2 = calculateM2(width, length, qty * crossgrainQty);
        const total = crossgrainM2 * price;
        
        document.getElementById(`top-crossgrain-m2-${index}`).value = crossgrainM2.toFixed(4);
        document.getElementById(`top-crossgrain-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`top-crossgrain-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.crossgrain = {
            name: 'Crossgrain',
            thickness: thickness,
            width: width,
            length: length,
            qty: qty * crossgrainQty,
            m2: crossgrainM2,
            price: price,
            total: total,
            material: 'MDF'
        };
    }
    
    // 5. Veneer Face
    const veneerFacePattern = document.getElementById(`top-veneer-face-pattern-${index}`).value;
    const veneerFaceSelect = document.getElementById(`top-veneer-face-type-${index}`);
    const veneerFaceQty = parseFloat(document.getElementById(`top-veneer-face-qty-${index}`).value) || 1;
    
    if (veneerFaceSelect && veneerFaceSelect.value) {
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerFaceM2 = calculateM2(width, length, qty * veneerFaceQty);
        const total = veneerFaceM2 * price;
        
        document.getElementById(`top-veneer-face-m2-${index}`).value = veneerFaceM2.toFixed(4);
        document.getElementById(`top-veneer-face-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`top-veneer-face-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerFace = {
            name: 'TV. FACE',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerFaceQty,
            m2: veneerFaceM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerFacePattern,
            veneerType: veneerType
        };
    }
    
    // 6. Veneer Back
    const veneerBackPattern = document.getElementById(`top-veneer-back-pattern-${index}`).value;
    const veneerBackSelect = document.getElementById(`top-veneer-back-type-${index}`);
    const veneerBackQty = parseFloat(document.getElementById(`top-veneer-back-qty-${index}`).value) || 1;
    
    if (veneerBackSelect && veneerBackSelect.value) {
        const selectedOption = veneerBackSelect.options[veneerBackSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerBackPattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerBackM2 = calculateM2(width, length, qty * veneerBackQty);
        const total = veneerBackM2 * price;
        
        document.getElementById(`top-veneer-back-m2-${index}`).value = veneerBackM2.toFixed(4);
        document.getElementById(`top-veneer-back-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`top-veneer-back-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerBack = {
            name: 'TV. BACK',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerBackQty,
            m2: veneerBackM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerBackPattern,
            veneerType: veneerType
        };
    }
    
    // 7. Cleat
    const cleatQty = parseFloat(document.getElementById(`top-cleat-qty-${index}`).value) || 6;
    const cleatPricePerUnit = 1000;
    const cleatTotal = cleatQty * cleatPricePerUnit;
    
    document.getElementById(`top-cleat-total-${index}`).textContent = formatRupiah(cleatTotal);
    
    componentTotal += cleatTotal;
    
    subComponents.cleat = {
        name: 'Cleat',
        thickness: 11,
        width: 11,
        length: 60,
        qty: cleatQty,
        pricePerUnit: cleatPricePerUnit,
        total: cleatTotal,
        material: 'Mix'
    };
    
    document.getElementById(`total-top-panel-${index}`).textContent = formatRupiah(componentTotal);
    
    calculationData[`top-panel-${index}`] = {
        componentName: 'TOP PANEL',
        componentTotal: componentTotal,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            thicknessTotal: thicknessTotal,
            coreThickness: coreThickness
        },
        subComponents: subComponents
    };
    
    console.log('✅ Top Panel Calculation:', calculationData[`top-panel-${index}`]);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateGrandTotal();
    saveComponentValuesToDraft(index, 'Top Panel');
}

// ============================================
// ✅ CALCULATE SIDE PANEL L/R
// ============================================

function calculateSidePanel(index, prefix) {
    console.log(`=== CALCULATE SIDE PANEL (${prefix.toUpperCase()}) ===`);
    
    const width = parseFloat(document.getElementById(`${prefix}-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`${prefix}-length-${index}`).value) || 0;
    const thicknessTotal = parseFloat(document.getElementById(`${prefix}-thickness-${index}`).value) || 0;
    const qty = parseFloat(document.getElementById(`${prefix}-qty-${index}`).value) || 1;
    
    if (!width || !length || !thicknessTotal) {
        console.log('Width, Length, or Thickness not provided');
        return;
    }
    
    if (![18, 21, 28].includes(thicknessTotal)) {
        return;
    }
    
    const m2 = calculateM2(width, length, qty);
    document.getElementById(`${prefix}-m2-${index}`).value = m2.toFixed(4);
    
    const coreThickness = getCoreThickness(thicknessTotal);
    
    document.getElementById(`${prefix}-core-thickness-${index}`).value = coreThickness + 'mm';
    document.getElementById(`${prefix}-core-type-${index}`).value = initialFormData.panel_type;
    
    let componentTotal = 0;
    const subComponents = {};
    
    // 1. Patching Panel 1 (Core Panel)
    const corePanel = getPanelsByType(initialFormData.panel_type)
        .find(p => parseFloat(p.thickness) === parseFloat(coreThickness));
    
    if (!corePanel) {
        console.log(`Panel ${initialFormData.panel_type} ${coreThickness}mm tidak ditemukan`);
        return;
    }
    
    const patchingPanelQty = parseFloat(document.getElementById(`${prefix}-patching-panel-qty-${index}`).value) || 1;
    const patchingM2 = calculateM2(width, length, qty * patchingPanelQty);
    const patchingTotal = patchingM2 * corePanel.price;
    
    document.getElementById(`${prefix}-patching-panel-m2-${index}`).value = patchingM2.toFixed(4);
    document.getElementById(`${prefix}-patching-panel-price-${index}`).textContent = formatRupiah(corePanel.price);
    document.getElementById(`${prefix}-patching-panel-total-${index}`).textContent = formatRupiah(patchingTotal);
    
    componentTotal += patchingTotal;
    
    subComponents.patchingPanel = {
        name: 'Patching Panel 1',
        thickness: coreThickness,
        width: width,
        length: length,
        qty: qty * patchingPanelQty,
        m2: patchingM2,
        price: corePanel.price,
        total: patchingTotal,
        material: initialFormData.panel_type
    };
    
    // 2. Patching Solid 2 (PANJANG)
    const solid2Width = parseFloat(document.getElementById(`${prefix}-patching-solid2-width-${index}`).value) || 0;
    const solid2Select = document.getElementById(`${prefix}-patching-solid2-thickness-${index}`);
    const solid2Qty = parseFloat(document.getElementById(`${prefix}-patching-solid2-qty-${index}`).value) || 2;
    
    if (solid2Select && solid2Select.value && solid2Width) {
        const selectedOption = solid2Select.options[solid2Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid2M2 = calculateM2(solid2Width, length, qty * solid2Qty);
        const total = solid2M2 * price;
        
        document.getElementById(`${prefix}-patching-solid2-m2-${index}`).value = solid2M2.toFixed(4);
        document.getElementById(`${prefix}-patching-solid2-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-patching-solid2-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid2 = {
            name: 'Patching Solid 2 (PANJANG)',
            thickness: thickness,
            width: solid2Width,
            length: length,
            qty: qty * solid2Qty,
            m2: solid2M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 3. Patching Solid 3 (LEBAR)
    const solid3Width = parseFloat(document.getElementById(`${prefix}-patching-solid3-width-${index}`).value) || 0;
    const solid3Select = document.getElementById(`${prefix}-patching-solid3-thickness-${index}`);
    const solid3Qty = parseFloat(document.getElementById(`${prefix}-patching-solid3-qty-${index}`).value) || 2;
    
    if (solid3Select && solid3Select.value && solid3Width) {
        const selectedOption = solid3Select.options[solid3Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid3M2 = calculateM2(solid3Width, width, qty * solid3Qty);
        const total = solid3M2 * price;
        
        document.getElementById(`${prefix}-patching-solid3-m2-${index}`).value = solid3M2.toFixed(4);
        document.getElementById(`${prefix}-patching-solid3-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-patching-solid3-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid3 = {
            name: 'Patching Solid 3 (LEBAR)',
            thickness: thickness,
            width: solid3Width,
            length: width,
            qty: qty * solid3Qty,
            m2: solid3M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 4. Crossgrain (MDF)
    const crossgrainSelect = document.getElementById(`${prefix}-crossgrain-thickness-${index}`);
    const crossgrainQty = parseFloat(document.getElementById(`${prefix}-crossgrain-qty-${index}`).value) || 2;
    
    if (crossgrainSelect && crossgrainSelect.value) {
        const selectedOption = crossgrainSelect.options[crossgrainSelect.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const crossgrainM2 = calculateM2(width, length, qty * crossgrainQty);
        const total = crossgrainM2 * price;
        
        document.getElementById(`${prefix}-crossgrain-m2-${index}`).value = crossgrainM2.toFixed(4);
        document.getElementById(`${prefix}-crossgrain-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-crossgrain-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.crossgrain = {
            name: 'Crossgrain',
            thickness: thickness,
            width: width,
            length: length,
            qty: qty * crossgrainQty,
            m2: crossgrainM2,
            price: price,
            total: total,
            material: 'MDF'
        };
    }
    
    // 5. Veneer Face
    const veneerFacePattern = document.getElementById(`${prefix}-veneer-face-pattern-${index}`).value;
    const veneerFaceSelect = document.getElementById(`${prefix}-veneer-face-type-${index}`);
    const veneerFaceQty = parseFloat(document.getElementById(`${prefix}-veneer-face-qty-${index}`).value) || 1;
    
    if (veneerFaceSelect && veneerFaceSelect.value) {
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerFaceM2 = calculateM2(width, length, qty * veneerFaceQty);
        const total = veneerFaceM2 * price;
        
        document.getElementById(`${prefix}-veneer-face-m2-${index}`).value = veneerFaceM2.toFixed(4);
        document.getElementById(`${prefix}-veneer-face-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-veneer-face-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerFace = {
            name: 'TV. FACE',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerFaceQty,
            m2: veneerFaceM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerFacePattern,
            veneerType: veneerType
        };
    }
    
    // 6. Veneer Back
    const veneerBackPattern = document.getElementById(`${prefix}-veneer-back-pattern-${index}`).value;
    const veneerBackSelect = document.getElementById(`${prefix}-veneer-back-type-${index}`);
    const veneerBackQty = parseFloat(document.getElementById(`${prefix}-veneer-back-qty-${index}`).value) || 1;
    
    if (veneerBackSelect && veneerBackSelect.value) {
        const selectedOption = veneerBackSelect.options[veneerBackSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerBackPattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerBackM2 = calculateM2(width, length, qty * veneerBackQty);
        const total = veneerBackM2 * price;
        
        document.getElementById(`${prefix}-veneer-back-m2-${index}`).value = veneerBackM2.toFixed(4);
        document.getElementById(`${prefix}-veneer-back-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-veneer-back-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerBack = {
            name: 'TV. BACK',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerBackQty,
            m2: veneerBackM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerBackPattern,
            veneerType: veneerType
        };
    }
    
    document.getElementById(`total-${prefix}-panel-${index}`).textContent = formatRupiah(componentTotal);
    
    const componentName = prefix === 'sidel' ? 'SIDE PANEL L' : 'SIDE PANEL R';
    
    calculationData[`${prefix}-panel-${index}`] = {
        componentName: componentName,
        componentTotal: componentTotal,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            thicknessTotal: thicknessTotal,
            coreThickness: coreThickness
        },
        subComponents: subComponents
    };
    
    if (prefix === 'sidel') {
        sidePanelLData = calculationData[`${prefix}-panel-${index}`];
        console.log('✅ Side Panel L Data Auto-Saved:', sidePanelLData);
    }
    
    console.log(`✅ ${componentName} Calculation:`, calculationData[`${prefix}-panel-${index}`]);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateGrandTotal();
    saveComponentValuesToDraft(index, componentName);
}

// ============================================
// SIDE PANEL R - MODE CHANGE
// ============================================

function onSidePanelRModeChange(index) {
    const mode = document.getElementById(`sider-mode-${index}`).value;
    const manualInputs = document.getElementById(`sider-manual-inputs-${index}`);
    
    if (mode === 'copy') {
        manualInputs.style.display = 'none';
        
        if (!sidePanelLData) {
            alert('❌ Side Panel L belum diisi! Silakan isi Side Panel L terlebih dahulu.');
            document.getElementById(`sider-mode-${index}`).value = '';
            return;
        }
        
        copySidePanelLToR(index);
        
    } else if (mode === 'manual') {
        manualInputs.style.display = 'block';
        
        document.getElementById(`sider-width-${index}`).value = '';
        document.getElementById(`sider-length-${index}`).value = '';
        document.getElementById(`sider-thickness-${index}`).value = '';
        document.getElementById(`sider-qty-${index}`).value = '1';
        
        document.getElementById(`total-sider-panel-${index}`).textContent = 'Rp 0';
        
        delete calculationData[`sider-panel-${index}`];
        updateComponentCard(index, 'empty', 0);
        updateGrandTotal();
    } else {
        manualInputs.style.display = 'none';
    }
}

function copySidePanelLToR(index) {
    console.log('=== COPY SIDE PANEL L TO R ===');
    
    if (!sidePanelLData) {
        alert('❌ Data Side Panel L tidak ditemukan!');
        return;
    }
    
    calculationData[`sider-panel-${index}`] = {
        componentName: 'SIDE PANEL R',
        componentTotal: sidePanelLData.componentTotal,
        details: { ...sidePanelLData.details },
        subComponents: { ...sidePanelLData.subComponents },
        copiedFrom: 'Side Panel L'
    };
    
    document.getElementById(`total-sider-panel-${index}`).textContent = formatRupiah(sidePanelLData.componentTotal);
    
    updateComponentCard(index, 'copied', sidePanelLData.componentTotal);
    updateGrandTotal();
    saveComponentValuesToDraft(index, 'Side Panel R');
    
    alert('✅ Side Panel R berhasil di-copy dari Side Panel L!');
    console.log('Side Panel R (copied):', calculationData[`sider-panel-${index}`]);
}
// ============================================
// ✅ CALCULATE BOTTOM PANEL
// ============================================

function calculateBottomPanel(index) {
    console.log('=== CALCULATE BOTTOM PANEL ===');
    
    const width = parseFloat(document.getElementById(`bottom-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`bottom-length-${index}`).value) || 0;
    const thicknessTotal = parseFloat(document.getElementById(`bottom-thickness-${index}`).value) || 0;
    const qty = parseFloat(document.getElementById(`bottom-qty-${index}`).value) || 1;
    
    if (!width || !length || !thicknessTotal) {
        console.log('Width, Length, or Thickness not provided');
        return;
    }
    
    if (![16, 19, 26].includes(thicknessTotal)) {
        return;
    }
    
    const m2 = calculateM2(width, length, qty);
    document.getElementById(`bottom-m2-${index}`).value = m2.toFixed(4);
    
    const coreThickness = getCoreThickness(thicknessTotal);
    
    document.getElementById(`bottom-core-thickness-${index}`).value = coreThickness + 'mm';
    document.getElementById(`bottom-core-type-${index}`).value = initialFormData.panel_type;
    
    let componentTotal = 0;
    const subComponents = {};
    
    // 1. Core Panel
    const corePanel = getPanelsByType(initialFormData.panel_type)
        .find(p => parseFloat(p.thickness) === parseFloat(coreThickness));
    
    if (!corePanel) {
        console.log(`Panel ${initialFormData.panel_type} ${coreThickness}mm tidak ditemukan`);
        return;
    }
    
    const coreQty = parseFloat(document.getElementById(`bottom-core-qty-${index}`).value) || 1;
    const coreM2 = calculateM2(width, length, qty * coreQty);
    const coreTotal = coreM2 * corePanel.price;
    
    document.getElementById(`bottom-core-m2-${index}`).value = coreM2.toFixed(4);
    document.getElementById(`bottom-core-price-${index}`).textContent = formatRupiah(corePanel.price);
    document.getElementById(`bottom-core-total-${index}`).textContent = formatRupiah(coreTotal);
    
    componentTotal += coreTotal;
    
    subComponents.core = {
        name: 'Core Panel',
        thickness: coreThickness,
        width: width,
        length: length,
        qty: qty * coreQty,
        m2: coreM2,
        price: corePanel.price,
        total: coreTotal,
        material: initialFormData.panel_type
    };
    
    // 2. Veneer Face
    const veneerFacePattern = document.getElementById(`bottom-veneer-face-pattern-${index}`).value;
    const veneerFaceSelect = document.getElementById(`bottom-veneer-face-type-${index}`);
    const veneerFaceQty = parseFloat(document.getElementById(`bottom-veneer-face-qty-${index}`).value) || 1;
    
    if (veneerFaceSelect && veneerFaceSelect.value) {
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerFaceM2 = calculateM2(width, length, qty * veneerFaceQty);
        const total = veneerFaceM2 * price;
        
        document.getElementById(`bottom-veneer-face-m2-${index}`).value = veneerFaceM2.toFixed(4);
        document.getElementById(`bottom-veneer-face-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`bottom-veneer-face-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerFace = {
            name: 'Veneer Face',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerFaceQty,
            m2: veneerFaceM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerFacePattern,
            veneerType: veneerType
        };
    }
    
    // 3. Veneer Back
    const veneerBackPattern = document.getElementById(`bottom-veneer-back-pattern-${index}`).value;
    const veneerBackSelect = document.getElementById(`bottom-veneer-back-type-${index}`);
    const veneerBackQty = parseFloat(document.getElementById(`bottom-veneer-back-qty-${index}`).value) || 1;
    
    if (veneerBackSelect && veneerBackSelect.value) {
        const selectedOption = veneerBackSelect.options[veneerBackSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerBackPattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerBackM2 = calculateM2(width, length, qty * veneerBackQty);
        const total = veneerBackM2 * price;
        
        document.getElementById(`bottom-veneer-back-m2-${index}`).value = veneerBackM2.toFixed(4);
        document.getElementById(`bottom-veneer-back-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`bottom-veneer-back-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerBack = {
            name: 'Veneer Back',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerBackQty,
            m2: veneerBackM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerBackPattern,
            veneerType: veneerType
        };
    }
    
    // 4. E.Band
    const ebandQty = parseFloat(document.getElementById(`bottom-eband-qty-${index}`).value) || 1;
    
    if (veneerFaceSelect && veneerFaceSelect.value) {
        const ebandM2 = ((thicknessTotal * length * qty * ebandQty) / 1000000);
        
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const total = ebandM2 * price;
        
        document.getElementById(`bottom-eband-m2-${index}`).value = ebandM2.toFixed(4);
        document.getElementById(`bottom-eband-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`bottom-eband-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.eband = {
            name: 'E.Band',
            thickness: thicknessTotal,
            width: thicknessTotal,
            length: length,
            qty: qty * ebandQty,
            m2: ebandM2,
            price: price,
            total: total,
            material: 'E.Band',
            pattern: veneerFacePattern
        };
    }
    
    document.getElementById(`total-bottom-panel-${index}`).textContent = formatRupiah(componentTotal);
    
    calculationData[`bottom-panel-${index}`] = {
        componentName: 'BOTTOM PANEL',
        componentTotal: componentTotal,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            thicknessTotal: thicknessTotal,
            coreThickness: coreThickness
        },
        subComponents: subComponents
    };
    
    console.log('✅ Bottom Panel Calculation:', calculationData[`bottom-panel-${index}`]);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateGrandTotal();
    saveComponentValuesToDraft(index, 'Bottom Panel');
}

// ============================================
// ✅ CALCULATE BACK PANEL (WITH SAFE ELEMENT CHECK)
// ============================================

function calculateBackPanel(index) {
    console.log('=== CALCULATE BACK PANEL ===');
    
    const width = parseFloat(document.getElementById(`back-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`back-length-${index}`).value) || 0;
    const thicknessTotal = parseFloat(document.getElementById(`back-thickness-${index}`).value) || 0;
    const qty = parseFloat(document.getElementById(`back-qty-${index}`).value) || 1;
    
    console.log('Width:', width, 'Length:', length, 'Thickness Total:', thicknessTotal, 'Qty:', qty);
    
    if (!width || !length || !thicknessTotal) {
        console.log('Width, Length, or Thickness not provided');
        return;
    }
    
    if (![6, 19, 26].includes(thicknessTotal)) {
        console.log('Invalid thickness total:', thicknessTotal);
        return;
    }
    
    const m2 = calculateM2(width, length, qty);
    const m2Element = document.getElementById(`back-m2-${index}`);
    if (m2Element) {
        m2Element.value = m2.toFixed(4);
    }
    
    const coreThickness = getCoreThickness(thicknessTotal);
    
    console.log('Core Thickness:', coreThickness);
    
    // ✅ Update thickness info (SAFE CHECK)
    const thicknessInfoElement = document.getElementById(`back-core-thickness-info-${index}`);
    if (thicknessInfoElement) {
        thicknessInfoElement.textContent = 
            `${coreThickness}mm (Core) + 0.5mm (Veneer Face) + 0.5mm (Veneer Back) = ${thicknessTotal}mm`;
    }
    
    let componentTotal = 0;
    const subComponents = {};
    
    // 1. Core Panel (MDF/PLYWOOD selectable)
    const coreTypeSelect = document.getElementById(`back-core-type-${index}`);
    const coreType = coreTypeSelect ? coreTypeSelect.value : '';
    
    console.log('Core Type Selected:', coreType);
    
    if (!coreType) {
        console.log('Core type not selected yet');
        // Reset displays (SAFE)
        const coreM2El = document.getElementById(`back-core-m2-${index}`);
        const corePriceEl = document.getElementById(`back-core-price-${index}`);
        const coreTotalEl = document.getElementById(`back-core-total-${index}`);
        const totalEl = document.getElementById(`total-back-panel-${index}`);
        
        if (coreM2El) coreM2El.value = '0';
        if (corePriceEl) corePriceEl.textContent = 'Rp 0';
        if (coreTotalEl) coreTotalEl.textContent = 'Rp 0';
        if (totalEl) totalEl.textContent = 'Rp 0';
        
        return;
    }
    
    const corePanel = getPanelsByType(coreType)
        .find(p => parseFloat(p.thickness) === parseFloat(coreThickness));
    
    console.log('Core Panel Found:', corePanel);
    
    if (!corePanel) {
        console.log(`Panel ${coreType} ${coreThickness}mm tidak ditemukan`);
        alert(`❌ Panel ${coreType} dengan tebal ${coreThickness}mm tidak ditemukan di database!`);
        return;
    }
    
    const coreQty = parseFloat(document.getElementById(`back-core-qty-${index}`).value) || 1;
    const coreM2 = calculateM2(width, length, qty * coreQty);
    const coreTotal = coreM2 * parseFloat(corePanel.price);
    
    console.log('Core M2:', coreM2, 'Core Price:', corePanel.price, 'Core Total:', coreTotal);
    
    // Update UI (SAFE)
    const coreM2El = document.getElementById(`back-core-m2-${index}`);
    const corePriceEl = document.getElementById(`back-core-price-${index}`);
    const coreTotalEl = document.getElementById(`back-core-total-${index}`);
    
    if (coreM2El) coreM2El.value = coreM2.toFixed(4);
    if (corePriceEl) corePriceEl.textContent = formatRupiah(corePanel.price);
    if (coreTotalEl) coreTotalEl.textContent = formatRupiah(coreTotal);
    
    componentTotal += coreTotal;
    
    subComponents.core = {
        name: 'Core Panel',
        thickness: coreThickness,
        width: width,
        length: length,
        qty: qty * coreQty,
        m2: coreM2,
        price: parseFloat(corePanel.price),
        total: coreTotal,
        material: coreType
    };
    
    // 2. Veneer Face (ALWAYS PRESENT)
    const veneerFacePattern = document.getElementById(`back-veneer-face-pattern-${index}`).value;
    const veneerFaceSelect = document.getElementById(`back-veneer-face-type-${index}`);
    const veneerFaceQty = parseFloat(document.getElementById(`back-veneer-face-qty-${index}`).value) || 1;
    
    if (veneerFaceSelect && veneerFaceSelect.value) {
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerFaceM2 = calculateM2(width, length, qty * veneerFaceQty);
        const total = veneerFaceM2 * price;
        
        const vfM2El = document.getElementById(`back-veneer-face-m2-${index}`);
        const vfPriceEl = document.getElementById(`back-veneer-face-price-${index}`);
        const vfTotalEl = document.getElementById(`back-veneer-face-total-${index}`);
        
        if (vfM2El) vfM2El.value = veneerFaceM2.toFixed(4);
        if (vfPriceEl) vfPriceEl.textContent = formatRupiah(price);
        if (vfTotalEl) vfTotalEl.textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerFace = {
            name: 'Veneer Face',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerFaceQty,
            m2: veneerFaceM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerFacePattern,
            veneerType: veneerType
        };
    }
    
    // 3. Veneer Back (ALWAYS PRESENT)
    const veneerBackPattern = document.getElementById(`back-veneer-back-pattern-${index}`).value;
    const veneerBackSelect = document.getElementById(`back-veneer-back-type-${index}`);
    const veneerBackQty = parseFloat(document.getElementById(`back-veneer-back-qty-${index}`).value) || 1;
    
    if (veneerBackSelect && veneerBackSelect.value) {
        const selectedOption = veneerBackSelect.options[veneerBackSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerBackPattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerBackM2 = calculateM2(width, length, qty * veneerBackQty);
        const total = veneerBackM2 * price;
        
        const vbM2El = document.getElementById(`back-veneer-back-m2-${index}`);
        const vbPriceEl = document.getElementById(`back-veneer-back-price-${index}`);
        const vbTotalEl = document.getElementById(`back-veneer-back-total-${index}`);
        
        if (vbM2El) vbM2El.value = veneerBackM2.toFixed(4);
        if (vbPriceEl) vbPriceEl.textContent = formatRupiah(price);
        if (vbTotalEl) vbTotalEl.textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerBack = {
            name: 'Veneer Back',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerBackQty,
            m2: veneerBackM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerBackPattern,
            veneerType: veneerType
        };
    }
    
    console.log('Component Total:', componentTotal);
    
    const totalEl = document.getElementById(`total-back-panel-${index}`);
    if (totalEl) {
        totalEl.textContent = formatRupiah(componentTotal);
    }
    
    calculationData[`back-panel-${index}`] = {
        componentName: 'BACK PANEL',
        componentTotal: componentTotal,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            thicknessTotal: thicknessTotal,
            coreThickness: coreThickness
        },
        subComponents: subComponents
    };
    
    console.log('✅ Back Panel Calculation:', calculationData[`back-panel-${index}`]);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateGrandTotal();
    saveComponentValuesToDraft(index, 'Back Panel');
}

// ============================================
// ✅ CALCULATE FRAME (FRONT & BACK)
// ============================================

function onFrameSolidTypeChange(index, prefix) {
    const solidTypeSelect = document.getElementById(`${prefix}-solid-type-${index}`);
    const solidSelect = document.getElementById(`${prefix}-solid-thickness-${index}`);
    
    const selectedType = solidTypeSelect.value;
    
    if (!selectedType) {
        solidSelect.innerHTML = '<option value="">Pilih Tebal</option>';
        return;
    }
    
    const solids = getSolidsByType(selectedType);
    
    let options = '<option value="">Pilih Tebal</option>';
    solids.forEach(s => {
        options += `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}</option>`;
    });
    
    solidSelect.innerHTML = options;
    
    console.log(`${prefix.toUpperCase()} Frame Solid Type changed to: ${selectedType}`);
}

function calculateFrame(index, prefix) {
    console.log(`=== CALCULATE FRAME (${prefix.toUpperCase()}) ===`);
    
    const width = parseFloat(document.getElementById(`${prefix}-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`${prefix}-length-${index}`).value) || 0;
    
    if (!width || !length) {
        console.log('Width or Length not provided');
        return;
    }
    
    let componentTotal = 0;
    const subComponents = {};
    
    const solidTypeSelect = document.getElementById(`${prefix}-solid-type-${index}`);
    const solidSelect = document.getElementById(`${prefix}-solid-thickness-${index}`);
    const solidQty = parseFloat(document.getElementById(`${prefix}-solid-qty-${index}`).value) || 1;
    
    if (solidSelect && solidSelect.value && solidTypeSelect && solidTypeSelect.value) {
        const selectedOption = solidSelect.options[solidSelect.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solidM2 = calculateM2(width, length, solidQty);
        const total = solidM2 * price;
        
        document.getElementById(`${prefix}-solid-m2-${index}`).value = solidM2.toFixed(4);
        document.getElementById(`${prefix}-solid-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-solid-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.coreSolid = {
            name: 'Core Solid',
            thickness: thickness,
            width: width,
            length: length,
            qty: solidQty,
            m2: solidM2,
            price: price,
            total: total,
            material: `SL.${solidTypeSelect.value}`
        };
    }
    
    document.getElementById(`total-${prefix}-frame-${index}`).textContent = formatRupiah(componentTotal);
    
    const componentName = prefix === 'front' ? 'FRONT FRAME' : 'BACK FRAME';
    
    calculationData[`${prefix}-frame-${index}`] = {
        componentName: componentName,
        componentTotal: componentTotal,
        details: {
            width: width,
            length: length
        },
        subComponents: subComponents
    };
    
    console.log(`${componentName} Calculation:`, calculationData[`${prefix}-frame-${index}`]);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateGrandTotal();
    saveComponentValuesToDraft(index, componentName);
}

// ============================================
// ✅ CALCULATE FACE DRAWER
// ============================================

function calculateFaceDrawer(index) {
    console.log('=== CALCULATE FACE DRAWER ===');
    
    const width = parseFloat(document.getElementById(`face-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`face-length-${index}`).value) || 0;
    const thicknessTotal = parseFloat(document.getElementById(`face-thickness-${index}`).value) || 0;
    const qty = parseFloat(document.getElementById(`face-qty-${index}`).value) || 1;
    
    if (!width || !length || !thicknessTotal) {
        console.log('Width, Length, or Thickness not provided');
        return;
    }
    
    if (![18, 21, 28].includes(thicknessTotal)) {
        return;
    }
    
    const oldQty = faceDrawerQty;
    faceDrawerQty = qty;
    
    console.log('Face Drawer Qty Updated:', oldQty, '→', faceDrawerQty);
    
    const m2 = calculateM2(width, length, qty);
    document.getElementById(`face-m2-${index}`).value = m2.toFixed(4);
    
    const coreThickness = getCoreThickness(thicknessTotal);
    
    document.getElementById(`face-core-thickness-${index}`).value = coreThickness + 'mm';
    document.getElementById(`face-core-type-${index}`).value = initialFormData.panel_type;
    
    document.getElementById(`face-patching-panel-qty-${index}`).value = qty;
    document.getElementById(`face-patching-solid2-qty-${index}`).value = qty * 2;
    document.getElementById(`face-patching-solid3-qty-${index}`).value = qty * 2;
    document.getElementById(`face-crossgrain-qty-${index}`).value = qty * 2;
    document.getElementById(`face-veneer-face-qty-${index}`).value = qty;
    document.getElementById(`face-veneer-back-qty-${index}`).value = qty;
    
    let componentTotal = 0;
    const subComponents = {};
    
    // 1. Patching Panel 1
    const corePanel = getPanelsByType(initialFormData.panel_type)
        .find(p => parseFloat(p.thickness) === parseFloat(coreThickness));
    
    if (!corePanel) {
        console.log(`Panel ${initialFormData.panel_type} ${coreThickness}mm tidak ditemukan`);
        return;
    }
    
    const patchingPanelQty = parseFloat(document.getElementById(`face-patching-panel-qty-${index}`).value) || qty;
    const patchingM2 = calculateM2(width, length, patchingPanelQty);
    const patchingTotal = patchingM2 * corePanel.price;
    
    document.getElementById(`face-patching-panel-m2-${index}`).value = patchingM2.toFixed(4);
    document.getElementById(`face-patching-panel-price-${index}`).textContent = formatRupiah(corePanel.price);
    document.getElementById(`face-patching-panel-total-${index}`).textContent = formatRupiah(patchingTotal);
    
    componentTotal += patchingTotal;
    
    subComponents.patchingPanel = {
        name: 'Patching Panel 1',
        thickness: coreThickness,
        width: width,
        length: length,
        qty: patchingPanelQty,
        m2: patchingM2,
        price: corePanel.price,
        total: patchingTotal,
        material: initialFormData.panel_type
    };
    
    // 2. Patching Solid 2
    const solid2Width = parseFloat(document.getElementById(`face-patching-solid2-width-${index}`).value) || 0;
    const solid2Select = document.getElementById(`face-patching-solid2-thickness-${index}`);
    const solid2Qty = parseFloat(document.getElementById(`face-patching-solid2-qty-${index}`).value) || (qty * 2);
    
    if (solid2Select && solid2Select.value && solid2Width) {
        const selectedOption = solid2Select.options[solid2Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid2M2 = calculateM2(solid2Width, length, solid2Qty);
        const total = solid2M2 * price;
        
        document.getElementById(`face-patching-solid2-m2-${index}`).value = solid2M2.toFixed(4);
        document.getElementById(`face-patching-solid2-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`face-patching-solid2-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid2 = {
            name: 'Patching Solid 2 (PANJANG)',
            thickness: thickness,
            width: solid2Width,
            length: length,
            qty: solid2Qty,
            m2: solid2M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 3. Patching Solid 3
    const solid3Width = parseFloat(document.getElementById(`face-patching-solid3-width-${index}`).value) || 0;
    const solid3Select = document.getElementById(`face-patching-solid3-thickness-${index}`);
    const solid3Qty = parseFloat(document.getElementById(`face-patching-solid3-qty-${index}`).value) || (qty * 2);
    
    if (solid3Select && solid3Select.value && solid3Width) {
        const selectedOption = solid3Select.options[solid3Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid3M2 = calculateM2(solid3Width, width, solid3Qty);
        const total = solid3M2 * price;
        
        document.getElementById(`face-patching-solid3-m2-${index}`).value = solid3M2.toFixed(4);
        document.getElementById(`face-patching-solid3-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`face-patching-solid3-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid3 = {
            name: 'Patching Solid 3 (LEBAR)',
            thickness: thickness,
            width: solid3Width,
            length: width,
            qty: solid3Qty,
            m2: solid3M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 4. Crossgrain
    const crossgrainSelect = document.getElementById(`face-crossgrain-thickness-${index}`);
    const crossgrainQty = parseFloat(document.getElementById(`face-crossgrain-qty-${index}`).value) || (qty * 2);
    
    if (crossgrainSelect && crossgrainSelect.value) {
        const selectedOption = crossgrainSelect.options[crossgrainSelect.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const crossgrainM2 = calculateM2(width, length, crossgrainQty);
        const total = crossgrainM2 * price;
        
        document.getElementById(`face-crossgrain-m2-${index}`).value = crossgrainM2.toFixed(4);
        document.getElementById(`face-crossgrain-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`face-crossgrain-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.crossgrain = {
            name: 'Crossgrain',
            thickness: thickness,
            width: width,
            length: length,
            qty: crossgrainQty,
            m2: crossgrainM2,
            price: price,
            total: total,
            material: 'MDF'
        };
    }
    
    // 5. Veneer Face
    const veneerFacePattern = document.getElementById(`face-veneer-face-pattern-${index}`).value;
    const veneerFaceSelect = document.getElementById(`face-veneer-face-type-${index}`);
    const veneerFaceQty = parseFloat(document.getElementById(`face-veneer-face-qty-${index}`).value) || qty;
    
    if (veneerFaceSelect && veneerFaceSelect.value) {
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerFaceM2 = calculateM2(width, length, veneerFaceQty);
        const total = veneerFaceM2 * price;
        
        document.getElementById(`face-veneer-face-m2-${index}`).value = veneerFaceM2.toFixed(4);
        document.getElementById(`face-veneer-face-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`face-veneer-face-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerFace = {
            name: 'TV. FACE',
            thickness: 0.5,
            width: width,
            length: length,
            qty: veneerFaceQty,
            m2: veneerFaceM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerFacePattern,
            veneerType: veneerType
        };
    }
    
    // 6. Veneer Back
    const veneerBackPattern = document.getElementById(`face-veneer-back-pattern-${index}`).value;
    const veneerBackSelect = document.getElementById(`face-veneer-back-type-${index}`);
    const veneerBackQty = parseFloat(document.getElementById(`face-veneer-back-qty-${index}`).value) || qty;
    
    if (veneerBackSelect && veneerBackSelect.value) {
        const selectedOption = veneerBackSelect.options[veneerBackSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerBackPattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerBackM2 = calculateM2(width, length, veneerBackQty);
        const total = veneerBackM2 * price;
        
        document.getElementById(`face-veneer-back-m2-${index}`).value = veneerBackM2.toFixed(4);
        document.getElementById(`face-veneer-back-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`face-veneer-back-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerBack = {
            name: 'TV. BACK',
            thickness: 0.5,
            width: width,
            length: length,
            qty: veneerBackQty,
            m2: veneerBackM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerBackPattern,
            veneerType: veneerType
        };
    }
    
    document.getElementById(`total-face-drawer-${index}`).textContent = formatRupiah(componentTotal);
    
    calculationData[`face-drawer-${index}`] = {
        componentName: 'FACE DRAWER',
        componentTotal: componentTotal,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            thicknessTotal: thicknessTotal,
            coreThickness: coreThickness
        },
        subComponents: subComponents
    };
    
    console.log('✅ Face Drawer Calculation:', calculationData[`face-drawer-${index}`]);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateGrandTotal();
    saveComponentValuesToDraft(index, 'Face Drawer');
    
    if (oldQty !== faceDrawerQty) {
        console.log('🔄 Face Drawer Qty changed, recalculating all drawers...');
        setTimeout(() => {
            recalculateAllDrawers();
        }, 500);
    }
}

function recalculateAllDrawers() {
    console.log('=== RECALCULATE ALL DRAWERS ===');
    console.log('New Face Drawer Qty:', faceDrawerQty);
    
    const drawerComponents = [
        { index: 8, prefix: 'frontdrawer', name: 'Front Drawer' },
        { index: 9, prefix: 'sidedrawerl', name: 'Side Drawer L' },
        { index: 10, prefix: 'sidedrawerr', name: 'Side Drawer R' },
        { index: 11, prefix: 'backdrawer', name: 'Back Drawer' },
        { index: 12, prefix: 'basedrawer', name: 'Base Drawer' }
    ];
    
    drawerComponents.forEach(drawer => {
        const dimensionSelect = document.getElementById(`${drawer.prefix}-dimension-${drawer.index}`);
        
        if (dimensionSelect && dimensionSelect.value) {
            console.log(`♻️ Recalculating ${drawer.name}...`);
            
            if (drawer.prefix === 'sidedrawerr') {
                const mode = document.getElementById(`sidedrawerr-mode-${drawer.index}`)?.value;
                
                if (mode === 'copy') {
                    const sideLDimension = document.getElementById(`sidedrawerl-dimension-9`);
                    if (sideLDimension && sideLDimension.value) {
                        calculateDrawer(9, 'sidedrawerl');
                    }
                    
                    copySideDrawerLToR(drawer.index);
                } else if (mode === 'manual') {
                    calculateDrawer(drawer.index, drawer.prefix);
                }
            } else {
                calculateDrawer(drawer.index, drawer.prefix);
            }
        }
    });
    
    console.log('✅ All drawers recalculated!');
}
// ============================================
// ✅ CALCULATE DRAWER (FRONT, SIDE L, BACK, BASE)
// ============================================

function calculateDrawer(index, prefix) {
    console.log(`=== CALCULATE DRAWER (${prefix.toUpperCase()}) ===`);
    
    const dimensionSelect = document.getElementById(`${prefix}-dimension-${index}`);
    
    if (!dimensionSelect || !dimensionSelect.value) {
        console.log('Dimension not selected');
        return;
    }
    
    const selectedOption = dimensionSelect.options[dimensionSelect.selectedIndex];
    const width = parseFloat(selectedOption.dataset.width);
    const length = parseFloat(selectedOption.dataset.length);
    const price = parseFloat(selectedOption.dataset.price);
    
    const qty = faceDrawerQty;
    
    console.log(`Drawer Qty from Face Drawer: ${qty}`);
    
    const m2 = calculateM2(width, length, qty);
    const total = m2 * price;
    
    document.getElementById(`${prefix}-m2-${index}`).value = m2.toFixed(4);
    document.getElementById(`${prefix}-price-${index}`).textContent = formatRupiah(price);
    document.getElementById(`${prefix}-total-${index}`).textContent = formatRupiah(total);
    
    document.getElementById(`total-${prefix}-${index}`).textContent = formatRupiah(total);
    
    const componentNames = {
        'frontdrawer': 'FRONT DRAWER',
        'sidedrawerl': 'SIDE DRAWER L',
        'sidedrawerr': 'SIDE DRAWER R',
        'backdrawer': 'BACK DRAWER',
        'basedrawer': 'BASE DRAWER'
    };
    
    calculationData[`${prefix}-${index}`] = {
        componentName: componentNames[prefix] || prefix.toUpperCase(),
        componentTotal: total,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            price: price,
            material: initialFormData.box_drawer_type
        }
    };
    
    if (prefix === 'sidedrawerl') {
        sideDrawerLData = calculationData[`${prefix}-${index}`];
        console.log('✅ Side Drawer L Data Auto-Saved:', sideDrawerLData);
    }
    
    console.log(`✅ ${componentNames[prefix]} Calculation:`, calculationData[`${prefix}-${index}`]);
    
    updateComponentCard(index, 'filled', total);
    updateGrandTotal();
    saveComponentValuesToDraft(index, componentNames[prefix]);
}

// ============================================
// ✅ SIDE DRAWER R - MODE CHANGE
// ============================================

function onSideDrawerRModeChange(index) {
    const mode = document.getElementById(`sidedrawerr-mode-${index}`).value;
    const manualInputs = document.getElementById(`sidedrawerr-manual-inputs-${index}`);
    
    if (mode === 'copy') {
        manualInputs.style.display = 'none';
        
        if (!sideDrawerLData) {
            alert('❌ Side Drawer L belum diisi! Silakan isi Side Drawer L terlebih dahulu.');
            document.getElementById(`sidedrawerr-mode-${index}`).value = '';
            return;
        }
        
        copySideDrawerLToR(index);
        
    } else if (mode === 'manual') {
        manualInputs.style.display = 'block';
        
        document.getElementById(`sidedrawerr-dimension-${index}`).value = '';
        document.getElementById(`total-sidedrawerr-${index}`).textContent = 'Rp 0';
        
        delete calculationData[`sidedrawerr-${index}`];
        updateComponentCard(index, 'empty', 0);
        updateGrandTotal();
        
    } else {
        manualInputs.style.display = 'none';
    }
}

function copySideDrawerLToR(index) {
    console.log('=== COPY SIDE DRAWER L TO R ===');
    
    if (!sideDrawerLData) {
        alert('❌ Data Side Drawer L tidak ditemukan!');
        return;
    }
    
    calculationData[`sidedrawerr-${index}`] = {
        componentName: 'SIDE DRAWER R',
        componentTotal: sideDrawerLData.componentTotal,
        details: { ...sideDrawerLData.details },
        copiedFrom: 'Side Drawer L'
    };
    
    document.getElementById(`total-sidedrawerr-${index}`).textContent = formatRupiah(sideDrawerLData.componentTotal);
    
    updateComponentCard(index, 'copied', sideDrawerLData.componentTotal);
    updateGrandTotal();
    saveComponentValuesToDraft(index, 'Side Drawer R');
    
    alert('✅ Side Drawer R berhasil di-copy dari Side Drawer L!');
    console.log('Side Drawer R (copied):', calculationData[`sidedrawerr-${index}`]);
}

// ============================================
// ✅ CALCULATE SUPPORT DRAWER (QTY × RP 2.200)
// ============================================

function calculateSupportDrawer(index) {
    console.log('=== CALCULATE SUPPORT DRAWER ===');
    
    const qty = parseFloat(document.getElementById(`support-qty-${index}`).value) || 1;
    const pricePerUnit = 2200;
    const total = qty * pricePerUnit;
    
    document.getElementById(`support-total-${index}`).textContent = formatRupiah(total);
    
    calculationData[`support-drawer-${index}`] = {
        componentName: 'SUPPORT DRAWER',
        componentTotal: total,
        details: {
            width: 11,
            qty: qty,
            pricePerUnit: pricePerUnit,
            material: 'Mix'
        }
    };
    
    console.log('✅ Support Drawer Calculation:', calculationData[`support-drawer-${index}`]);
    
    updateComponentCard(index, 'filled', total);
    updateGrandTotal();
    saveComponentValuesToDraft(index, 'Support Drawer');
}

// ============================================
// ✅ CALCULATE CLEAT DRAWER (QTY × RP 400)
// ============================================

function calculateCleatDrawer(index) {
    console.log('=== CALCULATE CLEAT DRAWER ===');
    
    const qty = parseFloat(document.getElementById(`cleat-qty-${index}`).value) || 1;
    const pricePerUnit = 400;
    const total = qty * pricePerUnit;
    
    document.getElementById(`cleat-total-${index}`).textContent = formatRupiah(total);
    
    calculationData[`cleat-drawer-${index}`] = {
        componentName: 'CLEAT DRAWER',
        componentTotal: total,
        details: {
            width: 11,
            qty: qty,
            pricePerUnit: pricePerUnit,
            material: 'Mix'
        }
    };
    
    console.log('✅ Cleat Drawer Calculation:', calculationData[`cleat-drawer-${index}`]);
    
    updateComponentCard(index, 'filled', total);
    updateGrandTotal();
    saveComponentValuesToDraft(index, 'Cleat Drawer');
}

// ============================================
// ✅ UPDATE INVOICE BOM (PUT REQUEST)
// ============================================

function updateInvoice() {
    console.log('=== UPDATE INVOICE BOM ===');
    
    if (Object.keys(calculationData).length === 0) {
        if (!confirm('⚠️ Belum ada komponen yang diisi. Yakin ingin melanjutkan?')) {
            return;
        }
    }
    
    const confirmText = `Apakah Anda yakin ingin mengupdate invoice ini?\n\nGrand Total: ${formatRupiah(grandTotal)}\n\nSetelah diupdate, perubahan akan tersimpan ke database.`;
    if (!confirm(confirmText)) {
        return;
    }
    
    document.getElementById('loadingOverlay').style.display = 'flex';
    
    const formData = new FormData();
    
    // ✅ Initial Form Data (LOCKED - tidak bisa diubah)
    Object.keys(initialFormData).forEach(key => {
        formData.append(key, initialFormData[key]);
    });
    
    // ✅ Calculation Data (JSON)
    const calculationDataJson = JSON.stringify(calculationData);
    formData.append('calculation_data', calculationDataJson);
    formData.append('grand_total', grandTotal);
    
    // ✅ Method spoofing untuk PUT
    formData.append('_method', 'PUT');
    
    console.log('=== UPDATE INVOICE DATA ===');
    console.log('Calculation Data:', calculationData);
    console.log('Grand Total:', grandTotal);
    
    fetch('{{ route("invoice.bom.update", $invoice->id) }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        },
        body: formData
    })
    .then(response => {
        if (!response.ok) {
            return response.json().then(err => {
                throw new Error(err.message || 'Update failed');
            });
        }
        return response.json();
    })
    .then(data => {
        console.log('✅ Server Response:', data);
        
        if (data.success) {
            // Clear session storage for this edit
            for (let i = 0; i < 15; i++) {
                sessionStorage.removeItem(`bom_component_values_edit_${i}`);
            }
            
            alert('✅ Invoice BOM berhasil diupdate!');
            window.location.href = '{{ route("invoice.bom.show", $invoice->id) }}';
        } else {
            throw new Error(data.message || 'Failed to update invoice');
        }
    })
    .catch(error => {
        console.error('❌ Error:', error);
        alert('❌ Terjadi kesalahan: ' + error.message);
    })
    .finally(() => {
        document.getElementById('loadingOverlay').style.display = 'none';
    });
}

console.log('✅ All Calculation Functions Loaded Successfully');
// ============================================
// ✅ GET TOP PANEL CONTENT - COMPLETE HTML
// ============================================

function getTopPanelContent(index) {
    const solids = getSolidsByType(initialFormData.solid_type);
    const mdfPanels = getPanelsByType('MDF');
    const allVeneers = getAllVeneers();
    
    return `
        <div class="p-3">
            <div class="alert alert-info mb-4">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Top Panel:</strong> Input dimensi dan pilih material. Perhitungan otomatis saat input berubah.
            </div>

            <h6 class="mb-3"><i class="fas fa-ruler-combined me-2"></i>Dimensi Panel</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-3">
                    <label class="form-label">Lebar (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="top-width-${index}" 
                           onchange="calculateTopPanel(${index})" 
                           onkeyup="calculateTopPanel(${index})"
                           placeholder="Lebar" step="0.01">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Panjang (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="top-length-${index}" 
                           onchange="calculateTopPanel(${index})"
                           onkeyup="calculateTopPanel(${index})"
                           placeholder="Panjang" step="0.01">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness Total <span class="text-danger">*</span></label>
                    <select class="form-select" id="top-thickness-${index}" onchange="calculateTopPanel(${index})">
                        <option value="">-- Pilih Thickness --</option>
                        <option value="18">18mm (Core 15mm + Veneer 1mm + Patching 2mm)</option>
                        <option value="21">21mm (Core 18mm + Veneer 1mm + Patching 2mm)</option>
                        <option value="28">28mm (Core 25mm + Veneer 1mm + Patching 2mm)</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Qty <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="top-qty-${index}" value="1" min="1" 
                           onchange="calculateTopPanel(${index})"
                           onkeyup="calculateTopPanel(${index})">
                </div>
            </div>
            
            <div class="row g-3 mb-4">
                <div class="col-md-12">
                    <label class="form-label">M² Total (Auto-calculated)</label>
                    <input type="text" class="form-control bg-light" id="top-m2-${index}" readonly>
                </div>
            </div>

            <hr class="my-4">

            <h6 class="mb-3"><i class="fas fa-layer-group me-2"></i>Sub Components</h6>

            <!-- 1. Patching Panel 1 (Core Panel - AUTO) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-cube me-2"></i>1. Patching Panel 1 (Core Panel - Auto Calculated)
                </h6>
                <div class="alert alert-success alert-sm mb-3">
                    <i class="fas fa-magic me-2"></i>
                    <strong>Core Panel otomatis dipilih berdasarkan Thickness Total yang Anda pilih</strong>
                </div>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Tebal Core (Auto)</label>
                        <input type="text" class="form-control bg-light" id="top-core-thickness-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Type Panel (Locked)</label>
                        <input type="text" class="form-control bg-light" id="top-core-type-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="top-patching-panel-qty-${index}" value="1" min="1" 
                               onchange="calculateTopPanel(${index})"
                               onkeyup="calculateTopPanel(${index})">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="top-patching-panel-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="top-patching-panel-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="top-patching-panel-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 2. Patching Solid 2 (PANJANG) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-ruler-horizontal me-2"></i>2. Patching Solid 2 (PANJANG)
                </h6>
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Lebar Pilihan (mm)</label>
                        <input type="number" class="form-control" id="top-patching-solid2-width-${index}" value="18" 
                               onchange="calculateTopPanel(${index})"
                               onkeyup="calculateTopPanel(${index})"
                               step="0.01">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Tebal Solid</label>
                        <select class="form-select" id="top-patching-solid2-thickness-${index}" onchange="calculateTopPanel(${index})">
                            <option value="">Pilih Tebal</option>
                            ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="top-patching-solid2-qty-${index}" value="2" min="0" 
                               onchange="calculateTopPanel(${index})"
                               onkeyup="calculateTopPanel(${index})">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="top-patching-solid2-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="top-patching-solid2-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="top-patching-solid2-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 3. Patching Solid 3 (LEBAR) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-ruler-vertical me-2"></i>3. Patching Solid 3 (LEBAR)
                </h6>
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Lebar Pilihan (mm)</label>
                        <input type="number" class="form-control" id="top-patching-solid3-width-${index}" value="18" 
                               onchange="calculateTopPanel(${index})"
                               onkeyup="calculateTopPanel(${index})"
                               step="0.01">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Tebal Solid</label>
                        <select class="form-select" id="top-patching-solid3-thickness-${index}" onchange="calculateTopPanel(${index})">
                            <option value="">Pilih Tebal</option>
                            ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="top-patching-solid3-qty-${index}" value="2" min="0" 
                               onchange="calculateTopPanel(${index})"
                               onkeyup="calculateTopPanel(${index})">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="top-patching-solid3-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="top-patching-solid3-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="top-patching-solid3-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 4. Crossgrain (MDF) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-grip-lines me-2"></i>4. Crossgrain (MDF)
                </h6>
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Tebal MDF</label>
                        <select class="form-select" id="top-crossgrain-thickness-${index}" onchange="calculateTopPanel(${index})">
                            <option value="">Pilih Tebal</option>
                            ${mdfPanels.map(p => `<option value="${p.id}" data-thickness="${p.thickness}" data-price="${p.price}">${p.thickness}mm - ${formatRupiah(p.price)}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="top-crossgrain-qty-${index}" value="2" min="0" 
                               onchange="calculateTopPanel(${index})"
                               onkeyup="calculateTopPanel(${index})">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="top-crossgrain-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="top-crossgrain-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="top-crossgrain-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 5. Veneer Face -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-square me-2"></i>5. Veneer Face
                </h6>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="top-veneer-face-pattern-${index}" onchange="calculateTopPanel(${index})">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Veneer Type (Bisa Diganti)</label>
                        <select class="form-select" id="top-veneer-face-type-${index}" onchange="calculateTopPanel(${index})">
                            <option value="">Pilih Veneer</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="top-veneer-face-qty-${index}" value="1" min="0" 
                               onchange="calculateTopPanel(${index})"
                               onkeyup="calculateTopPanel(${index})">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="top-veneer-face-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="top-veneer-face-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="top-veneer-face-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 6. Veneer Back -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-square me-2"></i>6. Veneer Back
                </h6>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="top-veneer-back-pattern-${index}" onchange="calculateTopPanel(${index})">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Veneer Type (Bisa Diganti)</label>
                        <select class="form-select" id="top-veneer-back-type-${index}" onchange="calculateTopPanel(${index})">
                            <option value="">Pilih Veneer</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="top-veneer-back-qty-${index}" value="1" min="0" 
                               onchange="calculateTopPanel(${index})"
                               onkeyup="calculateTopPanel(${index})">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="top-veneer-back-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="top-veneer-back-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="top-veneer-back-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 7. Cleat -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-screwdriver me-2"></i>7. Cleat
                </h6>
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="top-cleat-qty-${index}" value="6" min="0" 
                               onchange="calculateTopPanel(${index})"
                               onkeyup="calculateTopPanel(${index})">
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Harga per Unit</label>
                        <input type="text" class="form-control bg-light" value="Rp 1.000" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-12 text-end">
                            <strong>Total:</strong> <span id="top-cleat-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total Top Panel:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-top-panel-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET SIDE PANEL L CONTENT
// ============================================

function getSidePanelContent(index, componentName) {
    const solids = getSolidsByType(initialFormData.solid_type);
    const mdfPanels = getPanelsByType('MDF');
    const allVeneers = getAllVeneers();
    const prefix = 'sidel';
    
    return `
        <div class="p-3">
            <div class="alert alert-info mb-4">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Side Panel L:</strong> Data ini bisa di-copy ke Side Panel R untuk efisiensi. Perhitungan otomatis saat input berubah.
            </div>

            <h6 class="mb-3"><i class="fas fa-ruler-combined me-2"></i>Dimensi Panel</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-3">
                    <label class="form-label">Lebar (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="${prefix}-width-${index}" 
                           onchange="calculateSidePanel(${index}, '${prefix}')" 
                           onkeyup="calculateSidePanel(${index}, '${prefix}')"
                           placeholder="Lebar" step="0.01">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Panjang (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="${prefix}-length-${index}" 
                           onchange="calculateSidePanel(${index}, '${prefix}')"
                           onkeyup="calculateSidePanel(${index}, '${prefix}')"
                           placeholder="Panjang" step="0.01">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness Total <span class="text-danger">*</span></label>
                    <select class="form-select" id="${prefix}-thickness-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                        <option value="">-- Pilih Thickness --</option>
                        <option value="18">18mm (Core 15mm + Veneer 1mm + Patching 2mm)</option>
                        <option value="21">21mm (Core 18mm + Veneer 1mm + Patching 2mm)</option>
                        <option value="28">28mm (Core 25mm + Veneer 1mm + Patching 2mm)</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Qty <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="${prefix}-qty-${index}" value="1" min="1" 
                           onchange="calculateSidePanel(${index}, '${prefix}')"
                           onkeyup="calculateSidePanel(${index}, '${prefix}')">
                </div>
            </div>
            
            <div class="row g-3 mb-4">
                <div class="col-md-12">
                    <label class="form-label">M² Total (Auto-calculated)</label>
                    <input type="text" class="form-control bg-light" id="${prefix}-m2-${index}" readonly>
                </div>
            </div>

            <hr class="my-4">

            <h6 class="mb-3"><i class="fas fa-layer-group me-2"></i>Sub Components</h6>

            <!-- Same structure as Top Panel but with prefix "sidel" -->
            <!-- 1. Patching Panel 1 -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-cube me-2"></i>1. Patching Panel 1 (Core Panel - Auto Calculated)
                </h6>
                <div class="alert alert-success alert-sm mb-3">
                    <i class="fas fa-magic me-2"></i>
                    <strong>Core Panel otomatis dipilih berdasarkan Thickness Total</strong>
                </div>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Tebal Core (Auto)</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-core-thickness-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Type Panel (Locked)</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-core-type-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-patching-panel-qty-${index}" value="1" min="1" 
                               onchange="calculateSidePanel(${index}, '${prefix}')"
                               onkeyup="calculateSidePanel(${index}, '${prefix}')">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-patching-panel-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="${prefix}-patching-panel-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="${prefix}-patching-panel-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 2. Patching Solid 2 -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-ruler-horizontal me-2"></i>2. Patching Solid 2 (PANJANG)
                </h6>
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Lebar Pilihan (mm)</label>
                        <input type="number" class="form-control" id="${prefix}-patching-solid2-width-${index}" value="18" 
                               onchange="calculateSidePanel(${index}, '${prefix}')"
                               onkeyup="calculateSidePanel(${index}, '${prefix}')"
                               step="0.01">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Tebal Solid</label>
                        <select class="form-select" id="${prefix}-patching-solid2-thickness-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                            <option value="">Pilih Tebal</option>
                            ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-patching-solid2-qty-${index}" value="2" min="0" 
                               onchange="calculateSidePanel(${index}, '${prefix}')"
                               onkeyup="calculateSidePanel(${index}, '${prefix}')">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-patching-solid2-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="${prefix}-patching-solid2-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="${prefix}-patching-solid2-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 3. Patching Solid 3 -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-ruler-vertical me-2"></i>3. Patching Solid 3 (LEBAR)
                </h6>
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Lebar Pilihan (mm)</label>
                        <input type="number" class="form-control" id="${prefix}-patching-solid3-width-${index}" value="18" 
                               onchange="calculateSidePanel(${index}, '${prefix}')"
                               onkeyup="calculateSidePanel(${index}, '${prefix}')"
                               step="0.01">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Tebal Solid</label>
                        <select class="form-select" id="${prefix}-patching-solid3-thickness-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                            <option value="">Pilih Tebal</option>
                            ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-patching-solid3-qty-${index}" value="2" min="0" 
                               onchange="calculateSidePanel(${index}, '${prefix}')"
                               onkeyup="calculateSidePanel(${index}, '${prefix}')">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-patching-solid3-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="${prefix}-patching-solid3-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="${prefix}-patching-solid3-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 4. Crossgrain -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-grip-lines me-2"></i>4. Crossgrain (MDF)
                </h6>
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Tebal MDF</label>
                        <select class="form-select" id="${prefix}-crossgrain-thickness-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                            <option value="">Pilih Tebal</option>
                            ${mdfPanels.map(p => `<option value="${p.id}" data-thickness="${p.thickness}" data-price="${p.price}">${p.thickness}mm - ${formatRupiah(p.price)}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-crossgrain-qty-${index}" value="2" min="0" 
                               onchange="calculateSidePanel(${index}, '${prefix}')"
                               onkeyup="calculateSidePanel(${index}, '${prefix}')">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-crossgrain-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="${prefix}-crossgrain-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="${prefix}-crossgrain-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 5. Veneer Face -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-square me-2"></i>5. Veneer Face
                </h6>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="${prefix}-veneer-face-pattern-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Veneer Type (Bisa Diganti)</label>
                        <select class="form-select" id="${prefix}-veneer-face-type-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                            <option value="">Pilih Veneer</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-veneer-face-qty-${index}" value="1" min="0" 
                               onchange="calculateSidePanel(${index}, '${prefix}')"
                               onkeyup="calculateSidePanel(${index}, '${prefix}')">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-veneer-face-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="${prefix}-veneer-face-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="${prefix}-veneer-face-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 6. Veneer Back -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-square me-2"></i>6. Veneer Back
                </h6>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="${prefix}-veneer-back-pattern-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Veneer Type (Bisa Diganti)</label>
                        <select class="form-select" id="${prefix}-veneer-back-type-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                            <option value="">Pilih Veneer</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-veneer-back-qty-${index}" value="1" min="0" 
                               onchange="calculateSidePanel(${index}, '${prefix}')"
                               onkeyup="calculateSidePanel(${index}, '${prefix}')">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-veneer-back-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="${prefix}-veneer-back-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="${prefix}-veneer-back-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total Side Panel L:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-${prefix}-panel-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}
// ============================================
// ✅ GET SIDE PANEL R CONTENT (COPY OR MANUAL MODE)
// ============================================

function getSidePanelRContent(index) {
    const solids = getSolidsByType(initialFormData.solid_type);
    const mdfPanels = getPanelsByType('MDF');
    const allVeneers = getAllVeneers();
    const prefix = 'sider';
    
    return `
        <div class="p-3">
            <div class="alert alert-warning mb-4">
                <i class="fas fa-copy me-2"></i>
                <strong>Side Panel R:</strong> Pilih mode Copy dari Side Panel L atau Input Manual.
            </div>

            <h6 class="mb-3"><i class="fas fa-cog me-2"></i>Pilih Mode Input</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-12">
                    <label class="form-label">Mode <span class="text-danger">*</span></label>
                    <select class="form-select form-select-lg" id="${prefix}-mode-${index}" onchange="onSidePanelRModeChange(${index})">
                        <option value="">-- Pilih Mode --</option>
                        <option value="copy">📋 Copy dari Side Panel L</option>
                        <option value="manual">✏️ Input Manual</option>
                    </select>
                    <small class="text-muted">Pilih "Copy" untuk menggunakan data Side Panel L, atau "Manual" untuk input sendiri</small>
                </div>
            </div>

            <hr class="my-4">

            <!-- MANUAL INPUTS (Hidden by default) -->
            <div id="${prefix}-manual-inputs-${index}" style="display: none;">
                <h6 class="mb-3"><i class="fas fa-ruler-combined me-2"></i>Dimensi Panel</h6>
                <div class="row g-3 mb-4">
                    <div class="col-md-3">
                        <label class="form-label">Lebar (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="${prefix}-width-${index}" 
                               onchange="calculateSidePanel(${index}, '${prefix}')" 
                               onkeyup="calculateSidePanel(${index}, '${prefix}')"
                               placeholder="Lebar" step="0.01">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Panjang (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="${prefix}-length-${index}" 
                               onchange="calculateSidePanel(${index}, '${prefix}')"
                               onkeyup="calculateSidePanel(${index}, '${prefix}')"
                               placeholder="Panjang" step="0.01">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Thickness Total <span class="text-danger">*</span></label>
                        <select class="form-select" id="${prefix}-thickness-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                            <option value="">-- Pilih Thickness --</option>
                            <option value="18">18mm (Core 15mm + Veneer 1mm + Patching 2mm)</option>
                            <option value="21">21mm (Core 18mm + Veneer 1mm + Patching 2mm)</option>
                            <option value="28">28mm (Core 25mm + Veneer 1mm + Patching 2mm)</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="${prefix}-qty-${index}" value="1" min="1" 
                               onchange="calculateSidePanel(${index}, '${prefix}')"
                               onkeyup="calculateSidePanel(${index}, '${prefix}')">
                    </div>
                </div>
                
                <div class="row g-3 mb-4">
                    <div class="col-md-12">
                        <label class="form-label">M² Total (Auto-calculated)</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-m2-${index}" readonly>
                    </div>
                </div>

                <hr class="my-4">

                <h6 class="mb-3"><i class="fas fa-layer-group me-2"></i>Sub Components</h6>

                <!-- 1. Patching Panel 1 -->
                <div class="sub-component-card mb-3">
                    <h6 class="sub-component-title">
                        <i class="fas fa-cube me-2"></i>1. Patching Panel 1 (Core Panel - Auto Calculated)
                    </h6>
                    <div class="alert alert-success alert-sm mb-3">
                        <i class="fas fa-magic me-2"></i>
                        <strong>Core Panel otomatis dipilih berdasarkan Thickness Total</strong>
                    </div>
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">Tebal Core (Auto)</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-core-thickness-${index}" readonly>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Type Panel (Locked)</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-core-type-${index}" readonly>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Qty</label>
                            <input type="number" class="form-control" id="${prefix}-patching-panel-qty-${index}" value="1" min="1" 
                                   onchange="calculateSidePanel(${index}, '${prefix}')"
                                   onkeyup="calculateSidePanel(${index}, '${prefix}')">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">M²</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-patching-panel-m2-${index}" readonly>
                        </div>
                    </div>
                    <div class="calculated-info mt-3">
                        <div class="row">
                            <div class="col-md-6">
                                <strong>Harga:</strong> <span id="${prefix}-patching-panel-price-${index}" class="text-primary">Rp 0</span>
                            </div>
                            <div class="col-md-6 text-end">
                                <strong>Total:</strong> <span id="${prefix}-patching-panel-total-${index}" class="text-success fw-bold">Rp 0</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- 2. Patching Solid 2 (PANJANG) -->
                <div class="sub-component-card mb-3">
                    <h6 class="sub-component-title">
                        <i class="fas fa-ruler-horizontal me-2"></i>2. Patching Solid 2 (PANJANG)
                    </h6>
                    <div class="row g-3">
                        <div class="col-md-4">
                            <label class="form-label">Lebar Pilihan (mm)</label>
                            <input type="number" class="form-control" id="${prefix}-patching-solid2-width-${index}" value="18" 
                                   onchange="calculateSidePanel(${index}, '${prefix}')"
                                   onkeyup="calculateSidePanel(${index}, '${prefix}')"
                                   step="0.01">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Tebal Solid</label>
                            <select class="form-select" id="${prefix}-patching-solid2-thickness-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                                <option value="">Pilih Tebal</option>
                                ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}</option>`).join('')}
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Qty</label>
                            <input type="number" class="form-control" id="${prefix}-patching-solid2-qty-${index}" value="2" min="0" 
                                   onchange="calculateSidePanel(${index}, '${prefix}')"
                                   onkeyup="calculateSidePanel(${index}, '${prefix}')">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">M²</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-patching-solid2-m2-${index}" readonly>
                        </div>
                    </div>
                    <div class="calculated-info mt-3">
                        <div class="row">
                            <div class="col-md-6">
                                <strong>Harga:</strong> <span id="${prefix}-patching-solid2-price-${index}" class="text-primary">Rp 0</span>
                            </div>
                            <div class="col-md-6 text-end">
                                <strong>Total:</strong> <span id="${prefix}-patching-solid2-total-${index}" class="text-success fw-bold">Rp 0</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- 3. Patching Solid 3 (LEBAR) -->
                <div class="sub-component-card mb-3">
                    <h6 class="sub-component-title">
                        <i class="fas fa-ruler-vertical me-2"></i>3. Patching Solid 3 (LEBAR)
                    </h6>
                    <div class="row g-3">
                        <div class="col-md-4">
                            <label class="form-label">Lebar Pilihan (mm)</label>
                            <input type="number" class="form-control" id="${prefix}-patching-solid3-width-${index}" value="18" 
                                   onchange="calculateSidePanel(${index}, '${prefix}')"
                                   onkeyup="calculateSidePanel(${index}, '${prefix}')"
                                   step="0.01">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Tebal Solid</label>
                            <select class="form-select" id="${prefix}-patching-solid3-thickness-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                                <option value="">Pilih Tebal</option>
                                ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}</option>`).join('')}
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Qty</label>
                            <input type="number" class="form-control" id="${prefix}-patching-solid3-qty-${index}" value="2" min="0" 
                                   onchange="calculateSidePanel(${index}, '${prefix}')"
                                   onkeyup="calculateSidePanel(${index}, '${prefix}')">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">M²</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-patching-solid3-m2-${index}" readonly>
                        </div>
                    </div>
                    <div class="calculated-info mt-3">
                        <div class="row">
                            <div class="col-md-6">
                                <strong>Harga:</strong> <span id="${prefix}-patching-solid3-price-${index}" class="text-primary">Rp 0</span>
                            </div>
                            <div class="col-md-6 text-end">
                                <strong>Total:</strong> <span id="${prefix}-patching-solid3-total-${index}" class="text-success fw-bold">Rp 0</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- 4. Crossgrain (MDF) -->
                <div class="sub-component-card mb-3">
                    <h6 class="sub-component-title">
                        <i class="fas fa-grip-lines me-2"></i>4. Crossgrain (MDF)
                    </h6>
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Tebal MDF</label>
                            <select class="form-select" id="${prefix}-crossgrain-thickness-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                                <option value="">Pilih Tebal</option>
                                ${mdfPanels.map(p => `<option value="${p.id}" data-thickness="${p.thickness}" data-price="${p.price}">${p.thickness}mm - ${formatRupiah(p.price)}</option>`).join('')}
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Qty</label>
                            <input type="number" class="form-control" id="${prefix}-crossgrain-qty-${index}" value="2" min="0" 
                                   onchange="calculateSidePanel(${index}, '${prefix}')"
                                   onkeyup="calculateSidePanel(${index}, '${prefix}')">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">M²</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-crossgrain-m2-${index}" readonly>
                        </div>
                    </div>
                    <div class="calculated-info mt-3">
                        <div class="row">
                            <div class="col-md-6">
                                <strong>Harga:</strong> <span id="${prefix}-crossgrain-price-${index}" class="text-primary">Rp 0</span>
                            </div>
                            <div class="col-md-6 text-end">
                                <strong>Total:</strong> <span id="${prefix}-crossgrain-total-${index}" class="text-success fw-bold">Rp 0</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- 5. Veneer Face -->
                <div class="sub-component-card mb-3">
                    <h6 class="sub-component-title">
                        <i class="fas fa-square me-2"></i>5. Veneer Face
                    </h6>
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">Pattern</label>
                            <select class="form-select" id="${prefix}-veneer-face-pattern-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                                <option value="LURUS">LURUS</option>
                                <option value="MOTIF">MOTIF</option>
                            </select>
                        </div>
                        <div class="col-md-5">
                            <label class="form-label">Veneer Type (Bisa Diganti)</label>
                            <select class="form-select" id="${prefix}-veneer-face-type-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                                <option value="">Pilih Veneer</option>
                                ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Qty</label>
                            <input type="number" class="form-control" id="${prefix}-veneer-face-qty-${index}" value="1" min="0" 
                                   onchange="calculateSidePanel(${index}, '${prefix}')"
                                   onkeyup="calculateSidePanel(${index}, '${prefix}')">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">M²</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-veneer-face-m2-${index}" readonly>
                        </div>
                    </div>
                    <div class="calculated-info mt-3">
                        <div class="row">
                            <div class="col-md-6">
                                <strong>Harga:</strong> <span id="${prefix}-veneer-face-price-${index}" class="text-primary">Rp 0</span>
                            </div>
                            <div class="col-md-6 text-end">
                                <strong>Total:</strong> <span id="${prefix}-veneer-face-total-${index}" class="text-success fw-bold">Rp 0</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- 6. Veneer Back -->
                <div class="sub-component-card mb-3">
                    <h6 class="sub-component-title">
                        <i class="fas fa-square me-2"></i>6. Veneer Back
                    </h6>
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">Pattern</label>
                            <select class="form-select" id="${prefix}-veneer-back-pattern-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                                <option value="LURUS">LURUS</option>
                                <option value="MOTIF">MOTIF</option>
                            </select>
                        </div>
                        <div class="col-md-5">
                            <label class="form-label">Veneer Type (Bisa Diganti)</label>
                            <select class="form-select" id="${prefix}-veneer-back-type-${index}" onchange="calculateSidePanel(${index}, '${prefix}')">
                                <option value="">Pilih Veneer</option>
                                ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Qty</label>
                            <input type="number" class="form-control" id="${prefix}-veneer-back-qty-${index}" value="1" min="0" 
                                   onchange="calculateSidePanel(${index}, '${prefix}')"
                                   onkeyup="calculateSidePanel(${index}, '${prefix}')">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">M²</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-veneer-back-m2-${index}" readonly>
                        </div>
                    </div>
                    <div class="calculated-info mt-3">
                        <div class="row">
                            <div class="col-md-6">
                                <strong>Harga:</strong> <span id="${prefix}-veneer-back-price-${index}" class="text-primary">Rp 0</span>
                            </div>
                            <div class="col-md-6 text-end">
                                <strong>Total:</strong> <span id="${prefix}-veneer-back-total-${index}" class="text-success fw-bold">Rp 0</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total Side Panel R:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-${prefix}-panel-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET BOTTOM PANEL CONTENT
// ============================================

function getBottomPanelContent(index) {
    const allVeneers = getAllVeneers();
    
    return `
        <div class="p-3">
            <div class="alert alert-info mb-4">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Bottom Panel:</strong> Input dimensi dan pilih material. Perhitungan otomatis saat input berubah.
            </div>

            <h6 class="mb-3"><i class="fas fa-ruler-combined me-2"></i>Dimensi Panel</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-3">
                    <label class="form-label">Lebar (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="bottom-width-${index}" 
                           onchange="calculateBottomPanel(${index})" 
                           onkeyup="calculateBottomPanel(${index})"
                           placeholder="Lebar" step="0.01">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Panjang (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="bottom-length-${index}" 
                           onchange="calculateBottomPanel(${index})"
                           onkeyup="calculateBottomPanel(${index})"
                           placeholder="Panjang" step="0.01">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness Total <span class="text-danger">*</span></label>
                    <select class="form-select" id="bottom-thickness-${index}" onchange="calculateBottomPanel(${index})">
                        <option value="">-- Pilih Thickness --</option>
                        <option value="16">16mm (Core 15mm + Veneer 1mm)</option>
                        <option value="19">19mm (Core 18mm + Veneer 1mm)</option>
                        <option value="26">26mm (Core 25mm + Veneer 1mm)</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Qty <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="bottom-qty-${index}" value="1" min="1" 
                           onchange="calculateBottomPanel(${index})"
                           onkeyup="calculateBottomPanel(${index})">
                </div>
            </div>
            
            <div class="row g-3 mb-4">
                <div class="col-md-12">
                    <label class="form-label">M² Total (Auto-calculated)</label>
                    <input type="text" class="form-control bg-light" id="bottom-m2-${index}" readonly>
                </div>
            </div>

            <hr class="my-4">

            <h6 class="mb-3"><i class="fas fa-layer-group me-2"></i>Sub Components</h6>

            <!-- 1. Core Panel -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-cube me-2"></i>1. Core Panel (Auto Calculated)
                </h6>
                <div class="alert alert-success alert-sm mb-3">
                    <i class="fas fa-magic me-2"></i>
                    <strong>Core Panel otomatis dipilih berdasarkan Thickness Total</strong>
                </div>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Tebal Core (Auto)</label>
                        <input type="text" class="form-control bg-light" id="bottom-core-thickness-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Type Panel (Locked)</label>
                        <input type="text" class="form-control bg-light" id="bottom-core-type-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="bottom-core-qty-${index}" value="1" min="1" 
                               onchange="calculateBottomPanel(${index})"
                               onkeyup="calculateBottomPanel(${index})">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="bottom-core-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="bottom-core-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="bottom-core-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 2. Veneer Face -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-square me-2"></i>2. Veneer Face
                </h6>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="bottom-veneer-face-pattern-${index}" onchange="calculateBottomPanel(${index})">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Veneer Type (Bisa Diganti)</label>
                        <select class="form-select" id="bottom-veneer-face-type-${index}" onchange="calculateBottomPanel(${index})">
                            <option value="">Pilih Veneer</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="bottom-veneer-face-qty-${index}" value="1" min="0" 
                               onchange="calculateBottomPanel(${index})"
                               onkeyup="calculateBottomPanel(${index})">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="bottom-veneer-face-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="bottom-veneer-face-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="bottom-veneer-face-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 3. Veneer Back -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-square me-2"></i>3. Veneer Back
                </h6>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="bottom-veneer-back-pattern-${index}" onchange="calculateBottomPanel(${index})">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Veneer Type (Bisa Diganti)</label>
                        <select class="form-select" id="bottom-veneer-back-type-${index}" onchange="calculateBottomPanel(${index})">
                            <option value="">Pilih Veneer</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="bottom-veneer-back-qty-${index}" value="1" min="0" 
                               onchange="calculateBottomPanel(${index})"
                               onkeyup="calculateBottomPanel(${index})">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="bottom-veneer-back-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="bottom-veneer-back-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="bottom-veneer-back-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 4. E.Band -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-border-style me-2"></i>4. E.Band
                </h6>
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="bottom-eband-qty-${index}" value="1" min="0" 
                               onchange="calculateBottomPanel(${index})"
                               onkeyup="calculateBottomPanel(${index})">
                        <small class="text-muted">Formula: (Thickness × Length × Qty) ÷ 1,000,000</small>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="bottom-eband-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="bottom-eband-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="bottom-eband-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total Bottom Panel:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-bottom-panel-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}
// ============================================
// ✅ GET BACK PANEL CONTENT (COMPLETE FIX)
// ============================================

function getBackPanelContent(index) {
    const allVeneers = getAllVeneers();
    
    return `
        <div class="p-3">
            <div class="alert alert-info mb-4">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Back Panel:</strong> Tebal thickness otomatis dipilih dari Core (MDF/PLYWOOD). Core type bisa dipilih sendiri.
            </div>

            <h6 class="mb-3"><i class="fas fa-ruler-combined me-2"></i>Dimensi Panel</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-3">
                    <label class="form-label">Lebar (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="back-width-${index}" 
                           onchange="calculateBackPanel(${index})" 
                           onkeyup="calculateBackPanel(${index})"
                           placeholder="Lebar" step="0.01">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Panjang (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="back-length-${index}" 
                           onchange="calculateBackPanel(${index})"
                           onkeyup="calculateBackPanel(${index})"
                           placeholder="Panjang" step="0.01">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness Total <span class="text-danger">*</span></label>
                    <select class="form-select" id="back-thickness-${index}" onchange="calculateBackPanel(${index})">
                        <option value="">-- Pilih Thickness --</option>
                        <option value="6">6mm (Core 5mm + Veneer 1mm)</option>
                        <option value="19">19mm (Core 18mm + Veneer 1mm)</option>
                        <option value="26">26mm (Core 25mm + Veneer 1mm)</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Qty <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="back-qty-${index}" value="1" min="1" 
                           onchange="calculateBackPanel(${index})"
                           onkeyup="calculateBackPanel(${index})">
                </div>
            </div>
            
            <div class="row g-3 mb-4">
                <div class="col-md-12">
                    <label class="form-label">M² Total (Auto-calculated)</label>
                    <input type="text" class="form-control bg-light" id="back-m2-${index}" readonly>
                </div>
            </div>

            <hr class="my-4">

            <!-- ✅ TAMBAHAN INI YANG HILANG! -->
            <div class="alert alert-success mb-3">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Thickness Info:</strong> <span id="back-core-thickness-info-${index}">Pilih Thickness Total terlebih dahulu</span>
            </div>

            <h6 class="mb-3"><i class="fas fa-layer-group me-2"></i>Sub Components</h6>

            <!-- 1. Core Panel (MDF/PLYWOOD Selectable) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-cube me-2"></i>1. Core Panel (Bisa Pilih MDF atau PLYWOOD)
                </h6>
                <div class="alert alert-warning alert-sm mb-3">
                    <i class="fas fa-hand-pointer me-2"></i>
                    <strong>Pilih Type Panel (MDF/PLYWOOD), tebal core otomatis dari Thickness Total</strong>
                </div>
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Type Panel <span class="text-danger">*</span></label>
                        <select class="form-select" id="back-core-type-${index}" onchange="calculateBackPanel(${index})">
                            <option value="">Pilih Type</option>
                            <option value="MDF">MDF</option>
                            <option value="PLYWOOD">PLYWOOD</option>
                        </select>
                        <small class="text-muted">Bisa pilih MDF atau PLYWOOD</small>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="back-core-qty-${index}" value="1" min="1" 
                               onchange="calculateBackPanel(${index})"
                               onkeyup="calculateBackPanel(${index})">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="back-core-m2-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Harga</label>
                        <div class="form-control bg-light" id="back-core-price-${index}">Rp 0</div>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-12 text-end">
                            <strong>Total:</strong> <span id="back-core-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 2. Veneer Face (ALWAYS PRESENT) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-square me-2"></i>2. Veneer Face (Always Present)
                </h6>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="back-veneer-face-pattern-${index}" onchange="calculateBackPanel(${index})">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Veneer Type (Bisa Diganti)</label>
                        <select class="form-select" id="back-veneer-face-type-${index}" onchange="calculateBackPanel(${index})">
                            <option value="">Pilih Veneer</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="back-veneer-face-qty-${index}" value="1" min="0" 
                               onchange="calculateBackPanel(${index})"
                               onkeyup="calculateBackPanel(${index})">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="back-veneer-face-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="back-veneer-face-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="back-veneer-face-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 3. Veneer Back (ALWAYS PRESENT) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-square me-2"></i>3. Veneer Back (Always Present)
                </h6>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="back-veneer-back-pattern-${index}" onchange="calculateBackPanel(${index})">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Veneer Type (Bisa Diganti)</label>
                        <select class="form-select" id="back-veneer-back-type-${index}" onchange="calculateBackPanel(${index})">
                            <option value="">Pilih Veneer</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="back-veneer-back-qty-${index}" value="1" min="0" 
                               onchange="calculateBackPanel(${index})"
                               onkeyup="calculateBackPanel(${index})">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="back-veneer-back-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="back-veneer-back-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="back-veneer-back-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total Back Panel:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-back-panel-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET FRAME CONTENT (FRONT & BACK)
// ============================================

function getFrameContent(index, componentName) {
    const prefix = componentName === 'Front Frame' ? 'front' : 'back';
    const solidTypes = ['MAHONI', 'JATI', 'MERBAU', 'SUNGKAI', 'ALBASIA', 'RUBBER', 'MIX'];
    
    return `
        <div class="p-3">
            <div class="alert alert-info mb-4">
                <i class="fas fa-info-circle me-2"></i>
                <strong>${componentName}:</strong> Input dimensi frame dan pilih solid type + tebal solid.
            </div>

            <h6 class="mb-3"><i class="fas fa-ruler-combined me-2"></i>Dimensi Frame</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-6">
                    <label class="form-label">Lebar (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="${prefix}-width-${index}" 
                           onchange="calculateFrame(${index}, '${prefix}')" 
                           onkeyup="calculateFrame(${index}, '${prefix}')"
                           placeholder="Lebar" step="0.01">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Panjang (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="${prefix}-length-${index}" 
                           onchange="calculateFrame(${index}, '${prefix}')"
                           onkeyup="calculateFrame(${index}, '${prefix}')"
                           placeholder="Panjang" step="0.01">
                </div>
            </div>

            <hr class="my-4">

            <h6 class="mb-3"><i class="fas fa-layer-group me-2"></i>Sub Components</h6>

            <!-- Core Solid -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-cube me-2"></i>Core Solid
                </h6>
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Solid Type <span class="text-danger">*</span></label>
                        <select class="form-select" id="${prefix}-solid-type-${index}" onchange="onFrameSolidTypeChange(${index}, '${prefix}')">
                            <option value="">Pilih Solid Type</option>
                            ${solidTypes.map(type => `<option value="${type}">${type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Tebal Solid <span class="text-danger">*</span></label>
                        <select class="form-select" id="${prefix}-solid-thickness-${index}" onchange="calculateFrame(${index}, '${prefix}')">
                            <option value="">Pilih Tebal</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-solid-qty-${index}" value="1" min="0" 
                               onchange="calculateFrame(${index}, '${prefix}')"
                               onkeyup="calculateFrame(${index}, '${prefix}')">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-solid-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="${prefix}-solid-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="${prefix}-solid-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total ${componentName}:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-${prefix}-frame-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET FACE DRAWER CONTENT
// ============================================

function getFaceDrawerContent(index) {
    const solids = getSolidsByType(initialFormData.solid_type);
    const mdfPanels = getPanelsByType('MDF');
    const allVeneers = getAllVeneers();
    
    return `
        <div class="p-3">
            <div class="alert alert-danger mb-4">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>PENTING!</strong> Qty Face Drawer akan mempengaruhi semua komponen drawer lainnya. Pastikan input dengan benar!
            </div>

            <h6 class="mb-3"><i class="fas fa-ruler-combined me-2"></i>Dimensi Face Drawer</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-3">
                    <label class="form-label">Lebar (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="face-width-${index}" 
                           onchange="calculateFaceDrawer(${index})" 
                           onkeyup="calculateFaceDrawer(${index})"
                           placeholder="Lebar" step="0.01">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Panjang (mm) <span class="text-danger">*</span></label>
                    <input type="number" class="form-control" id="face-length-${index}" 
                           onchange="calculateFaceDrawer(${index})"
                           onkeyup="calculateFaceDrawer(${index})"
                           placeholder="Panjang" step="0.01">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness Total <span class="text-danger">*</span></label>
                    <select class="form-select" id="face-thickness-${index}" onchange="calculateFaceDrawer(${index})">
                        <option value="">-- Pilih Thickness --</option>
                        <option value="18">18mm (Core 15mm + Veneer 1mm + Patching 2mm)</option>
                        <option value="21">21mm (Core 18mm + Veneer 1mm + Patching 2mm)</option>
                        <option value="28">28mm (Core 25mm + Veneer 1mm + Patching 2mm)</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Qty <span class="text-danger">*</span></label>
                    <input type="number" class="form-control bg-warning" id="face-qty-${index}" value="1" min="1" 
                           onchange="calculateFaceDrawer(${index})"
                           onkeyup="calculateFaceDrawer(${index})">
                    <small class="text-danger fw-bold">⚠️ Affects all drawers!</small>
                </div>
            </div>
            
            <div class="row g-3 mb-4">
                <div class="col-md-12">
                    <label class="form-label">M² Total (Auto-calculated)</label>
                    <input type="text" class="form-control bg-light" id="face-m2-${index}" readonly>
                </div>
            </div>

            <hr class="my-4">

            <h6 class="mb-3"><i class="fas fa-layer-group me-2"></i>Sub Components</h6>

            <!-- 1. Patching Panel 1 (Core Panel - AUTO) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-cube me-2"></i>1. Patching Panel 1 (Core Panel - Auto Calculated)
                </h6>
                <div class="alert alert-success alert-sm mb-3">
                    <i class="fas fa-magic me-2"></i>
                    <strong>Qty otomatis = Face Drawer Qty</strong>
                </div>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Tebal Core (Auto)</label>
                        <input type="text" class="form-control bg-light" id="face-core-thickness-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Type Panel (Locked)</label>
                        <input type="text" class="form-control bg-light" id="face-core-type-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty (Auto)</label>
                        <input type="number" class="form-control bg-light" id="face-patching-panel-qty-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="face-patching-panel-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="face-patching-panel-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="face-patching-panel-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 2. Patching Solid 2 (PANJANG) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-ruler-horizontal me-2"></i>2. Patching Solid 2 (PANJANG)
                </h6>
                <div class="alert alert-success alert-sm mb-3">
                    <i class="fas fa-magic me-2"></i>
                    <strong>Qty otomatis = Face Drawer Qty × 2</strong>
                </div>
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Lebar Pilihan (mm)</label>
                        <input type="number" class="form-control" id="face-patching-solid2-width-${index}" value="18" 
                               onchange="calculateFaceDrawer(${index})"
                               onkeyup="calculateFaceDrawer(${index})"
                               step="0.01">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Tebal Solid</label>
                        <select class="form-select" id="face-patching-solid2-thickness-${index}" onchange="calculateFaceDrawer(${index})">
                            <option value="">Pilih Tebal</option>
                            ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty (Auto)</label>
                        <input type="number" class="form-control bg-light" id="face-patching-solid2-qty-${index}" readonly>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="face-patching-solid2-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="face-patching-solid2-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="face-patching-solid2-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 3. Patching Solid 3 (LEBAR) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-ruler-vertical me-2"></i>3. Patching Solid 3 (LEBAR)
                </h6>
                <div class="alert alert-success alert-sm mb-3">
                    <i class="fas fa-magic me-2"></i>
                    <strong>Qty otomatis = Face Drawer Qty × 2</strong>
                </div>
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Lebar Pilihan (mm)</label>
                        <input type="number" class="form-control" id="face-patching-solid3-width-${index}" value="18" 
                               onchange="calculateFaceDrawer(${index})"
                               onkeyup="calculateFaceDrawer(${index})"
                               step="0.01">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Tebal Solid</label>
                        <select class="form-select" id="face-patching-solid3-thickness-${index}" onchange="calculateFaceDrawer(${index})">
                            <option value="">Pilih Tebal</option>
                            ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty (Auto)</label>
                        <input type="number" class="form-control bg-light" id="face-patching-solid3-qty-${index}" readonly>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="face-patching-solid3-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="face-patching-solid3-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="face-patching-solid3-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 4. Crossgrain (MDF) -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-grip-lines me-2"></i>4. Crossgrain (MDF)
                </h6>
                <div class="alert alert-success alert-sm mb-3">
                    <i class="fas fa-magic me-2"></i>
                    <strong>Qty otomatis = Face Drawer Qty × 2</strong>
                </div>
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Tebal MDF</label>
                        <select class="form-select" id="face-crossgrain-thickness-${index}" onchange="calculateFaceDrawer(${index})">
                            <option value="">Pilih Tebal</option>
                            ${mdfPanels.map(p => `<option value="${p.id}" data-thickness="${p.thickness}" data-price="${p.price}">${p.thickness}mm - ${formatRupiah(p.price)}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty (Auto)</label>
                        <input type="number" class="form-control bg-light" id="face-crossgrain-qty-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="face-crossgrain-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="face-crossgrain-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="face-crossgrain-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 5. Veneer Face -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-square me-2"></i>5. Veneer Face
                </h6>
                <div class="alert alert-success alert-sm mb-3">
                    <i class="fas fa-magic me-2"></i>
                    <strong>Qty otomatis = Face Drawer Qty</strong>
                </div>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="face-veneer-face-pattern-${index}" onchange="calculateFaceDrawer(${index})">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Veneer Type (Bisa Diganti)</label>
                        <select class="form-select" id="face-veneer-face-type-${index}" onchange="calculateFaceDrawer(${index})">
                            <option value="">Pilih Veneer</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty (Auto)</label>
                        <input type="number" class="form-control bg-light" id="face-veneer-face-qty-${index}" readonly>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="face-veneer-face-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="face-veneer-face-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="face-veneer-face-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 6. Veneer Back -->
            <div class="sub-component-card mb-3">
                <h6 class="sub-component-title">
                    <i class="fas fa-square me-2"></i>6. Veneer Back
                </h6>
                <div class="alert alert-success alert-sm mb-3">
                    <i class="fas fa-magic me-2"></i>
                    <strong>Qty otomatis = Face Drawer Qty</strong>
                </div>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="face-veneer-back-pattern-${index}" onchange="calculateFaceDrawer(${index})">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Veneer Type (Bisa Diganti)</label>
                        <select class="form-select" id="face-veneer-back-type-${index}" onchange="calculateFaceDrawer(${index})">
                            <option value="">Pilih Veneer</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty (Auto)</label>
                        <input type="number" class="form-control bg-light" id="face-veneer-back-qty-${index}" readonly>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="face-veneer-back-m2-${index}" readonly>
                    </div>
                </div>
                <div class="calculated-info mt-3">
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Harga:</strong> <span id="face-veneer-back-price-${index}" class="text-primary">Rp 0</span>
                        </div>
                        <div class="col-md-6 text-end">
                            <strong>Total:</strong> <span id="face-veneer-back-total-${index}" class="text-success fw-bold">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total Face Drawer:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-face-drawer-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}
// ============================================
// ✅ GET DRAWER CONTENT (FRONT, SIDE L, BACK, BASE)
// ============================================

function getDrawerContent(index, componentName) {
    const prefixMap = {
        'Front Drawer': 'frontdrawer',
        'Side Drawer L': 'sidedrawerl',
        'Back Drawer': 'backdrawer',
        'Base Drawer': 'basedrawer'
    };
    
    const prefix = prefixMap[componentName];
    
    const drawerFunctionMap = {
        'Front Drawer': 'getFrontBackDrawers',
        'Side Drawer L': 'getSideDrawers',
        'Back Drawer': 'getFrontBackDrawers',
        'Base Drawer': 'getBaseDrawers'
    };
    
    const drawerFunction = drawerFunctionMap[componentName];
    const drawers = window[drawerFunction](initialFormData.box_drawer_type);
    
    return `
        <div class="p-3">
            <div class="alert alert-warning mb-4">
                <i class="fas fa-info-circle me-2"></i>
                <strong>${componentName}:</strong> Qty otomatis mengikuti Face Drawer Qty. Material: <strong>${initialFormData.box_drawer_type}</strong>
            </div>

            <div class="alert alert-info mb-4">
                <i class="fas fa-calculator me-2"></i>
                <strong>Current Face Drawer Qty:</strong> <span class="badge bg-primary fs-6">${faceDrawerQty}</span>
                <br><small>Qty untuk ${componentName} akan otomatis = ${faceDrawerQty}</small>
            </div>

            <h6 class="mb-3"><i class="fas fa-ruler-combined me-2"></i>Pilih Dimensi ${componentName}</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-12">
                    <label class="form-label">Dimensi (Lebar × Panjang) <span class="text-danger">*</span></label>
                    <select class="form-select form-select-lg" id="${prefix}-dimension-${index}" onchange="calculateDrawer(${index}, '${prefix}')">
                        <option value="">-- Pilih Dimensi --</option>
                        ${drawers.map(d => `<option value="${d.id}" data-width="${d.width}" data-length="${d.length}" data-price="${d.price}">${d.width}mm × ${d.length}mm - ${formatRupiah(d.price)}</option>`).join('')}
                    </select>
                </div>
            </div>

            <hr class="my-4">

            <h6 class="mb-3"><i class="fas fa-info-circle me-2"></i>Calculation Details</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-4">
                    <label class="form-label">Qty (Auto from Face Drawer)</label>
                    <input type="text" class="form-control bg-warning text-dark fw-bold" value="${faceDrawerQty}" readonly>
                </div>
                <div class="col-md-4">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="${prefix}-m2-${index}" readonly>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Harga/M²</label>
                    <div class="form-control bg-light" id="${prefix}-price-${index}">Rp 0</div>
                </div>
            </div>

            <div class="calculated-info">
                <div class="row">
                    <div class="col-md-12 text-end">
                        <strong>Total ${componentName}:</strong> <span id="${prefix}-total-${index}" class="text-success fw-bold fs-5">Rp 0</span>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total ${componentName}:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-${prefix}-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET SIDE DRAWER R CONTENT (COPY OR MANUAL MODE)
// ============================================

function getSideDrawerRContent(index) {
    const prefix = 'sidedrawerr';
    const drawers = getSideDrawers(initialFormData.box_drawer_type);
    
    return `
        <div class="p-3">
            <div class="alert alert-warning mb-4">
                <i class="fas fa-copy me-2"></i>
                <strong>Side Drawer R:</strong> Pilih mode Copy dari Side Drawer L atau Input Manual. Material: <strong>${initialFormData.box_drawer_type}</strong>
            </div>

            <div class="alert alert-info mb-4">
                <i class="fas fa-calculator me-2"></i>
                <strong>Current Face Drawer Qty:</strong> <span class="badge bg-primary fs-6">${faceDrawerQty}</span>
                <br><small>Qty untuk Side Drawer R akan otomatis = ${faceDrawerQty}</small>
            </div>

            <h6 class="mb-3"><i class="fas fa-cog me-2"></i>Pilih Mode Input</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-12">
                    <label class="form-label">Mode <span class="text-danger">*</span></label>
                    <select class="form-select form-select-lg" id="${prefix}-mode-${index}" onchange="onSideDrawerRModeChange(${index})">
                        <option value="">-- Pilih Mode --</option>
                        <option value="copy">📋 Copy dari Side Drawer L</option>
                        <option value="manual">✏️ Input Manual</option>
                    </select>
                    <small class="text-muted">Pilih "Copy" untuk menggunakan data Side Drawer L, atau "Manual" untuk input sendiri</small>
                </div>
            </div>

            <hr class="my-4">

            <!-- MANUAL INPUTS (Hidden by default) -->
            <div id="${prefix}-manual-inputs-${index}" style="display: none;">
                <h6 class="mb-3"><i class="fas fa-ruler-combined me-2"></i>Pilih Dimensi Side Drawer R</h6>
                <div class="row g-3 mb-4">
                    <div class="col-md-12">
                        <label class="form-label">Dimensi (Lebar × Panjang) <span class="text-danger">*</span></label>
                        <select class="form-select form-select-lg" id="${prefix}-dimension-${index}" onchange="calculateDrawer(${index}, '${prefix}')">
                            <option value="">-- Pilih Dimensi --</option>
                            ${drawers.map(d => `<option value="${d.id}" data-width="${d.width}" data-length="${d.length}" data-price="${d.price}">${d.width}mm × ${d.length}mm - ${formatRupiah(d.price)}</option>`).join('')}
                        </select>
                    </div>
                </div>

                <hr class="my-4">

                <h6 class="mb-3"><i class="fas fa-info-circle me-2"></i>Calculation Details</h6>
                <div class="row g-3 mb-4">
                    <div class="col-md-4">
                        <label class="form-label">Qty (Auto from Face Drawer)</label>
                        <input type="text" class="form-control bg-warning text-dark fw-bold" value="${faceDrawerQty}" readonly>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-m2-${index}" readonly>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Harga/M²</label>
                        <div class="form-control bg-light" id="${prefix}-price-${index}">Rp 0</div>
                    </div>
                </div>

                <div class="calculated-info">
                    <div class="row">
                        <div class="col-md-12 text-end">
                            <strong>Total Side Drawer R:</strong> <span id="${prefix}-total-${index}" class="text-success fw-bold fs-5">Rp 0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total Side Drawer R:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-${prefix}-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET SUPPORT DRAWER CONTENT
// ============================================

function getSupportDrawerContent(index) {
    return `
        <div class="p-3">
            <div class="alert alert-info mb-4">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Support Drawer:</strong> Input qty support drawer. Harga fixed Rp 2.200 per unit.
            </div>

            <h6 class="mb-3"><i class="fas fa-calculator me-2"></i>Support Drawer Calculation</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-6">
                    <label class="form-label">Qty <span class="text-danger">*</span></label>
                    <input type="number" class="form-control form-control-lg" id="support-qty-${index}" 
                           value="1" min="1" 
                           onchange="calculateSupportDrawer(${index})"
                           onkeyup="calculateSupportDrawer(${index})"
                           placeholder="Masukkan qty">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Harga per Unit (Fixed)</label>
                    <input type="text" class="form-control form-control-lg bg-light" value="Rp 2.200" readonly>
                </div>
            </div>

            <div class="alert alert-secondary">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Info:</strong> Width = 11mm, Material = Mix
            </div>

            <hr class="my-4">

            <div class="calculated-info">
                <div class="row">
                    <div class="col-md-12 text-end">
                        <strong>Total Support Drawer:</strong> <span id="support-total-${index}" class="text-success fw-bold fs-5">Rp 0</span>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total Support Drawer:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-support-drawer-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET CLEAT DRAWER CONTENT
// ============================================

function getCleatDrawerContent(index) {
    return `
        <div class="p-3">
            <div class="alert alert-info mb-4">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Cleat Drawer:</strong> Input qty cleat drawer. Harga fixed Rp 400 per unit.
            </div>

            <h6 class="mb-3"><i class="fas fa-calculator me-2"></i>Cleat Drawer Calculation</h6>
            <div class="row g-3 mb-4">
                <div class="col-md-6">
                    <label class="form-label">Qty <span class="text-danger">*</span></label>
                    <input type="number" class="form-control form-control-lg" id="cleat-qty-${index}" 
                           value="1" min="1" 
                           onchange="calculateCleatDrawer(${index})"
                           onkeyup="calculateCleatDrawer(${index})"
                           placeholder="Masukkan qty">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Harga per Unit (Fixed)</label>
                    <input type="text" class="form-control form-control-lg bg-light" value="Rp 400" readonly>
                </div>
            </div>

            <div class="alert alert-secondary">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Info:</strong> Width = 11mm, Material = Mix
            </div>

            <hr class="my-4">

            <div class="calculated-info">
                <div class="row">
                    <div class="col-md-12 text-end">
                        <strong>Total Cleat Drawer:</strong> <span id="cleat-total-${index}" class="text-success fw-bold fs-5">Rp 0</span>
                    </div>
                </div>
            </div>

            <!-- Component Total -->
            <div class="total-component mt-4">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Total Cleat Drawer:</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h4 class="mb-0 text-success" id="total-cleat-drawer-${index}">Rp 0</h4>
                    </div>
                </div>
            </div>
        </div>
    `;
}

console.log('✅ All Modal Content Functions Loaded Successfully');
</script>
@endpush
