@extends('layouts.app')

@section('title', 'Buat Invoice Handle - Cost Furniture')

@section('page-title', 'Buat Invoice Handle')

@section('content')

<!-- Alert jika tidak ada Invoice BOM -->
@if($invoiceBoms->isEmpty())
<div class="alert alert-warning">
    <i class="fas fa-exclamation-triangle me-2"></i>
    <strong>Tidak ada Invoice BOM yang tersedia!</strong> 
    Pastikan ada Invoice BOM yang sudah di-fix dan belum memiliki data handle.
    <a href="{{ route('invoice.bom.index') }}" class="alert-link">Lihat Invoice BOM</a>
</div>
@endif

<div class="card modern-card">
    <div class="card-header">
        <h5><i class="fas fa-hand-paper me-2"></i>Form Buat Invoice Handle</h5>
    </div>
    <div class="card-body">
        <form id="handleForm">
            @csrf
            
            <!-- Step 1: Pilih Invoice BOM -->
            <div class="section-card">
                <h6 class="section-title"><i class="fas fa-file-invoice me-2"></i>PILIH INVOICE BOM</h6>
                
                <div class="row">
                    <div class="col-md-12">
                        <label class="form-label">Invoice BOM <span class="text-danger">*</span></label>
                        <select name="invoice_bom_id" id="invoice_bom_id" class="form-select" required>
                            <option value="">-- Pilih Invoice BOM --</option>
                            @foreach($invoiceBoms as $bom)
                            <option value="{{ $bom->id }}" 
                                    data-product-name="{{ $bom->product_name }}"
                                    data-product-code="{{ $bom->product_code }}"
                                    data-product-type="{{ $bom->product_type }}"
                                    data-grand-total="{{ $bom->grand_total }}">
                                BOM #{{ $bom->id }} - {{ $bom->product_name }} ({{ $bom->product_code }}) - {{ \App\Helpers\CasegoodHelper::formatRupiah($bom->grand_total) }}
                            </option>
                            @endforeach
                        </select>
                    </div>
                </div>
                
                <!-- BOM Info Display -->
                <div id="bom-info" style="display: none;" class="mt-3">
                    <div class="info-box">
                        <div class="row">
                            <div class="col-md-3">
                                <strong>Product Name:</strong>
                                <div id="bom-product-name">-</div>
                            </div>
                            <div class="col-md-3">
                                <strong>Product Code:</strong>
                                <div id="bom-product-code">-</div>
                            </div>
                            <div class="col-md-3">
                                <strong>Product Type:</strong>
                                <div id="bom-product-type">-</div>
                            </div>
                            <div class="col-md-3">
                                <strong>BOM Grand Total:</strong>
                                <div id="bom-grand-total" class="text-success fw-bold">-</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Step 2: Manual Items -->
            <div class="section-card">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h6 class="section-title mb-0">
                        <i class="fas fa-edit me-2"></i>HANDLE ITEMS (Manual Input)
                    </h6>
                    <button type="button" class="btn btn-sm btn-primary" onclick="addManualItem()">
                        <i class="fas fa-plus-circle me-2"></i>Tambah Item Handle
                    </button>
                </div>
                
                <div id="manual-items-container">
                    <!-- Manual items will be added here -->
                </div>
                
                <div id="no-items-message" class="text-center py-4">
                    <i class="fas fa-info-circle fa-2x text-muted mb-2"></i>
                    <p class="text-muted">Belum ada item handle. Klik "Tambah Item Handle" untuk mulai.</p>
                </div>
            </div>
            
            <!-- Grand Total -->
            <div class="section-card bg-gradient">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>GRAND TOTAL HANDLE</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h3 class="mb-0 text-white" id="grand-total-display">Rp 0</h3>
                        <input type="hidden" name="grand_total" id="grand_total" value="0">
                    </div>
                </div>
            </div>
            
            <!-- Action Buttons -->
            <div class="d-flex gap-2 justify-content-end">
                <a href="{{ route('invoice.handle.index') }}" class="btn btn-secondary">
                    <i class="fas fa-times me-2"></i>Batal
                </a>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save me-2"></i>Simpan Invoice Handle
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Modal Input Manual Item -->
<div class="modal fade" id="manualItemModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title"><i class="fas fa-plus-circle me-2"></i>Tambah Item Handle</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <input type="hidden" id="edit-index">
                
                <div class="mb-3">
                    <label class="form-label">Material Code <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="material-code" placeholder="e.g. HANDLE-001" required>
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Material Description <span class="text-danger">*</span></label>
                    <textarea class="form-control" id="material-desc" rows="2" placeholder="Deskripsi material..." required></textarea>
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <label class="form-label">Qty <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="material-qty" min="1" value="1" required>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Price <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="material-price" min="0" step="0.01" placeholder="0" required>
                    </div>
                </div>
                
                <div class="mt-3">
                    <strong>Total:</strong> <span id="material-total" class="text-success fw-bold">Rp 0</span>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-primary" onclick="saveManualItem()">
                    <i class="fas fa-save me-2"></i>Simpan
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Loading Overlay -->
<div class="loading-overlay" id="loadingOverlay" style="display: none;">
    <div class="spinner-border text-light" role="status">
        <span class="visually-hidden">Loading...</span>
    </div>
    <p class="text-light mt-3">Menyimpan invoice handle...</p>
</div>

<style>
/* Modern Card */
.modern-card {
    border: none;
    border-radius: 15px;
    box-shadow: 0 5px 20px rgba(0,0,0,0.05);
}

.modern-card .card-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 15px 15px 0 0;
    padding: 20px 25px;
    border: none;
}

.modern-card .card-header h5 {
    margin: 0;
    font-weight: 700;
    color: white;
}

.modern-card .card-body {
    padding: 30px;
}

/* Section Card */
.section-card {
    background: #fff;
    border: 2px solid #f0f0f0;
    border-radius: 12px;
    padding: 25px;
    margin-bottom: 25px;
}

.section-title {
    font-weight: 700;
    color: #667eea;
    font-size: 16px;
    margin-bottom: 20px;
}

/* Info Box */
.info-box {
    background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%);
    border: 2px solid rgba(102, 126, 234, 0.2);
    border-radius: 10px;
    padding: 20px;
}

.info-box strong {
    color: #666;
    font-size: 12px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.info-box div:not(strong) {
    color: #2c3e50;
    font-size: 16px;
    font-weight: 600;
    margin-top: 5px;
}

/* Material Card */
.material-card {
    background: #fff;
    border: 2px solid #e9ecef;
    border-left: 4px solid #667eea;
    border-radius: 8px;
    padding: 15px;
    margin-bottom: 10px;
    transition: all 0.2s ease;
    position: relative;
}

.material-card:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    transform: translateY(-1px);
}

.material-code {
    font-size: 11px;
    color: #666;
    font-weight: 600;
    background: #f8f9fa;
    padding: 2px 6px;
    border-radius: 3px;
    display: inline-block;
    margin-bottom: 5px;
}

.material-desc {
    font-size: 14px;
    color: #2c3e50;
    font-weight: 500;
    margin-bottom: 5px;
}

.material-price {
    font-size: 12px;
    color: #27ae60;
    font-weight: 600;
}

.material-qty,
.material-total {
    min-width: 120px;
}

.material-total {
    text-align: right;
}

.material-total .fw-bold {
    font-size: 14px;
    color: #667eea;
}

.material-actions {
    display: flex;
    align-items: center;
}

/* Grand Total */
.bg-gradient {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

/* Form Controls */
.form-label {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 8px;
    font-size: 14px;
}

.form-control, .form-select {
    padding: 12px 15px;
    border: 2px solid #e0e0e0;
    border-radius: 8px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.form-control:focus, .form-select:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.1);
    outline: none;
}

/* Buttons */
.btn {
    padding: 12px 24px;
    border-radius: 8px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border: none;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #764ba2 0%, #667eea 100%);
}

.btn-sm {
    padding: 6px 12px;
    font-size: 13px;
}

/* Loading Overlay */
.loading-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.7);
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    z-index: 9999;
}

/* Alert */
.alert {
    border-radius: 10px;
    border: none;
    margin-bottom: 20px;
}

.alert-warning {
    background: linear-gradient(135deg, rgba(241, 196, 15, 0.1) 0%, rgba(243, 156, 18, 0.1) 100%);
    border-left: 4px solid #f39c12;
    color: #d68910;
}

/* Modal */
.modal-header.bg-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
}
</style>

<script>
// ============================================
// GLOBAL VARIABLES
// ============================================

let manualItems = [];
let editIndex = null;

// ============================================
// INVOICE BOM SELECTION
// ============================================

document.getElementById('invoice_bom_id').addEventListener('change', function() {
    const selectedOption = this.options[this.selectedIndex];
    
    if (this.value) {
        document.getElementById('bom-product-name').textContent = selectedOption.dataset.productName;
        document.getElementById('bom-product-code').textContent = selectedOption.dataset.productCode;
        document.getElementById('bom-product-type').textContent = selectedOption.dataset.productType;
        document.getElementById('bom-grand-total').textContent = formatRupiah(parseFloat(selectedOption.dataset.grandTotal));
        document.getElementById('bom-info').style.display = 'block';
    } else {
        document.getElementById('bom-info').style.display = 'none';
    }
});

// ============================================
// ADD MANUAL ITEM
// ============================================

function addManualItem() {
    editIndex = null;
    
    // Reset modal
    document.getElementById('edit-index').value = '';
    document.getElementById('material-code').value = '';
    document.getElementById('material-desc').value = '';
    document.getElementById('material-qty').value = 1;
    document.getElementById('material-price').value = '';
    document.getElementById('material-total').textContent = 'Rp 0';
    
    // Show modal
    const modal = new bootstrap.Modal(document.getElementById('manualItemModal'));
    modal.show();
}

// Auto calculate total in modal
document.getElementById('material-qty').addEventListener('input', calculateMaterialTotal);
document.getElementById('material-price').addEventListener('input', calculateMaterialTotal);

function calculateMaterialTotal() {
    const qty = parseFloat(document.getElementById('material-qty').value) || 0;
    const price = parseFloat(document.getElementById('material-price').value) || 0;
    const total = qty * price;
    
    document.getElementById('material-total').textContent = formatRupiah(total);
}

function saveManualItem() {
    const materialCode = document.getElementById('material-code').value.trim();
    const materialDesc = document.getElementById('material-desc').value.trim();
    const qty = parseFloat(document.getElementById('material-qty').value) || 0;
    const price = parseFloat(document.getElementById('material-price').value) || 0;
    
    if (!materialCode || !materialDesc || qty <= 0 || price <= 0) {
        alert('❌ Semua field harus diisi dengan benar!');
        return;
    }
    
    const total = qty * price;
    
    const editIdx = document.getElementById('edit-index').value;
    
    if (editIdx !== '') {
        // Edit existing item
        const index = parseInt(editIdx);
        manualItems[index] = {
            material_code: materialCode,
            material_description: materialDesc,
            qty: qty,
            price: price,
            total: total
        };
    } else {
        // Add new item
        manualItems.push({
            material_code: materialCode,
            material_description: materialDesc,
            qty: qty,
            price: price,
            total: total
        });
    }
    
    renderManualItems();
    updateGrandTotal();
    
    // Close modal
    bootstrap.Modal.getInstance(document.getElementById('manualItemModal')).hide();
    
    console.log('✅ Manual items:', manualItems);
}

function renderManualItems() {
    const container = document.getElementById('manual-items-container');
    
    if (manualItems.length === 0) {
        document.getElementById('no-items-message').style.display = 'block';
        return;
    }
    
    document.getElementById('no-items-message').style.display = 'none';
    
    container.innerHTML = manualItems.map((item, index) => `
        <div class="material-card">
            <div class="d-flex align-items-start gap-3">
                <div class="material-info flex-grow-1">
                    <div class="material-code">${item.material_code}</div>
                    <div class="material-desc">${item.material_description}</div>
                    <div class="material-price">@ ${formatRupiah(item.price)}</div>
                </div>
                <div class="material-qty">
                    <label class="form-label small mb-1">Qty:</label>
                    <div class="fw-bold">${item.qty}</div>
                </div>
                <div class="material-total">
                    <label class="form-label small mb-1">Total:</label>
                    <div class="fw-bold">${formatRupiah(item.total)}</div>
                </div>
                <div class="material-actions">
                    <button type="button" 
                            class="btn btn-sm btn-warning me-1 mt-4" 
                            onclick="editManualItem(${index})"
                            title="Edit">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button type="button" 
                            class="btn btn-sm btn-danger mt-4" 
                            onclick="removeManualItem(${index})"
                            title="Hapus">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
        </div>
    `).join('');
}

function editManualItem(index) {
    const item = manualItems[index];
    
    document.getElementById('edit-index').value = index;
    document.getElementById('material-code').value = item.material_code;
    document.getElementById('material-desc').value = item.material_description;
    document.getElementById('material-qty').value = item.qty;
    document.getElementById('material-price').value = item.price;
    calculateMaterialTotal();
    
    const modal = new bootstrap.Modal(document.getElementById('manualItemModal'));
    modal.show();
}

function removeManualItem(index) {
    if (!confirm('Yakin ingin menghapus item ini?')) return;
    
    manualItems.splice(index, 1);
    
    renderManualItems();
    updateGrandTotal();
}

function updateGrandTotal() {
    const grandTotal = manualItems.reduce((sum, item) => sum + item.total, 0);
    
    document.getElementById('grand-total-display').textContent = formatRupiah(grandTotal);
    document.getElementById('grand_total').value = grandTotal;
}

// ============================================
// FORMAT RUPIAH
// ============================================

function formatRupiah(amount) {
    return 'Rp ' + Math.round(amount).toLocaleString('id-ID');
}

// ============================================
// SUBMIT FORM
// ============================================

document.getElementById('handleForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const invoiceBomId = document.getElementById('invoice_bom_id').value;
    
    if (!invoiceBomId) {
        alert('❌ Pilih Invoice BOM terlebih dahulu!');
        return;
    }
    
    if (manualItems.length === 0) {
        alert('❌ Minimal harus ada 1 item handle!');
        return;
    }
    
    // Prepare data for submission
    const handleDataToSave = {
        items: manualItems,
        grand_total: parseFloat(document.getElementById('grand_total').value)
    };
    
    console.log('Submitting Handle Data:', handleDataToSave);
    
    document.getElementById('loadingOverlay').style.display = 'flex';
    
    fetch('{{ route("invoice.handle.store") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: JSON.stringify({
            invoice_bom_id: invoiceBomId,
            handle_data: JSON.stringify(handleDataToSave),
            grand_total: handleDataToSave.grand_total
        })
    })
    .then(response => response.json())
    .then(data => {
        console.log('Server Response:', data);
        
        if (data.success) {
            alert('✅ Invoice Handle berhasil dibuat!');
            window.location.href = '{{ route("invoice.handle.index") }}';
        } else {
            throw new Error(data.message || 'Failed to create invoice handle');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('❌ Terjadi kesalahan: ' + error.message);
    })
    .finally(() => {
        document.getElementById('loadingOverlay').style.display = 'none';
    });
});

// ============================================
// PAGE LOAD
// ============================================

document.addEventListener('DOMContentLoaded', function() {
    console.log('Handle Create Page Loaded');
});
</script>

@endsection