<?php
require_once '../config/connection.php';
checkUserType(['borrower']);

$message = '';
$message_type = '';
$asset_data = null;
$borrower_organization = getBorrowerOrganization($pdo, $_SESSION['employee_id']);

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['ajax'])) {
    header('Content-Type: application/json');
    
    if ($_POST['action'] == 'scan_asset') {
        $asset_number = sanitize($_POST['asset_number']);
        
        $stmt = $pdo->prepare("
            SELECT a.*, 
                   CASE 
                       WHEN a.status_borrower = 'available' AND a.kondisi IN ('baik', 'rusak') THEN 1
                       ELSE 0
                   END as is_available
            FROM assets a 
            WHERE a.asset_number = ?
        ");
        $stmt->execute([$asset_number]);
        $asset = $stmt->fetch();
        
        if ($asset) {
            echo json_encode([
                'success' => true,
                'asset' => $asset,
                'message' => $asset['is_available'] ? 'Asset ditemukan dan tersedia!' : 'Asset ditemukan tapi tidak tersedia untuk dipinjam.'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Asset tidak ditemukan!'
            ]);
        }
        exit;
    }
    
    if ($_POST['action'] == 'add_to_cart') {
        try {
            $asset_id = (int)$_POST['asset_id'];
            $session_id = session_id();
            
            // Check if asset is available
            if (isAssetAvailableForBorrow($pdo, $asset_id)) {
                // Check if already in cart
                $stmt = $pdo->prepare("
                    SELECT id FROM borrow_cart 
                    WHERE borrower_employee_id = ? AND asset_id = ?
                ");
                $stmt->execute([$_SESSION['employee_id'], $asset_id]);
                
                if (!$stmt->fetch()) {
                    // Add to cart
                    $stmt = $pdo->prepare("
                        INSERT INTO borrow_cart (session_id, borrower_employee_id, asset_id) 
                        VALUES (?, ?, ?)
                    ");
                    $stmt->execute([$session_id, $_SESSION['employee_id'], $asset_id]);
                    
                    echo json_encode(['success' => true, 'message' => 'Asset berhasil ditambahkan ke keranjang!']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Asset sudah ada di keranjang!']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Asset tidak tersedia untuk dipinjam!']);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
        exit;
    }
    
    if ($_POST['action'] == 'remove_from_cart') {
        $cart_id = (int)$_POST['cart_id'];
        
        $stmt = $pdo->prepare("
            DELETE FROM borrow_cart 
            WHERE id = ? AND borrower_employee_id = ?
        ");
        $stmt->execute([$cart_id, $_SESSION['employee_id']]);
        
        echo json_encode(['success' => true, 'message' => 'Asset dihapus dari keranjang!']);
        exit;
    }
    
    if ($_POST['action'] == 'get_cart') {
        try {
            // Check if borrow_cart table exists, if not create it temporarily in session
            $stmt = $pdo->prepare("SHOW TABLES LIKE 'borrow_cart'");
            $stmt->execute();
            $table_exists = $stmt->fetch();
            
            if (!$table_exists) {
                // Create the table if it doesn't exist
                $pdo->exec("
                    CREATE TABLE `borrow_cart` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `session_id` varchar(255) NOT NULL,
                        `borrower_employee_id` varchar(20) NOT NULL,
                        `asset_id` int(11) NOT NULL,
                        `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
                        PRIMARY KEY (`id`),
                        KEY `asset_id` (`asset_id`),
                        KEY `borrower_employee_id` (`borrower_employee_id`),
                        CONSTRAINT `borrow_cart_ibfk_1` FOREIGN KEY (`asset_id`) REFERENCES `assets` (`id`) ON DELETE CASCADE,
                        CONSTRAINT `borrow_cart_ibfk_2` FOREIGN KEY (`borrower_employee_id`) REFERENCES `borrowers` (`employee_id`) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                ");
            }
            
            $stmt = $pdo->prepare("
                SELECT bc.id as cart_id, a.* 
                FROM borrow_cart bc
                JOIN assets a ON bc.asset_id = a.id
                WHERE bc.borrower_employee_id = ?
                ORDER BY bc.created_at DESC
            ");
            $stmt->execute([$_SESSION['employee_id']]);
            $cart_items = $stmt->fetchAll();
            
            echo json_encode(['success' => true, 'items' => $cart_items]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage(), 'items' => []]);
        }
        exit;
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST' && !isset($_POST['ajax'])) {
    if (isset($_POST['submit_borrow'])) {
        $preference_pro = sanitize($_POST['preference_pro']);
        $mesin = sanitize($_POST['mesin']);
        $notes = sanitize($_POST['notes']);
        
        if (empty($preference_pro)) {
            $message = 'Preference Pro harus diisi!';
            $message_type = 'error';
        } elseif (empty($mesin)) {
            $message = 'Mesin harus diisi!';
            $message_type = 'error';
        } else {
            // Get cart items
            $stmt = $pdo->prepare("
                SELECT bc.id as cart_id, bc.asset_id, a.asset_name, a.asset_number
                FROM borrow_cart bc
                JOIN assets a ON bc.asset_id = a.id
                WHERE bc.borrower_employee_id = ?
            ");
            $stmt->execute([$_SESSION['employee_id']]);
            $cart_items = $stmt->fetchAll();
            
            if (empty($cart_items)) {
                $message = 'Keranjang kosong! Silakan tambahkan asset terlebih dahulu.';
                $message_type = 'error';
            } else {
                try {
                    $pdo->beginTransaction();
                    
                    $current_timestamp = date('Y-m-d H:i:s');
                    
                    foreach ($cart_items as $item) {
                        // Check availability one more time
                        if (!isAssetAvailableForBorrow($pdo, $item['asset_id'])) {
                            throw new Exception("Asset {$item['asset_name']} tidak lagi tersedia!");
                        }
                        
                        // Insert borrow request with status 'pending_approval'
                        // TIDAK mengubah status asset sampai admin menyetujui
                        $stmt = $pdo->prepare("
                            INSERT INTO borrow_history 
                            (asset_id, borrower_employee_id, borrow_date, notes, preference_pro, mesin, organization, created_by, status, created_at) 
                            VALUES (?, ?, CURDATE(), ?, ?, ?, ?, ?, 'pending_approval', ?)
                        ");
                        $stmt->execute([
                            $item['asset_id'],
                            $_SESSION['employee_id'],
                            $notes,
                            $preference_pro,
                            $mesin,
                            $borrower_organization,
                            $_SESSION['employee_id'],
                            $current_timestamp
                        ]);
                        
                        // ASSET TETAP STATUS 'available' sampai admin menyetujui
                        // Tidak ada perubahan pada tabel assets di sini
                    }
                    
                    // Clear cart
                    clearBorrowCart($pdo, $_SESSION['employee_id']);
                    
                    $pdo->commit();
                    
                    $message = 'Permintaan peminjaman berhasil diajukan untuk ' . count($cart_items) . ' asset! Menunggu persetujuan admin.';
                    $message_type = 'success';
                    
                } catch (Exception $e) {
                    $pdo->rollback();
                    $message = 'Terjadi kesalahan: ' . $e->getMessage();
                    $message_type = 'error';
                }
            }
        }
    }
}

// Get available assets for display
$stmt = $pdo->prepare("
    SELECT * FROM assets 
    WHERE status_borrower = 'available' AND kondisi IN ('baik', 'rusak')
    ORDER BY asset_name
");
$stmt->execute();
$available_assets = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ajukan Peminjaman - Peminjaman Tools Room</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/html5-qrcode@2.3.8/html5-qrcode.min.js"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'leaf-green': '#90EE90',
                        'dark-leaf': '#32CD32',
                        'soft-gray': '#F5F5F5',
                        'medium-gray': '#9CA3AF'
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-soft-gray min-h-screen">
    <!-- Header -->
    <header class="bg-white shadow-sm border-b-2 border-leaf-green">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center py-4">
                <div class="flex items-center space-x-4">
                    <a href="dashboard.php" class="text-dark-leaf hover:text-green-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"></path>
                        </svg>
                    </a>
                    <img src="../images/logo.png" alt="Logo" class="h-10 w-10" onerror="this.style.display='none'">
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">Ajukan Peminjaman</h1>
                        <p class="text-sm text-medium-gray">Pinjam tools yang Anda butuhkan</p>
                    </div>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-sm text-gray-600"><?php echo $_SESSION['user_name']; ?></span>
                    <a href="../logout.php" class="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded-lg text-sm transition duration-200">
                        Logout
                    </a>
                </div>
            </div>
        </div>
    </header>

    <div class="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Message -->
        <?php if ($message): ?>
        <div class="mb-6 p-4 rounded-lg <?php echo $message_type == 'success' ? 'bg-green-100 border border-green-400 text-green-700' : 'bg-red-100 border border-red-400 text-red-700'; ?>">
            <?php echo $message; ?>
        </div>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <!-- Left Column: Scanner & Manual Input -->
            <div class="space-y-6">
                <!-- Scanner Section -->
                <div class="bg-white rounded-xl shadow-lg p-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Scan Barcode/QR Code</h3>
                    
                    <div class="mb-4">
                        <button id="startScanBtn" onclick="startScanner()" 
                                class="bg-blue-500 hover:bg-blue-600 text-white px-6 py-3 rounded-lg transition duration-200 mr-2">
                            <svg class="w-5 h-5 inline mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v1m6 11h2m-6 0h-2v4m0-11v3m0 0h.01M12 12h4.01M16 16h4"></path>
                            </svg>
                            Mulai Scan
                        </button>
                        <button id="stopScanBtn" onclick="stopScanner()" 
                                class="bg-red-500 hover:bg-red-600 text-white px-6 py-3 rounded-lg transition duration-200 hidden">
                            Stop Scan
                        </button>
                    </div>
                    
                    <div id="scanner-container" class="hidden">
                        <div id="reader" class="w-full"></div>
                    </div>
                    
                    <div id="scan-result" class="hidden mt-4 p-4 bg-gray-50 rounded-lg">
                        <h4 class="font-semibold text-gray-800 mb-2">Asset Ditemukan:</h4>
                        <div id="scanned-asset-info"></div>
                        <button id="addToCartBtn" onclick="addScannedAssetToCart()" 
                                class="mt-3 bg-dark-leaf hover:bg-green-600 text-white px-4 py-2 rounded-lg transition duration-200">
                            Tambah ke Keranjang
                        </button>
                    </div>
                </div>

                <!-- Manual Input Section -->
                <div class="bg-white rounded-xl shadow-lg p-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Input Manual</h3>
                    
                    <div class="space-y-4">
                        <div>
                            <label for="manual_asset_number" class="block text-sm font-medium text-gray-700 mb-2">
                                Nomor Asset
                            </label>
                            <div class="flex space-x-2">
                                <input type="text" id="manual_asset_number" 
                                       class="flex-1 px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-dark-leaf focus:border-dark-leaf"
                                       placeholder="Contoh: AST001">
                                <button onclick="searchManualAsset()"
                                        class="bg-blue-500 hover:bg-blue-600 text-white px-6 py-3 rounded-lg transition duration-200">
                                    Cari
                                </button>
                            </div>
                        </div>
                        
                        <div id="manual-result" class="hidden p-4 bg-gray-50 rounded-lg">
                            <div id="manual-asset-info"></div>
                            <button onclick="addManualAssetToCart()" 
                                    class="mt-3 bg-dark-leaf hover:bg-green-600 text-white px-4 py-2 rounded-lg transition duration-200">
                                Tambah ke Keranjang
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Right Column: Cart & Form -->
            <div class="space-y-6">
                <!-- Cart Section -->
                <div class="bg-white rounded-xl shadow-lg p-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Keranjang Peminjaman</h3>
                    
                    <div id="cart-items" class="space-y-3 mb-4">
                        <!-- Cart items will be loaded here -->
                    </div>
                    
                    <button onclick="loadCart()" class="text-sm text-blue-600 hover:text-blue-800">
                        Refresh Keranjang
                    </button>
                </div>

                <!-- Form Section -->
                <div class="bg-white rounded-xl shadow-lg p-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Detail Peminjaman</h3>
                    
                    <form method="POST" class="space-y-4">
                        <div>
                            <label for="preference_pro" class="block text-sm font-medium text-gray-700 mb-2">
                                Preference Pro <span class="text-red-500">*</span>
                            </label>
                            <input type="text" id="preference_pro" name="preference_pro" required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-dark-leaf focus:border-dark-leaf"
                                   placeholder="Masukkan preference pro">
                        </div>

                        <div>
                            <label for="mesin" class="block text-sm font-medium text-gray-700 mb-2">
                                Mesin <span class="text-red-500">*</span>
                            </label>
                            <input type="text" id="mesin" name="mesin" required
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-dark-leaf focus:border-dark-leaf"
                                   placeholder="Masukkan nama mesin">
                        </div>

                        <div>
                            <label for="notes" class="block text-sm font-medium text-gray-700 mb-2">
                                Catatan (Opsional)
                            </label>
                            <textarea id="notes" name="notes" rows="3"
                                      class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-dark-leaf focus:border-dark-leaf"
                                      placeholder="Tambahkan catatan jika diperlukan..."></textarea>
                        </div>

                        <div class="flex space-x-4">
                            <button type="submit" name="submit_borrow"
                                    class="bg-dark-leaf hover:bg-green-600 text-white px-6 py-3 rounded-lg transition duration-200 font-semibold">
                                Ajukan Peminjaman
                            </button>
                            <a href="dashboard.php"
                               class="bg-medium-gray hover:bg-gray-600 text-white px-6 py-3 rounded-lg transition duration-200 font-semibold">
                                Batal
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Global variables for scanner -->
    <script>
        let html5QrcodeScanner = null;
        let currentScannedAsset = null;
        let currentManualAsset = null;

        // Load cart on page load
        document.addEventListener('DOMContentLoaded', function() {
            loadCart();
        });

        function startScanner() {
            const scannerContainer = document.getElementById('scanner-container');
            const startBtn = document.getElementById('startScanBtn');
            const stopBtn = document.getElementById('stopScanBtn');
            
            scannerContainer.classList.remove('hidden');
            startBtn.classList.add('hidden');
            stopBtn.classList.remove('hidden');
            
            html5QrcodeScanner = new Html5Qrcode("reader");
            
            const config = {
                fps: 10,
                qrbox: { width: 250, height: 250 },
                aspectRatio: 1.0
            };
            
            html5QrcodeScanner.start(
                { facingMode: "environment" },
                config,
                (decodedText, decodedResult) => {
                    // Stop scanner and process result
                    stopScanner();
                    processScannedCode(decodedText);
                },
                (errorMessage) => {
                    // Handle scan error (can be ignored for continuous scanning)
                }
            ).catch(err => {
                console.error("Error starting scanner:", err);
                alert("Error starting scanner: " + err);
                stopScanner();
            });
        }

        function stopScanner() {
            const scannerContainer = document.getElementById('scanner-container');
            const startBtn = document.getElementById('startScanBtn');
            const stopBtn = document.getElementById('stopScanBtn');
            
            if (html5QrcodeScanner) {
                html5QrcodeScanner.stop().then(() => {
                    html5QrcodeScanner.clear();
                    html5QrcodeScanner = null;
                }).catch(err => {
                    console.error("Error stopping scanner:", err);
                });
            }
            
            scannerContainer.classList.add('hidden');
            startBtn.classList.remove('hidden');
            stopBtn.classList.add('hidden');
        }

        function processScannedCode(decodedText) {
            // Extract asset number from scanned code
            let assetNumber = decodedText.trim();
            
            // If the scanned code contains additional data, try to extract asset number
            if (assetNumber.includes('|')) {
                assetNumber = assetNumber.split('|')[0];
            }
            
            searchAsset(assetNumber, 'scan');
        }

        function searchManualAsset() {
            const assetNumber = document.getElementById('manual_asset_number').value.trim();
            if (!assetNumber) {
                alert('Masukkan nomor asset terlebih dahulu!');
                return;
            }
            searchAsset(assetNumber, 'manual');
        }

        function searchAsset(assetNumber, type) {
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'scan_asset');
            formData.append('asset_number', assetNumber);
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    displayAssetInfo(data.asset, type);
                } else {
                    alert(data.message);
                    if (type === 'scan') {
                        document.getElementById('scan-result').classList.add('hidden');
                    } else {
                        document.getElementById('manual-result').classList.add('hidden');
                    }
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Terjadi kesalahan saat mencari asset');
            });
        }

        function displayAssetInfo(asset, type) {
            const isAvailable = asset.is_available == 1;
            const availabilityText = isAvailable ? 
                '<span class="text-green-600 font-medium">Tersedia</span>' : 
                '<span class="text-red-600 font-medium">Tidak Tersedia</span>';
            
            const kondisiColor = asset.kondisi === 'baik' ? 'text-green-600' : 
                                asset.kondisi === 'rusak' ? 'text-yellow-600' : 'text-red-600';
            
            const assetInfo = `
                <div class="space-y-2">
                    <div><strong>Nama:</strong> ${asset.asset_name}</div>
                    <div><strong>Nomor Asset:</strong> ${asset.asset_number}</div>
                    <div><strong>Kategori:</strong> ${asset.asset_category}</div>
                    <div><strong>Kondisi:</strong> <span class="${kondisiColor}">${asset.kondisi}</span></div>
                    <div><strong>Status:</strong> ${availabilityText}</div>
                    ${asset.description ? `<div><strong>Deskripsi:</strong> ${asset.description}</div>` : ''}
                </div>
            `;
            
            if (type === 'scan') {
                currentScannedAsset = asset;
                document.getElementById('scanned-asset-info').innerHTML = assetInfo;
                document.getElementById('scan-result').classList.remove('hidden');
                document.getElementById('addToCartBtn').style.display = isAvailable ? 'block' : 'none';
            } else {
                currentManualAsset = asset;
                document.getElementById('manual-asset-info').innerHTML = assetInfo;
                document.getElementById('manual-result').classList.remove('hidden');
                const addBtn = document.querySelector('#manual-result button');
                addBtn.style.display = isAvailable ? 'block' : 'none';
            }
        }

        function addScannedAssetToCart() {
            if (currentScannedAsset) {
                addToCart(currentScannedAsset.id);
            }
        }

        function addManualAssetToCart() {
            if (currentManualAsset) {
                addToCart(currentManualAsset.id);
            }
        }

        function addToCart(assetId) {
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'add_to_cart');
            formData.append('asset_id', assetId);
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert(data.message);
                    loadCart();
                    // Clear forms
                    document.getElementById('scan-result').classList.add('hidden');
                    document.getElementById('manual-result').classList.add('hidden');
                    document.getElementById('manual_asset_number').value = '';
                } else {
                    alert(data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Terjadi kesalahan saat menambahkan ke keranjang');
            });
        }

        function removeFromCart(cartId) {
            if (!confirm('Hapus asset dari keranjang?')) return;
            
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'remove_from_cart');
            formData.append('cart_id', cartId);
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    loadCart();
                } else {
                    alert('Gagal menghapus dari keranjang');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Terjadi kesalahan');
            });
        }

        function loadCart() {
            const formData = new FormData();
            formData.append('ajax', '1');
            formData.append('action', 'get_cart');
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                const cartContainer = document.getElementById('cart-items');
                
                if (data.success && data.items && data.items.length > 0) {
                    cartContainer.innerHTML = data.items.map(item => `
                        <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                            <div>
                                <div class="font-medium text-gray-800">${item.asset_name}</div>
                                <div class="text-sm text-gray-600">${item.asset_number} • ${item.asset_category}</div>
                                <div class="text-xs text-gray-500">Kondisi: ${item.kondisi}</div>
                            </div>
                            <button onclick="removeFromCart(${item.cart_id})" 
                                    class="text-red-600 hover:text-red-800 p-1">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                                </svg>
                            </button>
                        </div>
                    `).join('');
                } else {
                    cartContainer.innerHTML = '<p class="text-gray-500 text-center py-4">Keranjang kosong</p>';
                }
            })
            .catch(error => {
                console.error('Error:', error);
                const cartContainer = document.getElementById('cart-items');
                cartContainer.innerHTML = `
                    <div class="text-center py-4">
                        <p class="text-red-500">Error loading cart</p>
                        <button onclick="loadCart()" class="mt-2 text-sm text-blue-600 hover:text-blue-800">Coba lagi</button>
                    </div>
                `;
            });
        }

        // Form validation
        document.querySelector('form').addEventListener('submit', function(e) {
            const preference_pro = document.getElementById('preference_pro').value.trim();
            const mesin = document.getElementById('mesin').value.trim();
            
            if (!preference_pro) {
                e.preventDefault();
                alert('Preference Pro harus diisi!');
                return false;
            }
            
            if (!mesin) {
                e.preventDefault();
                alert('Mesin harus diisi!');
                return false;
            }
        });
    </script>
</body>
</html>