<?php
// Start session
session_start();

// Include database connection and helper functions
require_once "../config/connection.php";
require_once "transaction_actions.php";

// Redirect if not admin
if (!is_logged_in() || !is_user_type('admin')) {
    header("Location: ../login.php");
    exit();
}

// Get admin ID
$admin_id = $_SESSION['user_id'];

// UPDATED: Update overdue status for borrowed items automatically
updateOverdueStatus($conn);

// Log activity
log_activity($conn, $admin_id, 'admin', 'Approval Form Access', 'Admin accessed borrowing approval page');

// Set page title
$page_title = "Persetujuan Formulir Peminjaman";

// Handle AJAX actions (including bulk actions)
if (isset($_POST['action'])) {
    if ($_POST['action'] === 'bulk_action') {
        $response = handleBulkTransactionAction($conn);
    } else {
        $response = handleTransactionAction($conn);
    }
    echo json_encode($response);
    exit();
}

// UPDATED: Check for show_reservation parameter
$show_reservation = isset($_GET['show_reservation']) ? $_GET['show_reservation'] : '';

// Prepare filters
$filter_borrower = isset($_GET['borrower']) ? $_GET['borrower'] : '';
$filter_status = isset($_GET['status']) ? $_GET['status'] : '';
$filter_date_start = isset($_GET['date_start']) ? $_GET['date_start'] : '';
$filter_date_end = isset($_GET['date_end']) ? $_GET['date_end'] : '';
$tab = isset($_GET['tab']) ? $_GET['tab'] : 'process'; // Default to 'process' tab

// Get transactions for current tab
$transactions = getTransactions($conn, $tab, $filter_borrower, $filter_status, $filter_date_start, $filter_date_end);

// UPDATED: If show_reservation parameter exists, find the transaction ID
$auto_show_transaction_id = null;
if (!empty($show_reservation)) {
    $reservation_stmt = $conn->prepare("SELECT id FROM stock_transactions_semarang WHERE reservation_number = ?");
    $reservation_stmt->bind_param("s", $show_reservation);
    $reservation_stmt->execute();
    $reservation_result = $reservation_stmt->get_result();
    if ($reservation_result->num_rows > 0) {
        $auto_show_transaction_id = $reservation_result->fetch_assoc()['id'];
    }
    $reservation_stmt->close();
}

// Get borrowers for filter dropdown
$borrowers_query = "SELECT * FROM borrower_users_semarang ORDER BY username";
$borrowers_result = $conn->query($borrowers_query);

// Include header
include '../template_admin/header.php';
?>

<!-- Main Content Area -->
<main class="flex-1 p-6 overflow-auto bg-gradient-to-br from-blue-100 via-teal-100 to-green-100">
    <!-- CSS for transitions -->
    <style>
        .transition-all {
            transition-property: all;
            transition-timing-function: cubic-bezier(0.4, 0, 0.2, 1);
        }

        .duration-1000 {
            transition-duration: 1000ms;
        }

        .updated-item {
            animation: highlight-fade 2s ease-out;
        }

        @keyframes highlight-fade {
            0% { background-color: rgba(254, 249, 195, 1); } /* Yellow */
            100% { background-color: rgba(249, 250, 251, 1); } /* Gray-50 */
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }

        .tab-button {
            transition: all 0.3s ease;
        }

        .tab-button.active {
            background-color: rgb(59 130 246);
            color: white;
        }

        .tab-button:not(.active) {
            background-color: rgb(229 231 235);
            color: rgb(75 85 99);
        }

        .tab-button:not(.active):hover {
            background-color: rgb(209 213 219);
        }

        /* UPDATED: Add highlight styles for target reservation */
        .highlight-reservation {
            background-color: rgba(254, 249, 195, 1) !important;
            border-left: 4px solid #f59e0b !important;
            animation: pulse-highlight 3s ease-out;
        }

        @keyframes pulse-highlight {
            0% { 
                background-color: rgba(254, 249, 195, 1);
                transform: scale(1);
            }
            50% { 
                background-color: rgba(251, 191, 36, 0.3);
                transform: scale(1.02);
            }
            100% { 
                background-color: rgba(254, 249, 195, 1);
                transform: scale(1);
            }
        }

        /* NEW: Styles for bulk selection */
        .bulk-checkbox {
            transform: scale(1.2);
            margin-right: 8px;
        }

        .bulk-actions {
            position: sticky;
            top: 0;
            z-index: 40;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(5px);
            border-bottom: 2px solid #e5e7eb;
        }

        .item-checkbox:checked {
            background-color: #3b82f6;
            border-color: #3b82f6;
        }

        .item-disabled {
            opacity: 0.5;
            pointer-events: none;
        }

        .select-all-section {
            background: linear-gradient(135deg, #f3f4f6 0%, #e5e7eb 100%);
            border: 2px dashed #9ca3af;
        }

        /* Checkbox styling */
        .checkbox-container {
            position: relative;
            display: inline-block;
        }

        .checkbox-container input[type="checkbox"] {
            appearance: none;
            width: 20px;
            height: 20px;
            border: 2px solid #d1d5db;
            border-radius: 4px;
            background-color: white;
            cursor: pointer;
            position: relative;
            transition: all 0.2s ease;
        }

        .checkbox-container input[type="checkbox"]:checked {
            background-color: #3b82f6;
            border-color: #3b82f6;
        }

        .checkbox-container input[type="checkbox"]:checked::before {
            content: '✓';
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            color: white;
            font-size: 14px;
            font-weight: bold;
        }

        .checkbox-container input[type="checkbox"]:disabled {
            background-color: #f3f4f6;
            border-color: #d1d5db;
            cursor: not-allowed;
        }
    </style>
    
    <!-- Notification Container -->
    <div id="notification-container" class="fixed top-20 right-5 z-50" style="display: none;">
        <div id="notification" class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded shadow-md">
            <span id="notification-message"></span>
        </div>
    </div>

    <!-- UPDATED: Add reservation highlight notification -->
    <?php if (!empty($show_reservation)): ?>
    <div id="reservation-highlight-notification" class="fixed top-20 left-1/2 transform -translate-x-1/2 z-40 bg-yellow-100 border border-yellow-400 text-yellow-800 px-6 py-3 rounded-lg shadow-lg">
        <div class="flex items-center">
            <i class="fas fa-search mr-2"></i>
            <span>Menampilkan detail untuk reservasi: <strong><?php echo htmlspecialchars($show_reservation); ?></strong></span>
            <button onclick="closeReservationNotification()" class="ml-4 text-yellow-600 hover:text-yellow-800">
                <i class="fas fa-times"></i>
            </button>
        </div>
    </div>
    <?php endif; ?>

    <!-- Page Title -->
    <div class="bg-white rounded-lg shadow-md p-6 mb-6">
        <h2 class="text-2xl font-bold text-gray-800 mb-4">Persetujuan Formulir Peminjaman</h2>
        
        <!-- Tab Navigation -->
        <div class="flex space-x-2 mb-6">
            <button id="tab-process" class="tab-button px-6 py-2 rounded-lg font-medium <?php echo $tab === 'process' ? 'active' : ''; ?>" onclick="switchTab('process')">
                <i class="fas fa-clock mr-2"></i>Transaksi Proses
            </button>
            <button id="tab-completed" class="tab-button px-6 py-2 rounded-lg font-medium <?php echo $tab === 'completed' ? 'active' : ''; ?>" onclick="switchTab('completed')">
                <i class="fas fa-check-circle mr-2"></i>Transaksi Selesai
            </button>
        </div>
        
        <!-- Filters -->
        <form id="filter-form" method="GET" class="mb-6">
            <input type="hidden" name="tab" id="current-tab" value="<?php echo $tab; ?>">
            <?php if (!empty($show_reservation)): ?>
            <input type="hidden" name="show_reservation" value="<?php echo htmlspecialchars($show_reservation); ?>">
            <?php endif; ?>
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <div>
                    <label for="borrower" class="block text-sm font-medium text-gray-700 mb-1">Peminjam</label>
                    <select name="borrower" id="borrower" class="w-full p-2 border rounded-md">
                        <option value="">Semua Peminjam</option>
                        <?php 
                        $borrowers_result->data_seek(0); // Reset pointer
                        while ($borrower = $borrowers_result->fetch_assoc()): ?>
                            <option value="<?php echo $borrower['id']; ?>" <?php echo ($filter_borrower == $borrower['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($borrower['username'] . ' (' . $borrower['lokasi_posisi'] . ')'); ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div id="status-filter" style="<?php echo $tab === 'completed' ? 'display: none;' : ''; ?>">
                    <label for="status" class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                    <select name="status" id="status" class="w-full p-2 border rounded-md">
                        <option value="">Semua Status</option>
                        <option value="Diajukan" <?php echo ($filter_status == 'Diajukan') ? 'selected' : ''; ?>>Diajukan</option>
                        <option value="Disetujui" <?php echo ($filter_status == 'Disetujui') ? 'selected' : ''; ?>>Disetujui</option>
                        <option value="Ditolak" <?php echo ($filter_status == 'Ditolak') ? 'selected' : ''; ?>>Ditolak</option>
                        <option value="OnProccesing" <?php echo ($filter_status == 'OnProccesing') ? 'selected' : ''; ?>>OnProcessing</option>
                        <option value="Dipinjam" <?php echo ($filter_status == 'Dipinjam') ? 'selected' : ''; ?>>Dipinjam</option>
                    </select>
                </div>
                
                <div>
                    <label for="date_start" class="block text-sm font-medium text-gray-700 mb-1">Tanggal Mulai</label>
                    <input type="date" name="date_start" id="date_start" value="<?php echo $filter_date_start; ?>" class="w-full p-2 border rounded-md">
                </div>
                
                <div>
                    <label for="date_end" class="block text-sm font-medium text-gray-700 mb-1">Tanggal Akhir</label>
                    <input type="date" name="date_end" id="date_end" value="<?php echo $filter_date_end; ?>" class="w-full p-2 border rounded-md">
                </div>
            </div>
            
            <div class="mt-4 flex justify-end">
                <button type="submit" class="px-4 py-2 bg-blue-500 text-white rounded-md hover:bg-blue-600">
                    <i class="fas fa-filter mr-2"></i> Filter
                </button>
                <button type="button" onclick="resetFilters()" class="ml-2 px-4 py-2 bg-gray-500 text-white rounded-md hover:bg-gray-600">
                    <i class="fas fa-sync mr-2"></i> Reset
                </button>
            </div>
        </form>
    </div>
    
    <!-- Transactions Table -->
    <div class="bg-white rounded-lg shadow-md p-6">
        <h3 class="text-xl font-semibold text-gray-800 mb-4">
            <?php echo $tab === 'process' ? 'Daftar Transaksi Proses' : 'Daftar Transaksi Selesai'; ?>
        </h3>
        
        <div class="overflow-x-auto">
            <table class="min-w-full bg-white">
                <thead class="bg-gray-100">
                    <tr>
                        <th class="py-3 px-4 text-left" style="display: none;">ID</th>
                        <th class="py-3 px-4 text-left">No. Reservasi</th>
                        <th class="py-3 px-4 text-left">Peminjam</th>
                        <th class="py-3 px-4 text-left">Tanggal Transaksi</th>
                        <th class="py-3 px-4 text-left">Tanggal Pengembalian</th>
                        <?php if ($tab === 'completed'): ?>
                        <th class="py-3 px-4 text-left">Status Waktu</th>
                        <?php endif; ?>
                        <th class="py-3 px-4 text-left">Status</th>
                        <th class="py-3 px-4 text-left">Status Peminjaman</th>
                        <th class="py-3 px-4 text-left">Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($transactions->num_rows > 0): ?>
                        <?php while ($transaction = $transactions->fetch_assoc()): ?>
                            <tr class="border-b hover:bg-gray-50 <?php echo (!empty($show_reservation) && $transaction['reservation_number'] === $show_reservation) ? 'target-reservation' : ''; ?>" 
                                data-transaction-id="<?php echo $transaction['id']; ?>" 
                                data-reservation="<?php echo htmlspecialchars($transaction['reservation_number']); ?>">
                                <td class="py-3 px-4" style="display: none;"><?php echo $transaction['id']; ?></td>
                                <td class="py-3 px-4 <?php echo (!empty($show_reservation) && $transaction['reservation_number'] === $show_reservation) ? 'font-bold text-blue-600' : ''; ?>">
                                    <?php echo htmlspecialchars($transaction['reservation_number']); ?>
                                </td>
                                <td class="py-3 px-4"><?php echo htmlspecialchars($transaction['borrower_name']); ?> (<?php echo htmlspecialchars($transaction['lokasi_posisi']); ?>)</td>
                                <td class="py-3 px-4"><?php echo date('d M Y H:i', strtotime($transaction['tanggal_transaksi'])); ?></td>
                                <td class="py-3 px-4"><?php echo date('d M Y H:i', strtotime($transaction['tanggal_pengembalian'])); ?></td>
                                <?php if ($tab === 'completed'): ?>
                                <td class="py-3 px-4">
                                    <?php 
                                    $status_waktu_class = '';
                                    $status_waktu_text = '-';
                                    
                                    // Check if we have a status_waktu from any of the transaction items
                                    $status_query = "SELECT status_waktu FROM transaction_items_semarang WHERE transaction_id = ? AND dikembalikan = 1 LIMIT 1";
                                    $status_stmt = $conn->prepare($status_query);
                                    $status_stmt->bind_param('i', $transaction['id']);
                                    $status_stmt->execute();
                                    $status_result = $status_stmt->get_result();
                                    
                                    if ($status_result->num_rows > 0) {
                                        $status_row = $status_result->fetch_assoc();
                                        switch ($status_row['status_waktu']) {
                                            case 'tepat':
                                                $status_waktu_text = 'Tepat Waktu';
                                                $status_waktu_class = 'bg-green-100 text-green-800';
                                                break;
                                            case 'overdue':
                                                $status_waktu_text = 'Terlambat';
                                                $status_waktu_class = 'bg-red-100 text-red-800';
                                                break;
                                            default:
                                                $status_waktu_text = '-';
                                                $status_waktu_class = 'bg-gray-100 text-gray-800';
                                        }
                                    }
                                    $status_stmt->close();
                                    ?>
                                    <span class="px-2 py-1 rounded-full text-xs font-semibold <?php echo $status_waktu_class; ?>">
                                        <?php echo $status_waktu_text; ?>
                                    </span>
                                </td>
                                <?php endif; ?>
                                <td class="py-3 px-4">
                                    <span class="px-2 py-1 rounded-full text-xs font-semibold
                                        <?php 
                                        switch ($transaction['status']) {
                                            case 'Diajukan':
                                                echo 'bg-yellow-100 text-yellow-800';
                                                break;
                                            case 'Disetujui':
                                                echo 'bg-green-100 text-green-800';
                                                break;
                                            case 'Ditolak':
                                                echo 'bg-red-100 text-red-800';
                                                break;
                                            case 'OnProccesing':
                                                echo 'bg-blue-100 text-blue-800';
                                                break;
                                            default:
                                                echo 'bg-gray-100 text-gray-800';
                                        }
                                        ?>">
                                        <?php echo $transaction['status']; ?>
                                    </span>
                                </td>
                                <td class="py-3 px-4">
                                    <span class="px-2 py-1 rounded-full text-xs font-semibold
                                        <?php 
                                        switch ($transaction['status_peminjaman']) {
                                            case 'None':
                                                echo 'bg-gray-100 text-gray-800';
                                                break;
                                            case 'Dipinjam':
                                                echo 'bg-blue-100 text-blue-800';
                                                break;
                                            case 'Dikembalikan':
                                                echo 'bg-green-100 text-green-800';
                                                break;
                                            default:
                                                echo 'bg-gray-100 text-gray-800';
                                        }
                                        ?>">
                                        <?php echo $transaction['status_peminjaman']; ?>
                                    </span>
                                </td>
                                <td class="py-3 px-4">
                                    <button type="button" class="px-3 py-1 bg-blue-500 text-white rounded hover:bg-blue-600 view-details" 
                                            data-transaction-id="<?php echo $transaction['id']; ?>">
                                        <i class="fas fa-eye mr-1"></i> Detail
                                    </button>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="<?php echo $tab === 'completed' ? '9' : '8'; ?>" class="py-4 px-4 text-center text-gray-500">
                                <?php echo $tab === 'process' ? 'Tidak ada transaksi proses ditemukan' : 'Tidak ada transaksi selesai ditemukan'; ?>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Transaction Details Modal -->
    <div id="details-modal" class="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center hidden">
        <div class="bg-white rounded-lg shadow-xl w-11/12 md:w-3/4 lg:w-2/3 max-h-[90vh] overflow-auto">
            <div class="p-6">
                <div class="flex justify-between items-center mb-4">
                    <div class="flex items-center space-x-4">
                        <h3 class="text-xl font-semibold text-gray-800">Detail Peminjaman</h3>
                        <span id="reservation-number-display" class="px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-sm font-medium"></span>
                    </div>
                    <button type="button" class="text-gray-500 hover:text-gray-700 close-modal">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <div id="transaction-details" class="mb-6">
                    <!-- Transaction details will be loaded here -->
                </div>
                
                <!-- NEW: Bulk Actions Section -->
                <div id="bulk-actions-section" class="bulk-actions p-4 mb-4 rounded-lg border hidden">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-4">
                            <span id="selected-count" class="text-sm font-medium text-gray-600">0 item dipilih</span>
                            <button type="button" id="select-all-available" class="text-sm text-blue-600 hover:text-blue-800">
                                Pilih Semua (Tersedia)
                            </button>
                            <button type="button" id="clear-selection" class="text-sm text-gray-600 hover:text-gray-800">
                                Bersihkan Pilihan
                            </button>
                        </div>
                        <div class="flex space-x-2">
                            <button type="button" id="bulk-approve" class="px-3 py-1 bg-green-500 text-white rounded hover:bg-green-600 text-sm">
                                <i class="fas fa-check mr-1"></i> Setujui Terpilih
                            </button>
                            <button type="button" id="bulk-reject" class="px-3 py-1 bg-red-500 text-white rounded hover:bg-red-600 text-sm">
                                <i class="fas fa-times mr-1"></i> Tolak Terpilih
                            </button>
                            <button type="button" id="bulk-onprocess" class="px-3 py-1 bg-blue-500 text-white rounded hover:bg-blue-600 text-sm">
                                <i class="fas fa-clock mr-1"></i> OnProcess Terpilih
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="border-t pt-4">
                    <h4 class="text-lg font-medium text-gray-700 mb-3">Daftar Item yang Dipinjam</h4>
                    <div id="transaction-items" class="space-y-4">
                        <!-- Transaction items will be loaded here -->
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Approval/Rejection Modal -->
    <div id="action-modal" class="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center hidden">
        <div class="bg-white rounded-lg shadow-xl w-11/12 md:w-2/3 lg:w-1/2 max-h-[90vh] overflow-auto">
            <div class="p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 id="action-modal-title" class="text-xl font-semibold text-gray-800">Aksi Peminjaman</h3>
                    <button type="button" class="text-gray-500 hover:text-gray-700 close-action-modal">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <form id="action-form" class="space-y-4">
                    <input type="hidden" id="action-item-id" name="item_id">
                    <input type="hidden" id="action-transaction-id" name="transaction_id">
                    <input type="hidden" id="action-type" name="action">
                    
                    <div id="note-container" class="hidden">
                        <label for="action-note" class="block text-sm font-medium text-gray-700 mb-1">Catatan</label>
                        <textarea id="action-note" name="note" rows="3" class="w-full p-2 border rounded-md"></textarea>
                    </div>
                    
                    <div id="condition-container" class="hidden">
                        <label class="block text-sm font-medium text-gray-700 mb-1">Kondisi Pengembalian</label>
                        <div class="flex space-x-4">
                            <label class="inline-flex items-center">
                                <input type="radio" name="condition" value="Baik" class="form-radio" checked>
                                <span class="ml-2">Baik</span>
                            </label>
                            <label class="inline-flex items-center">
                                <input type="radio" name="condition" value="Rusak" class="form-radio">
                                <span class="ml-2">Rusak</span>
                            </label>
                            <label class="inline-flex items-center">
                                <input type="radio" name="condition" value="Hilang" class="form-radio">
                                <span class="ml-2">Hilang</span>
                            </label>
                        </div>
                    </div>
                    
                    <div class="flex justify-end pt-4">
                        <button type="button" class="px-4 py-2 bg-gray-500 text-white rounded-md hover:bg-gray-600 mr-2 close-action-modal">
                            Batal
                        </button>
                        <button type="submit" id="submit-action" class="px-4 py-2 bg-blue-500 text-white rounded-md hover:bg-blue-600">
                            Simpan
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- NEW: Bulk Action Confirmation Modal -->
    <div id="bulk-action-modal" class="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center hidden">
        <div class="bg-white rounded-lg shadow-xl w-11/12 md:w-2/3 lg:w-1/2 max-h-[90vh] overflow-auto">
            <div class="p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 id="bulk-action-modal-title" class="text-xl font-semibold text-gray-800">Konfirmasi Aksi Massal</h3>
                    <button type="button" class="text-gray-500 hover:text-gray-700 close-bulk-action-modal">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <form id="bulk-action-form" class="space-y-4">
                    <input type="hidden" id="bulk-action-type" name="action" value="bulk_action">
                    <input type="hidden" id="bulk-action-operation" name="operation">
                    <input type="hidden" id="bulk-transaction-id" name="transaction_id">
                    <input type="hidden" id="bulk-item-ids" name="item_ids">
                    
                    <div class="bg-gray-50 p-4 rounded-lg">
                        <p id="bulk-action-message" class="text-gray-700"></p>
                        <div id="bulk-selected-items" class="mt-3 max-h-40 overflow-y-auto">
                            <!-- Selected items will be listed here -->
                        </div>
                    </div>
                    
                    <div id="bulk-note-container" class="hidden">
                        <label for="bulk-action-note" class="block text-sm font-medium text-gray-700 mb-1">Catatan (untuk item yang ditolak)</label>
                        <textarea id="bulk-action-note" name="note" rows="3" class="w-full p-2 border rounded-md" placeholder="Masukkan alasan penolakan..."></textarea>
                    </div>
                    
                    <div class="flex justify-end pt-4">
                        <button type="button" class="px-4 py-2 bg-gray-500 text-white rounded-md hover:bg-gray-600 mr-2 close-bulk-action-modal">
                            Batal
                        </button>
                        <button type="submit" id="submit-bulk-action" class="px-4 py-2 bg-blue-500 text-white rounded-md hover:bg-blue-600">
                            Konfirmasi
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</main>

<!-- UPDATED: Add auto-show variables for JavaScript -->
<script>
// Auto-show variables
const AUTO_SHOW_TRANSACTION_ID = <?php echo $auto_show_transaction_id ? $auto_show_transaction_id : 'null'; ?>;
const SHOW_RESERVATION = '<?php echo htmlspecialchars($show_reservation); ?>';

// UPDATED: Function to close reservation notification
function closeReservationNotification() {
    const notification = document.getElementById('reservation-highlight-notification');
    if (notification) {
        notification.style.display = 'none';
    }
    // Remove highlight from target reservation
    const targetRow = document.querySelector('.target-reservation');
    if (targetRow) {
        targetRow.classList.remove('target-reservation', 'highlight-reservation');
    }
}

// UPDATED: Add auto-show functionality when page loads
document.addEventListener('DOMContentLoaded', function() {
    // Highlight target reservation if exists
    if (SHOW_RESERVATION) {
        const targetRow = document.querySelector('.target-reservation');
        if (targetRow) {
            targetRow.classList.add('highlight-reservation');
            // Scroll to the row
            targetRow.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
    }
    
    // Auto-show modal if transaction ID is provided
    if (AUTO_SHOW_TRANSACTION_ID) {
        setTimeout(function() {
            // Find and click the detail button for the target transaction
            const detailButton = document.querySelector(`[data-transaction-id="${AUTO_SHOW_TRANSACTION_ID}"] .view-details`);
            if (detailButton) {
                detailButton.click();
            }
        }, 1000); // Delay to allow page to fully load
    }
    
    // Auto-hide reservation notification after 10 seconds
    if (SHOW_RESERVATION) {
        setTimeout(function() {
            closeReservationNotification();
        }, 10000);
    }
});
</script>

<!-- Include JavaScript functions -->
<?php include 'transaction_ui.php'; ?>

<?php include '../template_admin/footer.php'; ?>