<!-- profile_borrower.php -->
<?php
// Start session
session_start();

// Include database connection
require_once "../config/connection.php";

// Redirect if not borrower user - using functions from connection.php
if (!is_logged_in() || !is_user_type('borrower')) {
    header("Location: ../login.php");
    exit();
}

// Get borrower data
$borrower_id = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM borrower_users_semarang WHERE id = ?");
$stmt->bind_param("i", $borrower_id);
$stmt->execute();
$borrower_data = $stmt->get_result()->fetch_assoc();
$stmt->close();

// Check if account is active, redirect to logout if not
if ($borrower_data['status'] == 'nonactive') {
    // Log this attempt
    error_log("Nonactive borrower attempted to access profile: " . $_SESSION['username']);
    
    // Clear all session variables
    $_SESSION = array();
    
    // Destroy the session
    session_destroy();
    
    // Redirect to login with message
    header("Location: ../login.php?error=inactive");
    exit();
}

// Set page title
$page_title = "Profil Peminjam";

// Log activity using audit_logs_semarang
create_audit_log($conn, 'Profile Access', $borrower_id, 'borrower', 'borrower user accessed their profile');

// Get borrowing statistics
$stats_stmt = $conn->prepare("
    SELECT 
        COUNT(*) as total_peminjaman,
        COUNT(CASE WHEN status_peminjaman = 'Dipinjam' THEN 1 END) as sedang_dipinjam,
        COUNT(CASE WHEN status_peminjaman = 'Dikembalikan' THEN 1 END) as selesai_dipinjam
    FROM stock_transactions_semarang 
    WHERE borrower_id = ?
");
$stats_stmt->bind_param("i", $borrower_id);
$stats_stmt->execute();
$stats = $stats_stmt->get_result()->fetch_assoc();
$stats_stmt->close();

// Get last login
$login_stmt = $conn->prepare("
    SELECT timestamp 
    FROM audit_logs_semarang 
    WHERE user_id = ? AND user_type = 'borrower' AND action_type = 'Dashboard Access'
    ORDER BY timestamp DESC 
    LIMIT 2
");
$login_stmt->bind_param("i", $borrower_id);
$login_stmt->execute();
$login_result = $login_stmt->get_result();
$login_result->fetch_assoc(); // Skip current session
$last_login = $login_result->fetch_assoc();
$login_stmt->close();

// Include header
include '../template_peminjam/header.php';
?>

<!-- Main Content Area -->
<main class="flex-1 p-6 overflow-auto bg-gradient-to-br from-green-50 via-lime-50 to-emerald-50">
    <!-- Header Section -->
    <div class="mb-8">
        <div class="flex items-center mb-4">
            <div class="p-3 rounded-full bg-gradient-to-r from-green-500 to-emerald-600 text-white mr-4">
                <i class="fas fa-user-circle text-2xl"></i>
            </div>
            <div>
                <h1 class="text-3xl font-bold text-gray-800">Profil Saya</h1>
                <p class="text-gray-600">Kelola informasi profil dan lihat aktivitas peminjaman Anda</p>
            </div>
        </div>
    </div>

    <!-- Welcome Card -->
    <div class="bg-gradient-to-r from-green-500 to-emerald-600 rounded-xl shadow-lg p-6 mb-8 text-white">
        <div class="flex items-center">
            <div class="flex-1">
                <h2 class="text-2xl font-bold mb-2">Selamat Datang, <?php echo htmlspecialchars($borrower_data['nama_borrower']); ?>!</h2>
                <p class="text-green-100 mb-4">Sistem Manajemen Color Panel - Portal Peminjam</p>
                <div class="flex items-center">
                    <span class="px-3 py-1 bg-white bg-opacity-20 rounded-full text-sm font-medium">
                        <i class="fas fa-map-marker-alt mr-1"></i>
                        <?php echo htmlspecialchars($borrower_data['lokasi_posisi']); ?>
                    </span>
                </div>
            </div>
            <div class="hidden md:block">
                <div class="w-24 h-24 bg-white bg-opacity-20 rounded-full flex items-center justify-center">
                    <i class="fas fa-user text-4xl"></i>
                </div>
            </div>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
        <div class="bg-white rounded-xl shadow-md p-6 border-l-4 border-green-500">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-green-100 text-green-600 mr-4">
                    <i class="fas fa-clipboard-list text-xl"></i>
                </div>
                <div>
                    <p class="text-sm text-gray-500 mb-1">Total Peminjaman</p>
                    <p class="text-2xl font-bold text-gray-800"><?php echo $stats['total_peminjaman']; ?></p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-xl shadow-md p-6 border-l-4 border-lime-500">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-lime-100 text-lime-600 mr-4">
                    <i class="fas fa-clock text-xl"></i>
                </div>
                <div>
                    <p class="text-sm text-gray-500 mb-1">Sedang Dipinjam</p>
                    <p class="text-2xl font-bold text-gray-800"><?php echo $stats['sedang_dipinjam']; ?></p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-xl shadow-md p-6 border-l-4 border-emerald-500">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-emerald-100 text-emerald-600 mr-4">
                    <i class="fas fa-check-circle text-xl"></i>
                </div>
                <div>
                    <p class="text-sm text-gray-500 mb-1">Selesai Dipinjam</p>
                    <p class="text-2xl font-bold text-gray-800"><?php echo $stats['selesai_dipinjam']; ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- Profile Details -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
        <!-- Personal Information -->
        <div class="bg-white rounded-xl shadow-md p-6">
            <div class="flex items-center mb-6">
                <div class="p-2 rounded-lg bg-green-100 text-green-600 mr-3">
                    <i class="fas fa-user text-lg"></i>
                </div>
                <h3 class="text-xl font-semibold text-gray-800">Informasi Pribadi</h3>
            </div>
            
            <div class="space-y-6">
                <div class="flex items-start">
                    <div class="w-32 text-sm text-gray-500 font-medium">Nama Lengkap</div>
                    <div class="flex-1">
                        <p class="font-semibold text-gray-800"><?php echo htmlspecialchars($borrower_data['nama_borrower']); ?></p>
                    </div>
                </div>
                
                <div class="flex items-start">
                    <div class="w-32 text-sm text-gray-500 font-medium">NIK / ID</div>
                    <div class="flex-1">
                        <p class="font-semibold text-gray-800"><?php echo htmlspecialchars($borrower_data['nik']); ?></p>
                    </div>
                </div>
                
                <div class="flex items-start">
                    <div class="w-32 text-sm text-gray-500 font-medium">Username</div>
                    <div class="flex-1">
                        <p class="font-semibold text-gray-800"><?php echo htmlspecialchars($borrower_data['username']); ?></p>
                    </div>
                </div>
                

                
                <div class="flex items-start">
                    <div class="w-32 text-sm text-gray-500 font-medium">Lokasi Kerja</div>
                    <div class="flex-1">
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-green-100 text-green-800">
                            <i class="fas fa-map-marker-alt mr-1"></i>
                            <?php echo htmlspecialchars($borrower_data['lokasi_posisi']); ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Account Information -->
        <div class="bg-white rounded-xl shadow-md p-6">
            <div class="flex items-center mb-6">
                <div class="p-2 rounded-lg bg-emerald-100 text-emerald-600 mr-3">
                    <i class="fas fa-cog text-lg"></i>
                </div>
                <h3 class="text-xl font-semibold text-gray-800">Informasi Akun</h3>
            </div>
            
            <div class="space-y-6">
                <div class="flex items-start">
                    <div class="w-32 text-sm text-gray-500 font-medium">ID Pengguna</div>
                    <div class="flex-1">
                        <p class="font-semibold text-gray-800">#<?php echo str_pad($borrower_data['id'], 4, '0', STR_PAD_LEFT); ?></p>
                    </div>
                </div>
                
                <div class="flex items-start">
                    <div class="w-32 text-sm text-gray-500 font-medium">Tipe Akun</div>
                    <div class="flex-1">
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-lime-100 text-lime-800">
                            <i class="fas fa-user-tag mr-1"></i>
                            Peminjam
                        </span>
                    </div>
                </div>
                
                <div class="flex items-start">
                    <div class="w-32 text-sm text-gray-500 font-medium">Status Akun</div>
                    <div class="flex-1">
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium <?php echo $borrower_data['status'] == 'active' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                            <i class="fas fa-<?php echo $borrower_data['status'] == 'active' ? 'check-circle' : 'times-circle'; ?> mr-1"></i>
                            <?php echo $borrower_data['status'] == 'active' ? 'Aktif' : 'Nonaktif'; ?>
                        </span>
                    </div>
                </div>
                
                <div class="flex items-start">
                    <div class="w-32 text-sm text-gray-500 font-medium">Login Terakhir</div>
                    <div class="flex-1">
                        <p class="font-semibold text-gray-800">
                            <?php
                            if ($last_login) {
                                echo date('d M Y, H:i:s', strtotime($last_login['timestamp']));
                            } else {
                                echo "Tidak ada data";
                            }
                            ?>
                        </p>
                    </div>
                </div>
                
                <div class="flex items-start">
                    <div class="w-32 text-sm text-gray-500 font-medium">Bergabung Sejak</div>
                    <div class="flex-1">
                        <p class="font-semibold text-gray-800">
                            <?php
                            // Get registration date from first audit log
                            $reg_stmt = $conn->prepare("
                                SELECT timestamp 
                                FROM audit_logs_semarang 
                                WHERE user_id = ? AND user_type = 'borrower'
                                ORDER BY timestamp ASC 
                                LIMIT 1
                            ");
                            $reg_stmt->bind_param("i", $borrower_id);
                            $reg_stmt->execute();
                            $reg_result = $reg_stmt->get_result();
                            
                            if ($reg_row = $reg_result->fetch_assoc()) {
                                echo date('d M Y', strtotime($reg_row['timestamp']));
                            } else {
                                echo "Tidak diketahui";
                            }
                            $reg_stmt->close();
                            ?>
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>


</main>

<?php include '../template_peminjam/footer.php'; ?>