<?php
// Start session
session_start();

// Include database connection
require_once "../config/connection.php";

// Redirect if not IT user - using functions from connection.php
if (!is_logged_in() || !is_user_type('it')) {
    header("Location: ../login.php");
    exit();
}

// Get IT user data
$it_id = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM it_users_surabaya WHERE id = ?");
$stmt->bind_param("i", $it_id);
$stmt->execute();
$it_data = $stmt->get_result()->fetch_assoc();
$stmt->close();

// Set page title
$page_title = "Pengelolaan Buyer & collection_surabaya";

// Process form for adding collection_surabaya to existing buyer
if (isset($_POST['add_collection_surabaya'])) {
    $buyer_id = $_POST['buyer_id'];
    $collection_surabaya_name = $_POST['collection_surabaya_name'];
    
    // Validate input
    if (empty($collection_surabaya_name)) {
        $error_message = "Nama collection_surabaya tidak boleh kosong";
    } else {
        // Check if collection_surabaya already exists for this buyer
        $check_stmt = $conn->prepare("SELECT * FROM collection_surabayas WHERE nama_collection_surabaya = ? AND buyer_id = ?");
        $check_stmt->bind_param("si", $collection_surabaya_name, $buyer_id);
        $check_stmt->execute();
        $result = $check_stmt->get_result();
        
        if ($result->num_rows > 0) {
            $error_message = "collection_surabaya dengan nama tersebut sudah ada untuk buyer ini";
        } else {
            // Insert new collection_surabaya
            $insert_stmt = $conn->prepare("INSERT INTO collection_surabayas (nama_collection_surabaya, buyer_id) VALUES (?, ?)");
            $insert_stmt->bind_param("si", $collection_surabaya_name, $buyer_id);
            
            if ($insert_stmt->execute()) {
                $success_message = "collection_surabaya baru berhasil ditambahkan";
                // Log activity
                log_activity($conn, $it_id, 'it', 'collection_surabaya Added', "IT user added new collection_surabaya '$collection_surabaya_name' for buyer ID: $buyer_id");
            } else {
                $error_message = "Gagal menambahkan collection_surabaya: " . $conn->error;
            }
            $insert_stmt->close();
        }
        $check_stmt->close();
    }
}

// Process form for adding new buyer with collection_surabaya
if (isset($_POST['add_buyer'])) {
    $buyer_name = $_POST['buyer_name'];
    $buyer_code = $_POST['buyer_code'];
    $collection_surabaya_name = $_POST['new_collection_surabaya_name'];
    
    // Validate input
    if (empty($buyer_name) || empty($buyer_code) || empty($collection_surabaya_name)) {
        $error_message_new = "Semua field harus diisi";
    } else {
        // Check if buyer code already exists
        $check_stmt = $conn->prepare("SELECT * FROM buyers_surabaya WHERE kode_buyer = ?");
        $check_stmt->bind_param("s", $buyer_code);
        $check_stmt->execute();
        $result = $check_stmt->get_result();
        
        if ($result->num_rows > 0) {
            $error_message_new = "Kode buyer sudah digunakan";
        } else {
            // Begin transaction
            $conn->begin_transaction();
            
            try {
                // Insert new buyer
                $insert_buyer = $conn->prepare("INSERT INTO buyers_surabaya (nama_buyer, kode_buyer) VALUES (?, ?)");
                $insert_buyer->bind_param("ss", $buyer_name, $buyer_code);
                $insert_buyer->execute();
                $new_buyer_id = $conn->insert_id;
                
                // Insert new collection_surabaya for this buyer
                $insert_collection_surabaya = $conn->prepare("INSERT INTO collection_surabayas (nama_collection_surabaya, buyer_id) VALUES (?, ?)");
                $insert_collection_surabaya->bind_param("si", $collection_surabaya_name, $new_buyer_id);
                $insert_collection_surabaya->execute();
                
                // Commit transaction
                $conn->commit();
                
                $success_message_new = "Buyer baru dan collection_surabaya berhasil ditambahkan";
                // Log activity
                log_activity($conn, $it_id, 'it', 'Buyer and collection_surabaya Added', "IT user added new buyer '$buyer_name' with collection_surabaya '$collection_surabaya_name'");
                
            } catch (Exception $e) {
                // Rollback in case of error
                $conn->rollback();
                $error_message_new = "Gagal menambahkan data: " . $e->getMessage();
            }
        }
        $check_stmt->close();
    }
}

// Handle AJAX delete requests for collection_surabayas
if (isset($_POST['action']) && $_POST['action'] == 'delete_collection_surabaya') {
    $collection_surabaya_id = $_POST['collection_surabaya_id'];
    $response = array();
    
    try {
        // Begin transaction for safety
        $conn->begin_transaction();
        
        // Check if collection_surabaya is used in any transactions
        $check_stmt = $conn->prepare("
            SELECT COUNT(*) as count
            FROM stock_transactions_surabaya t
            WHERE t.collection_surabaya_id = ?
        ");
        $check_stmt->bind_param("i", $collection_surabaya_id);
        $check_stmt->execute();
        $count_result = $check_stmt->get_result()->fetch_assoc();
        $check_stmt->close();
        
        if ($count_result['count'] > 0) {
            // collection_surabaya is used in transactions, can't delete
            $response['status'] = 'error';
            $response['message'] = 'collection_surabaya tidak dapat dihapus karena terhubung dengan transaksi';
            $conn->rollback();
        } else {
            // Check if collection_surabaya is used in master_color_panel_surabaya
            $check_color_stmt = $conn->prepare("
                SELECT COUNT(*) as count
                FROM master_color_panel_surabaya mcp
                WHERE mcp.collection_surabaya_id = ?
            ");
            $check_color_stmt->bind_param("i", $collection_surabaya_id);
            $check_color_stmt->execute();
            $color_count = $check_color_stmt->get_result()->fetch_assoc();
            $check_color_stmt->close();
            
            if ($color_count['count'] > 0) {
                $response['status'] = 'error';
                $response['message'] = 'collection_surabaya tidak dapat dihapus karena terhubung dengan color panel';
                $conn->rollback();
            } else {
                // Safe to delete
                $delete_stmt = $conn->prepare("DELETE FROM collection_surabayas WHERE id = ?");
                $delete_stmt->bind_param("i", $collection_surabaya_id);
                
                if ($delete_stmt->execute() && $delete_stmt->affected_rows > 0) {
                    $conn->commit();
                    $response['status'] = 'success';
                    $response['message'] = 'collection_surabaya berhasil dihapus';
                    // Log activity
                    log_activity($conn, $it_id, 'it', 'collection_surabaya Deleted', "IT user deleted collection_surabaya ID: $collection_surabaya_id");
                } else {
                    $conn->rollback();
                    $response['status'] = 'error';
                    $response['message'] = 'collection_surabaya tidak ditemukan atau gagal dihapus';
                }
                $delete_stmt->close();
            }
        }
        
    } catch (Exception $e) {
        $conn->rollback();
        $response['status'] = 'error';
        $response['message'] = 'Terjadi kesalahan: ' . $e->getMessage();
        error_log("Delete collection_surabaya error: " . $e->getMessage());
    }
    
    // Return JSON response
    header('Content-Type: application/json');
    echo json_encode($response);
    exit();
}

// Handle AJAX delete requests for buyers_surabaya
if (isset($_POST['action']) && $_POST['action'] == 'delete_buyer') {
    $buyer_id = $_POST['buyer_id'];
    $response = array();
    
    try {
        // Begin transaction
        $conn->begin_transaction();
        
        // Check if any collection_surabayas from this buyer are used in transactions
        $check_stmt = $conn->prepare("
            SELECT COUNT(*) as count
            FROM stock_transactions_surabaya t
            JOIN collection_surabayas c ON t.collection_surabaya_id = c.id
            WHERE c.buyer_id = ?
        ");
        $check_stmt->bind_param("i", $buyer_id);
        $check_stmt->execute();
        $transaction_count = $check_stmt->get_result()->fetch_assoc();
        $check_stmt->close();
        
        if ($transaction_count['count'] > 0) {
            // Buyer's collection_surabayas are used in transactions, can't delete
            $response['status'] = 'error';
            $response['message'] = 'Buyer tidak dapat dihapus karena collection_surabaya-nya terhubung dengan transaksi';
            $conn->rollback();
        } else {
            // Check if any color panels are associated with this buyer
            $check_color_stmt = $conn->prepare("
                SELECT COUNT(*) as count
                FROM master_color_panel_surabaya mcp
                WHERE mcp.buyer_id = ?
            ");
            $check_color_stmt->bind_param("i", $buyer_id);
            $check_color_stmt->execute();
            $color_count = $check_color_stmt->get_result()->fetch_assoc();
            $check_color_stmt->close();
            
            if ($color_count['count'] > 0) {
                // Check if any of these color panels have stock items
                $check_items_stmt = $conn->prepare("
                    SELECT COUNT(*) as count
                    FROM color_stock_items_surabaya csi
                    JOIN master_color_panel_surabaya mcp ON csi.color_id = mcp.id
                    WHERE mcp.buyer_id = ?
                ");
                $check_items_stmt->bind_param("i", $buyer_id);
                $check_items_stmt->execute();
                $items_count = $check_items_stmt->get_result()->fetch_assoc();
                $check_items_stmt->close();
                
                if ($items_count['count'] > 0) {
                    $response['status'] = 'error';
                    $response['message'] = 'Buyer tidak dapat dihapus karena memiliki stock items';
                    $conn->rollback();
                } else {
                    // Delete color panel components first
                    $delete_components = $conn->prepare("
                        DELETE cpc FROM color_panel_components_surabaya cpc
                        JOIN master_color_panel_surabaya mcp ON cpc.color_id = mcp.id
                        WHERE mcp.buyer_id = ?
                    ");
                    $delete_components->bind_param("i", $buyer_id);
                    $delete_components->execute();
                    $delete_components->close();
                    
                    // Delete color panels
                    $delete_panels = $conn->prepare("DELETE FROM master_color_panel_surabaya WHERE buyer_id = ?");
                    $delete_panels->bind_param("i", $buyer_id);
                    $delete_panels->execute();
                    $delete_panels->close();
                    
                    // Delete all collection_surabayas for this buyer
                    $delete_collection_surabayas = $conn->prepare("DELETE FROM collection_surabayas WHERE buyer_id = ?");
                    $delete_collection_surabayas->bind_param("i", $buyer_id);
                    $delete_collection_surabayas->execute();
                    $delete_collection_surabayas->close();
                    
                    // Delete the buyer
                    $delete_buyer = $conn->prepare("DELETE FROM buyers_surabaya WHERE id = ?");
                    $delete_buyer->bind_param("i", $buyer_id);
                    
                    if ($delete_buyer->execute() && $delete_buyer->affected_rows > 0) {
                        $conn->commit();
                        $response['status'] = 'success';
                        $response['message'] = 'Buyer dan semua data terkaitnya berhasil dihapus';
                        // Log activity
                        log_activity($conn, $it_id, 'it', 'Buyer Deleted', "IT user deleted buyer ID: $buyer_id");
                    } else {
                        $conn->rollback();
                        $response['status'] = 'error';
                        $response['message'] = 'Buyer tidak ditemukan atau gagal dihapus';
                    }
                    $delete_buyer->close();
                }
            } else {
                // No color panels, safe to delete collection_surabayas and buyer
                // Delete all collection_surabayas for this buyer
                $delete_collection_surabayas = $conn->prepare("DELETE FROM collection_surabayas WHERE buyer_id = ?");
                $delete_collection_surabayas->bind_param("i", $buyer_id);
                $delete_collection_surabayas->execute();
                $delete_collection_surabayas->close();
                
                // Delete the buyer
                $delete_buyer = $conn->prepare("DELETE FROM buyers_surabaya WHERE id = ?");
                $delete_buyer->bind_param("i", $buyer_id);
                
                if ($delete_buyer->execute() && $delete_buyer->affected_rows > 0) {
                    $conn->commit();
                    $response['status'] = 'success';
                    $response['message'] = 'Buyer dan semua collection_surabaya-nya berhasil dihapus';
                    // Log activity
                    log_activity($conn, $it_id, 'it', 'Buyer Deleted', "IT user deleted buyer ID: $buyer_id");
                } else {
                    $conn->rollback();
                    $response['status'] = 'error';
                    $response['message'] = 'Buyer tidak ditemukan atau gagal dihapus';
                }
                $delete_buyer->close();
            }
        }
        
    } catch (Exception $e) {
        $conn->rollback();
        $response['status'] = 'error';
        $response['message'] = 'Terjadi kesalahan: ' . $e->getMessage();
        error_log("Delete buyer error: " . $e->getMessage());
    }
    
    // Return JSON response
    header('Content-Type: application/json');
    echo json_encode($response);
    exit();
}

// Fetch all buyers_surabaya for dropdown
$buyers_surabaya_query = $conn->query("SELECT * FROM buyers_surabaya ORDER BY nama_buyer");

// Include header
include '../template_it/header.php';
?>

<!-- Main Content Area -->
<main class="flex-1 p-6 overflow-auto bg-gradient-to-br from-blue-100 via-teal-100 to-green-100">
    <div class="mb-6">
        <h1 class="text-2xl font-bold text-gray-800">Pengelolaan Buyer & collection_surabaya</h1>
        <p class="text-gray-600">Kelola data buyer dan collection_surabaya untuk sistem color panel</p>
    </div>
    
    <!-- Tabs -->
    <div class="mb-6">
        <div class="border-b border-gray-200">
            <ul class="flex flex-wrap -mb-px text-sm font-medium text-center">
                <li class="mr-2">
                    <a href="#" class="inline-flex items-center justify-center p-4 text-blue-600 border-b-2 border-blue-600 rounded-t-lg active group tab-button" data-tab="existing-buyer">
                        <i class="fas fa-user-tag mr-2"></i>Buyer dengan collection_surabaya Baru
                    </a>
                </li>
                <li class="mr-2">
                    <a href="#" class="inline-flex items-center justify-center p-4 text-gray-500 hover:text-gray-600 hover:border-gray-300 border-b-2 border-transparent rounded-t-lg group tab-button" data-tab="new-buyer">
                        <i class="fas fa-user-plus mr-2"></i>Buyer Baru dengan collection_surabaya Baru
                    </a>
                </li>
            </ul>
        </div>
    </div>
    
    <!-- Existing Buyer - New collection_surabaya Tab -->
    <div id="existing-buyer-tab" class="tab-content">
        <div class="bg-white rounded-lg shadow-md p-6 mb-6">
            <h2 class="text-xl font-semibold text-gray-800 mb-4">Tambah collection_surabaya untuk Buyer yang Sudah Ada</h2>
            
            <?php if (isset($error_message)): ?>
                <div class="mb-4 p-4 bg-red-100 border-l-4 border-red-500 text-red-700">
                    <p><?php echo $error_message; ?></p>
                </div>
            <?php endif; ?>
            
            <?php if (isset($success_message)): ?>
                <div class="mb-4 p-4 bg-green-100 border-l-4 border-green-500 text-green-700">
                    <p><?php echo $success_message; ?></p>
                </div>
            <?php endif; ?>
            
            <form action="" method="POST" class="space-y-4">
                <div>
                    <label for="buyer_id" class="block text-sm font-medium text-gray-700 mb-1">Pilih Buyer</label>
                    <select id="buyer_id" name="buyer_id" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50">
                        <option value="">-- Pilih Buyer --</option>
                        <?php while ($buyer = $buyers_surabaya_query->fetch_assoc()): ?>
                            <option value="<?php echo $buyer['id']; ?>">
                                <?php echo htmlspecialchars($buyer['nama_buyer'] . ' (' . $buyer['kode_buyer'] . ')'); ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div>
                    <label for="collection_surabaya_name" class="block text-sm font-medium text-gray-700 mb-1">Nama collection_surabaya Baru</label>
                    <input type="text" id="collection_surabaya_name" name="collection_surabaya_name" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50" placeholder="Masukkan nama collection_surabaya">
                </div>
                
                <div class="pt-3">
                    <button type="submit" name="add_collection_surabaya" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-50">
                        <i class="fas fa-plus-circle mr-2"></i>Tambah collection_surabaya
                    </button>
                </div>
            </form>
            
            <!-- Display Existing collection_surabayas -->
            <div class="mt-8">
                <h3 class="text-lg font-medium text-gray-800 mb-3">collection_surabaya yang Sudah Ada</h3>
                
                <!-- Notification container for collection_surabaya delete -->
                <div id="collection_surabaya-notification" class="hidden mb-4"></div>
                
                <div class="overflow-x-auto">
                    <table class="min-w-full bg-white border border-gray-200">
                        <thead class="bg-gray-100">
                            <tr>
                                <th class="py-2 px-4 border-b text-left text-xs font-medium text-gray-500 uppercase">ID</th>
                                <th class="py-2 px-4 border-b text-left text-xs font-medium text-gray-500 uppercase">Buyer</th>
                                <th class="py-2 px-4 border-b text-left text-xs font-medium text-gray-500 uppercase">Kode</th>
                                <th class="py-2 px-4 border-b text-left text-xs font-medium text-gray-500 uppercase">collection_surabaya</th>
                                <th class="py-2 px-4 border-b text-center text-xs font-medium text-gray-500 uppercase">Aksi</th>
                            </tr>
                        </thead>
                        <tbody id="collection_surabaya-table-body">
                            <?php
                            // Get all collection_surabayas with buyer information
                            $collection_surabayas_query = $conn->query("
                                SELECT c.id, c.nama_collection_surabaya, b.nama_buyer, b.kode_buyer, b.id AS buyer_id
                                FROM collection_surabayas c
                                JOIN buyers_surabaya b ON c.buyer_id = b.id
                                ORDER BY b.nama_buyer, c.nama_collection_surabaya
                            ");
                            
                            while ($collection_surabaya = $collection_surabayas_query->fetch_assoc()):
                            ?>
                            <tr class="hover:bg-gray-50" id="collection_surabaya-row-<?php echo $collection_surabaya['id']; ?>">
                                <td class="py-2 px-4 border-b"><?php echo $collection_surabaya['id']; ?></td>
                                <td class="py-2 px-4 border-b"><?php echo htmlspecialchars($collection_surabaya['nama_buyer']); ?></td>
                                <td class="py-2 px-4 border-b"><?php echo htmlspecialchars($collection_surabaya['kode_buyer']); ?></td>
                                <td class="py-2 px-4 border-b"><?php echo htmlspecialchars($collection_surabaya['nama_collection_surabaya']); ?></td>
                                <td class="py-2 px-4 border-b text-center">
                                    <button 
                                        type="button" 
                                        class="delete-collection_surabaya-btn px-3 py-1 bg-red-500 text-white rounded-md hover:bg-red-600 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-opacity-50"
                                        data-id="<?php echo $collection_surabaya['id']; ?>"
                                        data-buyer-id="<?php echo $collection_surabaya['buyer_id']; ?>"
                                    >
                                        <i class="fas fa-trash-alt"></i> Hapus
                                    </button>
                                </td>
                            </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- New Buyer - New collection_surabaya Tab -->
    <div id="new-buyer-tab" class="tab-content hidden">
        <div class="bg-white rounded-lg shadow-md p-6 mb-6">
            <h2 class="text-xl font-semibold text-gray-800 mb-4">Tambah Buyer Baru dengan collection_surabaya Baru</h2>
            
            <?php if (isset($error_message_new)): ?>
                <div class="mb-4 p-4 bg-red-100 border-l-4 border-red-500 text-red-700">
                    <p><?php echo $error_message_new; ?></p>
                </div>
            <?php endif; ?>
            
            <?php if (isset($success_message_new)): ?>
                <div class="mb-4 p-4 bg-green-100 border-l-4 border-green-500 text-green-700">
                    <p><?php echo $success_message_new; ?></p>
                </div>
            <?php endif; ?>
            
            <form action="" method="POST" class="space-y-4">
                <div>
                    <label for="buyer_name" class="block text-sm font-medium text-gray-700 mb-1">Nama Buyer Baru</label>
                    <input type="text" id="buyer_name" name="buyer_name" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50" placeholder="Masukkan nama buyer">
                </div>
                
                <div>
                    <label for="buyer_code" class="block text-sm font-medium text-gray-700 mb-1">Kode Buyer (max 5 karakter)</label>
                    <input type="text" id="buyer_code" name="buyer_code" maxlength="5" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50" placeholder="Contoh: EA, CB, UT">
                    <p class="mt-1 text-sm text-gray-500">Kode buyer harus unik, maksimal 5 karakter</p>
                </div>
                
                <div>
                    <label for="new_collection_surabaya_name" class="block text-sm font-medium text-gray-700 mb-1">Nama collection_surabaya Baru</label>
                    <input type="text" id="new_collection_surabaya_name" name="new_collection_surabaya_name" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring focus:ring-blue-500 focus:ring-opacity-50" placeholder="Masukkan nama collection_surabaya">
                </div>
                
                <div class="pt-3">
                    <button type="submit" name="add_buyer" class="px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-opacity-50">
                        <i class="fas fa-user-plus mr-2"></i>Tambah Buyer dan collection_surabaya
                    </button>
                </div>
            </form>
            
            <!-- Display Existing buyers_surabaya -->
            <div class="mt-8">
                <h3 class="text-lg font-medium text-gray-800 mb-3">Buyer yang Sudah Ada</h3>
                
                <!-- Notification container for buyer delete -->
                <div id="buyer-notification" class="hidden mb-4"></div>
                
                <div class="overflow-x-auto">
                    <table class="min-w-full bg-white border border-gray-200">
                        <thead class="bg-gray-100">
                            <tr>
                                <th class="py-2 px-4 border-b text-left text-xs font-medium text-gray-500 uppercase">ID</th>
                                <th class="py-2 px-4 border-b text-left text-xs font-medium text-gray-500 uppercase">Nama Buyer</th>
                                <th class="py-2 px-4 border-b text-left text-xs font-medium text-gray-500 uppercase">Kode Buyer</th>
                                <th class="py-2 px-4 border-b text-left text-xs font-medium text-gray-500 uppercase">Total collection_surabaya</th>
                                <th class="py-2 px-4 border-b text-center text-xs font-medium text-gray-500 uppercase">Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            // Reset the query pointer
                            $buyers_surabaya_query->data_seek(0);
                            
                            while ($buyer = $buyers_surabaya_query->fetch_assoc()):
                                // Count collection_surabayas for this buyer
                                $count_stmt = $conn->prepare("SELECT COUNT(*) as total FROM collection_surabayas WHERE buyer_id = ?");
                                $count_stmt->bind_param("i", $buyer['id']);
                                $count_stmt->execute();
                                $count_result = $count_stmt->get_result()->fetch_assoc();
                                $count_stmt->close();
                            ?>
                            <tr class="hover:bg-gray-50" id="buyer-row-<?php echo $buyer['id']; ?>">
                                <td class="py-2 px-4 border-b"><?php echo $buyer['id']; ?></td>
                                <td class="py-2 px-4 border-b"><?php echo htmlspecialchars($buyer['nama_buyer']); ?></td>
                                <td class="py-2 px-4 border-b"><?php echo htmlspecialchars($buyer['kode_buyer']); ?></td>
                                <td class="py-2 px-4 border-b"><?php echo $count_result['total']; ?></td>
                                <td class="py-2 px-4 border-b text-center">
                                    <button 
                                        type="button" 
                                        class="delete-buyer-btn px-3 py-1 bg-red-500 text-white rounded-md hover:bg-red-600 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-opacity-50"
                                        data-id="<?php echo $buyer['id']; ?>"
                                    >
                                        <i class="fas fa-trash-alt"></i> Hapus
                                    </button>
                                </td>
                            </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- JavaScript for tab functionality and delete operations - IMPROVED VERSION -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Get all tab buttons
    const tabButtons = document.querySelectorAll('.tab-button');
    
    // Add click event to each tab button
    tabButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            
            const tabName = this.getAttribute('data-tab');
            switchTab(tabName);
        });
    });
    
    // Function to switch tabs
    function switchTab(tabName) {
        // Hide all tab contents
        document.querySelectorAll('.tab-content').forEach(tab => {
            tab.classList.add('hidden');
        });
        
        // Show the selected tab content
        const selectedTab = document.getElementById(tabName + '-tab');
        if (selectedTab) {
            selectedTab.classList.remove('hidden');
        }
        
        // Reset all tab buttons to inactive state
        tabButtons.forEach(btn => {
            btn.classList.remove('text-blue-600', 'border-blue-600');
            btn.classList.add('text-gray-500', 'border-transparent');
        });
        
        // Set active tab button
        const activeButton = document.querySelector(`.tab-button[data-tab="${tabName}"]`);
        if (activeButton) {
            activeButton.classList.remove('text-gray-500', 'border-transparent');
            activeButton.classList.add('text-blue-600', 'border-blue-600');
        }
    }
    
    // Function to show notification - IMPROVED VERSION
    function showNotification(containerId, type, message) {
        const container = document.getElementById(containerId);
        if (!container) {
            console.error(`Notification container with ID "${containerId}" not found`);
            return;
        }
        
        // Set appropriate classes based on notification type
        container.className = `mb-4 p-4 border-l-4 ${
            type === 'success' 
                ? 'bg-green-100 border-green-500 text-green-700' 
                : 'bg-red-100 border-red-500 text-red-700'
        }`;
        
        // Set notification message
        container.innerHTML = `<p>${message}</p>`;
        
        // Make notification visible
        container.classList.remove('hidden');
        
        // Scroll to notification to ensure it's visible
        container.scrollIntoView({ behavior: 'smooth', block: 'start' });
        
        // Hide notification after 5 seconds
        setTimeout(() => {
            container.classList.add('hidden');
        }, 5000);
    }
    
    // Process AJAX response - IMPROVED Helper function
    function processResponse(xhr, containerId, rowElement) {
        console.log('Response received:', xhr.responseText); // Debug log
        
        try {
            const response = JSON.parse(xhr.responseText);
            
            if (response.status === 'success') {
                // Only remove row from display if database operation was successful
                rowElement.style.transition = 'opacity 0.5s';
                rowElement.style.opacity = '0';
                
                setTimeout(() => {
                    rowElement.remove();
                    showNotification(containerId, 'success', response.message);
                }, 500);
                
                return { success: true, response: response };
            } else {
                // Show specific error from server
                showNotification(containerId, 'error', response.message);
                return { success: false, response: response };
            }
        } catch (e) {
            console.error('Response parsing error:', e);
            console.error('Raw response:', xhr.responseText);
            
            // Check for specific database errors in the response
            if (xhr.responseText.includes('foreign key constraint') || 
                xhr.responseText.includes('Cannot delete or update a parent row')) {
                showNotification(containerId, 'error', 'Data tidak dapat dihapus karena terhubung dengan data lain');
            } else {
                showNotification(containerId, 'error', 'Terjadi kesalahan saat menghapus data');
            }
            return { success: false };
        }
    }
    
    // Delete collection_surabaya functionality - IMPROVED VERSION
    const deletecollection_surabayaButtons = document.querySelectorAll('.delete-collection_surabaya-btn');
    if (deletecollection_surabayaButtons.length > 0) {
        deletecollection_surabayaButtons.forEach(button => {
            button.addEventListener('click', function() {
                if (confirm('Apakah Anda yakin ingin menghapus collection_surabaya ini?')) {
                    const collection_surabayaId = this.getAttribute('data-id');
                    const rowElement = document.getElementById('collection_surabaya-row-' + collection_surabayaId);
                    
                    if (!rowElement) {
                        console.error(`Row with ID "collection_surabaya-row-${collection_surabayaId}" not found`);
                        return;
                    }
                    
                    // Disable button to prevent multiple clicks
                    this.disabled = true;
                    this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Menghapus...';
                    
                    // Send AJAX request to delete the collection_surabaya
                    const xhr = new XMLHttpRequest();
                    xhr.open('POST', window.location.href, true);
                    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                    
                    xhr.onload = function() {
                        const result = processResponse(xhr, 'collection_surabaya-notification', rowElement);
                        
                        // Re-enable button if operation failed
                        if (!result.success) {
                            button.disabled = false;
                            button.innerHTML = '<i class="fas fa-trash-alt"></i> Hapus';
                        }
                    };
                    
                    xhr.onerror = function() {
                        showNotification('collection_surabaya-notification', 'error', 'Gagal menghubungi server');
                        button.disabled = false;
                        button.innerHTML = '<i class="fas fa-trash-alt"></i> Hapus';
                    };
                    
                    xhr.send('action=delete_collection_surabaya&collection_surabaya_id=' + collection_surabayaId);
                }
            });
        });
    }
    
    // Delete Buyer functionality - IMPROVED VERSION
    const deleteBuyerButtons = document.querySelectorAll('.delete-buyer-btn');
    if (deleteBuyerButtons.length > 0) {
        deleteBuyerButtons.forEach(button => {
            button.addEventListener('click', function() {
                if (confirm('Apakah Anda yakin ingin menghapus buyer ini? Semua collection_surabaya milik buyer ini juga akan dihapus.')) {
                    const buyerId = this.getAttribute('data-id');
                    const rowElement = document.getElementById('buyer-row-' + buyerId);
                    
                    if (!rowElement) {
                        console.error(`Row with ID "buyer-row-${buyerId}" not found`);
                        return;
                    }
                    
                    // Disable button to prevent multiple clicks
                    this.disabled = true;
                    this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Menghapus...';
                    
                    // Send AJAX request to delete the buyer
                    const xhr = new XMLHttpRequest();
                    xhr.open('POST', window.location.href, true);
                    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                    
                    xhr.onload = function() {
                        const result = processResponse(xhr, 'buyer-notification', rowElement);
                        
                        // If deletion was successful, also remove related collection_surabaya rows
                        if (result.success) {
                            const collection_surabayaRows = document.querySelectorAll(`.delete-collection_surabaya-btn[data-buyer-id="${buyerId}"]`);
                            collection_surabayaRows.forEach(collection_surabayaButton => {
                                const collection_surabayaId = collection_surabayaButton.getAttribute('data-id');
                                const collection_surabayaRow = document.getElementById('collection_surabaya-row-' + collection_surabayaId);
                                if (collection_surabayaRow) {
                                    collection_surabayaRow.style.transition = 'opacity 0.5s';
                                    collection_surabayaRow.style.opacity = '0';
                                    setTimeout(() => {
                                        collection_surabayaRow.remove();
                                    }, 500);
                                }
                            });
                            
                            // Also show notification in collection_surabayas tab
                            showNotification('collection_surabaya-notification', 'success', 'collection_surabaya dari buyer ini juga telah dihapus');
                        } else {
                            // Re-enable button if operation failed
                            button.disabled = false;
                            button.innerHTML = '<i class="fas fa-trash-alt"></i> Hapus';
                        }
                    };
                    
                    xhr.onerror = function() {
                        showNotification('buyer-notification', 'error', 'Gagal menghubungi server');
                        button.disabled = false;
                        button.innerHTML = '<i class="fas fa-trash-alt"></i> Hapus';
                    };
                    
                    xhr.send('action=delete_buyer&buyer_id=' + buyerId);
                }
            });
        });
    }
});
</script>

<?php include '../template_it/footer.php'; ?>