<?php
// approval/profile_approval.php

// Start session only if not already started
if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}

// Include database connection
require_once "../config/connection.php";

// Helper fallback: redirect_if_not_approval (jika sudah ada di project, tidak akan dioverride)
if (!function_exists('redirect_if_not_approval')) {
    function redirect_if_not_approval() {
        // Pastikan ada session user dan role approval
        if (empty($_SESSION['user_id']) || empty($_SESSION['role']) || $_SESSION['role'] !== 'approval') {
            header('Location: ../auth/login.php');
            exit;
        }
    }
}

// Helper fallback: verify_active_status
if (!function_exists('verify_active_status')) {
    function verify_active_status($conn) {
        if (empty($_SESSION['user_id'])) return;
        $id = (int) $_SESSION['user_id'];
        $stmt = $conn->prepare("SELECT status FROM approval_users WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $row = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        if (!$row || $row['status'] !== 'active') {
            // logout & redirect to login
            session_unset();
            session_destroy();
            header('Location: ../auth/login.php?msg=account_inactive');
            exit;
        }
    }
}

// Pastikan user approval
redirect_if_not_approval();

// Pastikan akun aktif
verify_active_status($conn);

// Ambil data approval user
$approval_id = (int) ($_SESSION['user_id'] ?? 0);
$approval_data = null;

if ($approval_id) {
    $stmt = $conn->prepare("SELECT id, username, nama_lengkap, nik, email, jabatan, status, created_at FROM approval_users WHERE id = ?");
    $stmt->bind_param("i", $approval_id);
    $stmt->execute();
    $approval_data = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

// Set page title
$page_title = "Profile Approval";

// Include header (sesuaikan path jika berbeda)
include '../template_approval/header.php';
?>

<main class="flex-1 p-6 overflow-auto bg-gradient-to-br from-purple-100 via-pink-100 to-purple-200">
    <div class="max-w-4xl mx-auto">
        <!-- Optional message from redirects -->
        <?php if (isset($_GET['msg'])): ?>
            <?php
                $msg = $_GET['msg'];
                $alert_text = '';
                $alert_type = 'yellow';
                if ($msg === 'not_allowed_change_password') {
                    $alert_text = 'Anda tidak memiliki izin untuk mengubah password. Hubungi admin jika perlu perubahan.';
                    $alert_type = 'yellow';
                } elseif ($msg === 'account_inactive') {
                    $alert_text = 'Akun Anda tidak aktif. Silakan hubungi admin.';
                    $alert_type = 'red';
                } elseif ($msg === 'updated_profile') {
                    $alert_text = 'Profile berhasil diperbarui.';
                    $alert_type = 'green';
                } else {
                    $alert_text = htmlspecialchars($msg);
                }
            ?>
            <div class="mb-6 p-4 rounded-lg <?php echo $alert_type === 'green' ? 'bg-green-100 border border-green-400 text-green-700' : ($alert_type === 'red' ? 'bg-red-100 border border-red-400 text-red-700' : 'bg-yellow-100 border border-yellow-400 text-yellow-700'); ?>">
                <?php echo $alert_text; ?>
            </div>
        <?php endif; ?>

        <div class="bg-white rounded-lg shadow-md overflow-hidden">
            <!-- Header -->
            <div class="bg-gradient-to-r from-purple-500 to-purple-600 px-6 py-8 text-white text-center">
                <div class="w-24 h-24 mx-auto bg-white rounded-full flex items-center justify-center text-4xl font-bold text-purple-600 mb-4">
                    <?php
                        $initial = '-';
                        if (!empty($approval_data['nama_lengkap'])) {
                            $initial = strtoupper(substr($approval_data['nama_lengkap'], 0, 1));
                        }
                        echo htmlspecialchars($initial);
                    ?>
                </div>
                <h2 class="text-2xl font-bold"><?php echo htmlspecialchars($approval_data['nama_lengkap'] ?? '-'); ?></h2>
                <p class="text-purple-100 mt-1"><?php echo htmlspecialchars($approval_data['jabatan'] ?? '-'); ?></p>
            </div>

            <!-- Profile Details -->
            <div class="p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4 flex items-center">
                    <i class="fas fa-user-circle text-purple-600 mr-2"></i>
                    Informasi Profile
                </h3>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-medium text-gray-600 mb-1">Username</label>
                        <div class="p-3 bg-gray-50 rounded-lg border border-gray-200">
                            <p class="text-gray-900"><?php echo htmlspecialchars($approval_data['username'] ?? '-'); ?></p>
                        </div>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-600 mb-1">NIK</label>
                        <div class="p-3 bg-gray-50 rounded-lg border border-gray-200">
                            <p class="text-gray-900"><?php echo htmlspecialchars($approval_data['nik'] ?? '-'); ?></p>
                        </div>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-600 mb-1">Email</label>
                        <div class="p-3 bg-gray-50 rounded-lg border border-gray-200">
                            <p class="text-gray-900"><?php echo htmlspecialchars($approval_data['email'] ?? '-'); ?></p>
                        </div>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-600 mb-1">Jabatan</label>
                        <div class="p-3 bg-gray-50 rounded-lg border border-gray-200">
                            <p class="text-gray-900"><?php echo htmlspecialchars($approval_data['jabatan'] ?? '-'); ?></p>
                        </div>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-600 mb-1">Status Akun</label>
                        <div class="p-3 bg-gray-50 rounded-lg border border-gray-200">
                            <?php if (!empty($approval_data['status']) && $approval_data['status'] == 'active'): ?>
                                <span class="px-3 py-1 bg-green-100 text-green-800 text-sm rounded-full">
                                    <i class="fas fa-check-circle"></i> Aktif
                                </span>
                            <?php else: ?>
                                <span class="px-3 py-1 bg-red-100 text-red-800 text-sm rounded-full">
                                    <i class="fas fa-times-circle"></i> Non-Aktif
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-600 mb-1">Terdaftar Sejak</label>
                        <div class="p-3 bg-gray-50 rounded-lg border border-gray-200">
                            <p class="text-gray-900">
                                <?php
                                    if (!empty($approval_data['created_at'])) {
                                        echo date('d M Y', strtotime($approval_data['created_at']));
                                    } else {
                                        echo '-';
                                    }
                                ?>
                            </p>
                        </div>
                    </div>
                </div>

                <!-- Action Buttons: hanya Kembali -->
                <div class="mt-6 pt-6 border-t border-gray-200">
                    <div class="flex flex-col sm:flex-row gap-3">
                        <a href="dashboard_approval.php"
                           class="flex-1 flex items-center justify-center px-6 py-3 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300 transition-colors">
                            <i class="fas fa-arrow-left mr-2"></i>
                            Kembali ke Dashboard
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</main>

<?php include '../template_approval/footer.php'; ?>
