<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\BusinessCentralService;
use App\Services\InventoryBC;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use GuzzleHttp\Promise;
use GuzzleHttp\Promise\Create;
use GuzzleHttp\Promise\Utils;
use GuzzleHttp\Client;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Illuminate\Support\Str;
use GuzzleHttp\Exception\RequestException;
use App\Models\ItemRequest;
use App\Models\Company;
class BusinessCentralController extends Controller
{   

     public function logPurchaseOrderLine()
    {
        // Buat instance service langsung
        $bcService = app(BusinessCentralService::class);

        // Ambil metadata
        $entities = $bcService->getMetadata();

        if (isset($entities['purchaseOrderLine'])) {
            $logData = [];
            foreach ($entities['purchaseOrderLine'] as $field) {
                $logData[] = [
                    'name'     => $field['name'],
                    'type'     => $field['type'],
                    'nullable' => $field['nullable'],
                ];
            }

            Log::info('📦 purchaseOrderLine Metadata', $logData);

            return response()->json([
                'status' => 'success',
                'message' => 'Metadata logged in laravel.log',
                'count' => count($logData)
            ]);
        } else {
            Log::warning('Entity purchaseOrderLine tidak ditemukan di metadata');

            return response()->json([
                'status' => 'warning',
                'message' => 'Entity purchaseOrderLine tidak ditemukan'
            ]);
        }
    }
    
    public function getAccessToken()
    {
        $client = new Client();
        $response = $client->post("https://login.microsoftonline.com/".env('AZURE_TENANT_ID')."/oauth2/v2.0/token", [
            'form_params' => [
                'grant_type' => 'client_credentials',
                'client_id' => env('AZURE_CLIENT_ID'),
                'client_secret' => env('AZURE_CLIENT_SECRET'),
                'scope' => 'https://api.businesscentral.dynamics.com/.default'
            ]
        ]);
        $body = json_decode((string)$response->getBody(), true);

        return $body['access_token'];
    }

    public function getCompanyId($token)
    {
        $client = new Client();
        $response = $client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies", [
            'headers' => [
                'Authorization' => "Bearer {$token}",
                'Accept'        => 'application/json'
            ]
        ]);
        $companies = json_decode((string)$response->getBody(), true);
        $role = session('user')['role'] ?? null; 
        $companyId1 = $companies['value'][0]['id'];
        $companyId2 = $companies['value'][1]['id'];

        return [
            'companyId1' => $companyId1,
            'companyId2' => $companyId2
        ];
    }

    public function __construct()
    {
     $this->token = $this->getAccessToken();
     $companyIds = $this->getCompanyId($this->token);
     $this->companyRegent = $companyIds['companyId1'];
     $this->companyHIN    = $companyIds['companyId2'];
     
     $activeCompany = session('current_company_name') ?? session('user')['role'];
    if ($activeCompany === 'Regent' or $activeCompany === 'SUPER') {
        $this->companyId = $this->companyRegent;
    } elseif ($activeCompany === 'HIN') {
        $this->companyId = $this->companyHIN;
    }
     $this->client = new Client();
    }


    public function createAndPostPO()
    {
        $bc = new BusinessCentralService();
        $purchaseOrder= $bc->CreatePurchaseOrder();
        return response()->json($purchaseOrder);
    }

    public function showPoSuggestions(Request $request)
    {
        $user = session('user');
        $userName = $user['email'] ?? null;

        $service = new BusinessCentralService();

        $cacheKey = 'po_suggestions_' . md5($userName);
        if (Cache::has($cacheKey)) {
            Log::info("CACHE HIT: $cacheKey");
        } else {
            Log::info("CACHE MISS: $cacheKey");
        }
        $poData = Cache::remember($cacheKey, now()->addMinutes(480), function () use ($userName) {
            $service = new BusinessCentralService();
            return $service->getPoSuggestions($userName);
        });

        $allItems = $poData['items'];
        $vendorMap = $poData['vendors'];
        $vendorOptionsByItemNo = [];
        $uniqueItems = [];
        foreach ($allItems as $item) {
            $itemNo = $item['item_no'] ?? null;
            $vendorNo = $item['vendor_no'] ?? null;
            $vendorName = $item['vendor_name'] ?? null;
            $unitCost = $item['unit_cost'] ?? null;
            if ($vendorNo && $vendorName) {
                $vendorOptionsByItemNo[$itemNo][$vendorNo] = [
                    'name' => $vendorName,
                    'unit_cost' => $unitCost,
                ];
            } elseif ($vendorNo === '0') {
                $vendorOptionsByItemNo[$itemNo][$vendorNo] = [
                    'name' => 'No Vendor',
                    'unit_cost' => 0,
                ];
            }

            if (!isset($uniqueItems[$itemNo])) {
                $uniqueItems[$itemNo] = $item;
            }
        }

        $vendorSet = collect($allItems)
            ->filter(fn ($item) => $item['vendor_no'] && $item['vendor_name'])
            ->pluck('vendor_name', 'vendor_no')
            ->unique()
            ->sortBy(fn ($name, $no) => $name)
            ->toArray();

        $locationSet = collect($allItems)->pluck('location_code')->filter()->unique()->sort()->values()->all();

        $statusSet = collect($allItems)
        ->pluck('status')
        ->filter()
        ->unique()
        ->sort()
        ->values()
        ->all();

        $statusItemSet = collect($allItems)
        ->pluck('Status')
        ->filter()
        ->unique()
        ->sort()
        ->values()
        ->all();

        
        session([
            'po_items' => array_values($uniqueItems),
            'vendor_map' => $vendorMap
        ]);

        return view('po-suggestion', [
            'items'                => array_values($uniqueItems),
            'vendors'              => $vendorSet,
            'locations'            => $locationSet,
            'vendorOptionsByItem'  => $vendorOptionsByItemNo,
            'statuses'             => $statusSet,
            'statusItems'           => $statusItemSet
        ]);
    }

    public function refreshPoSuggestions(Request $request)
    {
        $user = session('user');
        $userName = $user['email'] ?? null;

        $cacheKey = 'po_suggestions_' . md5($userName);
        Cache::forget($cacheKey);

        return redirect()->route('po-suggestions')->with('success', 'Cache has been refreshed.');
    }

    public function refreshMinMaxSuggestions(Request $request)
    {
        Cache::forget('sku_mapping_cache');
       return redirect()->route('sku-mapping')->with('success', 'Cache has been refreshed.');
    }


    public function viewSkuMapping()
    {
        $start = microtime(true);

        $bcService = new InventoryBC(); // adjust if injected
        $data = $bcService->getItemLedgerAndSkuMappingAsync();

        $executionTime = round(microtime(true) - $start, 2);
        
        $currentCompany = session('current_company_name', session('user')['role'] ?? '');

        if ($currentCompany === 'Regent' || $currentCompany === 'SUPER') {
            $locationOptions = ['CI.1010', 'CI.1011' , 'CI.1020'];
        } elseif ($currentCompany === 'HIN') {
            $locationOptions = ['HIN.1200', 'HIN.1300', 'HIN.1000', 'HIN.3000'];
        }
        \Log::info("viewSkuMapping executed in {$executionTime} seconds");
        return view('sku-mapping', [
            'result' => $data['result'],
            'executionTime' => $executionTime,
            'locationOptions' => $locationOptions
        ]);
    }
    
    public function exportSkuMapping(Request $request)
    {
        $bcService = new InventoryBC();
        $data = $bcService->getItemLedgerAndSkuMappingAsync();
        $rows = collect($data['result'] ?? []);
        $baseOdata = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/ODataV4/Company('" . $this->companyId . "')";
        $headers = [
            'Authorization' => "Bearer {$this->token}",
            'Accept'        => 'application/json',
            'Prefer'        => 'odata.maxpagesize=20000',
        ];

        $today = now()->toDateString();
        $twoMonthsAgo = now()->subMonths(2)->toDateString();

        $pricePromise = $this->client->getAsync(
            "$baseOdata/Price_List_Lines?" .
            '$filter=' . urlencode("SourceNo ne null and StartingDate le $today and EndingDate ge $twoMonthsAgo") .
            '&$select=Asset_No,Product_No,Unit_of_Measure_Code,DirectUnitCost,SourceNo',
            ['headers' => $headers]
        );
        $uomPromise = $this->client->getAsync(
            "$baseOdata/Item_UOM?" .
            '$select=Item_No,Code,Qty_per_Unit_of_Measure',
            ['headers' => $headers]
        );

        $priceResp = $pricePromise->wait();
        $uomResp   = $uomPromise->wait();

        $priceLines = json_decode((string) $priceResp->getBody(), true)['value'] ?? [];
        $itemUoms   = json_decode((string) $uomResp->getBody(), true)['value'] ?? [];

        // ---- UoM map per item + detect Base UoM (qty == 1 preferred) ----
        $uomMapByItem = []; // [itemNo => ['codes'=>[uom=>qty], 'base'=>uom]]
        foreach ($itemUoms as $u) {
            $item = $u['Item_No'] ?? null;
            $code = $u['Code'] ?? null;
            $qty  = (float)($u['Qty_per_Unit_of_Measure'] ?? 0);
            if (!$item || !$code) continue;

            if (!isset($uomMapByItem[$item])) $uomMapByItem[$item] = ['codes'=>[], 'base'=>null];
            $uomMapByItem[$item]['codes'][$code] = $qty;
            if ($qty == 1.0 && $uomMapByItem[$item]['base'] === null) {
                $uomMapByItem[$item]['base'] = $code;
            }
        }
        foreach ($uomMapByItem as $item => &$info) {
            if ($info['base'] === null && !empty($info['codes'])) {
                $info['base'] = array_key_first($info['codes']);
            }
        }
        unset($info);

        $bestPriceByItem = []; 
        foreach ($priceLines as $pl) {
            $itemNo   = $pl['Asset_No'] ?? ($pl['Product_No'] ?? null);
            if (!$itemNo) continue;

            $priceUom = $pl['Unit_of_Measure_Code'] ?? null;
            $direct   = (float)($pl['DirectUnitCost'] ?? 0);
            if ($direct <= 0) continue;

            $uomInfo  = $uomMapByItem[$itemNo] ?? null;
            if (!$uomInfo) continue;

            $baseUom  = $uomInfo['base'] ?? null;
            $qtyBase  = (float)($baseUom ? ($uomInfo['codes'][$baseUom] ?? 1) : 1);
            $qtyPrice = (float)($priceUom ? ($uomInfo['codes'][$priceUom] ?? 0) : 0);
            if (!$baseUom || $qtyPrice <= 0) continue;

            $pricePerBase = $direct / $qtyPrice * $qtyBase; // normalize to base uom

            if (!isset($bestPriceByItem[$itemNo]) || $pricePerBase < $bestPriceByItem[$itemNo]['price']) {
                $bestPriceByItem[$itemNo] = [
                    'price'    => $pricePerBase,
                    'priceUom' => $priceUom,
                    'baseUom'  => $baseUom,
                ];
            }
        }

        $location     = $request->query('location');
        $search       = mb_strtolower(trim($request->query('search', '')));
        $statusFilter = mb_strtolower(trim($request->query('status', '')));
        $selectedKeys = $request->input('selected', []);

        $rows = $rows->map(function ($r) {
            if (!isset($r['Consumption'])) {
                $r['Consumption'] = $r['Consumption'] ?? ($r['Consumption / 2 months'] ?? 0);
            }
            return $r;
        });

        if (!empty($location)) {
            $rows = $rows->filter(fn ($r) => ($r['Location'] ?? '') === $location);
        }
        if ($search !== '') {
            $rows = $rows->filter(function ($r) use ($search) {
                $item  = mb_strtolower($r['Item_No'] ?? '');
                $desc  = mb_strtolower($r['Description'] ?? '');
                return str_contains($item, $search) || str_contains($desc, $search);
            });
        }
        if ($statusFilter !== '') {
            $rows = $rows->filter(function ($r) use ($statusFilter) {
                $isActive = (float)($r['Consumption'] ?? 0) !== 0.0;
                return $statusFilter === 'active' ? $isActive : !$isActive;
            });
        }
        if (!empty($selectedKeys) && is_array($selectedKeys)) {
            $keySet = collect($selectedKeys)->flip();
            $rows = $rows->filter(function ($r) use ($keySet) {
                $k = ($r['Item_No'] ?? '') . '|' . ($r['Location'] ?? '');
                return $keySet->has($k);
            });
        }
        $rows = $rows->values();

        $fileName = 'sku-mapping_' . now()->format('Ymd_His') . '.xlsx';

        return \Maatwebsite\Excel\Facades\Excel::download(
            new class($rows, $bestPriceByItem, $uomMapByItem) implements
                \Maatwebsite\Excel\Concerns\FromCollection,
                \Maatwebsite\Excel\Concerns\WithHeadings,
                \Maatwebsite\Excel\Concerns\WithMapping,
                \Maatwebsite\Excel\Concerns\WithColumnFormatting,
                \Maatwebsite\Excel\Concerns\WithStyles,
                \Maatwebsite\Excel\Concerns\WithEvents,
                \Maatwebsite\Excel\Concerns\ShouldAutoSize
            {
                private $rows;
                private $bestPriceByItem;
                private $uomMapByItem;

                public function __construct($rows, $bestPriceByItem, $uomMapByItem) {
                    $this->rows = $rows;
                    $this->bestPriceByItem = $bestPriceByItem;
                    $this->uomMapByItem = $uomMapByItem;
                }

                public function headings(): array
                {
                    return [
                        'Item No',
                        'Description',
                        'Location',
                        'Stock For (Days)',
                        'Lead Time',
                        'Consumption (2 Months)',

                        'Suggest Min',
                        'Temp Min',
                        'Actual Min',
                        'Forecast Min',

                        'Suggest Max',
                        'Temp Max',
                        'Actual Max',
                        'Forecast Max',

                        'Price / Base UoM',

                        'Cost (Suggest Min)',
                        'Cost (Temp Min)',
                        'Cost (Actual Min)',
                        'Cost (Forecast Min)',

                        'Cost (Suggest Max)',
                        'Cost (Temp Max)',
                        'Cost (Actual Max)',
                        'Cost (Forecast Max)',

                        'Base UoM',
                        'Price UoM (Listed)',
                    ];
                }

                public function collection()
                {
                    return $this->rows;
                }

                public function map($r): array
                {
                    $itemNo = $r['Item_No'] ?? '';

                    $priceInfo = $this->bestPriceByItem[$itemNo] ?? null;
                    $priceBase = isset($priceInfo['price']) ? (float)$priceInfo['price'] : null;
                    $baseUom   = $priceInfo['baseUom']  ?? ($this->uomMapByItem[$itemNo]['base'] ?? '');
                    $priceUom  = $priceInfo['priceUom'] ?? '';

                    // Quantities (ensure floats)
                    $sMin = (float)($r['SuggestMinQty'] ?? 0);
                    $tMin = (float)($r['ActMinQty'] ?? 0);
                    $aMin = (float)($r['RealMinQty'] ?? 0);
                    $fMin = (float)($r['ForecMin'] ?? 0);

                    $sMax = (float)($r['SuggestMaxQty'] ?? 0);
                    $tMax = (float)($r['ActMaxQty'] ?? 0);
                    $aMax = (float)($r['RealMaxQty'] ?? 0);
                    $fMax = (float)($r['ForecMax'] ?? 0);

                    $cost = function (?float $p, float $q) {
                        return is_null($p) ? null : (float)($p * $q);
                    };

                    return [
                        $itemNo,
                        $r['Description'] ?? '',
                        $r['Location'] ?? '',
                        (int) ($r['Stock_For'] ?? 0),
                        (int) ($r['Lead Time'] ?? 0),
                        (float) ($r['Consumption'] ?? 0),

                        $sMin,
                        $tMin,
                        $aMin,
                        $fMin,

                        $sMax,
                        $tMax,
                        $aMax,
                        $fMax,

                        $priceBase,              // Price / Base UoM

                        $cost($priceBase, $sMin),
                        $cost($priceBase, $tMin),
                        $cost($priceBase, $aMin),
                        $cost($priceBase, $fMin),

                        $cost($priceBase, $sMax),
                        $cost($priceBase, $tMax),
                        $cost($priceBase, $aMax),
                        $cost($priceBase, $fMax),

                        $baseUom,
                        $priceUom,
                    ];
                }

                public function columnFormats(): array
                {
                    return [
                        'F'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, 
                        'G'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, 
                        'H'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Temp Min
                        'I'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Actual Min
                        'J'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Forecast Min
                        'K'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Suggest Max
                        'L'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Temp Max
                        'M'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Actual Max
                        'N'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Forecast Max

                        'O'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Price / Base UoM

                        'P'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Suggest Min)
                        'Q'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Temp Min)
                        'R'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Actual Min)
                        'S'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Forecast Min)
                        'T'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Suggest Max)
                        'U'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Temp Max)
                        'V'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Actual Max)
                        'W'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Forecast Max)
                        // X (Base UoM) & Y (Price UoM) are text
                    ];
                }

                public function styles(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet)
                {
                    $sheet->getStyle('A1:Y1')->getFont()->setBold(true);
                    return [];
                }

                public function registerEvents(): array
                {
                    return [
                        \Maatwebsite\Excel\Events\AfterSheet::class => function ($event) {
                            $event->sheet->getDelegate()->freezePane('A2'); // freeze header
                        },
                    ];
                }
            },
            $fileName
        );
    }

public function storeTransferLine(Request $request, BusinessCentralService $bcService)
{
    $company = Company::find(2);
    if (!$company) {
        return response()->json([
            'status' => 'error',
            'message' => 'Company dengan ID 2 tidak ditemukan'
        ], 400);
    }

    $companyId = $company->bc_company_id;

    // === Validasi header ===
    $validatedHeader = $request->validate([
        'noSeries' => 'nullable|string|max:20',
        'transSpec' => 'nullable|string|max:10',
        'transType' => 'nullable|string',
        'transferFromCode' => 'nullable|string|max:10',
        'transfertoCode' => 'nullable|string|max:10',
        'businessUnit' => 'nullable|string|max:20',
        'status' => 'nullable|string',
        'department' => 'nullable|string|max:20',
        'directTransfer' => 'nullable|boolean',
        'postingDate' => 'nullable|date',
        'assignedUserID' => 'nullable|string|max:50',
        'inventoryShipment' => 'nullable|boolean',
        'lines' => 'required|array|min:1',
        'lines.*.itemNo' => 'required|string|max:50',
        'lines.*.quantity' => 'required|numeric|min:1',
        'lines.*.uomCode' => 'required|string|max:10',
    ]);

    $bcService = new BusinessCentralService(env('BC_ACCESS_TOKEN'), $companyId);

try {
    // === STEP 1: Create Header + Lines ===
    $headerResult = $bcService->createTransferHeader($validatedHeader);

    if (!isset($headerResult['status']) || $headerResult['status'] !== 'success') {
        return response()->json([
            'status' => 'error',
            'stage' => 'header',
            'message' => $headerResult['message'] ?? 'Gagal membuat header',
        ], 400);
    }

    // Pastikan format array-nya benar
    $headerData = $headerResult['header'] ?? [];
    $documentNo = $headerData['no'] ?? null;

    if (!$documentNo) {
        \Log::error('❌ Nomor dokumen tidak ditemukan di hasil header.', [
            'headerResult' => $headerResult,
        ]);
        return response()->json([
            'status' => 'error',
            'stage' => 'header',
            'message' => 'Nomor dokumen tidak ditemukan setelah create header.'
        ], 400);
    }

    \Log::info("✅ Header berhasil dibuat. DocumentNo: {$documentNo}");

    // Jika service sudah membuat line, cukup tampilkan saja hasilnya
    if (!empty($headerResult['lines'])) {
        \Log::info("ℹ️ Service sudah mengembalikan " . count($headerResult['lines']) . " line.");
        return response()->json([
            'status' => 'success',
            'stage' => 'completed',
            'header' => $headerData,
            'lines'  => $headerResult['lines'],
            'message' => $headerResult['message'] ?? 'Transfer berhasil dibuat.',
        ]);
    }

    // === STEP 2: Jika service hanya buat header, kita lanjut buat lines manual ===
    $lines = $validatedHeader['lines'] ?? [];
    $lineResults = [];

    foreach ($lines as $index => $line) {
        \Log::info("➡️ Membuat line " . ($index + 1) . " dari " . count($lines), $line);

        // Step 2.1 – POST itemNo
        $createLine = $bcService->createTransferLine($documentNo, $line['itemNo']);
        if (!isset($createLine['status']) || $createLine['status'] !== 'success') {
            $lineResults[] = [
                'line' => $index + 1,
                'status' => 'error',
                'message' => $createLine['message'] ?? 'Gagal create line item',
            ];
            continue;
        }

        // Ambil lineNo dari hasil POST
        $lineNo = $createLine['data']['lineNo'] ?? null;
        if (!$lineNo) {
            $lineResults[] = [
                'line' => $index + 1,
                'status' => 'error',
                'message' => 'lineNo tidak ditemukan setelah create line item',
            ];
            continue;
        }

        // Step 2.2 – PATCH quantity
        $patchQty = $bcService->updateTransferLineQty($documentNo, $lineNo, $line['quantity']);
        // Step 2.3 – PATCH UoM
        $patchUom = $bcService->updateTransferLineUom($documentNo, $lineNo, $line['uomCode']);

        $qtySuccess = isset($patchQty['status']) && $patchQty['status'] === 'success';
        $uomSuccess = isset($patchUom['status']) && $patchUom['status'] === 'success';

        if ($qtySuccess && $uomSuccess) {
            $lineResults[] = [
                'line' => $index + 1,
                'status' => 'success',
                'message' => 'Line item berhasil ditambahkan.'
            ];
        } else {
            $lineResults[] = [
                'line' => $index + 1,
                'status' => 'error',
                'message' => 'Gagal update qty/UoM line item.'
            ];
        }
    }

    // === Final Response ===
    return response()->json([
        'status' => 'success',
        'stage' => 'completed',
        'header' => $headerData,
        'lines' => $lineResults,
        'message' => "Transfer {$documentNo} berhasil dibuat dengan " . count($lineResults) . " line."
    ]);

} catch (\Throwable $th) {
    \Log::error("❌ Error create transfer order: " . $th->getMessage(), [
        'trace' => $th->getTraceAsString(),
    ]);

    return response()->json([
        'status' => 'error',
        'stage' => 'exception',
        'message' => $th->getMessage(),
    ], 500);
}
}





    public function exportMissingVendor(Request $request)
    {
        $start = microtime(true);
        $user = session('user');
        $userName = $user['email'] ?? null;
        $items = session('po_items', []);
        $locationCode = $request->query('location_code');
        $baseOdata = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/ODataV4/Company('" . $this->companyId . "')";
        $baseApi = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})";
        $headers = [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=5000'
            ];
        $step1 = microtime(true);
       

        $today = now()->toDateString();
        $items = collect($items); 
        if ($locationCode) {
            $items = $items->filter(function ($item) use ($locationCode) {
                return isset($item['location_code']) && $item['location_code'] === $locationCode;
            });
        }
         $missingVendorItems = collect($items)->filter(function ($item) {
            return empty($item['vendor_no']) || $item['vendor_no'] == 0;
        });
        $itemNos = $items->pluck('item_no')->unique()->filter()->values();
        $priceList = [];

        $step2 = microtime(true);   
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=5000'
            ]
        ];

        foreach ($itemNos->chunk(200) as $chunk) {
            $assetFilter = $chunk->map(fn($no) => "Asset_No eq '$no'")->implode(' or ');
            $filter = "SourceNo ne '' and StartingDate le $today and EndingDate lt $today and ($assetFilter)";
            $encodedFilter = urlencode($filter);

            $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
                "/ODataV4/Company('" . $this->companyId . "')/Price_List_Lines?\$filter={$encodedFilter}";

            try {
                $response = $this->client->get($url, $headers);
                $data = json_decode((string)$response->getBody(), true);
                $priceList = array_merge($priceList, $data['value'] ?? []);
            } catch (\Throwable $e) {
                \Log::error("Failed to fetch price list chunk: " . $e->getMessage());
            }
        }

        
        
        $vendorNameMap = session('vendor_map', []);
        
        $inactiveVendorsByItem = collect($priceList)->groupBy('Asset_No');
        
         return Excel::download(new class($missingVendorItems, $inactiveVendorsByItem, $vendorNameMap) implements FromCollection, WithHeadings {
            private $items, $vendorMap, $vendorNameMap;

            public function __construct($items, $vendorMap, $vendorNameMap)
            {
                $this->items = $items;
                $this->vendorMap = $vendorMap;
                $this->vendorNameMap = $vendorNameMap;
            }

            public function collection()
            {
                return $this->items->map(function ($item) {
                    $itemNo = $item['item_no'] ?? '';
                    $vendorInfo = $this->vendorMap[$itemNo][0] ?? null;
                    $vendorNo = $vendorInfo['AssignToNo'] ?? '';
                    $vendorName = $this->vendorNameMap[$vendorNo] ?? 'N/A';

                    return [
                        'Kode'     => $itemNo,
                        'Item' => $item['description'] ?? '',
                        'Supplier' => $vendorName,
                        'Harga Non PPN'   => $vendorInfo['DirectUnitCost'] ?? 'N/A',
                        'UoM'   => $vendorInfo['Unit_of_Measure_Code'] ?? 'N/A'
                    ];
                });
            }

            public function headings(): array
            {
                return ['Kode', 'Item', 'Supplier', 'Harga Non PPN', 'UoM'];
            }
        }, 'missing-vendors.xlsx');
    }

    public function exportPoLines(Request $request)
    {
        $items = collect(session('po_items', []));
        $locationCode = $request->query('location_code');

        $rawStatus = $request->input('status', $request->input('statuses', 'Follow Up PO'));
        if (is_string($rawStatus)) {
            $statusList = array_filter(array_map('trim', explode(',', $rawStatus)));
        } elseif (is_array($rawStatus)) {
            $statusList = array_map('trim', $rawStatus);
        } else {
            $statusList = ['Follow Up PO'];
        }
        $statusSet = collect($statusList)->map(fn ($s) => mb_strtolower($s))->flip(); // O(1) lookup

        if ($locationCode) {
            $items = $items->filter(fn ($it) =>
                isset($it['location_code']) && $it['location_code'] === $locationCode
            );
        }

        $items = $items
        ->filter(fn ($it) => !empty($it['po_lines']) && is_array($it['po_lines']))
        ->filter(function ($it) use ($statusSet) {
            $itemStatus = mb_strtolower(trim($it['status'] ?? $it['item_status'] ?? ''));
            return $itemStatus !== '' && $statusSet->has($itemStatus);
        })
        ->values();

        return \Maatwebsite\Excel\Facades\Excel::download(
            new class($items) implements \Maatwebsite\Excel\Concerns\FromCollection,
                                    \Maatwebsite\Excel\Concerns\WithHeadings,
                                    \Maatwebsite\Excel\Concerns\ShouldAutoSize {
                private $items;
                public function __construct($items) { $this->items = $items; }

                public function headings(): array
                {
                    return [
                        'Item No',
                        'Item Description',
                        'Location',
                        'Document No',
                        'Approved Date',
                        'Vendor',
                        'Status',
                        'Outstanding Qty',
                    ];
                }

                public function collection()
                {
                    $rows = collect();

                    foreach ($this->items as $item) {
                        $itemNo = $item['item_no'] ?? '';
                        $desc   = $item['description'] ?? '';
                        $loc    = $item['location_code'] ?? '';

                        foreach ($item['po_lines'] as $po) {
                            $rows->push([
                                $itemNo,
                                $desc,
                                $loc,
                                $po['document_no'] ?? '',
                                ($po['approved_date'] ?? 'No Date'),
                                $po['vendor'] ?? '',
                                $po['status'] ?? '',
                                $po['outstanding_qty'] ?? 0,
                            ]);
                        }
                    }

                    return $rows;
                }
            },
            'po-lines.xlsx'
        );
    }

    public function exportPoLinesVisible(Request $request)
    {
        $items = collect(session('po_items', []));
        $vendorOptionsByItem = session('vendor_options_by_item', []);
        $vendorNameMap       = session('vendor_map', []);
    
        $selectedLocation = $request->query('location_code');
        $selectedVendors     = (array) $request->input('selected_vendors', []);
        $vendorKeywordsInput = trim((string) $request->query('vendor_keywords', ''));
        $vendorKeywords = collect(explode('|', $vendorKeywordsInput))
                            ->map('trim')->filter()->map(fn($v) => mb_strtolower($v))->values()->all();
    
        $itemKeywordsInput = trim((string) $request->query('item_keywords', ''));
        $itemKeywords = collect(explode('|', $itemKeywordsInput))
                            ->map('trim')->filter()->map(fn($v) => mb_strtolower($v))->values()->all();
    
        $selectedStatus      = (array) $request->input('selected_status', []);
        $selectedStatusItem  = (array) $request->input('selected_status_item', []);
        $selectedDocumentNo  = $request->query('document_no');
        $shipFrom            = $request->query('ship_from');
        $shipTo              = $request->query('ship_to');
    
        $activeParam   = $request->query('active', null);
        $selectedActive = is_null($activeParam) ? null
                        : filter_var($activeParam, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        $needShipmentParam = $request->query('need_shipment', null);
        $selectedNeedShipment = is_null($needShipmentParam)
            ? null
            : filter_var($needShipmentParam, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);

    
        $requestedColumns = (array) $request->query('columns', []);
    
        $columnMap = [
            'item_no'        => ['Item No',        fn($i) => $i['item_no'] ?? ''],
            'description'    => ['Item Description', fn($i) => $i['description'] ?? ''],
            'location_code'  => ['Location',       fn($i) => $i['location_code'] ?? ''],
            'min_qty'        => ['min_qty',        fn($i) => $i['min_qty'] ?? ''],
            'max_qty'        => ['max_qty',        fn($i) => $i['max_qty'] ?? ''],
            'stock'          => ['stock',          fn($i) => $i['stock'] ?? ''],
            'on_po'          => ['on_po',          fn($i) => $i['on_po'] ?? ''],
            'in_transfer'    => ['in_transfer',    fn($i) => $i['in_transfer'] ?? ''],
            'qty_to_order'   => ['qty_to_order',   fn($i) => $i['qty_to_order'] ?? ''],
            'status'         => ['Item Status',    fn($i) => $i['status'] ?? ''],
            'Status'         => ['Status (Item)',  fn($i) => $i['Status'] ?? ''],
            'Active'         => ['Active',         fn($i) => isset($i['Active']) ? (is_bool($i['Active']) ? ($i['Active'] ? 'true' : 'false') : (string)$i['Active']) : ''],
            'vendor_no'      => ['Vendor No',      fn($i) => $i['vendor_no'] ?? ''],
            'vendor_name'    => ['Vendor Name',    fn($i) => function() use ($i, $vendorOptionsByItem, $vendorNameMap) {
                                                      $itemNo = $i['item_no'] ?? '';
                                                      $vno    = $i['vendor_no'] ?? '';
                                                      $opt    = $vendorOptionsByItem[$itemNo][$vno]['name'] ?? ($vendorNameMap[$vno] ?? '');
                                                      return $opt ?: '';
                                                  }],
            'unit_cost'      => ['Unit Cost',      fn($i) => function() use ($i, $vendorOptionsByItem) {
                                                      $itemNo = $i['item_no'] ?? '';
                                                      $vno    = $i['vendor_no'] ?? '';
                                                      return $vendorOptionsByItem[$itemNo][$vno]['unit_cost'] ?? '';
                                                  }],
            'transfer_details' => ['Transfer Details', fn($i) => implode(' | ',
                array_map(fn($t) => ($t['document_no'] ?? '').' '.($t['shipment_date'] ?? '').' Qty:'.($t['quantity'] ?? ''),
                    (array)($i['transfer_lines'] ?? [])
                )
            )],
            'po_details' => ['PO details', fn($i) => implode(' | ',
                array_map(fn($p) => ($p['document_no'] ?? '').' '.($p['approved_date'] ?? 'No Date').' '.($p['status'] ?? '').' Qty:'.($p['outstanding_qty'] ?? 0),
                    (array)($i['po_lines'] ?? [])
                )
            )],
            'open_po_details' => ['Open PO details', fn($i) => implode(' | ',
                array_map(fn($p) => ($p['document_no'] ?? '').' '.($p['approved_date'] ?? 'No Date').' '.($p['status'] ?? '').' Qty:'.($p['outstanding_qty'] ?? 0),
                    (array)($i['openPo'] ?? [])
                )
            )],
        ];
    
        $defaultOrder = [
            'item_no','description','location_code','min_qty','max_qty','stock','on_po','in_transfer','qty_to_order','status','Status'
        ];
    
        $columns = array_values(array_filter(
            $requestedColumns ?: $defaultOrder,
            fn($k) => array_key_exists($k, $columnMap)
        ));
    

        $filtered = $items->filter(function ($item) use (
            $selectedLocation, $selectedVendors, $vendorKeywords, $itemKeywords,
            $selectedStatus, $selectedStatusItem, $selectedDocumentNo, $shipFrom,$shipTo, $selectedActive, $vendorOptionsByItem, $vendorNameMap,$selectedNeedShipment
        ) {
            $desc = strtoupper(trim($item['description'] ?? ''));
            if (str_contains($desc, 'FRUIT & VEGETABLE') ||
                str_contains($desc, 'MEAT -') ||
                str_contains($desc, 'SEAFOOD -')) {
                return false;
            }
            if ($selectedLocation && ($item['location_code'] ?? null) !== $selectedLocation) return false;
    
            $itemNo = trim((string)($item['item_no'] ?? ''));
            $itemVendorsMap = $vendorOptionsByItem[$itemNo] ?? [];
            $itemVendorNos  = array_keys($itemVendorsMap);
    
            if (!empty($selectedVendors)) {
                $hasVendor = count(array_intersect($selectedVendors, $itemVendorNos)) > 0
                          || in_array($item['vendor_no'] ?? '', $selectedVendors, true);
                if (!$hasVendor) return false;
            }
    
            if (!empty($vendorKeywords)) {
                $allVendorNamesLower = [];
                if (!empty($itemVendorsMap)) {
                    foreach ($itemVendorsMap as $vno => $vi) {
                        $name = is_array($vi) && isset($vi['name']) ? $vi['name'] : ($vendorNameMap[$vno] ?? null);
                        if ($name) $allVendorNamesLower[] = mb_strtolower($name);
                    }
                } elseif (!empty($item['vendor_no'])) {
                    $name = $vendorNameMap[$item['vendor_no']] ?? null;
                    if ($name) $allVendorNamesLower[] = mb_strtolower($name); // <- fixed typo
                }
                if (!empty($allVendorNamesLower)) {
                    $matched = false;
                    foreach ($vendorKeywords as $kw) {
                        foreach ($allVendorNamesLower as $nm) {
                            if (str_contains($nm, $kw)) { $matched = true; break 2; }
                        }
                    }
                    if (!$matched) return false;
                } else {
                    return false;
                }
            }
    
            if (!empty($itemKeywords)) {
                $searchText = mb_strtolower(($item['item_no'] ?? '') . ' ' . ($item['description'] ?? ''));
                $found = false;
                foreach ($itemKeywords as $kw) {
                    if ($kw !== '' && str_contains($searchText, $kw)) { $found = true; break; }
                }
                if (!$found) return false;
            }
    
            if (!empty($selectedStatus) && !in_array($item['status'] ?? null, $selectedStatus, true)) return false;
            if (!empty($selectedStatusItem) && !in_array($item['Status'] ?? null, $selectedStatusItem, true)) return false;
    
            if ($selectedDocumentNo) {
                $lines = $item['transfer_lines'] ?? [];
                $hasDoc = false;
                foreach ($lines as $line) {
                    if (($line['document_no'] ?? null) === $selectedDocumentNo) { $hasDoc = true; break; }
                }
                if (!$hasDoc) return false;
            }
    
            if ($shipFrom && $shipTo) {
                $lines = $item['transfer_lines'] ?? [];
                $inRange = false;
                foreach ($lines as $line) {
                    $sd = $line['shipment_date'] ?? null;
                    if ($sd && $sd >= $shipFrom && $sd <= $shipTo) { $inRange = true; break; }
                }
                if (!$inRange) return false;
            }
            
            if (!is_null($selectedNeedShipment)) {
                $needVal = (bool)($item['need_shipment'] ?? false);
                if ($needVal !== (bool)$selectedNeedShipment) return false;
            }
    
            if (!is_null($selectedActive)) {
                $activeValue = $item['Active'] ?? null;
                if (is_string($activeValue)) {
                    $activeValue = filter_var($activeValue, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
                }
                if ((bool)$activeValue !== (bool)$selectedActive) return false;
            }
    
            return true;
        })->values();

        $labels = array_map(fn($k) => $columnMap[$k][0], $columns);
    
       return Excel::download(
            new class($filtered, $columns, $columnMap)
                implements \Maatwebsite\Excel\Concerns\FromCollection,
                           \Maatwebsite\Excel\Concerns\WithHeadings,
                           \Maatwebsite\Excel\Concerns\WithEvents
            {
                private $items; private $cols; private $map;
                public function __construct($items, $cols, $map){ $this->items=$items; $this->cols=$cols; $this->map=$map; }
        
                public function headings(): array {
                    return array_map(fn($k) => $this->map[$k][0], $this->cols);
                }
        
                public function collection() {
                    return collect($this->items)->map(function($i){
                        $row = [];
                        foreach ($this->cols as $k) {
                            $val = $this->map[$k][1]($i);
                            if ($val instanceof \Closure) { $val = $val(); }
                            $row[] = is_bool($val) ? ($val ? 'true' : 'false') : $val;
                        }
                        return $row;
                    });
                }
        
                public function registerEvents(): array
                {
                    // widths per DATA KEY (not per letter) so it works with any column order
                    $widthMap = [
                        'description'       => 35, // Item Description
                        'status'            => 18, // Item Status
                        'transfer_details'  => 32, // Transfer Details
                        'po_details'        => 32, // PO details
                        'open_po_details'   => 32, // Open PO details
                        // other columns will get default width below
                    ];
        
                    // bring selected keys into the closure
                    $cols = $this->cols;
        
                    return [
                        \Maatwebsite\Excel\Events\AfterSheet::class => function(\Maatwebsite\Excel\Events\AfterSheet $event) use ($cols, $widthMap) {
                            $sheet = $event->sheet->getDelegate();
        
                            // full used range
                            $lastCol = $sheet->getHighestColumn();
                            $lastRow = $sheet->getHighestRow();
                            $range   = "A1:{$lastCol}{$lastRow}";
        
                            // borders so they show on print
                            $sheet->getStyle($range)->applyFromArray([
                                'borders' => [
                                    'allBorders' => [
                                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                                        'color'       => ['argb' => 'FF000000'],
                                    ],
                                ],
                            ]);
        
                            // header style
                            $sheet->getStyle("A1:{$lastCol}1")->applyFromArray([
                                'font'      => ['bold' => true],
                                'alignment' => ['horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER],
                            ]);
        
                            // wrap + vertical top for all
                            $sheet->getStyle($range)->getAlignment()->setWrapText(true);
                            $sheet->getStyle($range)->getAlignment()->setVertical(
                                \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_TOP
                            );
        
                            // column widths by key (robust to column order)
                            $sheet->getDefaultColumnDimension()->setWidth(10); // baseline
                            foreach ($cols as $idx => $key) {
                                $letter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($idx + 1);
                                // make sure autosize can’t override manual widths
                                $sheet->getColumnDimension($letter)->setAutoSize(false);
                                if (isset($widthMap[$key])) {
                                    $sheet->getColumnDimension($letter)->setWidth($widthMap[$key]);
                                } else {
                                    // narrow numeric-ish columns look nicer around 10–12
                                    $sheet->getColumnDimension($letter)->setWidth(10);
                                }
                            }
        
                            // page setup – A4, portrait, keep 100% scale (no "fit to 1 page" shrinking)
                            $pageSetup = $sheet->getPageSetup();
                            $pageSetup->setPaperSize(\PhpOffice\PhpSpreadsheet\Worksheet\PageSetup::PAPERSIZE_A4);
                            $pageSetup->setOrientation(\PhpOffice\PhpSpreadsheet\Worksheet\PageSetup::ORIENTATION_PORTRAIT);
                            $pageSetup->setScale(100); // << do NOT use setFitToWidth here
                            $sheet->getPageMargins()->setTop(0.4)->setRight(0.3)->setBottom(0.4)->setLeft(0.3);
        
                            // repeat header row on each page
                            $pageSetup->setRowsToRepeatAtTopByStartAndEnd(1, 1);
        
                            // borders are enough; hide gridlines on screen/print
                            $sheet->setShowGridlines(false);
                            $sheet->setPrintGridlines(false);
        
                            // print area
                            $sheet->getPageSetup()->setPrintArea($range);
                        },
                    ];
                }
            },
            'visible-table.xlsx'
        );
    }

    public function patchItemField(Request $request)
    {
        $itemNo = $request->input('item_no');
        $variantCode = $request->input('variant_code', '');
        $location = $request->input('location_code');

        $fieldsToUpdate = [];
        $fieldaToUpdate = [];

        if ($request->has('Status')) {
            $fieldsToUpdate['Status'] = $request->input('Status');
            $fieldaToUpdate['Status'] = $request->input('Status');
        }
        if ($request->has('Comment')) {
            $fieldsToUpdate['Comment'] = $request->input('Comment');
            $fieldaToUpdate['Comment'] = $request->input('Comment');
        }
        if ($request->has('Active')) {
            $fieldsToUpdate['Active'] = filter_var($request->input('Active'), FILTER_VALIDATE_BOOLEAN);
            $fieldaToUpdate['Active'] = filter_var($request->input('Active'), FILTER_VALIDATE_BOOLEAN);
        }
        
        
        try {
            $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
            $company = "/ODataV4/Company('$this->companyId')/Price_List_Lines";
            $patchUrl = $baseUrl . $company .
                "(Item_No='" . rawurlencode($itemNo) . "'," .
                "Variant_Code='" . rawurlencode($variantCode) . "'," .
                "Location_Code='" . rawurlencode($location) . "')";

            $headers = [
                'Authorization' => "Bearer " . $this->getAccessToken(),
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'If-Match' => '*'
            ];

            $response = $this->client->request('PATCH', $patchUrl, [
                'headers' => $headers,
                'body' => json_encode($fieldsToUpdate)
            ]);

            return response()->json(['message' => 'Item updated.']);
        } catch (\Exception $e) {
             \Log::warning("PATCH to APIStockkeeping failed for $itemNo at $location: " . $e->getMessage());
            
            try {
                $fallbackUrl = $baseUrl . "/ODataV4/Company('$this->companyId')/ItemInvenLoc" .
                    "(ItemNo='" . rawurlencode($itemNo) . "'," .
                    "Location='" . rawurlencode($location) . "')";

                $this->client->request('PATCH', $fallbackUrl, [
                    'headers' => $headers,
                    'body'    => json_encode($fieldaToUpdate)
                ]);
                \Log::info("Fallback PATCH succeeded for $itemNo at $location");
                return response()->json("message' => 'Item updated.");
            } catch (\Exception $fallbackError) {
                \Log::error("Both APIStockkeeping and fallback ItemInvenLoc PATCH failed for $itemNo at $location: " . $fallbackError->getMessage());
                return response()->json(['error' => 'Failed to update item.'], 500);
            }
            return response()->json(['error' => 'Failed to update item.'], 500);
        }
    }


    public function patchPrice(Request $request)
{
    $productNo     = $request->input('product_no');
    $priceListCode = $request->input('price_list_code');
    $newPrice      = $request->input('direct_unit_cost');
    $itemRequestId = $request->input('item_request_id');
    $ending_date_input  = $request->input('ending_date'); // e.g. "05/01/2026"
    $actionType    = $request->input('action');

    $baseUrl   = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
    $companyId = $this->companyId;
    $priceListUrl = $baseUrl . "/ODataV4/Company('$companyId')/Price_List_Lines";

    $headers = [
        'Authorization' => 'Bearer ' . $this->getAccessToken(),
        'Content-Type'  => 'application/json'
    ];
    
    try {
        // ===== Normalisasi tanggal (mendukung format d/m/Y seperti "05/01/2026") =====
        $ending_date = null;
        if (!empty($ending_date_input)) {
            try {
                // jika format cocok d/m/Y gunakan createFromFormat
                if (preg_match('#^\d{1,2}/\d{1,2}/\d{4}$#', trim($ending_date_input))) {
                    // pakai Carbon jika ada
                    if (class_exists(\Carbon\Carbon::class)) {
                        $ending_date = \Carbon\Carbon::createFromFormat('d/m/Y', $ending_date_input)->format('Y-m-d');
                    } else {
                        $parts = explode('/', $ending_date_input);
                        // day/month/year -> Y-m-d
                        $day = str_pad($parts[0], 2, '0', STR_PAD_LEFT);
                        $month = str_pad($parts[1], 2, '0', STR_PAD_LEFT);
                        $year = $parts[2];
                        $ending_date = "{$year}-{$month}-{$day}";
                        // boleh juga validate dengan DateTime
                        $dt = \DateTime::createFromFormat('Y-m-d', $ending_date);
                        if (!$dt || $dt->format('Y-m-d') !== $ending_date) {
                            throw new \Exception('Invalid date after parsing.');
                        }
                    }
                } else {
                    // fallback: coba parse generik (Carbon atau DateTime)
                    if (class_exists(\Carbon\Carbon::class)) {
                        $ending_date = \Carbon\Carbon::parse($ending_date_input)->toDateString(); // "YYYY-MM-DD"
                    } else {
                        $ending_date = (new \DateTime($ending_date_input))->format('Y-m-d');
                    }
                }
            } catch (\Exception $e) {
                return response()->json(['error' => 'Format ending_date tidak valid. Contoh yang diterima: "05/01/2026" atau "2026-01-05".'], 422);
            }
        }

        // ===== Bangun filter OData =====
        $filterParts = [
            "Price_List_Code eq '{$priceListCode}'",
            "Product_No eq '{$productNo}'"
        ];
        if ($ending_date !== null) {
            // Edm.Date -> literal YYYY-MM-DD tanpa kutip
            $filterParts[] = "EndingDate eq {$ending_date}";
        }
        $filterString = implode(' and ', $filterParts);

        // urlencode seluruh filter agar aman di URL
        $filter = rawurlencode($filterString);
        $getUrl = "$priceListUrl?\$filter=$filter";

        // --- 1️⃣ GET data Price_List_Line dari BC
        $response = $this->client->request('GET', $getUrl, ['headers' => $headers]);
        $data = json_decode($response->getBody()->getContents(), true);

        if (empty($data['value'])) {
            return response()->json(['error' => 'Price list line tidak ditemukan.'], 404);
        }

        $line     = $data['value'][0];
        $lineNo   = $line['Line_No'] ?? null;
        $etag     = $line['@odata.etag'] ?? '*';

        if (!$lineNo) {
            return response()->json(['error' => 'Line_No tidak ditemukan.'], 404);
        }

        // --- 2️⃣ PATCH ke BC
        $patchUrl = $priceListUrl . "(Price_List_Code='" . rawurlencode($priceListCode) . "',Line_No=$lineNo)";
        $headers['If-Match'] = $etag;

        $fieldsToUpdate = [
            'DirectUnitCost' => (float) $newPrice
        ];

        $this->client->request('PATCH', $patchUrl, [
            'headers' => $headers,
            'body'    => json_encode($fieldsToUpdate)
        ]);
    
        // --- 4️⃣ Log request
        \App\Models\RequestsLog::create([
            'item_request_id' => $itemRequestId,
            'request_type'    => 'changeprice',
            'action'          => $actionType ?? 'update',
            'requestby'       => auth()->id(),
        ]);

        return response()->json([
            'message'   => 'Harga berhasil diperbarui',
            'new_price' => $newPrice
        ]);

    } catch (\GuzzleHttp\Exception\RequestException $e) {
        $responseBody = $e->hasResponse() ? (string) $e->getResponse()->getBody() : 'No response body';
        $decoded = json_decode($responseBody, true);
        return response()->json([
            'error'   => 'Gagal memperbarui harga.',
            'details' => $decoded ?? $responseBody
        ], 500);
    } catch (\Exception $e) {
        return response()->json([
            'error' => 'Terjadi kesalahan internal.',
            'message' => $e->getMessage()
        ], 500);
    }
}




   public function patchEndingDate(Request $request)
{
    $productNo     = $request->input('product_no');
    $priceListCode = $request->input('price_list_code');
    $newEndingDate = $request->input('new_ending_date');
    $unitCost      = $request->input('direct_unit_cost');
    $itemRequestId = $request->input('item_request_id'); // tambahkan item_request_id dari request

    $baseUrl   = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
    $companyId = $this->companyId;
    $priceListUrl = $baseUrl . "/ODataV4/Company('$companyId')/Price_List_Lines";

    // Format tanggal BC
    $formattedDate = date('Y-m-d', strtotime($newEndingDate));

    $headers = [
        'Authorization' => 'Bearer ' . $this->getAccessToken(),
        'Content-Type'  => 'application/json'
    ];

    try {
        // --- 1️⃣ GET data Price_List_Line berdasarkan Product_No, Price_List_Code, DirectUnitCost
        $filter = urlencode("Price_List_Code eq '$priceListCode' and Product_No eq '$productNo' and DirectUnitCost eq $unitCost");
        $getUrl = "$priceListUrl?\$filter=$filter";

        \Log::info("🔍 Mencari Price_List_Line: $getUrl");

        $response = $this->client->request('GET', $getUrl, ['headers' => $headers]);
        $data = json_decode($response->getBody()->getContents(), true);

        if (empty($data['value'])) {
            \Log::warning("❌ Tidak ditemukan Price_List_Line untuk $productNo / $priceListCode / $unitCost");
            return response()->json(['error' => 'Price list line tidak ditemukan.'], 404);
        }

        $line   = $data['value'][0];
        $lineNo = $line['Line_No'] ?? null;
        $etag   = $line['@odata.etag'] ?? '*';
        $oldEndingDate = $line['EndingDate'] ?? null; // ambil tanggal lama untuk log

        if (!$lineNo) {
            \Log::warning("⚠️ Line_No tidak ditemukan untuk $productNo / $priceListCode / $unitCost");
            return response()->json(['error' => 'Line_No tidak ditemukan.'], 404);
        }

        // --- 2️⃣ Siapkan URL PATCH dan header If-Match
        $patchUrl = $priceListUrl . "(Price_List_Code='" . rawurlencode($priceListCode) . "',Line_No=$lineNo)";
        $headers['If-Match'] = $etag;

        $fieldsToUpdate = [
            'EndingDate' => $formattedDate
        ];

        // --- 3️⃣ PATCH ke BC
        \Log::info("🚀 PATCH ke: $patchUrl dengan EndingDate=$formattedDate, If-Match=$etag");

        $patchResponse = $this->client->request('PATCH', $patchUrl, [
            'headers' => $headers,
            'body'    => json_encode($fieldsToUpdate)
        ]);

        // --- 4️⃣ Simpan log RequestsLog
        \App\Models\RequestsLog::create([
            'item_request_id' => $itemRequestId,
            'request_type'    => 'extdate',
            'action'          => 'update',
            'requestby'    => auth()->id(),
        ]);

        return response()->json([
            'message' => 'Ending Date berhasil diperbarui',
            'new_ending_date' => $formattedDate
        ]);

    } catch (\GuzzleHttp\Exception\RequestException $e) {
        $responseBody = $e->hasResponse() ? (string) $e->getResponse()->getBody() : 'No response body';

        \Log::error("❌ PATCH gagal untuk $productNo / $priceListCode / $unitCost: " . $e->getMessage());
        \Log::error("📦 Full Response Body: " . $responseBody);

        return response()->json([
            'error'   => 'Gagal memperbarui Ending Date.',
            'details' => json_decode($responseBody, true)
        ], 500);
    } catch (\Exception $e) {
        \Log::error("⚠️ PATCH gagal (non-HTTP) untuk $productNo / $priceListCode / $unitCost: " . $e->getMessage());
        return response()->json(['error' => $e->getMessage()], 500);
    }
}


public function createPrice(Request $request)
{
    $priceListCode     = $request->input('price_list_code');
    $productNo         = $request->input('product_no');
    $newPrice          = $request->input('direct_unit_cost');
    $itemRequestId     = $request->input('item_request_id');
    $ending_date_input = $request->input('ending_date');

    $baseUrl   = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
    $companyId = $this->companyId;
    $url       = $baseUrl . "/api/citbi/sku/v1.0/companies(" . rawurlencode($companyId) . ")/priceListLinesCustom";
    $priceListUrl = $baseUrl . "/ODataV4/Company('$companyId')/Price_List_Lines";

    $headers = [
        'Authorization' => 'Bearer ' . $this->getAccessToken(),
        'Content-Type'  => 'application/json',
        'Accept'        => 'application/json'
    ];

    try {
        // 🗓️ Normalisasi tanggal ending_date
        $ending_date = null;
        if (!empty($ending_date_input)) {
            try {
                if (preg_match('#^\d{1,2}/\d{1,2}/\d{4}$#', trim($ending_date_input))) {
                    $ending_date = \Carbon\Carbon::createFromFormat('d/m/Y', $ending_date_input)->format('Y-m-d');
                } else {
                    $ending_date = \Carbon\Carbon::parse($ending_date_input)->format('Y-m-d');
                }
            } catch (\Exception $e) {
                return response()->json(['error' => 'Format ending_date tidak valid.'], 422);
            }
        }

        // 🔍 Coba ambil oldLine jika ada
        $filter = "priceListCode eq '{$priceListCode}' and assetNo eq '{$productNo}'";
        $response = $this->client->get("$url?\$filter=" . rawurlencode($filter), ['headers' => $headers]);
        $data = json_decode($response->getBody(), true);
        $oldLine = $data['value'][0] ?? null;

        // 🔢 Ambil line terakhir
        $respLast = $this->client->get("$url?\$filter=" . rawurlencode("priceListCode eq '{$priceListCode}'") . "&\$orderby=lineNo desc&\$top=1", ['headers' => $headers]);
        $lastData = json_decode($respLast->getBody(), true);
        $newLineNo = (($lastData['value'][0]['lineNo'] ?? 0) + 10000);

        // === STEP 1: POST minimal line baru ===
        $postPayload = [
            'assignToType'  => $oldLine['assignToType'] ?? 'Vendor',
            'priceListCode' => $priceListCode,
            'assetNo'       => $productNo,
            'lineNo'        => $newLineNo,
            'unitCost'      => (float)$newPrice,
        ];

        $postResponse = $this->client->post($url, [
            'headers' => $headers,
            'body'    => json_encode($postPayload)
        ]);
        $postBody = json_decode($postResponse->getBody(), true);

        $lineNo = $newLineNo;
        $patchUrl = $priceListUrl . "(Price_List_Code='" . rawurlencode($priceListCode) . "',Line_No=$lineNo)";
        $headers['If-Match'] = '*';

        // === STEP 2: PATCH 1 - AssignToType ===
        $patch1 = [
            'AssignToType' => $oldLine['assignToType'] ?? 'Vendor',
        ];
        try {
            $this->client->patch($patchUrl, ['headers' => $headers, 'body' => json_encode($patch1)]);
            \Log::info('✅ Patch 1: AssignToType OK', ['line' => $lineNo]);
        } catch (\Throwable $e) {
            \Log::error('❌ Patch 1 gagal', ['err' => $e->getMessage()]);
        }

        // === STEP 3: PATCH 2 - AssignToNo ===
        $patch2 = [
            'AssignToNo' => $oldLine['priceListCode'] ?? $priceListCode,
        ];
        try {
            $this->client->patch($patchUrl, ['headers' => $headers, 'body' => json_encode($patch2)]);
            \Log::info('✅ Patch 2: AssignToNo OK', ['line' => $lineNo]);
        } catch (\Throwable $e) {
            \Log::error('❌ Patch 2 gagal', ['err' => $e->getMessage()]);
        }

        // === STEP 4: PATCH 3 - StartingDate ===
        $startingDate = now()->subDay()->toDateString();
        $patch3 = [
            'StartingDate' => $startingDate,
        ];
        try {
            $this->client->patch($patchUrl, ['headers' => $headers, 'body' => json_encode($patch3)]);
            \Log::info('✅ Patch 3: StartingDate OK', ['line' => $lineNo]);
        } catch (\Throwable $e) {
            \Log::error('❌ Patch 3 gagal', ['err' => $e->getMessage()]);
        }

        // === STEP 5: PATCH 4 - EndingDate + misc fields ===
        // 💡 Jika tidak ada oldLine dan tidak ada input ending_date, buat default 1 bulan dari startingDate
        if (!$oldLine && !$ending_date) {
            $ending_date = \Carbon\Carbon::parse($startingDate)->addMonth()->format('Y-m-d');
        }

        $patch4 = [
            'Allow_Invoice_Disc'     => true,
            'Allow_Line_Disc'        => true,
            'VATBusPostingGrPrice'   => $oldLine['vatPostingGroup'] ?? 'NON VAT',
            'Status'                 => 'Active'
        ];
        if ($ending_date) {
            $patch4['EndingDate'] = $ending_date;
        }

        try {
            $this->client->patch($patchUrl, ['headers' => $headers, 'body' => json_encode($patch4)]);
            \Log::info('✅ Patch 4: Misc fields OK', ['line' => $lineNo]);
        } catch (\Throwable $e) {
            \Log::error('❌ Patch 4 gagal', ['err' => $e->getMessage()]);
        }

        // === STEP 6: Simpan log ke DB ===
        \App\Models\RequestsLog::create([
            'item_request_id' => $itemRequestId,
            'request_type'    => 'changeprice',
            'action'          => 'create',
            'requestby'       => auth()->id(),
        ]);

        return response()->json([
            'success' => true,
            'message' => '✅ Line baru berhasil dibuat & semua patch sukses dijalankan',
            'data'    => $postBody
        ], 201);

    } catch (\GuzzleHttp\Exception\RequestException $e) {
        $body = $e->hasResponse() ? (string)$e->getResponse()->getBody() : 'No response';
        return response()->json(['error' => 'BC Request Error', 'details' => $body], 500);
    } catch (\Throwable $e) {
        return response()->json(['error' => $e->getMessage()], 500);
    }
}




 public function createExtDate(Request $request)
{
    $priceListCode = $request->input('price_list_code');
    $productNo     = $request->input('product_no');
    $newEndingDate = $request->input('new_ending_date');
    $itemRequestId = $request->input('item_request_id');
    $newPrice      = $request->input('direct_unit_cost');

    $baseUrl   = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
    $companyId = $this->companyId;
    $url       = $baseUrl . "/api/citbi/sku/v1.0/companies(" . rawurlencode($companyId) . ")/priceListLinesCustom";
    $priceListUrl = $baseUrl . "/ODataV4/Company('$companyId')/Price_List_Lines";

    $headers = [
        'Authorization' => 'Bearer ' . $this->getAccessToken(),
        'Content-Type'  => 'application/json',
        'Accept'        => 'application/json'
    ];

    try {
        // --- 1️⃣ Ambil line lama
        $filter = urlencode("priceListCode eq '$priceListCode' and assetNo eq '$productNo'");
        $getUrl = "$url?\$filter=$filter";
        $response = $this->client->request('GET', $getUrl, ['headers' => $headers]);
        $data = json_decode($response->getBody()->getContents(), true);

        if (empty($data['value'])) {
            return response()->json(['error' => 'Line lama tidak ditemukan di BC.'], 404);
        }

        $oldLine = $data['value'][0];

        // --- 2️⃣ Dapatkan line terakhir untuk lineNo baru
        $getLast = "$url?\$filter=" . urlencode("priceListCode eq '$priceListCode'") . "&\$orderby=lineNo desc&\$top=1";
        $respLast = $this->client->request('GET', $getLast, ['headers' => $headers]);
        $lastData = json_decode($respLast->getBody()->getContents(), true);
        $newLineNo = ($lastData['value'][0]['lineNo'] ?? 0) + 10000;

        // --- 3️⃣ Buat line baru (payload minimal)
        $payload = [
            'assignToType' => $oldLine['assignToType'],
            'priceListCode' => $oldLine['priceListCode'],
            'assetNo'       => $productNo,
            'lineNo'        => $newLineNo,
            'unitCost'      => (float)$newPrice,
        ];

        $postResponse = $this->client->request('POST', $url, [
            'headers' => $headers,
            'body'    => json_encode($payload)
        ]);
        $postBody = json_decode($postResponse->getBody()->getContents(), true);

        $lineNo = $newLineNo; // line baru hasil POST
        $patchUrl = $priceListUrl . "(Price_List_Code='" . rawurlencode($priceListCode) . "',Line_No=$lineNo)";
        $headers['If-Match'] = '*';

        // --- Patch 1: AssignToNo
        $patchFields1 = ['AssignToNo' => $oldLine['priceListCode']];
        try {
            $this->client->request('PATCH', $patchUrl, ['headers' => $headers, 'body' => json_encode($patchFields1)]);
            \Log::info('✅ Patch 1 berhasil', ['lineNo' => $lineNo, 'fields' => $patchFields1]);
        } catch (\GuzzleHttp\Exception\RequestException $e) {
            \Log::error('❌ Patch 1 gagal', ['lineNo' => $lineNo, 'error' => $e->getMessage()]);
        }

        // --- Patch 2: startingDate
        $patchFields2 = ['startingDate' => now()->subDay()->toDateString()];
        try {
            $this->client->request('PATCH', $patchUrl, ['headers' => $headers, 'body' => json_encode($patchFields2)]);
            \Log::info('✅ Patch 2 berhasil', ['lineNo' => $lineNo, 'fields' => $patchFields2]);
        } catch (\GuzzleHttp\Exception\RequestException $e) {
            \Log::error('❌ Patch 2 gagal', ['lineNo' => $lineNo, 'error' => $e->getMessage()]);
        }

        // --- Patch 3: EndingDate + fields lainnya
        $patchFields3 = [
            'EndingDate' => $newEndingDate,
            'Allow_Invoice_Disc' => 'true',
            'Allow_Line_Disc' => 'true',
            'VATBusPostingGrPrice' => $oldLine['vatPostingGroup'], 
        ];
        try {
            $this->client->request('PATCH', $patchUrl, ['headers' => $headers, 'body' => json_encode($patchFields3)]);
            \Log::info('✅ Patch 3 berhasil', ['lineNo' => $lineNo, 'fields' => $patchFields3]);
        } catch (\GuzzleHttp\Exception\RequestException $e) {
            $responseBody = $e->hasResponse() ? (string) $e->getResponse()->getBody() : 'No response body';
            \Log::error('❌ Patch 3 gagal', ['lineNo' => $lineNo, 'error' => $e->getMessage(), 'response' => $responseBody]);
        }

        // --- Ambil line terbaru untuk memastikan EndingDate tersimpan
        try {
            $getUpdatedLine = $this->client->request('GET', $patchUrl, ['headers' => $headers]);
            $updatedLine = json_decode($getUpdatedLine->getBody()->getContents(), true);
            $updatedEndingDate = $updatedLine['EndingDate'] ?? $newEndingDate;
        } catch (\Exception $e) {
            \Log::error('❌ Gagal ambil line terbaru', ['lineNo' => $lineNo, 'error' => $e->getMessage()]);
            $updatedEndingDate = $newEndingDate; // fallback
        }

        // --- Log request
        \App\Models\RequestsLog::create([
            'item_request_id' => $itemRequestId,
            'request_type'    => 'extdate',
            'action'          => 'create',
            'requestby'       => auth()->id(),
        ]);

        return response()->json([
            'message' => '✅ Line baru berhasil dibuat dan harga diperbarui',
            'new_line' => $postBody,
            'new_ending_date' => $updatedEndingDate
        ], 201);

    } catch (\GuzzleHttp\Exception\RequestException $e) {
        $responseBody = $e->hasResponse() ? (string) $e->getResponse()->getBody() : 'No response body';
        return response()->json(['error' => 'Gagal membuat/patch line baru.', 'details' => json_decode($responseBody, true)], 500);
    } catch (\Exception $e) {
        return response()->json(['error' => $e->getMessage()], 500);
    }
}


public function createItems(Request $request, BusinessCentralService $bcService)
{
    ini_set('max_execution_time', 300);
    set_time_limit(300);

    $request->validate([
        'company_id' => 'required|integer',
        'lines' => 'required|array|min:1',
    ]);

    $companyId = $request->input('company_id');
    $inputLines = $request->input('lines');
    $itemRequestId = $request->input('item_request_id');

    $bcService = new BusinessCentralService(env('BC_ACCESS_TOKEN'), $companyId);

    // Ambil semua vendor dari BC
    $vendorsResponse = $bcService->getVendor();
    $vendorList = collect($vendorsResponse['value'] ?? []);

    // Ambil data item untuk auto-number
    $rawBcData = $bcService->getData('items');
    $groupedItems = $rawBcData['data']['value'] ?? [];

    $results = [];

    // Track last number per type agar nomor lanjut saat multi-item
    $lastNumberPerType = [];

    foreach ($inputLines as $line) {
        $type = strtoupper($line['type'] ?? 'OTHER');

        // Ambil vendor_no dari vendor_name
        $vendorName = trim(strtolower($line['vendor_name'] ?? ''));
        $vendorNo = null;
        if ($vendorName !== '') {
            $foundVendor = $vendorList->first(fn($v) => strtolower($v['displayName'] ?? '') === $vendorName);
            $vendorNo = $foundVendor['number'] ?? null;
        }

        // Ambil last number dari BC atau dari track internal
        if (!isset($lastNumberPerType[$type])) {
            $group = $groupedItems[$type] ?? [];
            $lastItem = !empty($group) ? end($group) : null;
            $lastNumber = $lastItem['number'] ?? null;

            $num = 1;
            if ($lastNumber && preg_match('/\d+$/', $lastNumber, $matches)) {
                $num = (int)$matches[0] + 1;
            }

            $lastNumberPerType[$type] = $num;
        }

        // Ambil nomor untuk item saat ini
        $num = $lastNumberPerType[$type];
        $newNumber = $type . '.' . str_pad($num, 4, '0', STR_PAD_LEFT);

        // Increment untuk item berikutnya
        $lastNumberPerType[$type]++;

        // Unit price
        $unitPrice = isset($line['unitPrice']) && is_numeric($line['unitPrice']) ? (float)$line['unitPrice'] : 0.0;

        // Payload ke BC
        $itemPayload = [
            'number' => $newNumber,
            'displayName' => $line['displayName'] ?? 'No Name',
            'description' => $line['displayName'] ?? 'No Desc', 
            'baseUnitOfMeasureCode' => $line['baseUnitOfMeasureCode'] ?? 'PCS',
            'generalProductPostingGroupCode' => $line['generalProductPostingGroupCode'] ?? 'DEFAULT',
            'inventoryPostingGroupCode' => $line['inventoryPostingGroupCode'] ?? 'DEFAULT',
            'unitPrice' => $unitPrice,
        ];

        // Create Item ke BC
        $createdItem = $bcService->createItem($itemPayload);

        // Simpan hasil, gunakan + agar field dari line tidak hilang
       $results[] = array_merge($createdItem, [
        'vendor_no' => $vendorNo,
        'vendor_name' => $vendorName,
        'ending_date' => $line['ending_date'] ?? null,
        'unitPrice' => $unitPrice,
        'number' => $newNumber,
        'displayName' => $line['displayName'] ?? 'No Name',
        'description' => $line['displayName'] ?? 'No Desc',
        'baseUnitOfMeasureCode' => $line['baseUnitOfMeasureCode'] ?? 'PCS',
        'generalProductPostingGroupCode' => $line['generalProductPostingGroupCode'] ?? 'DEFAULT',
        'inventoryPostingGroupCode' => $line['inventoryPostingGroupCode'] ?? 'DEFAULT',
    ]);
    }

    // Update status request
    ItemRequest::where('id', $itemRequestId)->update(['up_to_bc' => 'yes']);

    return response()->json([
        'success' => true,
        'message' => 'Items created successfully',
        'data' => $results
    ]);
}


public function createVendor(Request $request, BusinessCentralService $bcService)
{
    ini_set('max_execution_time', 300);
    set_time_limit(300);

    $request->validate([
        'company_id' => 'required|integer',
        'lines' => 'required|array|min:1',
    ]);

    $companyId = $request->input('company_id');
    $inputLines = $request->input('lines');
    $itemRequestId = $request->input('item_request_id');

    $bcService = new BusinessCentralService(env('BC_ACCESS_TOKEN'), $companyId);

    // Ambil semua vendor dari BC dalam bentuk map
    $vendorMap = $bcService->getAllVendorsMap() ?? [];

    $results = [];
    $lastNumberPerType = [];

    foreach ($inputLines as $line) {
        $type = strtoupper($line['type'] ?? 'OTH');

        // Cari last number untuk type ini dari vendorMap
        if (!isset($lastNumberPerType[$type])) {
            $filteredKeys = array_filter(array_keys($vendorMap), fn($key) => str_starts_with($key, $type));
            if ($filteredKeys) {
                // Ambil key terakhir secara numerik
                usort($filteredKeys, function($a, $b) use ($type) {
                    $numA = (int)substr($a, strlen($type));
                    $numB = (int)substr($b, strlen($type));
                    return $numA <=> $numB;
                });
                $lastKey = end($filteredKeys);
                $lastNumber = (int)substr($lastKey, strlen($type));
                $lastNumberPerType[$type] = $lastNumber + 1;
            } else {
                $lastNumberPerType[$type] = 1;
            }
        }

        // Generate nomor baru
        $num = $lastNumberPerType[$type];
        $newNumber = $type . str_pad($num, 4, '0', STR_PAD_LEFT);
        $lastNumberPerType[$type]++; // increment untuk vendor berikutnya

        // Siapkan payload
        $vendorPayload = [
            'number' => $newNumber,
            'displayName' => $line['displayName'] ?? 'No Name',
            'addressLine1' => $line['addressLine1'] ?? null,
            'addressLine2' => $line['addressLine2'] ?? null,
            'city' => $line['city'] ?? null,
            'state' => $line['state'] ?? null,
            'country' => $line['country'] ?? null,
            'postalCode' => $line['postalCode'] ?? null,
            'phoneNumber' => $line['phoneNumber'] ?? null,
            'email' => $line['email'] ?? null,
            'website' => $line['website'] ?? null,
            'taxRegistrationNumber' => $line['taxRegistrationNumber'] ?? null,
            'currencyId' => $line['currencyId'] ?? null,
            'paymentTermsId' => $line['paymentTermsId'] ?? null,
            'paymentMethodId' => $line['paymentMethodId'] ?? null,
            'taxLiable' => $line['taxLiable'] ?? false,
            'blocked' => $line['blocked'] ?? null,
        ];

        // Create vendor ke BC
        $createdVendor = $bcService->createVendor($vendorPayload);

        // Simpan hasil
        $results[] = array_merge($createdVendor, [
            'number' => $newNumber,
            'type' => $type,
            'displayName' => $line['displayName'] ?? 'No Name',
            'phoneNumber' => $line['phoneNumber'] ?? null,
            'email' => $line['email'] ?? null,
        ]);
    }

    // Update status request
    if ($itemRequestId) {
        ItemRequest::where('id', $itemRequestId)->update(['up_to_bc' => 'yes']);
    }

    return response()->json([
        'success' => true,
        'message' => 'Vendors created successfully',
        'data' => $results
    ]);
}




    public function create(Request $request, BusinessCentralService $bcService)
    {
        $user = session('user');
        $userName = $user['email'] ?? null;
        $selectedItems = array_unique($request->input('selected_items', []));
        $vendorMap = $request->input('vendor_no', []);
        $qtyMap = $request->input('qty_to_order', []);
        $unitCost = $request->input('unit_cost', []);
        $locationMap = $request->input('location_code', []);

        $groupedByVendor = [];
        
        foreach ($selectedItems as $itemNo) {
            $vendor = $vendorMap[$itemNo] ?? null;
            $qty = $qtyMap[$itemNo] ?? 0;
            $cost = $unitCost[$itemNo];
            $loc  = $locationMap[$itemNo] ?? '';

            if (!$vendor || $vendor === '0' || $qty <= 0) continue;

            $groupedByVendor[$vendor][] = [
                'item_no' => $itemNo,
                'quantity' => $qty,
                'cost' => $cost,
                'location' => $loc
            ];
        }
        $results = [];
        $successes = [];
        $failures  = [];

        foreach ($groupedByVendor as $vendorNo => $items) {
             $poLocation = $items[0]['location'] ?? '';
             $po = $bcService->createPurchaseOrderForVendor($vendorNo, $userName, $poLocation);
             $promises = [];

             foreach ($items as $line) {
                $response = $bcService->addPurchaseLineToPO(
                    $po['number'],
                    $line['item_no'],
                    $line['quantity'],
                    $line['cost'],
                    $line['location'],
                    $userName,
                    $po['id']
                );
            }

             $results[] = [
                 'vendor'     => $vendorNo,
                 'po_no'      => $po['number'] ?? 'N/A',
                 'line_count' => count($items),
             ];
         }
    
        return response()->json([
             'success' => 'Purchase Orders created successfully.',
             'createdPOs' => $results,
        ]);
    }

    public function updateSku(Request $request)
    {
        set_time_limit(3000);
        $updatedRows = $request->input('rows', []);
        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
        $company = "/ODataV4/Company('" . $this->companyId . "')/SKUImport";
        $headers = [
            'Authorization' => "Bearer {$this->token}",
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
            'If-Match' => '*'
        ];

        foreach ($updatedRows as $row) {
            $patchUrl = $baseUrl . $company . "(Item_No='" . rawurlencode($row['Item_No']) . "',Location='" . rawurlencode($row['Location']) . "')";

            $body = [
                'StockFor' => $row['Stock_For'],
                'LeadTime' => $row['Lead Time'],
                'MinInven' => $row['ActMinQty'],
                'MaxInven' => $row['ActMaxQty'],
                'Comment' => $row['Comment'] ?? ''
            ];

            try {
                $this->client->request('PATCH', $patchUrl, [
                    'headers' => $headers,
                    'body' => json_encode($body)
                ]);
            } catch (\Exception $e) {
                \Log::warning("PATCH failed for SKU {$row['Item_No']}, trying POST. Reason: " . $e->getMessage());

                try {
                    $this->client->request('POST', $baseUrl . $company, [
                        'headers' => $headers,
                        'body' => json_encode(array_merge([
                            'Item_No' => $row['Item_No'],
                            'Location' => $row['Location']
                        ], $body))
                    ]);
                } catch (\Exception $e2) {
                    \Log::error("Failed to POST SKU {$row['Item_No']}: " . $e2->getMessage());
                }
            }
        }

        return response()->json(['message' => 'SKU update process completed']);
    }


    public function updateStockkeeping(Request $request)
    {
        set_time_limit(6000);

        $selectedRows = $request->input('rows', []);

        $this->updateSku($request);

        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
        $company = "/ODataV4/Company('" . $this->companyId . "')/APIStockkeeping";

        $headers = [
            'Authorization' => "Bearer {$this->token}", 
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
            'If-Match' => '*'
        ];

        foreach ($selectedRows as $row) {
            try {
                $patchUrl = $baseUrl . $company .
                    "(Item_No='" . rawurlencode($row['Item_No']) . "'," .
                    "Variant_Code='" . rawurlencode($row['Variant_Code'] ?? '') . "'," .
                    "Location_Code='" . rawurlencode($row['Location']) . "')";

                $body = [
                    'MinInven' => $row['ActMinQty'],
                    'MaxInven' => $row['ActMaxQty']
                ];

                $this->client->request('PATCH', $patchUrl, [
                    'headers' => $headers,
                    'body'    => json_encode($body)
                ]);
            } catch (\Exception $e) {
                \Log::error("Failed to patch Stockkeeping for {$row['Item_No']} - {$row['Location']}: " . $e->getMessage());
                return response()->json(['error' => 'Stockkeeping update failed. See logs.'], 500);
            }
        }

        return response()->json(['message' => 'Stockkeeping updated successfully']);
    }


public function getCombinedStock()
{
    $bc = new BusinessCentralService();
    try {
        $data = $bc->getCombinedStock();
        return response()->json($data);
    } catch (\Throwable $e) {
        return response()->json([
            'status' => 'error',
            'message' => $e->getMessage()
        ], 500);
    }
}

public function getCombinedBOMWithStock(BusinessCentralService $bc)
{
    $data = $bc->getCombinedBOMWithStock();
    return response()->json($data);
}

 public function createAssemblyHeader(Request $request, BusinessCentralService $bc)
    {
        $validated = $request->validate([
            'itemNo'       => 'required|string',
            'postingDate'  => 'required|date_format:Y-m-d',
            'quantity'     => 'required|numeric|min:1',
        ]);

        $result = $bc->createAssemblyHeader($validated);

        if ($result['status'] === 'success') {
            return response()->json($result, 200);
        }

        return response()->json($result, 500);
    }

}
