<?php
require_once '../config/connection.php';
checkUserType(['admin']);

// Debug: Check what's in session
// Uncomment line below to see session content
// echo "<!-- Debug Session: " . print_r($_SESSION, true) . " -->";

// Get admin's master organization - Try multiple approaches
$admin_data = null;

// Try 1: Using session user_id as ID
$stmt = $pdo->prepare("
    SELECT id, master_organization_id, organization, full_name, nik, employee_id
    FROM admins 
    WHERE id = ?
");
$stmt->execute([$_SESSION['user_id']]);
$admin_data = $stmt->fetch();

// Try 2: If not found, try using employee_id
if (!$admin_data && isset($_SESSION['employee_id'])) {
    $stmt = $pdo->prepare("
        SELECT id, master_organization_id, organization, full_name, nik, employee_id
        FROM admins 
        WHERE employee_id = ?
    ");
    $stmt->execute([$_SESSION['employee_id']]);
    $admin_data = $stmt->fetch();
}

// Try 3: If still not found, try using user_name
if (!$admin_data && isset($_SESSION['user_name'])) {
    $stmt = $pdo->prepare("
        SELECT id, master_organization_id, organization, full_name, nik, employee_id
        FROM admins 
        WHERE full_name = ?
    ");
    $stmt->execute([$_SESSION['user_name']]);
    $admin_data = $stmt->fetch();
}

// Final check
if (!$admin_data) {
    echo "<div style='padding: 20px; background: #fee; border: 1px solid red; margin: 20px;'>";
    echo "<h3>Debug Information:</h3>";
    echo "<pre>";
    echo "Session user_id: " . ($_SESSION['user_id'] ?? 'NOT SET') . "\n";
    echo "Session user_name: " . ($_SESSION['user_name'] ?? 'NOT SET') . "\n";
    echo "Session user_type: " . ($_SESSION['user_type'] ?? 'NOT SET') . "\n";
    echo "\nAll session data:\n";
    print_r($_SESSION);
    echo "</pre>";
    echo "<p><a href='../logout.php'>Logout dan Login Ulang</a></p>";
    echo "</div>";
    die("Error: Data admin tidak ditemukan di database.");
}

$admin_id = $admin_data['id'];
$admin_master_org_id = $admin_data['master_organization_id'];
$admin_organization = $admin_data['organization'] ?? 'Tidak ada';
$admin_name = $admin_data['full_name'];
$admin_nik = $admin_data['nik'] ?? '-';

// Get master organization info
$master_org_info = null;
if ($admin_master_org_id) {
    $stmt = $pdo->prepare("SELECT * FROM master_organization WHERE id = ?");
    $stmt->execute([$admin_master_org_id]);
    $master_org_info = $stmt->fetch();
}

// Get borrowers in the same master organization who are currently borrowing
$borrowers_borrowing = [];
if ($admin_master_org_id) {
    $stmt = $pdo->prepare("
        SELECT DISTINCT
            b.id,
            b.nik,
            b.employee_id,
            b.full_name,
            b.organization,
            b.branch_name,
            b.jabatan,
            COUNT(DISTINCT bh.id) as total_borrowed
        FROM borrowers b
        INNER JOIN borrow_history bh ON b.employee_id = bh.borrower_employee_id
        WHERE b.master_organization_id = ?
        AND bh.status = 'borrowed'
        GROUP BY b.id, b.nik, b.employee_id, b.full_name, b.organization, b.branch_name, b.jabatan
        ORDER BY b.full_name ASC
    ");
    $stmt->execute([$admin_master_org_id]);
    $borrowers_borrowing = $stmt->fetchAll();
}

// Get detailed borrow info for each borrower
$borrow_details = [];
foreach ($borrowers_borrowing as $borrower) {
    $stmt = $pdo->prepare("
        SELECT 
            bh.*,
            a.asset_name,
            a.asset_number,
            a.asset_category,
            a.kondisi,
            DATEDIFF(CURDATE(), bh.borrow_date) as days_borrowed
        FROM borrow_history bh
        JOIN assets a ON bh.asset_id = a.id
        WHERE bh.borrower_employee_id = ?
        AND bh.status = 'borrowed'
        ORDER BY bh.borrow_date DESC
    ");
    $stmt->execute([$borrower['employee_id']]);
    $borrow_details[$borrower['employee_id']] = $stmt->fetchAll();
}

// Statistics
$total_borrowers = count($borrowers_borrowing);
$total_items_borrowed = $total_borrowers > 0 ? array_sum(array_column($borrowers_borrowing, 'total_borrowed')) : 0;
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>View Borrower - Tools Room</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'leaf-green': '#90EE90',
                        'dark-leaf': '#32CD32',
                        'soft-gray': '#F5F5F5',
                        'medium-gray': '#9CA3AF'
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-soft-gray min-h-screen">
    <!-- Header -->
    <header class="bg-white shadow-sm border-b-2 border-leaf-green">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center py-4">
                <div class="flex items-center space-x-4">
                    <a href="dashboard.php" class="text-dark-leaf hover:text-green-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"></path>
                        </svg>
                    </a>
                    <img src="../images/logo.png" alt="Logo" class="h-10 w-10" onerror="this.style.display='none'">
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">View Borrower</h1>
                        <p class="text-sm text-medium-gray">Borrower yang sedang meminjam tools</p>
                    </div>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-sm text-gray-600"><?php echo htmlspecialchars($admin_name); ?></span>
                    <a href="../logout.php" class="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded-lg text-sm transition duration-200">
                        Logout
                    </a>
                </div>
            </div>
        </div>
    </header>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Master Organization Info -->
        <?php if ($master_org_info): ?>
        <div class="bg-white rounded-xl shadow-lg p-6 mb-6 border-l-4" style="border-color: <?php echo $master_org_info['color_code']; ?>">
            <div class="flex items-center space-x-4">
                <div class="w-16 h-16 rounded-full flex items-center justify-center" 
                     style="background-color: <?php echo $master_org_info['color_code']; ?>20;">
                    <span class="text-2xl font-bold" style="color: <?php echo $master_org_info['color_code']; ?>">
                        <?php echo strtoupper(substr($master_org_info['master_name'], 0, 2)); ?>
                    </span>
                </div>
                <div>
                    <h2 class="text-xl font-bold text-gray-800"><?php echo htmlspecialchars($master_org_info['master_name']); ?></h2>
                    <?php if ($master_org_info['sloc']): ?>
                    <p class="text-sm text-gray-600 font-mono">SLOC: <?php echo htmlspecialchars($master_org_info['sloc']); ?></p>
                    <?php endif; ?>
                    <p class="text-sm text-gray-600">Organization Anda: <?php echo htmlspecialchars($admin_organization); ?></p>
                    <p class="text-xs text-gray-500">Admin: <?php echo htmlspecialchars($admin_name); ?> (NIK: <?php echo htmlspecialchars($admin_nik); ?>)</p>
                </div>
            </div>
        </div>
        <?php else: ?>
        <div class="bg-yellow-100 border border-yellow-400 text-yellow-800 rounded-xl p-6 mb-6">
            <div class="flex items-center">
                <svg class="w-6 h-6 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
                </svg>
                <div>
                    <p class="font-semibold">Anda belum terikat dengan Master Organization</p>
                    <p class="text-sm">Silakan hubungi superadmin untuk mengatur master organization Anda.</p>
                    <p class="text-xs mt-2">Admin: <?php echo htmlspecialchars($admin_name); ?> (Organization: <?php echo htmlspecialchars($admin_organization); ?>)</p>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Statistics Cards -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-blue-400">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-medium-gray text-sm font-medium">Borrower Aktif</p>
                        <p class="text-3xl font-bold text-blue-600"><?php echo $total_borrowers; ?></p>
                    </div>
                    <div class="bg-blue-100 p-3 rounded-full">
                        <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"></path>
                        </svg>
                    </div>
                </div>
                <p class="text-xs text-medium-gray mt-2">Yang sedang meminjam</p>
            </div>

            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-green-400">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-medium-gray text-sm font-medium">Total Tools Dipinjam</p>
                        <p class="text-3xl font-bold text-green-600"><?php echo $total_items_borrowed; ?></p>
                    </div>
                    <div class="bg-green-100 p-3 rounded-full">
                        <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"></path>
                        </svg>
                    </div>
                </div>
                <p class="text-xs text-medium-gray mt-2">Item yang dipinjam</p>
            </div>

            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-purple-400">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-medium-gray text-sm font-medium">Rata-rata per Borrower</p>
                        <p class="text-3xl font-bold text-purple-600">
                            <?php echo $total_borrowers > 0 ? number_format($total_items_borrowed / $total_borrowers, 1) : '0'; ?>
                        </p>
                    </div>
                    <div class="bg-purple-100 p-3 rounded-full">
                        <svg class="w-6 h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"></path>
                        </svg>
                    </div>
                </div>
                <p class="text-xs text-medium-gray mt-2">Tools per borrower</p>
            </div>
        </div>

        <!-- Search Box -->
        <div class="bg-white rounded-xl shadow-lg p-4 mb-6">
            <div class="flex items-center space-x-2">
                <svg class="w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                </svg>
                <input type="text" id="searchBorrower" placeholder="Cari borrower (nama, NIK, organization)..." 
                       class="flex-1 px-3 py-2 border-0 focus:outline-none focus:ring-0"
                       onkeyup="filterBorrowers()">
            </div>
        </div>

        <!-- Borrowers List -->
        <?php if (empty($borrowers_borrowing)): ?>
        <div class="bg-white rounded-xl shadow-lg p-12 text-center">
            <svg class="w-16 h-16 text-gray-300 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"></path>
            </svg>
            <h3 class="text-lg font-medium text-gray-800 mb-2">Tidak Ada Borrower Aktif</h3>
            <p class="text-gray-600">
                <?php if (!$admin_master_org_id): ?>
                Anda belum terikat dengan master organization
                <?php else: ?>
                Belum ada borrower di master organization Anda yang sedang meminjam tools
                <?php endif; ?>
            </p>
        </div>
        <?php else: ?>
        <div class="space-y-6">
            <?php foreach ($borrowers_borrowing as $borrower): ?>
            <div class="bg-white rounded-xl shadow-lg overflow-hidden borrower-card"
                 data-name="<?php echo strtolower($borrower['full_name']); ?>"
                 data-nik="<?php echo strtolower($borrower['nik']); ?>"
                 data-org="<?php echo strtolower($borrower['organization']); ?>">
                <!-- Borrower Header -->
                <div class="p-6 border-l-4 border-blue-400 cursor-pointer hover:bg-gray-50 transition duration-200"
                     onclick="toggleBorrower('<?php echo $borrower['employee_id']; ?>')">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-4">
                            <div class="w-12 h-12 rounded-full bg-blue-100 flex items-center justify-center">
                                <span class="text-xl font-bold text-blue-600">
                                    <?php echo strtoupper(substr($borrower['full_name'], 0, 2)); ?>
                                </span>
                            </div>
                            <div>
                                <h3 class="text-lg font-bold text-gray-800"><?php echo htmlspecialchars($borrower['full_name']); ?></h3>
                                <p class="text-sm text-gray-600">
                                    NIK: <?php echo htmlspecialchars($borrower['nik']); ?> • 
                                    <?php echo htmlspecialchars($borrower['jabatan']); ?>
                                </p>
                                <p class="text-xs text-gray-500">
                                    <?php echo htmlspecialchars($borrower['organization']); ?> • 
                                    <?php echo htmlspecialchars($borrower['branch_name']); ?>
                                </p>
                            </div>
                        </div>
                        <div class="flex items-center space-x-4">
                            <div class="text-right">
                                <span class="bg-blue-100 text-blue-800 px-3 py-1 rounded-full text-sm font-medium">
                                    <?php echo $borrower['total_borrowed']; ?> tools
                                </span>
                            </div>
                            <svg class="w-6 h-6 text-gray-400 transition-transform duration-200 toggle-icon-<?php echo $borrower['employee_id']; ?>" 
                                 fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <!-- Borrowed Items (Hidden by default) -->
                <div id="borrower-items-<?php echo $borrower['employee_id']; ?>" class="hidden border-t border-gray-200">
                    <div class="p-6 bg-gray-50">
                        <h4 class="text-lg font-semibold text-gray-800 mb-4">Tools yang Dipinjam</h4>
                        <div class="space-y-3">
                            <?php foreach ($borrow_details[$borrower['employee_id']] as $item): ?>
                            <div class="bg-white p-4 rounded-lg border border-gray-200">
                                <div class="flex justify-between items-start">
                                    <div class="flex-1">
                                        <h5 class="font-semibold text-gray-800"><?php echo htmlspecialchars($item['asset_name']); ?></h5>
                                        <p class="text-sm text-gray-600">
                                            <?php echo htmlspecialchars($item['asset_number']); ?> • 
                                            <?php echo htmlspecialchars($item['asset_category']); ?>
                                        </p>
                                        <div class="flex items-center space-x-4 mt-2 text-xs">
                                            <span class="text-gray-600">
                                                <strong>Dipinjam:</strong> <?php echo date('d/m/Y', strtotime($item['borrow_date'])); ?>
                                            </span>
                                            <span class="text-gray-600">
                                                <strong>Durasi:</strong> <?php echo $item['days_borrowed']; ?> hari
                                            </span>
                                            <span class="px-2 py-1 rounded-full text-xs font-medium 
                                                <?php echo $item['kondisi'] == 'baik' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'; ?>">
                                                <?php echo ucfirst($item['kondisi']); ?>
                                            </span>
                                        </div>
                                        <?php if ($item['workcenter']): ?>
                                        <p class="text-xs text-gray-500 mt-1">
                                            <strong>Workcenter:</strong> <?php echo htmlspecialchars($item['workcenter']); ?>
                                        </p>
                                        <?php endif; ?>
                                    </div>
                                    <?php if ($item['days_borrowed'] > 7): ?>
                                    <span class="bg-red-100 text-red-800 px-2 py-1 rounded-full text-xs">
                                        Terlambat
                                    </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>

    <script>
        // Toggle borrower details
        function toggleBorrower(employeeId) {
            const content = document.getElementById(`borrower-items-${employeeId}`);
            const icon = document.querySelector(`.toggle-icon-${employeeId}`);
            
            if (content.classList.contains('hidden')) {
                content.classList.remove('hidden');
                icon.style.transform = 'rotate(180deg)';
            } else {
                content.classList.add('hidden');
                icon.style.transform = 'rotate(0deg)';
            }
        }

        // Filter borrowers
        function filterBorrowers() {
            const searchTerm = document.getElementById('searchBorrower').value.toLowerCase();
            const borrowerCards = document.querySelectorAll('.borrower-card');
            
            borrowerCards.forEach(card => {
                const name = card.getAttribute('data-name');
                const nik = card.getAttribute('data-nik');
                const org = card.getAttribute('data-org');
                
                if (name.includes(searchTerm) || nik.includes(searchTerm) || org.includes(searchTerm)) {
                    card.style.display = 'block';
                } else {
                    card.style.display = 'none';
                }
            });
        }
    </script>
</body>
</html>