<?php
require_once '../config/connection.php';
checkUserType(['superadmin']);

// Get all master organizations with their data
$stmt = $pdo->query("SELECT * FROM master_organization ORDER BY master_name");
$masters = $stmt->fetchAll();

// Get organization mappings for each master
$organization_mappings = [];
foreach ($masters as $master) {
    $stmt = $pdo->prepare("SELECT organization_name FROM organization_master_mapping WHERE master_organization_id = ? ORDER BY organization_name");
    $stmt->execute([$master['id']]);
    $organization_mappings[$master['id']] = $stmt->fetchAll(PDO::FETCH_COLUMN);
}

// Get users for each master (both admins and borrowers)
$users_by_master = [];
foreach ($masters as $master) {
    $stmt = $pdo->prepare("
        SELECT 'admin' as type, nik, employee_id, full_name, organization, branch_name, jabatan 
        FROM admins 
        WHERE master_organization_id = ?
        UNION ALL
        SELECT 'borrower' as type, nik, employee_id, full_name, organization, branch_name, jabatan 
        FROM borrowers 
        WHERE master_organization_id = ?
        ORDER BY full_name
    ");
    $stmt->execute([$master['id'], $master['id']]);
    $users_by_master[$master['id']] = $stmt->fetchAll();
}

// Get statistics
$total_masters = count($masters);
$total_organizations = 0;
$total_users = 0;
foreach ($organization_mappings as $orgs) {
    $total_organizations += count($orgs);
}
foreach ($users_by_master as $users) {
    $total_users += count($users);
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Data Organization - Tools Room</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'leaf-green': '#90EE90',
                        'dark-leaf': '#32CD32',
                        'soft-gray': '#F5F5F5',
                        'medium-gray': '#9CA3AF'
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-soft-gray min-h-screen">
    <!-- Header -->
    <header class="bg-white shadow-sm border-b-2 border-leaf-green">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center py-4">
                <div class="flex items-center space-x-4">
                    <a href="dashboard.php" class="text-dark-leaf hover:text-green-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"></path>
                        </svg>
                    </a>
                    <img src="../images/logo.png" alt="Logo" class="h-10 w-10" onerror="this.style.display='none'">
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">Data Organization</h1>
                        <p class="text-sm text-medium-gray">Hierarchy Master → Organization → Users</p>
                    </div>
                </div>
                <div class="flex items-center space-x-4">
                    <button onclick="expandAll()" class="bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded-lg text-sm transition duration-200">
                        Expand All
                    </button>
                    <button onclick="collapseAll()" class="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded-lg text-sm transition duration-200">
                        Collapse All
                    </button>
                    <span class="text-sm text-gray-600"><?php echo $_SESSION['user_name']; ?></span>
                    <a href="../logout.php" class="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded-lg text-sm transition duration-200">
                        Logout
                    </a>
                </div>
            </div>
        </div>
    </header>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Stats Cards -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-blue-400">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-medium-gray text-sm font-medium">Total Master Organization</p>
                        <p class="text-3xl font-bold text-blue-600"><?php echo $total_masters; ?></p>
                    </div>
                    <div class="bg-blue-100 p-3 rounded-full">
                        <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-green-400">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-medium-gray text-sm font-medium">Total Organizations</p>
                        <p class="text-3xl font-bold text-green-600"><?php echo $total_organizations; ?></p>
                    </div>
                    <div class="bg-green-100 p-3 rounded-full">
                        <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-purple-400">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-medium-gray text-sm font-medium">Total Users</p>
                        <p class="text-3xl font-bold text-purple-600"><?php echo $total_users; ?></p>
                    </div>
                    <div class="bg-purple-100 p-3 rounded-full">
                        <svg class="w-6 h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"></path>
                        </svg>
                    </div>
                </div>
            </div>
        </div>

        <!-- Search Box -->
        <div class="bg-white rounded-xl shadow-lg p-4 mb-6">
            <div class="flex items-center space-x-2">
                <svg class="w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                </svg>
                <input type="text" id="globalSearch" placeholder="Cari Master, Organization, atau User..." 
                       class="flex-1 px-3 py-2 border-0 focus:outline-none focus:ring-0"
                       onkeyup="performGlobalSearch()">
            </div>
        </div>

        <!-- Organization Hierarchy -->
        <div class="space-y-6">
            <?php foreach ($masters as $master): ?>
            <div class="bg-white rounded-xl shadow-lg overflow-hidden master-item" 
                 data-master-name="<?php echo strtolower($master['master_name']); ?>"
                 data-sloc="<?php echo strtolower($master['sloc'] ?? ''); ?>">
                <!-- Master Header -->
                <div class="p-6 border-l-4 cursor-pointer hover:bg-gray-50 transition duration-200" 
                     style="border-color: <?php echo $master['color_code']; ?>"
                     onclick="toggleMaster(<?php echo $master['id']; ?>)">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-4">
                            <div class="w-12 h-12 rounded-full flex items-center justify-center" 
                                 style="background-color: <?php echo $master['color_code']; ?>20;">
                                <span class="text-xl font-bold" style="color: <?php echo $master['color_code']; ?>">
                                    <?php echo strtoupper(substr($master['master_name'], 0, 2)); ?>
                                </span>
                            </div>
                            <div>
                                <h2 class="text-xl font-bold text-gray-800"><?php echo htmlspecialchars($master['master_name']); ?></h2>
                                <div class="flex items-center space-x-2 mt-1">
                                    <?php if ($master['sloc']): ?>
                                    <span class="text-xs font-mono bg-gray-100 px-2 py-1 rounded">SLOC: <?php echo htmlspecialchars($master['sloc']); ?></span>
                                    <?php endif; ?>
                                    <span class="text-xs text-gray-600">
                                        <?php echo count($organization_mappings[$master['id']]); ?> Organizations • 
                                        <?php echo count($users_by_master[$master['id']]); ?> Users
                                    </span>
                                </div>
                            </div>
                        </div>
                        <svg class="w-6 h-6 text-gray-400 transition-transform duration-200 toggle-icon-<?php echo $master['id']; ?>" 
                             fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                        </svg>
                    </div>
                </div>

                <!-- Master Content (Hidden by default) -->
                <div id="master-content-<?php echo $master['id']; ?>" class="hidden border-t border-gray-200">
                    <!-- Organizations Section -->
                    <?php if (!empty($organization_mappings[$master['id']])): ?>
                    <div class="p-6 bg-gray-50">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4 flex items-center">
                            <svg class="w-5 h-5 mr-2 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"></path>
                            </svg>
                            Organizations (<?php echo count($organization_mappings[$master['id']]); ?>)
                        </h3>
                        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                            <?php foreach ($organization_mappings[$master['id']] as $org): ?>
                            <div class="bg-white p-3 rounded-lg border border-gray-200 organization-item" 
                                 data-org-name="<?php echo strtolower($org); ?>">
                                <p class="text-sm font-medium text-gray-800"><?php echo htmlspecialchars($org); ?></p>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <?php else: ?>
                    <div class="p-6 bg-gray-50 text-center text-gray-500">
                        <p class="text-sm">Tidak ada organization yang di-assign</p>
                    </div>
                    <?php endif; ?>

                    <!-- Users Section -->
                    <?php if (!empty($users_by_master[$master['id']])): ?>
                    <div class="p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4 flex items-center">
                            <svg class="w-5 h-5 mr-2 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"></path>
                            </svg>
                            Users (<?php echo count($users_by_master[$master['id']]); ?>)
                        </h3>
                        <div class="overflow-x-auto">
                            <table class="min-w-full divide-y divide-gray-200">
                                <thead class="bg-gray-50">
                                    <tr>
                                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">NIK</th>
                                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nama Lengkap</th>
                                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Organization</th>
                                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Branch</th>
                                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Jabatan</th>
                                    </tr>
                                </thead>
                                <tbody class="bg-white divide-y divide-gray-200">
                                    <?php foreach ($users_by_master[$master['id']] as $user): ?>
                                    <tr class="hover:bg-gray-50 user-row" 
                                        data-user-name="<?php echo strtolower($user['full_name']); ?>"
                                        data-user-nik="<?php echo strtolower($user['nik']); ?>"
                                        data-user-org="<?php echo strtolower($user['organization']); ?>">
                                        <td class="px-4 py-3 whitespace-nowrap">
                                            <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium 
                                                <?php echo $user['type'] === 'admin' ? 'bg-green-100 text-green-800' : 'bg-blue-100 text-blue-800'; ?>">
                                                <?php echo ucfirst($user['type']); ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-3 whitespace-nowrap text-sm font-mono text-gray-900">
                                            <?php echo htmlspecialchars($user['nik']); ?>
                                        </td>
                                        <td class="px-4 py-3 whitespace-nowrap text-sm font-medium text-gray-900">
                                            <?php echo htmlspecialchars($user['full_name']); ?>
                                        </td>
                                        <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-600">
                                            <?php echo htmlspecialchars($user['organization']); ?>
                                        </td>
                                        <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-600">
                                            <?php echo htmlspecialchars($user['branch_name']); ?>
                                        </td>
                                        <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-600">
                                            <?php echo htmlspecialchars($user['jabatan']); ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <?php else: ?>
                    <div class="p-6 text-center text-gray-500">
                        <p class="text-sm">Tidak ada user yang di-assign</p>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>

        <!-- Empty State -->
        <?php if (empty($masters)): ?>
        <div class="bg-white rounded-xl shadow-lg p-12 text-center">
            <svg class="w-16 h-16 text-gray-300 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"></path>
            </svg>
            <h3 class="text-lg font-medium text-gray-800 mb-2">Belum Ada Master Organization</h3>
            <p class="text-gray-600 mb-4">Silakan tambahkan master organization terlebih dahulu</p>
            <a href="master_organization.php" class="inline-block bg-dark-leaf hover:bg-green-600 text-white px-6 py-2 rounded-lg transition duration-200">
                Kelola Master Organization
            </a>
        </div>
        <?php endif; ?>
    </div>

    <script>
        // Toggle Master Content
        function toggleMaster(masterId) {
            const content = document.getElementById(`master-content-${masterId}`);
            const icon = document.querySelector(`.toggle-icon-${masterId}`);
            
            if (content.classList.contains('hidden')) {
                content.classList.remove('hidden');
                icon.style.transform = 'rotate(180deg)';
            } else {
                content.classList.add('hidden');
                icon.style.transform = 'rotate(0deg)';
            }
        }

        // Expand All
        function expandAll() {
            document.querySelectorAll('[id^="master-content-"]').forEach(content => {
                content.classList.remove('hidden');
            });
            document.querySelectorAll('[class*="toggle-icon-"]').forEach(icon => {
                icon.style.transform = 'rotate(180deg)';
            });
        }

        // Collapse All
        function collapseAll() {
            document.querySelectorAll('[id^="master-content-"]').forEach(content => {
                content.classList.add('hidden');
            });
            document.querySelectorAll('[class*="toggle-icon-"]').forEach(icon => {
                icon.style.transform = 'rotate(0deg)';
            });
        }

        // Global Search
        function performGlobalSearch() {
            const searchTerm = document.getElementById('globalSearch').value.toLowerCase();
            
            // If search is empty, show all
            if (searchTerm === '') {
                document.querySelectorAll('.master-item').forEach(item => {
                    item.style.display = 'block';
                });
                document.querySelectorAll('.organization-item').forEach(item => {
                    item.style.display = 'block';
                });
                document.querySelectorAll('.user-row').forEach(item => {
                    item.style.display = 'table-row';
                });
                return;
            }

            // Search in masters
            document.querySelectorAll('.master-item').forEach(masterItem => {
                const masterName = masterItem.getAttribute('data-master-name');
                const sloc = masterItem.getAttribute('data-sloc');
                let hasMatch = false;

                // Check master name or SLOC
                if (masterName.includes(searchTerm) || sloc.includes(searchTerm)) {
                    hasMatch = true;
                }

                // Check organizations
                const organizations = masterItem.querySelectorAll('.organization-item');
                organizations.forEach(org => {
                    const orgName = org.getAttribute('data-org-name');
                    if (orgName.includes(searchTerm)) {
                        hasMatch = true;
                        org.style.display = 'block';
                    } else {
                        org.style.display = 'none';
                    }
                });

                // Check users
                const users = masterItem.querySelectorAll('.user-row');
                users.forEach(user => {
                    const userName = user.getAttribute('data-user-name');
                    const userNik = user.getAttribute('data-user-nik');
                    const userOrg = user.getAttribute('data-user-org');
                    
                    if (userName.includes(searchTerm) || userNik.includes(searchTerm) || userOrg.includes(searchTerm)) {
                        hasMatch = true;
                        user.style.display = 'table-row';
                    } else {
                        user.style.display = 'none';
                    }
                });

                // Show/hide master based on match
                if (hasMatch) {
                    masterItem.style.display = 'block';
                    // Auto expand if has match
                    const masterId = masterItem.querySelector('[id^="master-content-"]').id.split('-')[2];
                    const content = document.getElementById(`master-content-${masterId}`);
                    content.classList.remove('hidden');
                } else {
                    masterItem.style.display = 'none';
                }
            });
        }
    </script>
</body>
</html>