<?php
require_once '../config/connection.php';
checkUserType(['superadmin']);

$message = '';
$message_type = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['action'])) {
            if ($_POST['action'] === 'add_master') {
                // Add new master organization
                $master_name = sanitize($_POST['master_name']);
                $sloc = sanitize($_POST['sloc']);
                $description = sanitize($_POST['description']);
                $color_code = sanitize($_POST['color_code']);
                
                $stmt = $pdo->prepare("INSERT INTO master_organization (master_name, sloc, description, color_code) VALUES (?, ?, ?, ?)");
                $stmt->execute([$master_name, $sloc, $description, $color_code]);
                
                $message = 'Master Organization berhasil ditambahkan!';
                $message_type = 'success';
                
            } elseif ($_POST['action'] === 'update_master') {
                // Update master organization
                $id = (int)$_POST['master_id'];
                $master_name = sanitize($_POST['master_name']);
                $sloc = sanitize($_POST['sloc']);
                $description = sanitize($_POST['description']);
                $color_code = sanitize($_POST['color_code']);
                $is_active = (int)$_POST['is_active'];
                
                $stmt = $pdo->prepare("UPDATE master_organization SET master_name = ?, sloc = ?, description = ?, color_code = ?, is_active = ? WHERE id = ?");
                $stmt->execute([$master_name, $sloc, $description, $color_code, $is_active, $id]);
                
                $message = 'Master Organization berhasil diupdate!';
                $message_type = 'success';
                
            } elseif ($_POST['action'] === 'assign_organizations') {
                // Assign organizations to master
                $master_id = (int)$_POST['master_id'];
                $organizations = $_POST['organizations'] ?? [];
                
                // Delete existing mappings
                $stmt = $pdo->prepare("DELETE FROM organization_master_mapping WHERE master_organization_id = ?");
                $stmt->execute([$master_id]);
                
                // Insert new mappings
                $stmt = $pdo->prepare("INSERT INTO organization_master_mapping (master_organization_id, organization_name) VALUES (?, ?)");
                foreach ($organizations as $org) {
                    $stmt->execute([$master_id, sanitize($org)]);
                }
                
                $message = 'Organization berhasil di-assign ke Master!';
                $message_type = 'success';
                
            } elseif ($_POST['action'] === 'assign_users') {
                // Assign users to master organization
                $master_id = (int)$_POST['master_id'];
                $admin_niks = $_POST['admin_niks'] ?? [];
                $borrower_niks = $_POST['borrower_niks'] ?? [];
                
                // Update admins
                if (!empty($admin_niks)) {
                    $placeholders = str_repeat('?,', count($admin_niks) - 1) . '?';
                    $stmt = $pdo->prepare("UPDATE admins SET master_organization_id = ? WHERE nik IN ($placeholders)");
                    $stmt->execute(array_merge([$master_id], $admin_niks));
                }
                
                // Update borrowers
                if (!empty($borrower_niks)) {
                    $placeholders = str_repeat('?,', count($borrower_niks) - 1) . '?';
                    $stmt = $pdo->prepare("UPDATE borrowers SET master_organization_id = ? WHERE nik IN ($placeholders)");
                    $stmt->execute(array_merge([$master_id], $borrower_niks));
                }
                
                $message = 'User berhasil di-assign ke Master Organization!';
                $message_type = 'success';
                
            } elseif ($_POST['action'] === 'delete_master') {
                // Delete master organization
                $id = (int)$_POST['master_id'];
                
                // First, unassign all users
                $stmt = $pdo->prepare("UPDATE admins SET master_organization_id = NULL WHERE master_organization_id = ?");
                $stmt->execute([$id]);
                
                $stmt = $pdo->prepare("UPDATE borrowers SET master_organization_id = NULL WHERE master_organization_id = ?");
                $stmt->execute([$id]);
                
                // Delete master
                $stmt = $pdo->prepare("DELETE FROM master_organization WHERE id = ?");
                $stmt->execute([$id]);
                
                $message = 'Master Organization berhasil dihapus!';
                $message_type = 'success';
            }
        }
    } catch (Exception $e) {
        $message = 'Error: ' . $e->getMessage();
        $message_type = 'error';
    }
}

// Get all master organizations
$stmt = $pdo->query("SELECT * FROM master_organization ORDER BY master_name");
$masters = $stmt->fetchAll();

// Get all unique organizations
$stmt = $pdo->query("SELECT DISTINCT organization FROM admins WHERE organization IS NOT NULL AND organization != '' 
                     UNION 
                     SELECT DISTINCT organization FROM borrowers WHERE organization IS NOT NULL AND organization != '' 
                     ORDER BY organization");
$all_organizations = $stmt->fetchAll(PDO::FETCH_COLUMN);

// Get all users with NIK
$stmt = $pdo->query("SELECT 'admin' as type, nik, full_name, organization FROM admins WHERE nik IS NOT NULL 
                     UNION ALL 
                     SELECT 'borrower' as type, nik, full_name, organization FROM borrowers WHERE nik IS NOT NULL 
                     ORDER BY full_name");
$all_users = $stmt->fetchAll();

// Get statistics for each master
$master_stats = [];
foreach ($masters as $master) {
    $stmt = $pdo->prepare("
        SELECT 
            (SELECT COUNT(*) FROM admins WHERE master_organization_id = ?) + 
            (SELECT COUNT(*) FROM borrowers WHERE master_organization_id = ?) as total_users,
            (SELECT COUNT(*) FROM organization_master_mapping WHERE master_organization_id = ?) as total_orgs
    ");
    $stmt->execute([$master['id'], $master['id'], $master['id']]);
    $master_stats[$master['id']] = $stmt->fetch();
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kelola Master Organization - Tools Room</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'leaf-green': '#90EE90',
                        'dark-leaf': '#32CD32',
                        'soft-gray': '#F5F5F5',
                        'medium-gray': '#9CA3AF'
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-soft-gray min-h-screen">
    <!-- Header -->
    <header class="bg-white shadow-sm border-b-2 border-leaf-green">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center py-4">
                <div class="flex items-center space-x-4">
                    <a href="dashboard.php" class="text-dark-leaf hover:text-green-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"></path>
                        </svg>
                    </a>
                    <img src="../images/logo.png" alt="Logo" class="h-10 w-10" onerror="this.style.display='none'">
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">Kelola Master Organization</h1>
                        <p class="text-sm text-medium-gray">Management master organization, SLOC, mapping, dan assignment</p>
                    </div>
                </div>
                <div class="flex items-center space-x-4">
                    <button onclick="showAddMasterModal()" class="bg-dark-leaf hover:bg-green-600 text-white px-4 py-2 rounded-lg text-sm transition duration-200 flex items-center space-x-2">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                        </svg>
                        <span>Tambah Master</span>
                    </button>
                    <span class="text-sm text-gray-600"><?php echo $_SESSION['user_name']; ?></span>
                    <a href="../logout.php" class="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded-lg text-sm transition duration-200">
                        Logout
                    </a>
                </div>
            </div>
        </div>
    </header>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Alert Messages -->
        <?php if ($message): ?>
        <div class="mb-6 p-4 rounded-lg <?php echo $message_type === 'success' ? 'bg-green-100 text-green-700 border border-green-200' : 'bg-red-100 text-red-700 border border-red-200'; ?>">
            <?php echo htmlspecialchars($message); ?>
        </div>
        <?php endif; ?>

        <!-- Stats Cards -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-blue-400">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-medium-gray text-sm font-medium">Total Master</p>
                        <p class="text-3xl font-bold text-blue-600"><?php echo count($masters); ?></p>
                    </div>
                    <div class="bg-blue-100 p-3 rounded-full">
                        <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-green-400">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-medium-gray text-sm font-medium">Total Organizations</p>
                        <p class="text-3xl font-bold text-green-600"><?php echo count($all_organizations); ?></p>
                    </div>
                    <div class="bg-green-100 p-3 rounded-full">
                        <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-purple-400">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-medium-gray text-sm font-medium">Total Users</p>
                        <p class="text-3xl font-bold text-purple-600"><?php echo count($all_users); ?></p>
                    </div>
                    <div class="bg-purple-100 p-3 rounded-full">
                        <svg class="w-6 h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"></path>
                        </svg>
                    </div>
                </div>
            </div>
        </div>

        <!-- Master Organizations List -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <?php foreach ($masters as $master): ?>
            <div class="bg-white rounded-xl shadow-lg p-6 border-l-4" style="border-color: <?php echo $master['color_code']; ?>">
                <div class="flex justify-between items-start mb-4">
                    <div class="flex items-center space-x-3">
                        <div class="w-12 h-12 rounded-full flex items-center justify-center" style="background-color: <?php echo $master['color_code']; ?>20;">
                            <span class="text-xl font-bold" style="color: <?php echo $master['color_code']; ?>">
                                <?php echo strtoupper(substr($master['master_name'], 0, 2)); ?>
                            </span>
                        </div>
                        <div>
                            <h3 class="text-lg font-bold text-gray-800"><?php echo htmlspecialchars($master['master_name']); ?></h3>
                            <?php if ($master['sloc']): ?>
                            <p class="text-xs text-gray-600 font-mono bg-gray-100 px-2 py-1 rounded inline-block">SLOC: <?php echo htmlspecialchars($master['sloc']); ?></p>
                            <?php endif; ?>
                            <p class="text-sm text-medium-gray mt-1"><?php echo htmlspecialchars($master['description'] ?? 'Tidak ada deskripsi'); ?></p>
                        </div>
                    </div>
                    <div class="flex items-center space-x-2">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?php echo $master['is_active'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                            <?php echo $master['is_active'] ? 'Aktif' : 'Non-Aktif'; ?>
                        </span>
                    </div>
                </div>

                <!-- Stats -->
                <div class="grid grid-cols-2 gap-4 mb-4">
                    <div class="bg-gray-50 rounded-lg p-3">
                        <p class="text-xs text-medium-gray">Organizations</p>
                        <p class="text-xl font-bold text-gray-800"><?php echo $master_stats[$master['id']]['total_orgs']; ?></p>
                    </div>
                    <div class="bg-gray-50 rounded-lg p-3">
                        <p class="text-xs text-medium-gray">Users (NIK)</p>
                        <p class="text-xl font-bold text-gray-800"><?php echo $master_stats[$master['id']]['total_users']; ?></p>
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="flex flex-wrap gap-2">
                    <button onclick="showAssignOrganizationsModal(<?php echo $master['id']; ?>, '<?php echo htmlspecialchars($master['master_name']); ?>')" 
                            class="flex-1 bg-blue-500 hover:bg-blue-600 text-white px-3 py-2 rounded-lg text-xs transition duration-200">
                        <svg class="w-3 h-3 inline mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"></path>
                        </svg>
                        Assign Org
                    </button>
                    <button onclick="showAssignUsersModal(<?php echo $master['id']; ?>, '<?php echo htmlspecialchars($master['master_name']); ?>')" 
                            class="flex-1 bg-green-500 hover:bg-green-600 text-white px-3 py-2 rounded-lg text-xs transition duration-200">
                        <svg class="w-3 h-3 inline mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path>
                        </svg>
                        Assign NIK
                    </button>
                    <button onclick="showEditMasterModal(<?php echo htmlspecialchars(json_encode($master)); ?>)" 
                            class="bg-yellow-500 hover:bg-yellow-600 text-white px-3 py-2 rounded-lg text-xs transition duration-200">
                        <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                        </svg>
                    </button>
                    <button onclick="deleteMaster(<?php echo $master['id']; ?>, '<?php echo htmlspecialchars($master['master_name']); ?>')" 
                            class="bg-red-500 hover:bg-red-600 text-white px-3 py-2 rounded-lg text-xs transition duration-200">
                        <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                        </svg>
                    </button>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- Add/Edit Master Modal -->
    <div id="masterModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-5 border w-full max-w-md shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 id="masterModalTitle" class="text-lg font-medium text-gray-900 mb-4"></h3>
                <form method="POST">
                    <input type="hidden" name="action" id="masterAction" value="add_master">
                    <input type="hidden" name="master_id" id="masterId">
                    
                    <div class="space-y-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Nama Master <span class="text-red-500">*</span></label>
                            <input type="text" name="master_name" id="masterName" required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-dark-leaf"
                                   placeholder="Contoh: KMI 1">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">SLOC <span class="text-red-500">*</span></label>
                            <input type="text" name="sloc" id="masterSloc" required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-dark-leaf"
                                   placeholder="Contoh: 21T1">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Deskripsi</label>
                            <textarea name="description" id="masterDescription" rows="3"
                                      class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-dark-leaf"
                                      placeholder="Deskripsi master organization..."></textarea>
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Warna <span class="text-red-500">*</span></label>
                            <input type="color" name="color_code" id="masterColor" required
                                   class="w-full h-10 px-1 py-1 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-dark-leaf">
                        </div>
                        
                        <div id="masterStatusDiv" class="hidden">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                            <select name="is_active" id="masterStatus" 
                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-dark-leaf">
                                <option value="1">Aktif</option>
                                <option value="0">Non-Aktif</option>
                            </select>
                        </div>
                    </div>

                    <div class="flex justify-end space-x-3 mt-6">
                        <button type="button" onclick="hideMasterModal()" 
                                class="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 rounded-md hover:bg-gray-200">
                            Batal
                        </button>
                        <button type="submit" 
                                class="px-4 py-2 text-sm font-medium text-white bg-dark-leaf rounded-md hover:bg-green-600">
                            Simpan
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Assign Organizations Modal -->
    <div id="assignOrganizationsModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-5 border w-full max-w-2xl shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 id="assignOrgTitle" class="text-lg font-medium text-gray-900 mb-4"></h3>
                <form method="POST">
                    <input type="hidden" name="action" value="assign_organizations">
                    <input type="hidden" name="master_id" id="assignOrgMasterId">
                    
                    <div class="mb-4">
                        <input type="text" id="searchOrganization" placeholder="Cari organization..." 
                               class="w-full px-3 py-2 border border-gray-300 rounded-md mb-3"
                               onkeyup="filterOrganizations()">
                        
                        <div class="max-h-96 overflow-y-auto border border-gray-200 rounded-md p-3">
                            <div class="space-y-2" id="organizationsList">
                                <?php foreach ($all_organizations as $org): ?>
                                <label class="flex items-center p-2 hover:bg-gray-50 rounded cursor-pointer organization-item">
                                    <input type="checkbox" name="organizations[]" value="<?php echo htmlspecialchars($org); ?>" 
                                           class="mr-3 h-4 w-4 text-dark-leaf">
                                    <span class="text-sm"><?php echo htmlspecialchars($org); ?></span>
                                </label>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>

                    <div class="flex justify-end space-x-3">
                        <button type="button" onclick="hideAssignOrganizationsModal()" 
                                class="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 rounded-md hover:bg-gray-200">
                            Batal
                        </button>
                        <button type="submit" 
                                class="px-4 py-2 text-sm font-medium text-white bg-blue-500 rounded-md hover:bg-blue-600">
                            Assign Organizations
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Assign Users Modal -->
    <div id="assignUsersModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-5 border w-full max-w-2xl shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 id="assignUsersTitle" class="text-lg font-medium text-gray-900 mb-4"></h3>
                <form method="POST">
                    <input type="hidden" name="action" value="assign_users">
                    <input type="hidden" name="master_id" id="assignUsersMasterId">
                    
                    <div class="mb-4">
                        <input type="text" id="searchUser" placeholder="Cari NIK atau nama user..." 
                               class="w-full px-3 py-2 border border-gray-300 rounded-md mb-3"
                               onkeyup="filterUsers()">
                        
                        <div class="grid grid-cols-2 gap-4">
                            <!-- Admins -->
                            <div>
                                <h4 class="font-semibold text-gray-700 mb-2">Admins</h4>
                                <div class="max-h-80 overflow-y-auto border border-gray-200 rounded-md p-3">
                                    <div class="space-y-2" id="adminsList">
                                        <?php foreach (array_filter($all_users, fn($u) => $u['type'] === 'admin') as $user): ?>
                                        <label class="flex items-start p-2 hover:bg-gray-50 rounded cursor-pointer user-item" 
                                               data-name="<?php echo strtolower($user['full_name'] . ' ' . $user['nik']); ?>">
                                            <input type="checkbox" name="admin_niks[]" value="<?php echo htmlspecialchars($user['nik']); ?>" 
                                                   class="mr-3 mt-1 h-4 w-4 text-dark-leaf">
                                            <div class="text-sm">
                                                <div class="font-medium"><?php echo htmlspecialchars($user['full_name']); ?></div>
                                                <div class="text-gray-500"><?php echo htmlspecialchars($user['nik']); ?></div>
                                                <div class="text-xs text-gray-400"><?php echo htmlspecialchars($user['organization']); ?></div>
                                            </div>
                                        </label>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Borrowers -->
                            <div>
                                <h4 class="font-semibold text-gray-700 mb-2">Borrowers</h4>
                                <div class="max-h-80 overflow-y-auto border border-gray-200 rounded-md p-3">
                                    <div class="space-y-2" id="borrowersList">
                                        <?php foreach (array_filter($all_users, fn($u) => $u['type'] === 'borrower') as $user): ?>
                                        <label class="flex items-start p-2 hover:bg-gray-50 rounded cursor-pointer user-item" 
                                               data-name="<?php echo strtolower($user['full_name'] . ' ' . $user['nik']); ?>">
                                            <input type="checkbox" name="borrower_niks[]" value="<?php echo htmlspecialchars($user['nik']); ?>" 
                                                   class="mr-3 mt-1 h-4 w-4 text-dark-leaf">
                                            <div class="text-sm">
                                                <div class="font-medium"><?php echo htmlspecialchars($user['full_name']); ?></div>
                                                <div class="text-gray-500"><?php echo htmlspecialchars($user['nik']); ?></div>
                                                <div class="text-xs text-gray-400"><?php echo htmlspecialchars($user['organization']); ?></div>
                                            </div>
                                        </label>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="flex justify-end space-x-3">
                        <button type="button" onclick="hideAssignUsersModal()" 
                                class="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 rounded-md hover:bg-gray-200">
                            Batal
                        </button>
                        <button type="submit" 
                                class="px-4 py-2 text-sm font-medium text-white bg-green-500 rounded-md hover:bg-green-600">
                            Assign Users
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div id="deleteModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Konfirmasi Hapus</h3>
                <p class="text-sm text-gray-600 mb-4" id="deleteMessage"></p>
                <form method="POST">
                    <input type="hidden" name="action" value="delete_master">
                    <input type="hidden" name="master_id" id="deleteMasterId">
                    
                    <div class="flex justify-end space-x-3">
                        <button type="button" onclick="hideDeleteModal()" 
                                class="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 rounded-md hover:bg-gray-200">
                            Batal
                        </button>
                        <button type="submit" 
                                class="px-4 py-2 text-sm font-medium text-white bg-red-500 rounded-md hover:bg-red-600">
                            Hapus
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Add Master Modal
        function showAddMasterModal() {
            document.getElementById('masterModalTitle').textContent = 'Tambah Master Organization';
            document.getElementById('masterAction').value = 'add_master';
            document.getElementById('masterId').value = '';
            document.getElementById('masterName').value = '';
            document.getElementById('masterSloc').value = '';
            document.getElementById('masterDescription').value = '';
            document.getElementById('masterColor').value = '#3B82F6';
            document.getElementById('masterStatusDiv').classList.add('hidden');
            document.getElementById('masterModal').classList.remove('hidden');
        }

        // Edit Master Modal
        function showEditMasterModal(master) {
            document.getElementById('masterModalTitle').textContent = 'Edit Master Organization';
            document.getElementById('masterAction').value = 'update_master';
            document.getElementById('masterId').value = master.id;
            document.getElementById('masterName').value = master.master_name;
            document.getElementById('masterSloc').value = master.sloc || '';
            document.getElementById('masterDescription').value = master.description || '';
            document.getElementById('masterColor').value = master.color_code;
            document.getElementById('masterStatus').value = master.is_active;
            document.getElementById('masterStatusDiv').classList.remove('hidden');
            document.getElementById('masterModal').classList.remove('hidden');
        }

        function hideMasterModal() {
            document.getElementById('masterModal').classList.add('hidden');
        }

        // Assign Organizations Modal
        function showAssignOrganizationsModal(masterId, masterName) {
            document.getElementById('assignOrgTitle').textContent = `Assign Organizations ke ${masterName}`;
            document.getElementById('assignOrgMasterId').value = masterId;
            
            // Load current assignments
            fetch(`get_master_data.php?master_id=${masterId}&type=organizations`)
                .then(response => response.json())
                .then(data => {
                    document.querySelectorAll('input[name="organizations[]"]').forEach(checkbox => {
                        checkbox.checked = data.organizations.includes(checkbox.value);
                    });
                });
            
            document.getElementById('assignOrganizationsModal').classList.remove('hidden');
        }

        function hideAssignOrganizationsModal() {
            document.getElementById('assignOrganizationsModal').classList.add('hidden');
        }

        function filterOrganizations() {
            const searchTerm = document.getElementById('searchOrganization').value.toLowerCase();
            document.querySelectorAll('.organization-item').forEach(item => {
                const text = item.textContent.toLowerCase();
                item.style.display = text.includes(searchTerm) ? 'flex' : 'none';
            });
        }

        // Assign Users Modal
        function showAssignUsersModal(masterId, masterName) {
            document.getElementById('assignUsersTitle').textContent = `Assign Users (NIK) ke ${masterName}`;
            document.getElementById('assignUsersMasterId').value = masterId;
            
            // Load current assignments
            fetch(`get_master_data.php?master_id=${masterId}&type=users`)
                .then(response => response.json())
                .then(data => {
                    document.querySelectorAll('input[name="admin_niks[]"], input[name="borrower_niks[]"]').forEach(checkbox => {
                        checkbox.checked = data.niks.includes(checkbox.value);
                    });
                });
            
            document.getElementById('assignUsersModal').classList.remove('hidden');
        }

        function hideAssignUsersModal() {
            document.getElementById('assignUsersModal').classList.add('hidden');
        }

        function filterUsers() {
            const searchTerm = document.getElementById('searchUser').value.toLowerCase();
            document.querySelectorAll('.user-item').forEach(item => {
                const text = item.getAttribute('data-name');
                item.style.display = text.includes(searchTerm) ? 'flex' : 'none';
            });
        }

        // Delete Modal
        function deleteMaster(masterId, masterName) {
            document.getElementById('deleteMessage').textContent = 
                `Apakah Anda yakin ingin menghapus master organization "${masterName}"? Semua assignment akan dihapus.`;
            document.getElementById('deleteMasterId').value = masterId;
            document.getElementById('deleteModal').classList.remove('hidden');
        }

        function hideDeleteModal() {
            document.getElementById('deleteModal').classList.add('hidden');
        }

        // Close modals when clicking outside
        window.onclick = function(event) {
            const modals = ['masterModal', 'assignOrganizationsModal', 'assignUsersModal', 'deleteModal'];
            modals.forEach(modalId => {
                const modal = document.getElementById(modalId);
                if (event.target === modal) {
                    modal.classList.add('hidden');
                }
            });
        }
    </script>
</body>
</html>