<?php

namespace App\Http\Controllers;

use App\Models\ItemRequest;
use App\Models\ApprovalFlowTemplate;
use App\Models\User;
use App\Models\Company;
use App\Models\UserRequest;
use App\Models\ApprovalStepTemplate;
use App\Models\ApprovalStepApprover;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use App\Models\ApprovalAction;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;


class DashboardController extends Controller
{
    public function formatDurationHuman($seconds) {
    $hours = floor($seconds / 3600);
    $minutes = floor(($seconds % 3600) / 60);

    $parts = [];
    if ($hours > 0) $parts[] = $hours . ' jam';
    if ($minutes > 0) $parts[] = $minutes . ' menit';
    if (empty($parts)) $parts[] = '0 menit'; // jika durasi < 1 menit

    return implode(' ', $parts);
}

    public function __construct(
        private \App\Services\BusinessCentralService $bc,
        private \App\Services\InventoryBC $inv
    ) {}

    protected function activeCompanyId(): ?int
    {
        $key = session('current_company_name') ?? data_get(session('user'), 'role');

        $map = [
            'Regent' => 'CI',   
            'HIN'    => 'TBI',  
        ];

        $code = $map[$key] ?? null;

        if (!$code) {
            return null;
        }

        return Company::where('code', $code)->value('id');
    }


    protected function resolveCurrentUser(): User
    {
        $sess  = session('user') ?? [];
        $email = $sess['email'] ?? null;
        $name  = $sess['name'] ?? ($sess['displayName'] ?? 'MS User');
        abort_unless($email, 401, 'Not authenticated');

        return User::firstOrCreate(['email' => $email], ['name' => $name]);
    }

    public function index(Request $request)
{
    $formatDuration = function ($seconds) {
        if (!$seconds || $seconds <= 0) return '-';
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        return sprintf('%d jam %d menit', $hours, $minutes);
    };

    set_time_limit(300);
    $me = $this->resolveCurrentUser();
    $cid = $this->activeCompanyId();

    // ==========================================
    // Hitung start → finish & submit time per item_request
    // ==========================================
    $itemRequests = ItemRequest::with(['requester','approvalInstances'])
        ->get();

    $itemRequestData = $itemRequests->map(function ($ir) use ($formatDuration) {
        // Ambil approval instance yang relevan
        $instance = $ir->approvalInstances()
            ->whereNotNull('started_at')
            ->where('status', 'Approved')
            ->orderBy('started_at')
            ->first();

        $startToFinish = null;
        $submitTime    = null;

        if ($ir->requested_at && $instance?->ended_at) {
            $startToFinish = Carbon::parse($ir->requested_at)
                                ->diffInSeconds(Carbon::parse($instance->ended_at));
        }

        if ($ir->created_at && $instance?->started_at) {
            $submitTime = Carbon::parse($ir->created_at)
                            ->diffInSeconds(Carbon::parse($instance->started_at));
        }

        // Ambil from_user dari user_requests
        $fromUser = UserRequest::where('item_request_id', $ir->id)
            ->with('user')
            ->first()?->user?->name ?? '-';

        $stepTemplateId = ApprovalStepTemplate::where('flow_template_id', $ir->flow_template_id)
        ->where('step_no', $ir->current_step_no) // current step dari itemRequest
        ->value('id');
        $approverName = '-';
        if ($stepTemplateId) {
            $approverId = ApprovalStepApprover::where('step_template_id', $stepTemplateId)
                ->value('approver_user_id');
                    
            $approver = User::find($approverId);
            $approverName = $approver ? $approver->name : '-';
        }

         $currentStep = ApprovalStepTemplate::where('flow_template_id', $ir->flow_template_id)
        ->orderBy('step_no', 'asc')
        ->get()
        ->first(); // ambil step pertama, atau bisa modifikasi logikanya untuk current step sesuai kebutuhan

        return [
            'request_number'     => $ir->id,
            'vendor_name'        => $ir->vendor_name ?? '-', 
            'requester_name'     => $ir->requester?->name ?? '-',
            'from_user'          => $fromUser,
            'requested_at'       => $ir->requested_at,
            'start_to_finish'    => $startToFinish,
            'start_to_finish_hr' => $formatDuration($startToFinish),
            'submit_time'        => $submitTime,
            'submit_time_hr'     => $formatDuration($submitTime),
            'status'             => [
            'current_step' => $ir->status,
            'approver_name'=> $approverName
        ]
        ];
    });
   
    // ==========================================
    // Logic KPI & averageTimes lama
    // ==========================================
    $purchasers = DB::table('item_requests as ir')
        ->join('approval_instances as ai', 'ai.item_request_id', '=', 'ir.id')
        ->join('users as u', 'u.id', '=', 'ir.requester_id')
        ->where('ai.status', 'Approved')
        ->groupBy('ir.requester_id', 'u.name')
        ->select(
            'ir.requester_id',
            'u.name as requester_name',
            DB::raw('AVG(TIMESTAMPDIFF(SECOND, ir.requested_at, ai.ended_at)) as avg_total_seconds')
        )
        ->get();

    $averageTimes = $purchasers->map(function ($row) use ($formatDuration) {
        $sourcingSeconds = DB::table('item_requests as ir')
            ->join('approval_instances as ai', 'ai.item_request_id', '=', 'ir.id')
            ->where('ir.requester_id', $row->requester_id)
            ->whereIn('ir.id', function ($q) {
                $q->select('item_request_id')->from('user_requests');
            })
            ->whereNotNull('ir.requested_at')
            ->whereNotNull('ai.started_at')
            ->select(DB::raw('AVG(TIMESTAMPDIFF(SECOND, ir.requested_at, ai.started_at)) as avg_seconds'))
            ->value('avg_seconds');

        return (object) [
            'requester_id' => $row->requester_id,
            'requester_name' => $row->requester_name,
            'avg_total_seconds' => (int) $row->avg_total_seconds,
            'avg_total' => $formatDuration($row->avg_total_seconds),
            'avg_sourcing_seconds' => $sourcingSeconds ? (int) $sourcingSeconds : null,
            'avg_sourcing' => $sourcingSeconds ? $formatDuration($sourcingSeconds) : '-',
        ];
    });

    $averageTimesIndexed = $averageTimes->keyBy('requester_name');
    $totalRequestsByStatus = $this->getTotalRequestByStatusAndRequester();
    $totalRequestsThisWeek = $this->getWeeklyApprovalStats();

    // ==========================================
    // Hitung durasi per approval action
    // ==========================================
    $actions = ApprovalAction::orderBy('approval_instance_id')
                ->orderBy('acted_at')
                ->get();
    $grouped = $actions->groupBy('approval_instance_id');

    $durations = [];
    foreach ($grouped as $instanceId => $logs) {
        $logs = $logs->sortBy('acted_at')->values();
        $submitAction = $logs->firstWhere(fn($item) => strtolower($item->action) === 'submit');
        if (!$submitAction) continue;

        $startCounting = false;
        foreach ($logs as $i => $current) {
            if (strtolower($current->action) === 'submit') {
                $startCounting = true;
                continue;
            }
            if (!$startCounting) continue;
            if (strtolower($current->action) !== 'approve') continue;

            $previous = $logs[$i - 1];
            $durationSeconds = Carbon::parse($previous->acted_at)
                                ->diffInSeconds(Carbon::parse($current->acted_at));

            $durations[] = [
                'actor_user_id' => $current->actor_user_id,
                'instance_id'   => $instanceId,
                'duration_seconds' => $durationSeconds,
                'step_no'       => $current->step_no,
            ];
        }
    }

    $durationCollection = collect($durations);
    $avgByUser = $durationCollection
        ->groupBy('actor_user_id')
        ->map(function ($items, $userId) {
            $user = User::find($userId);
            return [
                'actor_user_id' => $userId,
                'actor_name'    => $user ? $user->name : null,
                'avg_seconds'   => round($items->avg('duration_seconds'), 2),
                'avg_hours'     => round($items->avg('duration_seconds') / 3600, 2),
                'avg_duration'  => $this->formatDurationHuman($items->avg('duration_seconds')),
                'total_approvals'=> $items->count(),
            ];
        })
        ->values();

    // ==========================================
    // KPI
    // ==========================================
    $kpi_inbox        = ItemRequest::where('status', 'InReview')->count();
    $kpi_my_open      = ItemRequest::where('requester_id', $me->id)->whereIn('status',['Draft','InReview'])->count();
    $kpi_approved_30d = ItemRequest::where('status','Approved')->where('updated_at','>=',now()->subDays(30))->count();
    $kpi_rejected_30d = ItemRequest::where('status','Rejected')->where('updated_at','>=',now()->subDays(30))->count();

            $approvedItems = ItemRequest::where('status', 'Approved')
            ->with('approvalInstances') // eager load untuk efisiensi
            ->get();

            $totalSeconds = $approvedItems->reduce(function ($carry, $item) {
                // Ambil approval instance terakhir yang ada ended_at
                $instance = $item->approvalInstances
                    ->whereNotNull('ended_at')
                    ->sortByDesc('ended_at') // ambil yang terbaru
                    ->first();

                if (!$instance) return $carry;

                return $carry + $item->created_at->diffInSeconds($instance->ended_at);
            }, 0);

            $count = $approvedItems->filter(fn($item) => $item->approvalInstances->whereNotNull('ended_at')->count() > 0)->count();

            $avgSeconds = $count > 0 ? $totalSeconds / $count : 0;
            $avgDuration = gmdate('H:i:s', $avgSeconds);
    return view('dashboard.index', compact(
        'kpi_inbox','kpi_my_open','kpi_approved_30d','kpi_rejected_30d',
        'avgByUser','totalRequestsByStatus','totalRequestsThisWeek','averageTimesIndexed',
        'itemRequestData','avgDuration'
    ));
}

    // public function index(Request $request)
    // {
        
    //     $formatDuration = function ($seconds) {
    //         if (!$seconds || $seconds <= 0) return '-';
    //         $hours = floor($seconds / 3600);
    //         $minutes = floor(($seconds % 3600) / 60);
    //         return sprintf('%d jam %d menit', $hours, $minutes);
    //     };

    //     // Ambil semua purchaser
    //     $purchasers = DB::table('item_requests as ir')
    //         ->join('approval_instances as ai', 'ai.item_request_id', '=', 'ir.id')
    //         ->join('users as u', 'u.id', '=', 'ir.requester_id')
    //         ->where('ai.status', 'Approved')
    //         ->groupBy('ir.requester_id', 'u.name')
    //         ->select(
    //             'ir.requester_id',
    //             'u.name as requester_name',
    //             DB::raw('AVG(TIMESTAMPDIFF(SECOND, ir.requested_at, ai.ended_at)) as avg_total_seconds')
    //         )
    //         ->get();

    //     // Loop semua purchaser dan hitung sourcing time miliknya
    //     $averageTimes = $purchasers->map(function ($row) use ($formatDuration) {
    //         // Ambil semua item_request milik purchaser ini yang juga punya user_request
    //         $sourcingSeconds = DB::table('item_requests as ir')
    //             ->join('approval_instances as ai', 'ai.item_request_id', '=', 'ir.id')
    //             ->where('ir.requester_id', $row->requester_id)
    //             ->whereIn('ir.id', function ($q) {
    //                 $q->select('item_request_id')->from('user_requests');
    //             })
    //             ->whereNotNull('ir.requested_at')
    //             ->whereNotNull('ai.started_at')
    //             ->select(DB::raw('AVG(TIMESTAMPDIFF(SECOND, ir.requested_at, ai.started_at)) as avg_seconds'))
    //             ->value('avg_seconds');
    //         return (object) [
    //             'requester_id' => $row->requester_id,
    //             'requester_name' => $row->requester_name,
    //             'avg_total_seconds' => (int) $row->avg_total_seconds,
    //             'avg_total' => $formatDuration($row->avg_total_seconds),
    //             'avg_sourcing_seconds' => $sourcingSeconds ? (int) $sourcingSeconds : null,
    //             'avg_sourcing' => $sourcingSeconds ? $formatDuration($sourcingSeconds) : '-',
    //         ];
    //     });
    //         $averageTimesIndexed = $averageTimes->keyBy('requester_name');
    //         $totalRequestsByStatus = $this->getTotalRequestByStatusAndRequester();
    //         $totalRequestsThisWeek = $this->getWeeklyApprovalStats();
    //         $actions = ApprovalAction::orderBy('approval_instance_id')
    //                     ->orderBy('acted_at')
    //                     ->get();
    //         // Kelompokkan berdasarkan instance
    //         $grouped = $actions->groupBy('approval_instance_id');

    //         $durations = [];

    //         foreach ($grouped as $instanceId => $logs) {
    //             $logs = $logs->sortBy('acted_at')->values();

    //             // Cari action 'submit'
    //             $submitAction = $logs->firstWhere(fn($item) => strtolower($item->action) === 'submit');
    //             if (!$submitAction) continue;

    //             // Cari index submit sebagai titik awal
    //            $startCounting = false;
    //             foreach ($logs as $i => $current) {
    //                 if (strtolower($current->action) === 'submit') {
    //                     $startCounting = true;
    //                     continue; // mulai hitung dari log berikutnya
    //                 }

    //                 if (!$startCounting) continue; // belum ketemu submit

    //                 if (strtolower($current->action) !== 'approve') continue;

    //                 $previous = $logs[$i - 1]; // aksi sebelumnya

    //                 $durationSeconds = Carbon::parse($previous->acted_at)
    //                                     ->diffInSeconds(Carbon::parse($current->acted_at));

    //                 $durations[] = [
    //                     'actor_user_id' => $current->actor_user_id,
    //                     'instance_id' => $instanceId,
    //                     'duration_seconds' => $durationSeconds,
    //                     'step_no' => $current->step_no,
    //                 ];
    //             }
    //         }
           
    //         // Collection untuk hitung rata-rata
    //         $durationCollection = collect($durations);
    //         $avgByUser = $durationCollection
    //             ->groupBy('actor_user_id')
    //             ->map(function ($items, $userId) {
    //                 // Ambil nama user dari model User
    //                 $user = User::find($userId);

    //                 return [
    //                     'actor_user_id' => $userId,
    //                     'actor_name' => $user ? $user->name : null, // jika user tidak ditemukan
    //                     'avg_seconds' => round($items->avg('duration_seconds'), 2),
    //                     'avg_hours' => round($items->avg('duration_seconds') / 3600, 2),
    //                     'avg_duration' => $this->formatDurationHuman($items->avg('duration_seconds')), // jika method controller
    //                     'total_approvals' => $items->count(),
    //                 ];
    //             })
    //             ->values();

           
    //     $approvedItems = ItemRequest::where('status', 'Approved')->get();
    //     $totalSeconds = $approvedItems->reduce(function ($carry, $item) {
    //         return $carry + $item->created_at->diffInSeconds($item->updated_at);
    //     }, 0);
    //     $count = $approvedItems->count();
    //     $avgSeconds = $count > 0 ? $totalSeconds / $count : 0;
    //     $avgDuration = gmdate('H:i:s', $avgSeconds);
    //     set_time_limit(300);
    //     $me = $this->resolveCurrentUser();
    //     $cid = $this->activeCompanyId();
    //     $inboxQuery = ItemRequest::query()
    //         ->where('status', 'InReview')
    //         ->whereExists(function ($q) {
    //             $q->from('approval_instances as ai')
    //               ->whereColumn('ai.item_request_id','item_requests.id')
    //               ->where('ai.status','InProgress');
    //         });

    //     if (!($me->role?->code === 'ADMIN'|| $me->role?->code === 'APPROVER_1' || $me->role?->code === 'APPROVER_2' )) {
    //         $inboxQuery->whereExists(function ($q) use ($me) {
    //             $q->from('approval_step_templates as ast')
    //               ->whereColumn('ast.flow_template_id','item_requests.flow_template_id')
    //               ->whereColumn('ast.step_no','item_requests.current_step_no')
    //               ->where(function ($qq) use ($me) {
    //                   $qq->where('ast.approver_user_id', $me->id)
    //                      ->orWhere('ast.approver_role_id', $me->role_id);
    //               });
    //         });
    //     }
    //     $kpi_inbox        = (clone $inboxQuery)->count();
    //     $kpi_my_open      = ItemRequest::where('requester_id',$me->id)->when($cid, fn($q)=>$q->where('company_id', $cid))->whereIn('status',['Draft','InReview'])->count();
    //     $kpi_approved_30d = ItemRequest::where('status','Approved')->when($cid, fn($q)=>$q->where('company_id', $cid))->where('updated_at','>=',now()->subDays(30))->count();
    //     $kpi_rejected_30d = ItemRequest::where('status','Rejected')->when($cid, fn($q)=>$q->where('company_id', $cid))->where('updated_at','>=',now()->subDays(30))->count();
        // $poData  = $this->bc->getPoSuggestions($me->name ?? $me->email);
        // $poItems = collect($poData['items'] ?? []);
        
        // $poVendors   = $poItems->pluck('vendor_name')->filter()->unique()->sort()->values();
        // $poLocations = $poItems->pluck('location_code')->filter()->unique()->sort()->values();
        // $poCounts = [
        //     'need_order'       => $poItems->where('status','Need Order')->count(),
        //     'need_order_for_to'=> $poItems->where('status','Need Order For TO')->count(),
        //     'follow_up'        => $poItems->where('status','Follow Up PO')->count(),
        //     'for_stock'        => $poItems->where('status','Order For Stock')->count(),
        //     'no_need'          => $poItems->where('status','No Need Order')->count(),
        // ];
        // $poRows = $poItems->map(function ($r) {
        //     return [
        //         'item_no'       => $r['item_no'] ?? null,
        //         'description'   => $r['description'] ?? null,
        //         'vendor_name'   => $r['vendor_name'] ?? null,
        //         'location_code' => $r['location_code'] ?? null,
        //         'status'        => $r['status'] ?? null,
        //         'qty_to_order'  => (float)($r['qty_to_order'] ?? 0),
        //     ];
        // });
        // $poTop = $poItems
        //     ->sortByDesc(fn($r) => (float)($r['qty_to_order'] ?? 0))
        //     ->take(8)
        //     ->values();
        // $rawSku = $this->inv->getItemLedgerAndSkuMappingAsync();
        // $top    = $rawSku instanceof Collection ? $rawSku->all() : (array) $rawSku;
        // if (isset($top['items']) && is_array($top['items'])) {
        //     $skuRows = collect($top['items']);
        // } else {
        //     $skuRows = collect($top)
        //         ->flatMap(fn($chunk) => is_array($chunk) && array_is_list($chunk) ? $chunk : (is_array($chunk) ? [$chunk] : []))
        //         ->values();
        // }

        // $skuModified30d = $skuRows->filter(function ($r) {
        //     $d = $r['LastModifiedAt'] ?? null;
        //     if (!$d) return false;
        //     try {
        //         return Carbon::createFromFormat('d F Y', $d)->gte(now()->subDays(30));
        //     } catch (\Throwable $e) {
        //         return false;
        //     }
        // })->count();

        // $skuCounts = [
        //     'rows'         => $skuRows->count(),
        //     'modified_30d' => $skuModified30d,
        // ];

        // $skuTop = $skuRows
        //     ->sortByDesc(fn($r) => (int)($r['Consumption / 2 months'] ?? 0))
        //     ->take(8)
        //     ->values();
        // 'poCounts','poTop','skuCounts','skuTop',
        //     'poRows','poVendors','poLocations'
    //     return view('dashboard.index', compact(
    //         'kpi_inbox','kpi_my_open','kpi_approved_30d','kpi_rejected_30d'
    //         ,'avgDuration','avgByUser','totalRequestsByStatus','totalRequestsThisWeek','averageTimesIndexed'
    //     ));
    // }

    public function getTotalRequestByStatusAndRequester()
{
    // Ambil request beserta requester (user) yang role_id = 4
    $requests = DB::table('item_requests as r')
        ->join('users as u', 'r.requester_id', '=', 'u.id')
        ->where('u.role_id', 4) // filter role_id 4
        ->select('r.status', 'u.id as user_id', 'u.name as requester_name')
        ->get();

    $data = [];

    foreach ($requests as $request) {
        // Normalisasi status
        switch (strtolower($request->status)) {
            case 'rejected':
                $statusKey = 'Rejected';
                break;
            case 'draft':
                $statusKey = 'Draft';
                break;
            case 'inreview':
            case 'in_review':
                $statusKey = 'InReview';
                break;
            case 'approved':
                $statusKey = 'Approved';
                break;
            case 'cancelled':
            case 'canceled':
                $statusKey = 'Cancelled';
                break;
            default:
                $statusKey = 'Other';
        }

        // Inisialisasi jika requester belum ada
        if (!isset($data[$request->requester_name])) {
            $data[$request->requester_name] = [
                'Rejected' =>0,
                'Draft' => 0,
                'InReview' => 0,
                'Approved' => 0,
                'Cancelled' => 0,
            ];
        }

        // Tambahkan count
        if (isset($data[$request->requester_name][$statusKey])) {
            $data[$request->requester_name][$statusKey]++;
        }
    }

    return $data;
}

public function getWeeklyApprovalStats()
{
    // Ambil semua approval action minggu ini
    $startOfWeek = Carbon::now()->startOfWeek();
    $endOfWeek = Carbon::now()->endOfWeek();

    $actions = ApprovalAction::whereBetween('created_at', [$startOfWeek, $endOfWeek])
        ->orderBy('approval_instance_id')
        ->orderBy('created_at')
        ->get();
    // Kelompokkan per approval_instance_id
    $grouped = $actions->groupBy('approval_instance_id');
    $counts = [
        'Submitted' => 0,
        'Approved' => 0,
        'Rejected' => 0,
    ];

    foreach ($grouped as $instanceId => $logs) {
        // Ambil unique action per instance berdasarkan id
        $uniqueActions = $logs->unique('action');

        foreach ($uniqueActions as $action) {
            switch ($action->action) {
                case 'Submit':
                    $counts['Submitted']++;
                    break;
                case 'Approve':
                    $counts['Approved']++;
                    break;
                case 'Reject':
                    $counts['Rejected']++;
                    break;
            }
        }
    }
    
    return $counts;
    
}
}
