<?php

namespace App\Http\Controllers;

use App\Models\ItemRequest;
use App\Models\ApprovalFlowTemplate;
use App\Models\User;
use App\Models\Company;
use App\Models\UserRequest;
use App\Models\ApprovalStepTemplate;
use App\Models\ApprovalStepApprover;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use App\Models\ApprovalAction;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;


class DashboardController extends Controller
{
    public function formatDurationHuman($seconds) {
    $hours = floor($seconds / 3600);
    $minutes = floor(($seconds % 3600) / 60);

    $parts = [];
    if ($hours > 0) $parts[] = $hours . ' jam';
    if ($minutes > 0) $parts[] = $minutes . ' menit';
    if (empty($parts)) $parts[] = '0 menit'; // jika durasi < 1 menit

    return implode(' ', $parts);
}

    public function __construct(
        private \App\Services\BusinessCentralService $bc,
        private \App\Services\InventoryBC $inv
    ) {}

    protected function activeCompanyId(): ?int
    {
        $key = session('current_company_name') ?? data_get(session('user'), 'role');

        $map = [
            'Regent' => 'CI',   
            'HIN'    => 'TBI',  
        ];

        $code = $map[$key] ?? null;

        if (!$code) {
            return null;
        }

        return Company::where('code', $code)->value('id');
    }


    protected function resolveCurrentUser(): User
    {
        $sess  = session('user') ?? [];
        $email = $sess['email'] ?? null;
        $name  = $sess['name'] ?? ($sess['displayName'] ?? 'MS User');
        abort_unless($email, 401, 'Not authenticated');

        return User::firstOrCreate(['email' => $email], ['name' => $name]);
    }


public function index()
{
    // ============================
    // 1. GET DATA AO, WAREHOUSE TO, OUTLET ORDER
    // ============================
    $assemblyOrders = $this->safeFetch(fn() => $this->bc->getAllAssemblyOrder());
    $warehouseTO    = $this->bc->getAllTransferWithLinesV2();
    $outletOrders   = $this->safeFetch(fn() => $this->bc->getAllTransferWithLines());
    $assemblyOrdersList = $assemblyOrders['data'];
    $outletOrdersList = $outletOrders['original']['data'];
    $transferDecoded = $warehouseTO instanceof \Illuminate\Http\JsonResponse
        ? $warehouseTO->getData(true)
        : $warehouseTO;

    $transferData = $transferDecoded['data'] ?? [];

    // ====================================================
    // 2. Filter hanya transfer line "ST. ORDER"
    // ====================================================
    $filteredLines = [];
    foreach ($transferData as $header) {
        $spec = $header['transSpec'] ?? '';

        if (trim(strtoupper($spec)) === 'ST. ORDER') {
            if (!empty($header['Lines']) && is_array($header['Lines'])) {
                foreach ($header['Lines'] as $line) {
                    $filteredLines[] = $line;
                }
            }
        }
    }

    // ====================================================
    // 3. HITUNG jumlah
    // ====================================================
    $assemblyCount  = $assemblyOrders['count'];
    $warehouseCount = count($filteredLines);
    $outletCount    = $outletOrders['original']['count'];

    // ====================================================
    // 4. GET ITEM LOT — FIX: TAMPILKAN SEMUA LOT, BUKAN PER ITEM
    // ====================================================
    $alllotitem = $this->safeFetch(fn() => $this->bc->getItemLot());  
    $itemLots    = $alllotitem['items'] ?? [];

    $allLotsFlattened = [];

    foreach ($itemLots as $item) {

        if (empty($item['lots'])) continue;

        foreach ($item['lots'] as $lot) {

            // Skip lot invalid
            if (!isset($lot['Expiration_Date']) || 
                $lot['Expiration_Date'] === "0001-01-01" || 
                empty($lot['Expiration_Date'])) {
                continue;
            }

            // Ambil quantity, support 2 jenis key
            $qty = $item['TotalQty'] 
                ?? $lot['Qty'] 
                ?? 0;

            // Skip jika Qty <= 0
            if ($qty <= 0) continue;

            // MASUKKAN SEMUA LOT TANPA DI-GROUP
            $allLotsFlattened[] = [
                'ItemNo'         => $item['ItemNo'],
                'Description'    => $item['ItemDescription'] ?? '',
                'Lot_No'         => $lot['Lot_No'] ?? '',
                'Qty'            => $qty,
                'ExpiredAt'      => $lot['Expiration_Date'],
            ];
        }
    }

    // ====================================================
    // 5. FILTER: expired masuk kategori H-1 → H-3
    // ====================================================
    $today = now()->startOfDay();

    $expiringSoon = array_filter($allLotsFlattened, function ($lot) use ($today) {
        $expDate = \Carbon\Carbon::parse($lot['ExpiredAt'])->startOfDay();
        $diff    = $today->diffInDays($expDate, false);

        return $diff >= 1 && $diff <= 7;
    });

    // ====================================================
    // 6. GROUP BY itemNo DAN AMBIL EXPIRED TERDEKAT
    // ====================================================
    $closestExpiring = [];

    foreach ($expiringSoon as $lot) {
        $itemNo = $lot['ItemNo'];
        $expDate = \Carbon\Carbon::parse($lot['ExpiredAt']);

        // Jika belum ada item ini → langsung simpan
        if (!isset($closestExpiring[$itemNo])) {
            $closestExpiring[$itemNo] = $lot;
            continue;
        }

        // Bandingkan tanggal expired yang sudah ada dengan yang baru
        $existingExpDate = \Carbon\Carbon::parse($closestExpiring[$itemNo]['ExpiredAt']);

        if ($expDate->lessThan($existingExpDate)) {
            $closestExpiring[$itemNo] = $lot;
        }
    }

    $expiringCount = count($closestExpiring);

    // ====================================================
    // 7. FILTER DATA OVERDUE
    // ====================================================
    // Overdue Assembly Orders
    $overdueAO = array_filter($assemblyOrdersList, function ($ao) use ($today) {
    if (empty($ao['dueDate'])) return false;
    $dueDate = \Carbon\Carbon::parse($ao['dueDate'])->startOfDay();
    // Hanya dihitung overdue jika tanggal sudah lewat hari ini
    return $dueDate->lt($today);
});

// Overdue Transfer Orders (tolist)
$overdueTO = array_filter($filteredLines, function ($line) use ($today) {
    if (empty($line['shipmentDate'])) return false;
    $shipDate = \Carbon\Carbon::parse($line['shipmentDate'])->startOfDay();
    return $shipDate->lt($today);
});

// Overdue Outlet Orders
$overdueOutlet = array_filter($outletOrdersList, function ($outlet) use ($today) {
    if (empty($outlet['shipmentDate'])) return false;
    $shipDate = \Carbon\Carbon::parse($outlet['shipmentDate'])->startOfDay();
    return $shipDate->lt($today);
});

    $overdueAOCount      = count($overdueAO);
    $overdueTOCount      = count($overdueTO);
    $overdueOutletCount  = count($overdueOutlet);

    // ====================================================
    // 8. RETURN VIEW
    // ====================================================
    return view('dashboard.index', [
        'assemblyCount'      => $assemblyCount,
        'warehouseCount'     => $warehouseCount,
        'outletCount'        => $outletCount,
        'aolist'             => $assemblyOrdersList,
        'outletlist'         => $outletOrdersList,
        'tolist'             => $filteredLines,
        'expiringCount'      => $expiringCount,
        'expiringItems'      => $closestExpiring,

        // Overdue data
        'overdueAOCount'     => $overdueAOCount,
        'overdueTOCount'     => $overdueTOCount,
        'overdueOutletCount' => $overdueOutletCount,
        'overdueAOList'      => $overdueAO,
        'overdueTOList'      => $overdueTO,
        'overdueOutletList'  => $overdueOutlet,

        'chartData'          => [],
    ]);
}




    // ==========================================================
    // AMANKAN FETCH AGAR SELALU MENGEMBALIKAN ARRAY
    // ==========================================================
    private function safeFetch($callback): array
    {
        try {
            $result = $callback();

            return $this->decodeResult($result);
        } catch (\Throwable $e) {
            \Log::error("BC fetch error: " . $e->getMessage());
            return [];
        }
    }

    // ==========================================================
    // NORMALISASI SEMUA OUTPUT → ARRAY
    // ==========================================================
    private function decodeResult($data): array
    {
        // Jika JsonResponse (Laravel)
        if ($data instanceof JsonResponse) {
            return $data->getData(true) ?? [];
        }

        // Jika response instance lain
        if ($data instanceof \Illuminate\Http\Response) {
            return json_decode($data->getContent(), true) ?? [];
        }

        // Jika JSON string
        if (is_string($data)) {
            $decoded = json_decode($data, true);
            return $decoded ?: [];
        }

        // Jika memang array → kembalikan apa adanya
        if (is_array($data)) {
            return $data;
        }

        // Jika object biasa
        if (is_object($data)) {
            return json_decode(json_encode($data), true) ?? [];
        }

        return [];
    }
}
