<?php
session_start();
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'it') {
    header("Location: ../login.php");
    exit();
}

// Set page title
$page_title = "Edit Tanggal - Date Management";

// Include header
include '../template_it/header.php';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'update_master') {
        $master_id = (int)$_POST['master_id'];
        $manufacturing_date = $_POST['manufacturing_date'];
        $expired_date = $_POST['expired_date'];
        
        // Check if master_color table has these columns
        $check_columns = "SHOW COLUMNS FROM master_color LIKE 'manufacturing_date'";
        $result = $conn->query($check_columns);
        
        if ($result->num_rows == 0) {
            // Add columns if they don't exist
            $alter_query = "ALTER TABLE master_color 
                           ADD COLUMN manufacturing_date DATE NULL COMMENT 'Tanggal pembuatan master color',
                           ADD COLUMN expired_date DATE NULL COMMENT 'Tanggal kedaluwarsa master color'";
            $conn->query($alter_query);
        }
        
        // Start transaction
        $conn->begin_transaction();
        
        try {
            // Update master_color
            $query = "UPDATE master_color SET 
                      manufacturing_date = ?, 
                      expired_date = ? 
                      WHERE id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("ssi", $manufacturing_date, $expired_date, $master_id);
            
            if (!$stmt->execute()) {
                throw new Exception("Gagal mengupdate Master Color");
            }
            
            // Check if master_color_panel table has date columns
            $check_panel_columns = "SHOW COLUMNS FROM master_color_panel LIKE 'manufacturing_date'";
            $panel_result = $conn->query($check_panel_columns);
            
            if ($panel_result->num_rows == 0) {
                // Add columns to master_color_panel if they don't exist
                $alter_panel_query = "ALTER TABLE master_color_panel 
                                     ADD COLUMN manufacturing_date DATE NULL COMMENT 'Tanggal pembuatan master color panel',
                                     ADD COLUMN expired_date DATE NULL COMMENT 'Tanggal kedaluwarsa master color panel'";
                $conn->query($alter_panel_query);
            }
            
            // Update related master_color_panel records
            $panel_query = "UPDATE master_color_panel SET 
                           manufacturing_date = ?, 
                           expired_date = ? 
                           WHERE master_color_id = ?";
            $panel_stmt = $conn->prepare($panel_query);
            $panel_stmt->bind_param("ssi", $manufacturing_date, $expired_date, $master_id);
            
            if (!$panel_stmt->execute()) {
                throw new Exception("Gagal mengupdate Master Color Panel");
            }
            
            // Check how many panel records were updated
            $updated_panels = $panel_stmt->affected_rows;
            
            // Commit transaction
            $conn->commit();
            
            $success_message = "Master Color tanggal berhasil diupdate! " . 
                              ($updated_panels > 0 ? "({$updated_panels} Master Panel juga ikut terupdate)" : "");
            
            // Log audit
            $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, ip_address) 
                           VALUES ('UPDATE_MASTER_DATE', ?, 'it', ?, ?)";
            $audit_stmt = $conn->prepare($audit_query);
            $details = "Updated Master Color ID: $master_id - Manufacturing Date: $manufacturing_date, Expired Date: $expired_date. Updated $updated_panels related master panels.";
            $ip = $_SERVER['REMOTE_ADDR'];
            $audit_stmt->bind_param("iss", $_SESSION['user_id'], $details, $ip);
            $audit_stmt->execute();
            
        } catch (Exception $e) {
            // Rollback transaction
            $conn->rollback();
            $error_message = "Gagal mengupdate data: " . $e->getMessage();
        }
    }
    
    if ($action === 'update_panel') {
        $panel_id = (int)$_POST['panel_id'];
        $manufacturing_date = $_POST['manufacturing_date'];
        $expired_date = $_POST['expired_date'];
        
        // Check if master_color_panel table has date columns
        $check_panel_columns = "SHOW COLUMNS FROM master_color_panel LIKE 'manufacturing_date'";
        $panel_result = $conn->query($check_panel_columns);
        
        if ($panel_result->num_rows == 0) {
            // Add columns if they don't exist
            $alter_panel_query = "ALTER TABLE master_color_panel 
                                 ADD COLUMN manufacturing_date DATE NULL COMMENT 'Tanggal pembuatan master color panel',
                                 ADD COLUMN expired_date DATE NULL COMMENT 'Tanggal kedaluwarsa master color panel'";
            $conn->query($alter_panel_query);
        }
        
        $query = "UPDATE master_color_panel SET 
                  manufacturing_date = ?, 
                  expired_date = ? 
                  WHERE id = ?";
        $stmt = $conn->prepare($query);
        $stmt->bind_param("ssi", $manufacturing_date, $expired_date, $panel_id);
        
        if ($stmt->execute()) {
            $success_message = "Master Panel tanggal berhasil diupdate!";
            
            // Log audit
            $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, ip_address) 
                           VALUES ('UPDATE_PANEL_DATE', ?, 'it', ?, ?)";
            $audit_stmt = $conn->prepare($audit_query);
            $details = "Updated Master Panel ID: $panel_id - Manufacturing Date: $manufacturing_date, Expired Date: $expired_date";
            $ip = $_SERVER['REMOTE_ADDR'];
            $audit_stmt->bind_param("iss", $_SESSION['user_id'], $details, $ip);
            $audit_stmt->execute();
        } else {
            $error_message = "Gagal mengupdate Master Panel!";
        }
    }
    
    if ($action === 'update_stock') {
        $stock_id = (int)$_POST['stock_id'];
        $manufacturing_date = $_POST['manufacturing_date'];
        $expired_date = $_POST['expired_date'];
        
        $query = "UPDATE color_stock_items SET 
                  manufacturing_date = ?, 
                  expired_date = ? 
                  WHERE id = ?";
        $stmt = $conn->prepare($query);
        $stmt->bind_param("ssi", $manufacturing_date, $expired_date, $stock_id);
        
        if ($stmt->execute()) {
            $success_message = "Copy Panel (Stock Item) tanggal berhasil diupdate!";
            
            // Log audit
            $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, ip_address) 
                           VALUES ('UPDATE_STOCK_DATE', ?, 'it', ?, ?)";
            $audit_stmt = $conn->prepare($audit_query);
            $details = "Updated Stock Item ID: $stock_id - Manufacturing Date: $manufacturing_date, Expired Date: $expired_date";
            $ip = $_SERVER['REMOTE_ADDR'];
            $audit_stmt->bind_param("iss", $_SESSION['user_id'], $details, $ip);
            $audit_stmt->execute();
        } else {
            $error_message = "Gagal mengupdate Copy Panel!";
        }
    }
}

// Get current tab
$current_tab = $_GET['tab'] ?? 'master';
?>

<div class="container mx-auto">
    <!-- Header Section -->
    <div class="bg-white rounded-lg shadow-md p-6 mb-6">
        <h2 class="text-2xl font-bold text-gray-800">Edit Tanggal Manufacturing & Expired</h2>
        <p class="text-gray-600 mt-1">Kelola tanggal manufacturing dan expired date untuk Master Color, Master Panel, dan Copy Panel</p>
    </div>

    <?php if (isset($success_message)): ?>
    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
        <?php echo htmlspecialchars($success_message); ?>
    </div>
    <?php endif; ?>

    <?php if (isset($error_message)): ?>
    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
        <?php echo htmlspecialchars($error_message); ?>
    </div>
    <?php endif; ?>

    <!-- Tab Navigation -->
    <div class="bg-white rounded-lg shadow-md mb-6">
        <div class="border-b border-gray-200">
            <nav class="-mb-px flex">
                <a href="?tab=master" 
                   class="<?php echo $current_tab === 'master' ? 'bg-blue-50 border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?> 
                          whitespace-nowrap py-4 px-6 border-b-2 font-medium text-sm">
                    <i class="fas fa-palette mr-2"></i>Master Color & Panel
                </a>
                <a href="?tab=copy" 
                   class="<?php echo $current_tab === 'copy' ? 'bg-blue-50 border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?> 
                          whitespace-nowrap py-4 px-6 border-b-2 font-medium text-sm">
                    <i class="fas fa-boxes mr-2"></i>Copy Panel (Stock Items)
                </a>
            </nav>
        </div>
    </div>

    <?php if ($current_tab === 'master'): ?>
    <!-- Master Color & Panel Section -->
    <div class="bg-white rounded-lg shadow-md p-6">
        <h3 class="text-xl font-bold text-gray-800 mb-4">
            <i class="fas fa-palette text-purple-600 mr-2"></i>Master Color & Panel - Edit Tanggal
        </h3>
        
        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-4">
            <div class="flex items-center">
                <i class="fas fa-info-circle text-blue-600 mr-2"></i>
                <p class="text-blue-800 text-sm">
                    <strong>Info:</strong> Ketika mengupdate tanggal Master Color, semua Master Panel yang terkait akan otomatis ikut terupdate dengan tanggal yang sama.
                </p>
            </div>
        </div>
        
        <!-- Search and Filter -->
        <div class="mb-4">
            <div class="flex flex-col md:flex-row gap-4">
                <input type="text" id="searchMaster" placeholder="Cari kode warna, buyer, collection, panel code..." 
                       class="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                <select id="filterBuyerMaster" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                    <option value="">Semua Buyer</option>
                    <?php
                    $buyer_query = "SELECT id, nama_buyer FROM buyers ORDER BY nama_buyer";
                    $buyer_result = $conn->query($buyer_query);
                    while ($buyer = $buyer_result->fetch_assoc()) {
                        echo "<option value='" . $buyer['id'] . "'>" . htmlspecialchars($buyer['nama_buyer']) . "</option>";
                    }
                    ?>
                </select>
            </div>
        </div>

        <!-- Master Color & Panel Table -->
        <div class="space-y-6">
            <?php
            // Check if manufacturing_date and expired_date columns exist in master_color
            $check_columns = "SHOW COLUMNS FROM master_color";
            $result = $conn->query($check_columns);
            $columns = [];
            while ($row = $result->fetch_assoc()) {
                $columns[] = $row['Field'];
            }
            
            $has_dates = in_array('manufacturing_date', $columns) && in_array('expired_date', $columns);
            
            // Check master_color_panel columns
            $check_panel_cols = "SHOW COLUMNS FROM master_color_panel";
            $panel_cols_result = $conn->query($check_panel_cols);
            $panel_columns = [];
            while ($row = $panel_cols_result->fetch_assoc()) {
                $panel_columns[] = $row['Field'];
            }
            $panel_has_dates = in_array('manufacturing_date', $panel_columns) && in_array('expired_date', $panel_columns);
            
            if ($has_dates) {
                $master_query = "SELECT mc.id, mc.code_color, mc.color_name, mc.manufacturing_date, mc.expired_date,
                                       b.nama_buyer, col.nama_collection,
                                       COUNT(mcp.id) as panel_count
                                FROM master_color mc
                                LEFT JOIN buyers b ON mc.buyer_id = b.id
                                LEFT JOIN collections col ON mc.collection_id = col.id
                                LEFT JOIN master_color_panel mcp ON mc.id = mcp.master_color_id
                                GROUP BY mc.id
                                ORDER BY mc.code_color";
            } else {
                $master_query = "SELECT mc.id, mc.code_color, mc.color_name, NULL as manufacturing_date, NULL as expired_date,
                                       b.nama_buyer, col.nama_collection,
                                       COUNT(mcp.id) as panel_count
                                FROM master_color mc
                                LEFT JOIN buyers b ON mc.buyer_id = b.id
                                LEFT JOIN collections col ON mc.collection_id = col.id
                                LEFT JOIN master_color_panel mcp ON mc.id = mcp.master_color_id
                                GROUP BY mc.id
                                ORDER BY mc.code_color";
            }
            
            $master_result = $conn->query($master_query);
            
            while ($master = $master_result->fetch_assoc()) {
                ?>
                <div class="border border-gray-200 rounded-lg overflow-hidden">
                    <!-- Master Color Header -->
                    <div class="bg-gradient-to-r from-purple-50 to-blue-50 p-4">
                        <div class="flex items-center justify-between">
                            <div class="flex items-center gap-4">
                                <div class="bg-purple-600 text-white px-4 py-2 rounded-lg font-bold text-lg">
                                    <?php echo htmlspecialchars($master['code_color']); ?>
                                </div>
                                <div>
                                    <h4 class="text-lg font-bold text-gray-800">
                                        <?php echo htmlspecialchars($master['color_name']); ?>
                                    </h4>
                                    <p class="text-sm text-gray-600">
                                        <span class="font-semibold"><?php echo htmlspecialchars($master['nama_buyer'] ?? 'No Buyer'); ?></span>
                                        <?php if ($master['nama_buyer']): ?> - <?php echo htmlspecialchars($master['nama_collection'] ?? 'No Collection'); ?><?php endif; ?>
                                    </p>
                                    <div class="flex items-center gap-4 text-xs text-gray-500 mt-1">
                                        <span>
                                            <i class="fas fa-calendar mr-1"></i>
                                            Manufacturing: <?php echo htmlspecialchars($master['manufacturing_date'] ?? '-'); ?>
                                        </span>
                                        <span>
                                            <i class="fas fa-clock mr-1"></i>
                                            Expired: <?php echo htmlspecialchars($master['expired_date'] ?? '-'); ?>
                                        </span>
                                        <span class="bg-blue-100 text-blue-800 px-2 py-1 rounded-full font-semibold">
                                            <i class="fas fa-clipboard-list mr-1"></i>
                                            <?php echo $master['panel_count']; ?> panel
                                        </span>
                                    </div>
                                </div>
                            </div>
                            <div class="flex gap-2">
                                <button onclick='editMaster(<?php echo $master["id"]; ?>, "<?php echo htmlspecialchars($master["code_color"]); ?>", "<?php echo ($master["manufacturing_date"] ?? ""); ?>", "<?php echo ($master["expired_date"] ?? ""); ?>", <?php echo $master["panel_count"]; ?>)' 
                                        class='bg-purple-600 hover:bg-purple-700 text-white px-4 py-2 rounded-lg text-sm font-semibold transition-colors'>
                                    <i class='fas fa-edit mr-1'></i>Edit Master
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Master Panels -->
                    <?php if ($master['panel_count'] > 0): ?>
                        <div class="bg-gray-50 p-4">
                            <h5 class="font-semibold text-gray-700 mb-3">
                                <i class="fas fa-clipboard-list text-orange-600 mr-2"></i>Master Panels
                            </h5>
                            
                            <?php
                            // Get panels for this master
                            if ($panel_has_dates) {
                                $panel_query = "SELECT mcp.id, mcp.panel_code, mcp.rack_number, mcp.manufacturing_date, mcp.expired_date
                                               FROM master_color_panel mcp
                                               WHERE mcp.master_color_id = " . $master['id'] . "
                                               ORDER BY mcp.panel_code";
                            } else {
                                $panel_query = "SELECT mcp.id, mcp.panel_code, mcp.rack_number, NULL as manufacturing_date, NULL as expired_date
                                               FROM master_color_panel mcp
                                               WHERE mcp.master_color_id = " . $master['id'] . "
                                               ORDER BY mcp.panel_code";
                            }
                            
                            $panel_result = $conn->query($panel_query);
                            ?>
                            
                            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                                <?php while ($panel = $panel_result->fetch_assoc()): ?>
                                    <div class="bg-white border border-gray-200 rounded-lg p-3">
                                        <div class="flex items-center justify-between">
                                            <div>
                                                <div class="font-semibold text-gray-800 text-sm">
                                                    <?php echo htmlspecialchars($panel['panel_code']); ?>
                                                </div>
                                                <div class="text-xs text-gray-500">
                                                    Rack: <?php echo str_pad($panel['rack_number'], 2, '0', STR_PAD_LEFT); ?>
                                                </div>
                                                <div class="text-xs text-gray-500 mt-1">
                                                    <div>Mfg: <?php echo htmlspecialchars($panel['manufacturing_date'] ?? '-'); ?></div>
                                                    <div>Exp: <?php echo htmlspecialchars($panel['expired_date'] ?? '-'); ?></div>
                                                </div>
                                            </div>
                                            <button onclick='editPanel(<?php echo $panel["id"]; ?>, "<?php echo htmlspecialchars($panel["panel_code"]); ?>", "<?php echo ($panel["manufacturing_date"] ?? ""); ?>", "<?php echo ($panel["expired_date"] ?? ""); ?>")' 
                                                    class='bg-orange-500 hover:bg-orange-600 text-white px-2 py-1 rounded text-xs transition-colors'>
                                                <i class='fas fa-edit'></i>
                                            </button>
                                        </div>
                                    </div>
                                <?php endwhile; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                <?php
            }
            ?>
        </div>
    </div>

    <?php else: ?>
    <!-- Copy Panel (Stock Items) Section -->
    <div class="bg-white rounded-lg shadow-md p-6">
        <h3 class="text-xl font-bold text-gray-800 mb-4">
            <i class="fas fa-boxes text-indigo-600 mr-2"></i>Copy Panel (Stock Items) - Edit Tanggal
        </h3>
        
        <!-- Search and Filter -->
        <div class="mb-4">
            <div class="flex flex-col md:flex-row gap-4">
                <input type="text" id="searchStock" placeholder="Cari kode barang, kode warna..." 
                       class="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                <select id="filterStatus" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                    <option value="">Semua Status</option>
                    <option value="available">Available</option>
                    <option value="borrowed">Borrowed</option>
                    <option value="expired">Expired</option>
                    <option value="renew">Renew</option>
                </select>
            </div>
        </div>

        <!-- Stock Items Table -->
        <div class="overflow-x-auto">
            <table class="min-w-full bg-white border border-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Code Barang</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Kode Warna</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Component</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Manufacturing Date</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Expired Date</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Aksi</th>
                    </tr>
                </thead>
                <tbody id="stockTableBody" class="bg-white divide-y divide-gray-200">
                    <?php
                    $stock_query = "SELECT csi.id, csi.code_barang_lemari5, csi.status, csi.manufacturing_date, csi.expired_date,
                                           mc.code_color, cc.component_name
                                    FROM color_stock_items csi
                                    LEFT JOIN master_color mc ON csi.master_color_id = mc.id
                                    LEFT JOIN color_components cc ON csi.component_id = cc.id
                                    ORDER BY csi.code_barang_lemari5";
                    $stock_result = $conn->query($stock_query);
                    
                    if ($stock_result && $stock_result->num_rows > 0) {
                        while ($stock = $stock_result->fetch_assoc()) {
                            $status_class = $stock['status'] === 'available' ? 'bg-green-100 text-green-800' : 
                                           ($stock['status'] === 'borrowed' ? 'bg-yellow-100 text-yellow-800' : 
                                           ($stock['status'] === 'expired' ? 'bg-red-100 text-red-800' : 'bg-blue-100 text-blue-800'));
                            
                            echo "<tr class='hover:bg-gray-50'>";
                            echo "<td class='px-4 py-3 text-sm font-medium text-gray-900'>" . htmlspecialchars($stock['code_barang_lemari5'] ?? '-') . "</td>";
                            echo "<td class='px-4 py-3 text-sm text-gray-500'>" . htmlspecialchars($stock['code_color'] ?? '-') . "</td>";
                            echo "<td class='px-4 py-3 text-sm text-gray-500'>" . htmlspecialchars($stock['component_name'] ?? '-') . "</td>";
                            echo "<td class='px-4 py-3 text-sm'><span class='px-2 py-1 rounded-full text-xs $status_class'>" . ucfirst($stock['status']) . "</span></td>";
                            echo "<td class='px-4 py-3 text-sm text-gray-500'>" . htmlspecialchars($stock['manufacturing_date'] ?? '-') . "</td>";
                            echo "<td class='px-4 py-3 text-sm text-gray-500'>" . htmlspecialchars($stock['expired_date'] ?? '-') . "</td>";
                            echo "<td class='px-4 py-3 text-sm'>";
                            echo "<button onclick='editStock(" . $stock['id'] . ", \"" . htmlspecialchars($stock['code_barang_lemari5'] ?? 'N/A') . "\", \"" . ($stock['manufacturing_date'] ?? '') . "\", \"" . ($stock['expired_date'] ?? '') . "\")' 
                                    class='bg-indigo-600 hover:bg-indigo-700 text-white px-3 py-1 rounded text-sm'>
                                    <i class='fas fa-edit mr-1'></i>Edit
                                  </button>";
                            echo "</td>";
                            echo "</tr>";
                        }
                    } else {
                        echo "<tr><td colspan='7' class='px-4 py-8 text-center text-gray-500'>Tidak ada data stock items</td></tr>";
                    }
                    ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endif; ?>
</div>

<!-- Edit Master Modal -->
<div id="editMasterModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <h3 class="text-lg font-medium text-gray-900 mb-4">
                <i class="fas fa-palette text-purple-600 mr-2"></i>Edit Tanggal Master Color
            </h3>
            <form method="POST">
                <input type="hidden" name="action" value="update_master">
                <input type="hidden" name="master_id" id="editMasterId">
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Kode Warna</label>
                    <input type="text" id="editMasterKode" readonly 
                           class="w-full px-3 py-2 border border-gray-300 rounded-md bg-gray-100">
                </div>
                
                <div id="panelWarning" class="mb-4 p-3 bg-yellow-50 border border-yellow-200 rounded-md hidden">
                    <div class="flex items-start">
                        <i class="fas fa-exclamation-triangle text-yellow-600 mr-2 mt-0.5"></i>
                        <div class="text-sm text-yellow-800">
                            <strong>Perhatian:</strong> <span id="panelCount"></span> Master Panel yang terkait juga akan ikut terupdate dengan tanggal yang sama.
                        </div>
                    </div>
                </div>
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Manufacturing Date</label>
                    <input type="date" name="manufacturing_date" id="editMasterManufacturingDate" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                </div>
                
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Expired Date</label>
                    <input type="date" name="expired_date" id="editMasterExpiredDate" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                </div>
                
                <div class="flex justify-end gap-3">
                    <button type="button" onclick="closeEditMasterModal()" 
                            class="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400">
                        Batal
                    </button>
                    <button type="submit" 
                            class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                        <i class="fas fa-save mr-1"></i>Simpan
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Panel Modal -->
<div id="editPanelModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <h3 class="text-lg font-medium text-gray-900 mb-4">
                <i class="fas fa-clipboard-list text-orange-600 mr-2"></i>Edit Tanggal Master Panel
            </h3>
            <form method="POST">
                <input type="hidden" name="action" value="update_panel">
                <input type="hidden" name="panel_id" id="editPanelId">
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Panel Code</label>
                    <input type="text" id="editPanelKode" readonly 
                           class="w-full px-3 py-2 border border-gray-300 rounded-md bg-gray-100">
                </div>
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Manufacturing Date</label>
                    <input type="date" name="manufacturing_date" id="editPanelManufacturingDate" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-orange-500 focus:border-orange-500">
                </div>
                
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Expired Date</label>
                    <input type="date" name="expired_date" id="editPanelExpiredDate" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-orange-500 focus:border-orange-500">
                </div>
                
                <div class="flex justify-end gap-3">
                    <button type="button" onclick="closeEditPanelModal()" 
                            class="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400">
                        Batal
                    </button>
                    <button type="submit" 
                            class="px-4 py-2 bg-orange-600 text-white rounded-md hover:bg-orange-700">
                        <i class="fas fa-save mr-1"></i>Simpan
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Stock Modal -->
<div id="editStockModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <h3 class="text-lg font-medium text-gray-900 mb-4">
                <i class="fas fa-boxes text-indigo-600 mr-2"></i>Edit Tanggal Copy Panel
            </h3>
            <form method="POST">
                <input type="hidden" name="action" value="update_stock">
                <input type="hidden" name="stock_id" id="editStockId">
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Kode Barang</label>
                    <input type="text" id="editStockKode" readonly 
                           class="w-full px-3 py-2 border border-gray-300 rounded-md bg-gray-100">
                </div>
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Manufacturing Date</label>
                    <input type="date" name="manufacturing_date" id="editStockManufacturingDate" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500">
                </div>
                
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Expired Date</label>
                    <input type="date" name="expired_date" id="editStockExpiredDate" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500">
                </div>
                
                <div class="flex justify-end gap-3">
                    <button type="button" onclick="closeEditStockModal()" 
                            class="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400">
                        Batal
                    </button>
                    <button type="submit" 
                            class="px-4 py-2 bg-indigo-600 text-white rounded-md hover:bg-indigo-700">
                        <i class="fas fa-save mr-1"></i>Simpan
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Edit Master Functions
function editMaster(id, kode, manufacturingDate, expiredDate, panelCount) {
    document.getElementById('editMasterId').value = id;
    document.getElementById('editMasterKode').value = kode;
    document.getElementById('editMasterManufacturingDate').value = manufacturingDate;
    document.getElementById('editMasterExpiredDate').value = expiredDate;
    
    // Show/hide panel warning
    const panelWarning = document.getElementById('panelWarning');
    const panelCountSpan = document.getElementById('panelCount');
    
    if (panelCount > 0) {
        panelCountSpan.textContent = panelCount;
        panelWarning.classList.remove('hidden');
    } else {
        panelWarning.classList.add('hidden');
    }
    
    document.getElementById('editMasterModal').classList.remove('hidden');
}

function closeEditMasterModal() {
    document.getElementById('editMasterModal').classList.add('hidden');
}

// Edit Panel Functions
function editPanel(id, kode, manufacturingDate, expiredDate) {
    document.getElementById('editPanelId').value = id;
    document.getElementById('editPanelKode').value = kode;
    document.getElementById('editPanelManufacturingDate').value = manufacturingDate;
    document.getElementById('editPanelExpiredDate').value = expiredDate;
    document.getElementById('editPanelModal').classList.remove('hidden');
}

function closeEditPanelModal() {
    document.getElementById('editPanelModal').classList.add('hidden');
}

// Edit Stock Functions
function editStock(id, kode, manufacturingDate, expiredDate) {
    document.getElementById('editStockId').value = id;
    document.getElementById('editStockKode').value = kode;
    document.getElementById('editStockManufacturingDate').value = manufacturingDate;
    document.getElementById('editStockExpiredDate').value = expiredDate;
    document.getElementById('editStockModal').classList.remove('hidden');
}

function closeEditStockModal() {
    document.getElementById('editStockModal').classList.add('hidden');
}

// Search functionality
document.getElementById('searchMaster')?.addEventListener('keyup', function() {
    const searchTerm = this.value.toLowerCase();
    const containers = document.querySelectorAll('.space-y-6 > div');
    
    containers.forEach(container => {
        const text = container.textContent.toLowerCase();
        container.style.display = text.includes(searchTerm) ? '' : 'none';
    });
});

document.getElementById('searchStock')?.addEventListener('keyup', function() {
    const searchTerm = this.value.toLowerCase();
    const rows = document.querySelectorAll('#stockTableBody tr');
    
    rows.forEach(row => {
        const text = row.textContent.toLowerCase();
        row.style.display = text.includes(searchTerm) ? '' : 'none';
    });
});

// Filter functionality
document.getElementById('filterStatus')?.addEventListener('change', function() {
    const status = this.value.toLowerCase();
    const rows = document.querySelectorAll('#stockTableBody tr');
    
    rows.forEach(row => {
        if (status === '') {
            row.style.display = '';
        } else {
            const statusCell = row.querySelector('td:nth-child(4)');
            const statusText = statusCell ? statusCell.textContent.toLowerCase() : '';
            row.style.display = statusText.includes(status) ? '' : 'none';
        }
    });
});

// Close modals when clicking outside
window.onclick = function(event) {
    const masterModal = document.getElementById('editMasterModal');
    const panelModal = document.getElementById('editPanelModal');
    const stockModal = document.getElementById('editStockModal');
    
    if (event.target === masterModal) {
        closeEditMasterModal();
    }
    if (event.target === panelModal) {
        closeEditPanelModal();
    }
    if (event.target === stockModal) {
        closeEditStockModal();
    }
}

// ESC key to close modals
document.addEventListener('keydown', function(event) {
    if (event.key === 'Escape') {
        closeEditMasterModal();
        closeEditPanelModal();
        closeEditStockModal();
    }
});
</script>

<?php
// Include footer
include '../template_it/footer.php';
?>