<?php
// history_archive.php - Halaman untuk melihat transaksi yang sudah selesai dan diarsipkan
session_start();
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'borrower') {
    header("Location: ../login.php");
    exit();
}

$page_title = "History Archive (Completed)";
include '../template_borrower/header.php';

$borrower_id = $_SESSION['user_id'];

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

// Search filter
$search = isset($_GET['search']) ? trim($_GET['search']) : '';

// Count total records
$count_query = "SELECT COUNT(*) as total FROM peminjaman_selesai WHERE borrower_id = ?";
$count_params = [$borrower_id];
$count_types = "i";

if ($search) {
    $count_query .= " AND (transaction_number LIKE ? OR borrower_name LIKE ?)";
    $search_param = "%{$search}%";
    $count_params[] = $search_param;
    $count_params[] = $search_param;
    $count_types .= "ss";
}

$count_stmt = $conn->prepare($count_query);
$count_stmt->bind_param($count_types, ...$count_params);
$count_stmt->execute();
$total_records = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_records / $limit);

// Get data
$query = "SELECT * FROM peminjaman_selesai WHERE borrower_id = ?";
$params = [$borrower_id];
$types = "i";

if ($search) {
    $query .= " AND (transaction_number LIKE ? OR borrower_name LIKE ?)";
    $params[] = $search_param;
    $params[] = $search_param;
    $types .= "ss";
}

$query .= " ORDER BY archived_at DESC LIMIT ? OFFSET ?";
$params[] = $limit;
$params[] = $offset;
$types .= "ii";

$stmt = $conn->prepare($query);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
?>

<div class="w-full max-w-7xl mx-auto px-2 sm:px-4">
    <!-- Header Section -->
    <div class="bg-white rounded-lg shadow-md p-4 sm:p-6 mb-4 sm:mb-6">
        <div class="flex items-center mb-2">
            <i class="fas fa-archive text-blue-500 text-2xl mr-3"></i>
            <h2 class="text-xl sm:text-2xl font-bold text-gray-800">History Archive (Completed)</h2>
        </div>
        <p class="text-sm text-gray-600">Riwayat peminjaman yang sudah selesai dan item telah dihapus dari stock</p>
    </div>

    <!-- Search Section -->
    <div class="bg-white rounded-lg shadow-md p-4 sm:p-6 mb-4 sm:mb-6">
        <form method="GET" action="" class="flex gap-2">
            <input type="text" 
                   name="search" 
                   value="<?php echo htmlspecialchars($search); ?>"
                   placeholder="Cari nomor transaksi atau nama..." 
                   class="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
            <button type="submit" 
                    class="px-6 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors">
                <i class="fas fa-search mr-2"></i>Cari
            </button>
            <?php if ($search): ?>
            <a href="history_archive.php" 
               class="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600 transition-colors">
                <i class="fas fa-times"></i>
            </a>
            <?php endif; ?>
        </form>
    </div>

    <!-- Statistics -->
    <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
        <?php
        $stats_query = "SELECT 
                          COUNT(*) as total_archived,
                          SUM(total_items) as total_items_deleted,
                          AVG(total_days_borrowed) as avg_days
                        FROM peminjaman_selesai 
                        WHERE borrower_id = ?";
        $stats_stmt = $conn->prepare($stats_query);
        $stats_stmt->bind_param("i", $borrower_id);
        $stats_stmt->execute();
        $stats = $stats_stmt->get_result()->fetch_assoc();
        ?>
        
        <div class="bg-white rounded-lg shadow-md p-4">
            <div class="flex items-center">
                <div class="bg-blue-100 p-3 rounded-full">
                    <i class="fas fa-archive text-blue-600 text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-gray-600 text-sm">Total Archived</p>
                    <p class="text-2xl font-bold text-gray-800"><?php echo $stats['total_archived'] ?? 0; ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-md p-4">
            <div class="flex items-center">
                <div class="bg-red-100 p-3 rounded-full">
                    <i class="fas fa-trash-alt text-red-600 text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-gray-600 text-sm">Items Deleted</p>
                    <p class="text-2xl font-bold text-gray-800"><?php echo $stats['total_items_deleted'] ?? 0; ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-md p-4">
            <div class="flex items-center">
                <div class="bg-green-100 p-3 rounded-full">
                    <i class="fas fa-calendar-check text-green-600 text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-gray-600 text-sm">Avg. Duration</p>
                    <p class="text-2xl font-bold text-gray-800"><?php echo round($stats['avg_days'] ?? 0); ?> days</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Archive Table -->
    <div class="bg-white rounded-lg shadow-md overflow-hidden">
        <div class="p-4 sm:p-6 border-b border-gray-200">
            <h3 class="text-lg font-semibold text-gray-800">
                <i class="fas fa-table mr-2"></i>Data Archive
            </h3>
        </div>

        <?php if ($result->num_rows > 0): ?>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">No. Transaksi</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Items</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Tanggal</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Durasi</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Archived</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Aksi</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php while ($row = $result->fetch_assoc()): ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-4 py-4 whitespace-nowrap">
                            <div class="text-sm font-medium text-gray-900">
                                <?php echo htmlspecialchars($row['transaction_number']); ?>
                            </div>
                            <div class="text-xs text-gray-500">
                                ID: <?php echo $row['original_transaction_id']; ?>
                            </div>
                        </td>
                        <td class="px-4 py-4">
                            <div class="text-sm font-semibold text-blue-600">
                                <?php echo $row['total_items']; ?> item(s) deleted
                            </div>
                        </td>
                        <td class="px-4 py-4 whitespace-nowrap">
                            <div class="text-sm text-gray-900">
                                Pinjam: <?php echo date('d M Y', strtotime($row['transaction_date'])); ?>
                            </div>
                            <div class="text-sm text-gray-500">
                                Kembali: <?php echo date('d M Y', strtotime($row['actual_return_date'])); ?>
                            </div>
                        </td>
                        <td class="px-4 py-4 whitespace-nowrap">
                            <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full bg-green-100 text-green-800">
                                <?php echo $row['total_days_borrowed']; ?> hari
                            </span>
                        </td>
                        <td class="px-4 py-4 whitespace-nowrap text-sm text-gray-500">
                            <?php echo date('d M Y H:i', strtotime($row['archived_at'])); ?>
                        </td>
                        <td class="px-4 py-4 whitespace-nowrap text-sm">
                            <button onclick="viewArchiveDetail(<?php echo $row['id']; ?>)" 
                                    class="bg-blue-500 hover:bg-blue-600 text-white px-3 py-1 rounded text-xs">
                                <i class="fas fa-eye mr-1"></i>Detail
                            </button>
                        </td>
                    </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
        <div class="bg-white px-4 py-3 border-t border-gray-200 sm:px-6">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm text-gray-700">
                        Showing <span class="font-medium"><?php echo $offset + 1; ?></span> to 
                        <span class="font-medium"><?php echo min($offset + $limit, $total_records); ?></span> of 
                        <span class="font-medium"><?php echo $total_records; ?></span> results
                    </p>
                </div>
                <div>
                    <nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px">
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <a href="?page=<?php echo $i; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?>" 
                           class="<?php echo $i == $page ? 'bg-blue-50 border-blue-500 text-blue-600' : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50'; ?> 
                                  relative inline-flex items-center px-4 py-2 border text-sm font-medium">
                            <?php echo $i; ?>
                        </a>
                        <?php endfor; ?>
                    </nav>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php else: ?>
        <div class="text-center py-12">
            <i class="fas fa-archive text-gray-300 text-6xl mb-4"></i>
            <p class="text-gray-500 text-lg mb-2">Belum Ada Archive</p>
            <p class="text-gray-400 text-sm">Belum ada transaksi yang diarsipkan</p>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Detail Modal -->
<div id="archiveDetailModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 max-w-5xl shadow-lg rounded-md bg-white">
        <div class="flex items-center justify-between mb-4 border-b pb-3">
            <h3 class="text-xl font-bold text-gray-800">
                <i class="fas fa-info-circle text-blue-600 mr-2"></i>
                Detail Archive
            </h3>
            <button onclick="closeArchiveModal()" class="text-gray-400 hover:text-gray-600">
                <i class="fas fa-times text-2xl"></i>
            </button>
        </div>
        
        <div id="archiveDetailContent">
            <!-- Content will be loaded here -->
        </div>
    </div>
</div>

<script>
function viewArchiveDetail(archiveId) {
    document.getElementById('archiveDetailModal').classList.remove('hidden');
    document.getElementById('archiveDetailContent').innerHTML = '<div class="text-center py-4"><i class="fas fa-spinner fa-spin text-2xl text-gray-400"></i></div>';
    
    fetch(`get_archive_detail.php?id=${archiveId}`)
        .then(response => response.text())
        .then(data => {
            document.getElementById('archiveDetailContent').innerHTML = data;
        })
        .catch(error => {
            document.getElementById('archiveDetailContent').innerHTML = '<div class="text-center py-4 text-red-500">Error loading detail</div>';
        });
}

function closeArchiveModal() {
    document.getElementById('archiveDetailModal').classList.add('hidden');
}

window.onclick = function(event) {
    const modal = document.getElementById('archiveDetailModal');
    if (event.target === modal) {
        closeArchiveModal();
    }
}
</script>

<?php include '../template_borrower/footer.php'; ?>