<?php

namespace App\Http\Controllers;

use App\Models\ItemRequest;
use App\Models\ApprovalFlowTemplate;
use App\Models\User;
use App\Models\Company;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;

class DashboardController extends Controller
{
    public function __construct(
        private \App\Services\BusinessCentralService $bc,
        private \App\Services\InventoryBC $inv
    ) {}

    protected function activeCompanyId(): ?int
    {
        $key = session('current_company_name') ?? data_get(session('user'), 'role');

        $map = [
            'Regent' => 'CI',   
            'HIN'    => 'TBI',  
        ];

        $code = $map[$key] ?? null;

        if (!$code) {
            return null;
        }

        return Company::where('code', $code)->value('id');
    }


    protected function resolveCurrentUser(): User
    {
        $sess  = session('user') ?? [];
        $email = $sess['email'] ?? null;
        $name  = $sess['name'] ?? ($sess['displayName'] ?? 'MS User');
        abort_unless($email, 401, 'Not authenticated');

        return User::firstOrCreate(['email' => $email], ['name' => $name]);
    }

    public function index(Request $request)
    {
        set_time_limit(300);
        $me = $this->resolveCurrentUser();
        $cid = $this->activeCompanyId();
        $inboxQuery = ItemRequest::query()
            ->where('status', 'InReview')
            ->whereExists(function ($q) {
                $q->from('approval_instances as ai')
                  ->whereColumn('ai.item_request_id','item_requests.id')
                  ->where('ai.status','InProgress');
            });

        if (!($me->role?->code === 'ADMIN')) {
            $inboxQuery->whereExists(function ($q) use ($me) {
                $q->from('approval_step_templates as ast')
                  ->whereColumn('ast.flow_template_id','item_requests.flow_template_id')
                  ->whereColumn('ast.step_no','item_requests.current_step_no')
                  ->where(function ($qq) use ($me) {
                      $qq->where('ast.approver_user_id', $me->id)
                         ->orWhere('ast.approver_role_id', $me->role_id);
                  });
            });
        }
        $kpi_inbox        = (clone $inboxQuery)->count();
        $kpi_my_open      = ItemRequest::where('requester_id',$me->id)->when($cid, fn($q)=>$q->where('company_id', $cid))->whereIn('status',['Draft','InReview'])->count();
        $kpi_approved_30d = ItemRequest::where('status','Approved')->when($cid, fn($q)=>$q->where('company_id', $cid))->where('updated_at','>=',now()->subDays(30))->count();
        $kpi_rejected_30d = ItemRequest::where('status','Rejected')->when($cid, fn($q)=>$q->where('company_id', $cid))->where('updated_at','>=',now()->subDays(30))->count();
        $poData  = $this->bc->getPoSuggestions($me->name ?? $me->email);
        $poItems = collect($poData['items'] ?? []);
        dd('2');
        $poVendors   = $poItems->pluck('vendor_name')->filter()->unique()->sort()->values();
        $poLocations = $poItems->pluck('location_code')->filter()->unique()->sort()->values();
        $poCounts = [
            'need_order'       => $poItems->where('status','Need Order')->count(),
            'need_order_for_to'=> $poItems->where('status','Need Order For TO')->count(),
            'follow_up'        => $poItems->where('status','Follow Up PO')->count(),
            'for_stock'        => $poItems->where('status','Order For Stock')->count(),
            'no_need'          => $poItems->where('status','No Need Order')->count(),
        ];
        $poRows = $poItems->map(function ($r) {
            return [
                'item_no'       => $r['item_no'] ?? null,
                'description'   => $r['description'] ?? null,
                'vendor_name'   => $r['vendor_name'] ?? null,
                'location_code' => $r['location_code'] ?? null,
                'status'        => $r['status'] ?? null,
                'qty_to_order'  => (float)($r['qty_to_order'] ?? 0),
            ];
        });
        $poTop = $poItems
            ->sortByDesc(fn($r) => (float)($r['qty_to_order'] ?? 0))
            ->take(8)
            ->values();
        dd(3);
        $rawSku = $this->inv->getItemLedgerAndSkuMappingAsync();
        $top    = $rawSku instanceof Collection ? $rawSku->all() : (array) $rawSku;
        if (isset($top['items']) && is_array($top['items'])) {
            $skuRows = collect($top['items']);
        } else {
            $skuRows = collect($top)
                ->flatMap(fn($chunk) => is_array($chunk) && array_is_list($chunk) ? $chunk : (is_array($chunk) ? [$chunk] : []))
                ->values();
        }

        $skuModified30d = $skuRows->filter(function ($r) {
            $d = $r['LastModifiedAt'] ?? null;
            if (!$d) return false;
            try {
                return Carbon::createFromFormat('d F Y', $d)->gte(now()->subDays(30));
            } catch (\Throwable $e) {
                return false;
            }
        })->count();

        $skuCounts = [
            'rows'         => $skuRows->count(),
            'modified_30d' => $skuModified30d,
        ];

        $skuTop = $skuRows
            ->sortByDesc(fn($r) => (int)($r['Consumption / 2 months'] ?? 0))
            ->take(8)
            ->values();
        return view('dashboard.index', compact(
            'kpi_inbox','kpi_my_open','kpi_approved_30d','kpi_rejected_30d',
            'poCounts','poTop','skuCounts','skuTop',
            'poRows','poVendors','poLocations'  
        ));
    }
}
