<?php

namespace App\Http\Controllers;

use App\Models\{ItemRequest, ItemRequestLine};
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\{ApprovalFlowTemplate, Company, User};
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Arr;

class ItemRequestController extends Controller
{
    public function create()
    {
        $me = $this->resolveCurrentUser();
        abort_unless($me->canCreateItemRequests(), 403, 'You are not allowed to create item requests.');
        $flows = ApprovalFlowTemplate::where('object_type','NewItem')
            ->where('is_active',1)->orderByDesc('version')->get();
        
        $companies = $me->isAdmin()
            ? Company::orderBy('name')->get()
            : Company::whereIn('id', $me->allowedCompanyIds())->orderBy('name')->get();

        $defaultCompanyId = $me->defaultCompanyId() ?? optional($companies->first())->id;

        return view('approvals.new', compact('flows','companies','defaultCompanyId'));
    }

    protected function resolveCurrentUser(): User
    {
        $sess  = session('user') ?? [];
        $email = $sess['email'] ?? (auth()->check() ? auth()->user()->email : null);
        $name  = $sess['name'] ?? ($sess['displayName'] ?? (auth()->check() ? (auth()->user()->name ?? 'MS User') : 'MS User'));

        abort_unless($email, 401, 'Not authenticated');

        return User::firstOrCreate(['email' => $email], ['name' => $name]);
    }

    public function store(Request $r)
    {
        $me = $this->resolveCurrentUser();

        $data = $r->validate([
            'company_id'        => 'required|integer|exists:companies,id',
            'flow_template_id'  => 'required|exists:approval_flow_templates,id',
            'posting_date'      => 'nullable|date',
            'remarks'           => 'nullable|string',
            'vendor_name'       => 'nullable|string|max:255',
            'lines'                           => 'required|array|min:1',
            'lines.*.article_name'            => 'required|string',
            'lines.*.type'                    => 'required|in:Inventory,Service,NonInventory',
            'lines.*.base_unit_code'          => 'required|string',
            'lines.*.gl_account_no'           => 'nullable|string',
            'lines.*.inventory_posting_group_code' => 'nullable|string',
            'lines.*.gen_prod_posting_group_code'  => 'nullable|string',
            'lines.*.l1'                      => 'nullable|string',
            'lines.*.l2'                      => 'nullable|string',
            'lines.*.article_no'              => 'nullable|string',
            'lines.*.line_remarks'            => 'nullable|string',
            'lines.*.vendor_quotes'           => 'nullable|array',
            'lines.*.vendor_quotes.*.name'    => 'nullable|string|max:120',
            'lines.*.vendor_quotes.*.price'   => 'nullable|numeric|min:0',
            'lines.*.vendor_quotes.*.currency'=> 'nullable|string|max:10',
            'lines.*.vendor_selected'         => 'nullable|integer',
            'attachments'   => ['array'],
            'attachments.*' => ['file','max:5120'],
        ]);

        foreach ($data['lines'] as $i => $ln) {
            if ($ln['type'] === 'Inventory' && !empty($ln['gl_account_no'])) {
                return back()->withErrors(["lines.$i.gl_account_no" => 'Inventory line cannot have GL account'])->withInput();
            }
            if (in_array($ln['type'], ['Service','NonInventory'], true) && empty($ln['gl_account_no'])) {
                return back()->withErrors(["lines.$i.gl_account_no" => 'Service/NonInventory requires GL account'])->withInput();
            }
        }
        $files = $r->hasFile('attachments') ? $r->file('attachments') : [];

        $req = null;
        DB::transaction(function () use ($me, $data, &$req) {
            $req = ItemRequest::create([
                'company_id'       => (int) $data['company_id'],
                'requester_id'     => $me->id,
                'flow_template_id' => $data['flow_template_id'],
                'posting_date'     => $data['posting_date'] ?? null,
                'remarks'          => $data['remarks'] ?? null,
                'vendor_name'      => $data['vendor_name'] ?? null,
                'status'           => 'Draft',
                'current_step_no'  => 0,
            ]);


            $payloads = [];
            foreach ($data['lines'] as $ln) {
                $quotesRaw   = $ln['vendor_quotes'] ?? [];
                $selectedIdx = $ln['vendor_selected'] ?? null;

                $quotes = [];
                foreach ($quotesRaw as $idx => $q) {
                    if (blank($q['name'] ?? null) && blank($q['price'] ?? null)) continue;
                    $quotes[] = [
                        'name'     => trim((string)($q['name'] ?? '')),
                        'price'    => is_numeric($q['price'] ?? null) ? (float)$q['price'] : null,
                        'currency' => $q['currency'] ?? 'IDR',
                        'selected' => (string)$idx === (string)$selectedIdx,
                    ];
                }

                $clean = Arr::only($ln, [
                    'article_name','type','base_unit_code','gl_account_no',
                    'inventory_posting_group_code','gen_prod_posting_group_code',
                    'l1','l2','article_no','line_remarks',
                ]);
                $clean['vendor_quotes']   = $quotes ?: null;
                $clean['item_request_id'] = $req->id;

                $payloads[] = $clean;
            }

            $req->lines()->createMany($payloads);
        });

        DB::afterCommit(function () use ($req, $files) {
            foreach ($files as $file) {
                $path = $file->store("requests/{$req->id}", 'public');
                $req->attachments()->create([
                    'uploader_user_id' => auth()->id(),
                    'original_name'    => $file->getClientOriginalName(),
                    'path'             => $path,
                    'mime'             => $file->getClientMimeType(),
                    'size'             => $file->getSize(),
                ]);
            }
        });

        return redirect()->route('approvals.show', $req->id)->with('success', 'Draft created');
    }

}