<?php

namespace App\Support;

use App\Models\ItemRequestComment;
use Illuminate\Support\Facades\Mail;

class CommentNotifier
{

    public static function send(ItemRequestComment $comment): void
    {
        $itemRequest = $comment->itemRequest;

        if ($itemRequest && $itemRequest->user && $itemRequest->user->email) {
            $subject = 'New Comment on Your Request';
            $to = $itemRequest->user->email;

            $body = self::buildHtmlMessage($comment);

            Mail::html($body, function ($message) use ($to, $subject) {
                $message->to($to)->subject($subject);
            });
        }
    }

    protected static function buildTextMessage(ItemRequestComment $comment): string
    {
        $ownerName = $comment->itemRequest->user->name;
        $commenterName = $comment->user->name;
        $commentBody = $comment->body;
        $link = route('approvals.show', $comment->itemRequest->id);

        return <<<EOT
Hi {$ownerName},

{$commenterName} commented on your request:

"{$commentBody}"

You can view the request here: {$link}

Thanks,
Your App Team
EOT;
    }

    protected static function buildHtmlMessage(ItemRequestComment $comment): string
    {
        $ownerName = e($comment->itemRequest->user->name);
        $commenterName = e($comment->user->name);
        $commentBody = nl2br(e($comment->body));
        $link = route('approvals.show', $comment->itemRequest->id);

        return <<<HTML
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.5; color: #333; }
        blockquote { border-left: 4px solid #ccc; margin: 10px 0; padding-left: 10px; color: #555; }
        .btn { display: inline-block; padding: 10px 15px; background: #4CAF50; color: white; text-decoration: none; border-radius: 5px; }
    </style>
</head>
<body>
    <h2>Hi {$ownerName},</h2>
    <p><strong>{$commenterName}</strong> commented on your request:</p>

    <blockquote>
        {$commentBody}
    </blockquote>

    <p>
        <a href="{$link}" class="btn">View Request</a>
    </p>

    <p>Thanks,<br>Your App Team</p>
</body>
</html>
HTML;
    }
}
