<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Quotation;
use App\Models\QuotationDetail;
use App\Models\Buyer;
use App\Models\Product;
use App\Models\Material;
use App\Models\Veneer;
use App\Models\OtherComponent;
use App\Models\ComponentDimension;
use App\Helpers\QuotationHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class QuotationController extends Controller
{
    /**
     * Display quotation history
     */
    public function index()
    {
        $quotations = Quotation::with(['buyer', 'product', 'user'])
            ->where('user_id', Auth::id())
            ->latest()
            ->paginate(10);
            
        return view('admin.quotations.index', compact('quotations'));
    }

    /**
     * Show the form for creating a new quotation - Step 1: Select Buyer
     */
    public function create()
    {
        $buyers = Buyer::where('is_active', true)->get();
        return view('admin.quotations.step1-buyer', compact('buyers'));
    }

    /**
     * Step 2: Select Product
     */
    public function selectProduct(Request $request)
    {
        $request->validate([
            'buyer_id' => 'required|exists:buyers,id',
        ]);

        $buyer = Buyer::findOrFail($request->buyer_id);
        $products = Product::where('is_active', true)->get();
        
        return view('admin.quotations.step2-product', compact('buyer', 'products'));
    }

    /**
     * Step 3: Build Components (TOP PANEL)
     */
    public function buildComponents(Request $request)
    {
        $request->validate([
            'buyer_id' => 'required|exists:buyers,id',
            'product_id' => 'required|exists:products,id',
        ]);

        $buyer = Buyer::findOrFail($request->buyer_id);
        $product = Product::findOrFail($request->product_id);
        
        // Get all materials for component selection
        $panels = Material::where('category', 'panel')->where('is_active', true)->get();
        $solids = Material::where('category', 'solid')->where('is_active', true)->get();
        $mdfs = Material::where('category', 'mdf')->where('is_active', true)->get();
        $veneers = Veneer::where('is_active', true)->get();
        $otherComponents = OtherComponent::where('is_active', true)->get();
        $dimensions = ComponentDimension::where('is_active', true)->get();
        
        return view('admin.quotations.step3-components', compact(
            'buyer', 
            'product', 
            'panels', 
            'solids', 
            'mdfs', 
            'veneers', 
            'otherComponents',
            'dimensions'
        ));
    }

    /**
     * Calculate component price via AJAX
     */
    public function calculateComponent(Request $request)
    {
        $componentType = $request->component_type;
        $data = $request->all();
        
        $result = [
            'success' => true,
            'component_name' => $componentType,
            'subtotal' => 0,
            'details' => []
        ];

        switch ($componentType) {
            case 'patching_panel_1':
                $result = $this->calculatePatchingPanel1($data);
                break;
            case 'patching_solid_2':
                $result = $this->calculatePatchingSolid2($data);
                break;
            case 'patching_solid_3':
                $result = $this->calculatePatchingSolid3($data);
                break;
            case 'crossgrain':
                $result = $this->calculateCrossgrain($data);
                break;
            case 'veneer_face':
                $result = $this->calculateVeneerFace($data);
                break;
            case 'veneer_back':
                $result = $this->calculateVeneerBack($data);
                break;
            case 'other_component':
                $result = $this->calculateOtherComponent($data);
                break;
        }

        return response()->json($result);
    }

    /**
     * Calculate Patching Panel 1
     */
    private function calculatePatchingPanel1($data)
    {
        $lebar = $data['lebar'];
        $panjang = $data['panjang'];
        $materialId = $data['material_id'];
        $thickness = $data['thickness'];
        
        // Calculate M2
        $m2 = ($lebar * $panjang) / 1000000;
        
        // Get material price
        $material = Material::find($materialId);
        $pricePerM2 = $material->price_per_m2;
        
        // Calculate subtotal
        $subtotal = $m2 * $pricePerM2;
        
        return [
            'success' => true,
            'component_name' => 'Patching Panel 1',
            'm2' => $m2,
            'price_per_m2' => $pricePerM2,
            'subtotal' => $subtotal,
            'formula' => "($lebar × $panjang) / 1,000,000 × $pricePerM2"
        ];
    }

    /**
     * Calculate Patching Solid 2 (menggunakan PANJANG)
     */
    private function calculatePatchingSolid2($data)
    {
        $tebal = $data['tebal'];
        $panjang = $data['panjang'];
        $qty = $data['qty'];
        $materialId = $data['material_id'];
        
        // Calculate M2
        $m2 = ($tebal * $panjang) / 1000000;
        
        // Get material price
        $material = Material::find($materialId);
        $pricePerM2 = $material->price_per_m2;
        
        // Calculate subtotal
        $subtotal = $m2 * $qty * $pricePerM2;
        
        return [
            'success' => true,
            'component_name' => 'Patching Solid 2',
            'm2' => $m2,
            'qty' => $qty,
            'price_per_m2' => $pricePerM2,
            'subtotal' => $subtotal,
            'formula' => "($tebal × $panjang) / 1,000,000 × $qty × $pricePerM2"
        ];
    }

    /**
     * Calculate Patching Solid 3 (menggunakan LEBAR)
     */
    private function calculatePatchingSolid3($data)
    {
        $tebal = $data['tebal'];
        $lebar = $data['lebar'];
        $qty = $data['qty'];
        $materialId = $data['material_id'];
        
        // Calculate M2
        $m2 = ($tebal * $lebar) / 1000000;
        
        // Get material price
        $material = Material::find($materialId);
        $pricePerM2 = $material->price_per_m2;
        
        // Calculate subtotal
        $subtotal = $m2 * $qty * $pricePerM2;
        
        return [
            'success' => true,
            'component_name' => 'Patching Solid 3',
            'm2' => $m2,
            'qty' => $qty,
            'price_per_m2' => $pricePerM2,
            'subtotal' => $subtotal,
            'formula' => "($tebal × $lebar) / 1,000,000 × $qty × $pricePerM2"
        ];
    }

    /**
     * Calculate Crossgrain
     */
    private function calculateCrossgrain($data)
    {
        $m2 = $data['m2_base']; // M2 dari Patching Panel 1
        $qty = $data['qty'];
        $materialId = $data['material_id'];
        
        // Get material price
        $material = Material::find($materialId);
        $pricePerM2 = $material->price_per_m2;
        
        // Calculate subtotal
        $subtotal = $m2 * $qty * $pricePerM2;
        
        return [
            'success' => true,
            'component_name' => 'Crossgrain',
            'm2' => $m2,
            'qty' => $qty,
            'price_per_m2' => $pricePerM2,
            'subtotal' => $subtotal,
            'formula' => "$m2 × $qty × $pricePerM2"
        ];
    }

    /**
     * Calculate Veneer Face
     */
    private function calculateVeneerFace($data)
    {
        $m2 = $data['m2_base']; // M2 dari Patching Panel 1
        $qty = $data['qty'];
        $veneerId = $data['veneer_id'];
        
        // Get veneer price
        $veneer = Veneer::find($veneerId);
        $pricePerM2 = $veneer->price_per_m2;
        
        // Calculate subtotal
        $subtotal = $m2 * $qty * $pricePerM2;
        
        return [
            'success' => true,
            'component_name' => 'Veneer Face',
            'm2' => $m2,
            'qty' => $qty,
            'price_per_m2' => $pricePerM2,
            'subtotal' => $subtotal,
            'formula' => "$m2 × $qty × $pricePerM2"
        ];
    }

    /**
     * Calculate Veneer Back
     */
    private function calculateVeneerBack($data)
    {
        $m2 = $data['m2_base']; // M2 dari Patching Panel 1
        $qty = $data['qty'];
        $veneerId = $data['veneer_id'];
        
        // Get veneer price
        $veneer = Veneer::find($veneerId);
        $pricePerM2 = $veneer->price_per_m2;
        
        // Calculate subtotal
        $subtotal = $m2 * $qty * $pricePerM2;
        
        return [
            'success' => true,
            'component_name' => 'Veneer Back',
            'm2' => $m2,
            'qty' => $qty,
            'price_per_m2' => $pricePerM2,
            'subtotal' => $subtotal,
            'formula' => "$m2 × $qty × $pricePerM2"
        ];
    }

    /**
     * Calculate Other Component (Cleat, Support Drawer, etc)
     */
    private function calculateOtherComponent($data)
    {
        $qty = $data['qty'];
        $componentId = $data['component_id'];
        
        // Get component price
        $component = OtherComponent::find($componentId);
        $pricePerUnit = $component->price_per_unit;
        
        // Calculate subtotal
        $subtotal = $qty * $pricePerUnit;
        
        return [
            'success' => true,
            'component_name' => $component->name,
            'qty' => $qty,
            'price_per_unit' => $pricePerUnit,
            'subtotal' => $subtotal,
            'formula' => "$qty × $pricePerUnit"
        ];
    }

    /**
     * Store quotation
     */
    public function store(Request $request)
{
    DB::beginTransaction();
    
    try {
        // Generate quotation number menggunakan helper
        $quotationNumber = QuotationHelper::generateQuotationNumber();
        
        // Create quotation
        $quotation = Quotation::create([
            'quotation_number' => $quotationNumber,
            'buyer_id' => $request->buyer_id,
            'product_id' => $request->product_id,
            'user_id' => Auth::id(),
            'product_code' => $request->product_code,
            'item_name' => $request->item_name,
            'total_price' => $request->total_price,
            'status' => 'draft',
            'notes' => $request->notes,
        ]);
        
        // Store quotation details
        $components = json_decode($request->components, true);
        
        foreach ($components as $component) {
            QuotationDetail::create([
                'quotation_id' => $quotation->id,
                'component_name' => $component['name'],
                'material_type' => $component['material_type'] ?? null,
                'length' => $component['length'] ?? null,
                'width' => $component['width'] ?? null,
                'thickness' => $component['thickness'] ?? null,
                'm2' => $component['m2'] ?? null,
                'qty' => $component['qty'],
                'price_per_unit' => $component['price_per_unit'],
                'subtotal' => $component['subtotal'],
                'note' => $component['note'] ?? null,
            ]);
        }
        
        DB::commit();
        
        return redirect()->route('admin.quotations.show', $quotation->id)
            ->with('success', 'Quotation berhasil dibuat!');
            
    } catch (\Exception $e) {
        DB::rollback();
        return back()->withErrors(['error' => 'Gagal membuat quotation: ' . $e->getMessage()]);
    }
}

    /**
     * Display quotation detail
     */
    public function show(Quotation $quotation)
    {
        // Make sure user can only see their own quotation
        if ($quotation->user_id !== Auth::id()) {
            abort(403, 'Unauthorized access');
        }
        
        $quotation->load(['buyer', 'product', 'details', 'user']);
        
        return view('admin.quotations.show', compact('quotation'));
    }

    /**
     * Delete quotation
     */
    public function destroy(Quotation $quotation)
    {
        // Make sure user can only delete their own quotation
        if ($quotation->user_id !== Auth::id()) {
            abort(403, 'Unauthorized access');
        }
        
        $quotation->delete();
        
        return redirect()->route('admin.quotations.index')
            ->with('success', 'Quotation berhasil dihapus!');
    }
}