<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use App\Models\{User, Role, Company};

class AdminController extends Controller
{
    public function manageUsers()
    {
        // Only allow admins
        if (!auth()->user()->isAdmin()) {
            abort(403, 'Access denied');
        }

        $users = User::orderBy('created_at', 'desc')->get();

        return view('admin.users', compact('users'));
    }

    public function createUser()
    {
        $roles = Role::orderBy('code')->get();
        return view('admin.create-user', compact('roles'));
    }

    // Store new user
    public function storeUser(Request $request)
    {
        $request->validate([
            'name'      => 'required|string|max:255',
            'email'     => 'required|email|unique:users,email',
            'role_id'   => 'required|exists:roles,id', // make sure the role exists
        ]);

        User::create([
            'name'      => $request->name,
            'email'     => $request->email,
            'role_id'   => $request->role_id, // save role id
            'active'    => true,
        ]);

        return redirect()->route('admin.users')->with('success', 'User created successfully!');
    }

    // Show Edit User page
    public function editUser(User $user)
    {
        $roles = Role::orderBy('code')->get();
        return view('admin.edit-user', compact('user', 'roles'));
    }

    // Update user
    public function updateUser(Request $request, User $user)
    {
        $request->validate([
            'name'      => 'required|string|max:255',
            'email'     => 'required|email|unique:users,email,'.$user->id,
            'role_id'   => 'required|exists:roles,id'
        ]);

        $user->name    = $request->name;
        $user->email   = $request->email;
        $user->role_id = $request->role_id; // save role id

        if ($request->password) {
            $user->password = Hash::make($request->password);
        }

        $user->save();

        return redirect()->route('admin.users')->with('success', 'User updated successfully!');
    }


    // Delete user
    public function deleteUser(User $user)
    {
        $user->delete();

        return redirect()->route('admin.users')->with('success', 'User deleted successfully!');
    }
}
