<?php

namespace App\Services;

use Google\Client;
use Google\Service\Drive;
use Google\Service\Drive\DriveFile;

class GoogleDriveService
{
    protected Drive $drive;

    public function __construct()
    {
        $client = new Client();

        // ✅ OAuth credentials (NOT service account)
        $client->setClientId(config('services.google.client_id'));
        $client->setClientSecret(config('services.google.client_secret'));

        $client->setAccessType('offline');
        $client->addScope(Drive::DRIVE_FILE);

        $tokenPath = storage_path('app/google/oauth-token.json');

        if (!file_exists($tokenPath)) {
            throw new \Exception('OAuth token not found. Connect Google Drive first.');
        }

        $client->setAccessToken(
            json_decode(file_get_contents($tokenPath), true)
        );

        if ($client->isAccessTokenExpired()) {
            $client->fetchAccessTokenWithRefreshToken(
                $client->getRefreshToken()
            );

            file_put_contents(
                $tokenPath,
                json_encode($client->getAccessToken())
            );
        }

        $this->drive = new Drive($client);
    }

    public function uploadPdf(string $path, string $name, string $folderId): array
    {
        $fileMetadata = new DriveFile([
            'name' => $name,
            'parents' => [$folderId],
        ]);

        $file = $this->drive->files->create($fileMetadata, [
            'data' => file_get_contents($path),
            'mimeType' => 'application/pdf',
            'uploadType' => 'multipart',
            'fields' => 'id, webViewLink',
        ]);

        return [
            'id'   => $file->id,
            'link' => $file->webViewLink,
        ];
    }
}
