<?php

namespace App\Http\Controllers;

use App\Models\ApprovalHistory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Response;

class AdminController extends Controller
{
    /**
     * Admin Dashboard - Show all approved documents
     */
    public function dashboard(Request $request)
    {
        $query = ApprovalHistory::where('action', 'approved')
            ->orderBy('action_date', 'desc');

        // Filter by date range if provided
        if ($request->filled('start_date')) {
            $query->whereDate('action_date', '>=', $request->start_date);
        }

        if ($request->filled('end_date')) {
            $query->whereDate('action_date', '<=', $request->end_date);
        }

        // Filter by company if provided
        if ($request->filled('company')) {
            $query->where('company', $request->company);
        }

        // Filter by category if provided
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        $approvals = $query->paginate(50);

        // Get statistics
        $stats = [
            'total_approved' => ApprovalHistory::where('action', 'approved')->count(),
            'total_rejected' => ApprovalHistory::where('action', 'rejected')->count(),
            'today_approved' => ApprovalHistory::where('action', 'approved')
                ->whereDate('action_date', today())
                ->count(),
        ];

        return view('admin.dashboard', compact('approvals', 'stats'));
    }

    /**
     * Export approvals to CSV
     */
    public function exportApprovals(Request $request)
    {
        $approvals = ApprovalHistory::where('action', 'approved')
            ->orderBy('action_date', 'desc')
            ->get();

        $filename = 'approvals_' . date('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($approvals) {
            $file = fopen('php://output', 'w');
            
            // Header
            fputcsv($file, [
                'Document No',
                'Company',
                'Category',
                'Approver ID',
                'Approver Email',
                'Action Date',
                'Remarks'
            ]);

            // Data
            foreach ($approvals as $approval) {
                fputcsv($file, [
                    $approval->document_no,
                    $approval->company,
                    $approval->category,
                    $approval->approver_id,
                    $approval->approver_email,
                    $approval->action_date->format('Y-m-d H:i:s'),
                    $approval->remarks ?? '-',
                ]);
            }

            fclose($file);
        };

        return Response::stream($callback, 200, $headers);
    }
}