<?php

namespace App\Http\Controllers\Talenta;

use App\Http\Controllers\Controller;
use App\Services\MekariTalentaClient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class EmployeeController extends Controller
{
    public function __construct(private MekariTalentaClient $talenta) {}

    /* ===================== helpers ===================== */

    private function norm(string $s): string
    {
        $s = strtolower(trim($s));
        return preg_replace('/\s+/', ' ', $s);
    }

    /**
     * Tambahkan baris employee ke index nama (associative):
     * - key full name (first + last)
     * - key first_name saja
     * value: array list employee (bisa >1)
     */
    private function addToNameIndex(array &$index, array $it): void
    {
        $fn   = $this->norm((string) data_get($it, 'personal.first_name', ''));
        $ln   = $this->norm((string) data_get($it, 'personal.last_name', ''));
        $full = trim($this->norm(trim($fn . ' ' . $ln)));

        foreach (array_filter([$fn, $full]) as $key) {
            if ($key === '') continue;
            if (!isset($index[$key])) $index[$key] = [];
            $index[$key][] = $it;
        }
    }

    /**
     * Susun urutan halaman spiral: 1, last, 2, last-1, ...
     * Ambil maksimal $max halaman.
     */
    private function spiralPages(int $lastPage, int $max): array
    {
        $res = [];
        $l = 1; $r = $lastPage;
        while ($l <= $r && count($res) < $max) {
            $res[] = $l++;
            if ($l <= $r && count($res) < $max) $res[] = $r--;
        }
        return $res;
    }

    /* ===================== actions ===================== */

    public function index(Request $request)
    {
        $limit      = max(1, (int) $request->query('limit', 20));
        $page       = max(1, (int) $request->query('page', 1));

        // 🔎 hanya 3 kunci + filter status
        $userId     = trim((string) $request->query('user_id', ''));
        $employeeId = strtoupper(trim((string) $request->query('employee_id', '')));
        $firstName  = trim((string) $request->query('first_name', ''));
        $status     = trim((string) $request->query('status', '')); // Active / Inactive / ''

        // performa (bisa override via .env)
        $SCAN_MAX_PAGES = (int) config('services.talenta.scan_pages', 500); // naikin default 500
        $SCAN_CHUNK     = (int) config('services.talenta.scan_chunk', 12);  // paralel 12 halaman

        $notice = null;

        try {
            // 1) user_id -> langsung detail
            if ($userId !== '') {
                return redirect()->route('talenta.employees.show', ['user' => $userId]);
            }

            // Helper: ambil first page utk dapat last_page
            $firstPageJson = $this->talenta->get('/v2/talenta/v2/employee', ['limit'=>100, 'page'=>1]);
            $last  = (int) data_get($firstPageJson, 'data.pagination.last_page', 1);
            if ($last < 1) $last = 1;

            // 2) employee_id -> cari cepat (paralel batch), kalau ketemu -> redirect
            if ($employeeId !== '') {
                $cacheKey = "talenta:map:emp:{$employeeId}";
                if ($uid = Cache::get($cacheKey)) {
                    return redirect()->route('talenta.employees.show', ['user' => $uid]);
                }

                $order = $this->spiralPages($last, min($SCAN_MAX_PAGES, $last));

                for ($i = 0; $i < count($order); $i += $SCAN_CHUNK) {
                    $pages = array_slice($order, $i, $SCAN_CHUNK);
                    $pool  = $this->talenta->getPool('/v2/talenta/v2/employee', $pages, ['limit'=>100]);

                    foreach ($pool as $json) {
                        $rows = (array) data_get($json, 'data.employees', []);
                        foreach ($rows as $it) {
                            $empId = strtoupper((string) (
                                $it['employment']['employee_id'] ?? $it['employee_id'] ?? $it['employee_number'] ?? ''
                            ));
                            if ($empId === $employeeId) {
                                $uid = $it['user_id'] ?? null;
                                if ($uid) {
                                    Cache::put($cacheKey, $uid, now()->addHours(12));
                                    return redirect()->route('talenta.employees.show', ['user' => $uid]);
                                }
                            }
                        }
                    }
                }

                $notice = 'Employee ID tidak ditemukan pada batch yang dipindai.';
                return view('talenta.employees.index', [
                    'items'      => [],
                    'raw'        => $firstPageJson,
                    'limit'      => $limit,
                    'page'       => 1,
                    'status'     => $status,
                    'pagination' => ['current_page'=>1,'last_page'=>$last,'total'=>0],
                    'notice'     => $notice,
                ]);
            }

            // 3) first_name -> associative index (full name & first name) + fallback "contains"
            if ($firstName !== '') {
                $needle      = $this->norm($firstName);         // "hambali" atau "hambali fitrianto"
                $needleFirst = $this->norm(strtok($needle, ' '));

                // cache nama→user (biar next time instan)
                $cacheKey = 'talenta:index:first:' . md5($needle);
                if ($uid = Cache::get($cacheKey)) {
                    return redirect()->route('talenta.employees.show', ['user' => $uid]);
                }

                $index      = []; // name_key => [employees]
                $candsExact = [];
                $candsAny   = [];

                $order = $this->spiralPages($last, min($SCAN_MAX_PAGES, $last));

                for ($i = 0; $i < count($order); $i += $SCAN_CHUNK) {
                    $pages = array_slice($order, $i, $SCAN_CHUNK);
                    $pool  = $this->talenta->getPool('/v2/talenta/v2/employee', $pages, ['limit'=>100]);

                    // build index & sekaligus early-match
                    foreach ($pool as $json) {
                        $rows = (array) data_get($json, 'data.employees', []);
                        foreach ($rows as $it) {
                            // index
                            $this->addToNameIndex($index, $it);

                            // early contains tanpa menunggu index (lebih aman utk nama parsial)
                            $fn   = $this->norm((string) data_get($it, 'personal.first_name', ''));
                            $ln   = $this->norm((string) data_get($it, 'personal.last_name', ''));
                            $full = trim($this->norm(trim($fn . ' ' . $ln)));
                            if ($fn !== '' && (str_contains($fn, $needle) || str_contains($full, $needle))) {
                                $candsAny[] = $it;
                            }
                        }
                    }

                    // exact dari index (kunci full & first)
                    $candsExact = array_merge(
                        $index[$needle]      ?? [],
                        $index[$needleFirst] ?? []
                    );

                    // kalau exact unik -> selesai
                    if (count($candsExact) === 1 && isset($candsExact[0]['user_id'])) {
                        $uid = $candsExact[0]['user_id'];
                        Cache::put($cacheKey, $uid, now()->addHours(12));
                        return redirect()->route('talenta.employees.show', ['user' => $uid]);
                    }

                    // kalau sudah ada kandidat (exact >1 atau contains ada) -> cukup tampilkan
                    if (!empty($candsExact) || !empty($candsAny)) break;
                }

                // gabungkan kandidat (prioritas exact)
                $candidates = !empty($candsExact) ? $candsExact : $candsAny;

                // unikkan by user_id
                $uniq = []; $byId = [];
                foreach ($candidates as $it) {
                    $id = $it['user_id'] ?? null;
                    if ($id && !isset($uniq[$id])) { $uniq[$id] = true; $byId[] = $it; }
                }

                if (count($byId) === 1 && isset($byId[0]['user_id'])) {
                    $uid = $byId[0]['user_id'];
                    Cache::put($cacheKey, $uid, now()->addHours(12));
                    return redirect()->route('talenta.employees.show', ['user' => $uid]);
                }

                if (!empty($byId)) {
                    return view('talenta.employees.index', [
                        'items'      => array_slice($byId, 0, 50),
                        'raw'        => [],
                        'limit'      => $limit,
                        'page'       => 1,
                        'status'     => $status,
                        'pagination' => ['current_page'=>1,'last_page'=>$last,'total'=>count($byId)],
                        'notice'     => 'Ditemukan beberapa kandidat nama. Klik ikon mata untuk melihat detail.',
                    ]);
                }

                $notice = 'Nama (first_name) tidak ditemukan pada batch yang dipindai.';
                return view('talenta.employees.index', [
                    'items'      => [],
                    'raw'        => $firstPageJson,
                    'limit'      => $limit,
                    'page'       => 1,
                    'status'     => $status,
                    'pagination' => ['current_page'=>1,'last_page'=>$last,'total'=>0],
                    'notice'     => $notice,
                ]);
            }

            // 4) list biasa + filter status + pagination
            $resp    = $this->talenta->get('/v2/talenta/v2/employee', ['limit'=>$limit, 'page'=>$page]);
            $raw     = $resp;
            $data    = $resp['data'] ?? [];
            $items   = $data['employees'] ?? [];
            $paging  = $data['pagination'] ?? [];

            if ($status !== '') {
                $items = array_values(array_filter($items, function ($it) use ($status) {
                    $st = strtolower((string)($it['employment']['status'] ?? $it['status'] ?? ''));
                    return $st === strtolower($status);
                }));
            }

            return view('talenta.employees.index', [
                'items'      => $items,
                'raw'        => $raw,
                'limit'      => $limit,
                'page'       => $page,
                'status'     => $status,
                'pagination' => $paging,
                'notice'     => $notice,
            ]);

        } catch (\Throwable $e) {
            Log::channel('talenta')->error('Employees index error', ['error' => $e->getMessage()]);
            return view('talenta.employees.index', [
                'items'      => [],
                'raw'        => ['error' => $e->getMessage()],
                'limit'      => $limit,
                'page'       => $page,
                'status'     => $status,
                'pagination' => [],
                'notice'     => 'Terjadi kesalahan saat memuat data.',
            ])->withErrors(['talenta' => $e->getMessage()]);
        }
    }

    public function show(string $user)
    {
        try {
            $resp = $this->talenta->get('/v2/talenta/v2/employee/' . urlencode($user), []);
            $emp  = $resp['data']['employee'] ?? null;

            if (!$emp) {
                return redirect()->route('talenta.employees.index', ['user_id' => $user])
                    ->withErrors(['talenta' => 'Employee tidak ditemukan.']);
            }

            return view('talenta.employees.show', ['emp' => $emp, 'raw' => $resp]);
        } catch (\Throwable $e) {
            Log::channel('talenta')->error('Employee show error', ['user' => $user, 'error' => $e->getMessage()]);
            return redirect()->route('talenta.employees.index')
                ->withErrors(['talenta' => $e->getMessage()]);
        }
    }
}
