<?php

namespace App\Http\Controllers\Talenta;

use App\Http\Controllers\Controller;
use App\Services\MekariTalentaSandboxClient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class SandboxEmployeeController extends Controller
{
    private MekariTalentaSandboxClient $talenta;
    private int $companyId = 2627; // SANDBOX kamu (branch: id=0, name="Pusat")
    private ?int $actorUserId = null;  // admin/actor Talenta (dibutuhkan PATCH)

    public function __construct(MekariTalentaSandboxClient $talenta)
    {
        $this->talenta = $talenta;
        $cfg = config('services.mekari_talenta_sandbox.actor_user_id');
        if (is_numeric($cfg)) {
            $id = (int) $cfg;
            if ($id > 0) $this->actorUserId = $id;
        }
    }

    /* ================= Helpers (ambil masters, pagination, mapping) ================= */

    private function safeData(string $path, array $query = []): array
    {
        try {
            $resp = $this->talenta->get($path, $query);
            return (is_array($resp) && isset($resp['data']) && is_array($resp['data'])) ? $resp['data'] : [];
        } catch (\Throwable $e) {
            Log::error('[SANDBOX] safeData', ['path'=>$path,'query'=>$query,'err'=>$e->getMessage()]);
            return [];
        }
    }

    private function looksLikeList(array $arr): bool
    {
        if ($arr === []) return true;
        $keys = array_keys($arr);
        $isSeq = ($keys === range(0, count($keys)-1));
        if ($isSeq) return true;
        foreach ($arr as $v) if (!is_array($v)) return false;
        return count($arr) > 1;
    }

    private function extractList(array $raw, array $cand = []): array
    {
        foreach ($cand as $k) if (isset($raw[$k]) && is_array($raw[$k])) return $raw[$k];
        foreach (['items','list','rows','result','data'] as $k) if (isset($raw[$k]) && is_array($raw[$k])) return $raw[$k];
        if ($this->looksLikeList($raw)) return $raw;
        foreach ($raw as $v) if (is_array($v) && $this->looksLikeList($v)) return $v;
        return [];
    }

    private function fetchAllPages(string $path, array $queryBase = [], int $limit = 200, array $wrapperKeys = []): array
    {
        $all = []; $page = 1;
        while (true) {
            $q = array_merge($queryBase, ['limit'=>$limit,'page'=>$page]);
            $data = $this->safeData($path, $q);
            $rows = $this->extractList($data, $wrapperKeys);
            if (!$rows) break;

            $all = array_merge($all, $rows);

            $pg = $data['pagination'] ?? ($data['meta']['pagination'] ?? null);
            $hasMore = false;
            if (is_array($pg)) {
                $cur = (int)($pg['current_page'] ?? $page);
                $tot = (int)($pg['total_pages'] ?? $page);
                $hasMore = $cur < $tot;
            } else {
                $hasMore = count($rows) >= $limit;
            }
            if (!$hasMore) break;
            if (++$page > 200) break;
        }
        return $all;
    }

    private function normalizeDateYmd(?string $val): ?string
    {
        if (!$val) return null;
        $v = trim($val);
        if (preg_match('~^\d{2}/\d{2}/\d{4}$~', $v)) { [$m,$d,$y] = explode('/', $v); return sprintf('%04d-%02d-%02d', $y,$m,$d); }
        if (preg_match('~^\d{4}-\d{2}-\d{2}$~', $v)) return $v;
        return $v;
    }

    private function getBranches(): array
    {
        $raw = $this->fetchAllPages("/v2/talenta/v2/company/{$this->companyId}/branch", [], 200, ['branches','branch']);
        $out = [];
        foreach ($raw as $r) {
            if (!is_array($r)) continue;
            $id   = $r['id'] ?? $r['branch_id'] ?? null;
            $name = $r['name'] ?? $r['branch'] ?? $r['branch_name'] ?? null;
            if ($name !== null) $out[] = ['id'=> (string)($id ?? '0'), 'name'=> (string)$name];
        }
        return $out;
    }

    private function getOrganizations(): array
    {
        $raw = $this->fetchAllPages("/v2/talenta/v2/company/{$this->companyId}/organization", [], 200, ['organizations','organization']);
        $out = [];
        foreach ($raw as $r) {
            if (!is_array($r)) continue;
            $id   = $r['organization_id'] ?? $r['id'] ?? null;
            $name = $r['organization_name'] ?? $r['name'] ?? null;
            if ($name !== null) $out[] = ['id'=>(string)($id ?? ''), 'name'=>(string)$name];
        }
        return $out;
    }

    private function getJobPositions(): array
    {
        $raw = $this->fetchAllPages("/v2/talenta/v2/company/{$this->companyId}/job-position", [], 200, ['job_positions','job_position']);
        $out = [];
        foreach ($raw as $r) {
            if (!is_array($r)) continue;
            $id   = $r['job_position_id'] ?? $r['id'] ?? null;
            $name = $r['job_position'] ?? $r['job_position_name'] ?? $r['name'] ?? null;
            if ($name !== null) $out[] = ['id'=>(string)($id ?? ''), 'name'=>(string)$name];
        }
        return $out;
    }

    private function getJobLevels(): array
    {
        $paths = [
            "/v2/talenta/v2/company/{$this->companyId}/job-level",
            "/v2/talenta/v2/company/me/job-level",
            "/v2/talenta/v2/job-level?company_id={$this->companyId}",
        ];
        $by=[];
        foreach ($paths as $p) {
            $rows = $this->fetchAllPages($p, [], 200, ['job_levels','job_level','levels']);
            foreach ($rows as $r) {
                if (!is_array($r)) continue;
                $id   = $r['job_level_id'] ?? $r['id'] ?? null;
                $name = $r['job_level'] ?? $r['job_level_name'] ?? $r['name'] ?? null;
                if ($name) $by[(string)($id ?? $name)] = ['id'=>(string)($id ?? $name), 'name'=>(string)$name];
            }
        }
        $levels = array_values($by);
        if (!count($levels)) {
            $levels = [];
            $seen=[]; $page=1; $limit=200;
            while (true) {
                $data = $this->safeData('/v2/talenta/v2/employee', ['limit'=>$limit,'page'=>$page]);
                $list = $this->extractList($data, ['employees','employee']);
                if (!$list) break;
                foreach ($list as $emp) {
                    $e = $emp['employment'] ?? [];
                    $nm = $e['job_level'] ?? null; $id = $e['job_level_id'] ?? null;
                    if ($nm && !isset($seen[mb_strtolower($nm)])) {
                        $seen[mb_strtolower($nm)] = true;
                        $levels[] = ['id'=>(string)($id ?? $nm), 'name'=>(string)$nm];
                    }
                }
                $pg = $data['pagination'] ?? null;
                $hasMore = is_array($pg)
                    ? (int)($pg['current_page'] ?? $page) < (int)($pg['total_pages'] ?? $page)
                    : count($list) >= $limit;
                if (!$hasMore || ++$page>50) break;
            }
        }
        return $levels;
    }

    private function getEmploymentStatuses(): array
    {
        $raw = $this->fetchAllPages("/v2/talenta/v2/company/me/employment-status", [], 200, ['employment_statuses','employment_status']);
        $out = [];
        foreach ($raw as $r) {
            if (!is_array($r)) continue;
            $id   = $r['employment_status_id'] ?? $r['id'] ?? null;
            $name = $r['employment_status'] ?? $r['employment_status_name'] ?? $r['name'] ?? (string)$id;
            if ($id !== null) $out[] = ['id'=>(string)$id, 'name'=>(string)$name];
        }
        return $out;
    }

    private function mustPick(string $field, string $input, array $opts): array
    {
        $val = trim($input);
        foreach ($opts as $o) if (mb_strtolower($o['name']) === mb_strtolower($val)) return ['id'=>$o['id'], 'name'=>$o['name']];
        foreach ($opts as $o) if ((string)$o['id'] === (string)$val) return ['id'=>$o['id'], 'name'=>$o['name']];
        $allowed = implode(', ', array_map(fn($o)=>$o['name'], $opts));
        throw ValidationException::withMessages([$field => "{$field} '{$input}' tidak ditemukan. Pilihan: {$allowed}"]);
    }

    private function mapGender(mixed $v): ?int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'male','pria','laki','laki-laki'=>1, 'female','wanita','perempuan'=>2, default=>null }; }
    private function mapMarital(mixed $v): ?int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'single'=>1,'married'=>2,'widow'=>3,'widower'=>4, default=>null }; }
    private function mapReligion(mixed $v): ?int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'islam'=>1,'christian'=>2,'catholic'=>3,'buddha'=>4,'hindu'=>5,'confucius','konghucu'=>6,'others','other'=>7, default=>null }; }
    private function mapPtkp(mixed $v): ?int
    { if (is_numeric($v)) return (int)$v; return match(strtoupper((string)$v)){ 'TK/0'=>1,'TK/1'=>2,'TK/2'=>3,'TK/3'=>4,'K/0'=>5,'K/1'=>6,'K/2'=>7,'K/3'=>8, default=>null }; }
    private function mapTaxConfig(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'gross'=>1,'net'=>2,'gross_up','gross-up'=>3, default=>1 }; }
    private function mapTypeSalary(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'monthly'=>1,'daily'=>2, default=>1 }; }
    private function mapSalaryConfig(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'gross'=>1,'net'=>2, default=>1 }; }
    private function mapYesNo0to3(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'none','no','tidak'=>0,'by employee','employee'=>1,'by company','company'=>2,'both','keduanya'=>3, default=>0 }; }
    private function mapOvertime(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'yes','ya'=>1,'no','tidak'=>2, default=>2 }; }
    private function mapBpjsClass(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ '1','class_1','kelas_1'=>1,'2','class_2','kelas_2'=>2,'3','class_3','kelas_3'=>3, default=>1 }; }

    /* ================= Views ================= */

    public function index(Request $req)
    {
        $limit = $req->get('limit', 10);
        $page  = $req->get('page', 1);
        try {
            $resp   = $this->talenta->get('/v2/talenta/v2/employee', ['limit'=>$limit,'page'=>$page]);
            $items  = is_array($resp) ? data_get($resp,'data.employees',[]) : [];
            $paging = is_array($resp) ? data_get($resp,'data.pagination',[]) : [];
            return view('talenta.sandbox.employees.index', compact('items','paging','limit','page','resp'));
        } catch (\Throwable $e) {
            Log::error('[SANDBOX] index', ['e'=>$e->getMessage()]);
            $items=$paging=[]; $resp=['error'=>$e->getMessage()];
            return view('talenta.sandbox.employees.index', compact('items','paging','limit','page','resp'))
                ->withErrors(['sandbox'=>$e->getMessage()]);
        }
    }

    public function create()
    {
        try {
            $branches = $this->getBranches();
            $orgs     = $this->getOrganizations();
            $jobs     = $this->getJobPositions();
            $levels   = $this->getJobLevels();
            $emps     = $this->getEmploymentStatuses();

            $genders = [['id'=>'Male','name'=>'Male'],['id'=>'Female','name'=>'Female']];
            $maritals = [['id'=>'Single','name'=>'Single'],['id'=>'Married','name'=>'Married'],['id'=>'Widow','name'=>'Widow'],['id'=>'Widower','name'=>'Widower']];
            $religs = [
                ['id'=>'Islam','name'=>'Islam'],['id'=>'Christian','name'=>'Christian'],['id'=>'Catholic','name'=>'Catholic'],
                ['id'=>'Buddha','name'=>'Buddha'],['id'=>'Hindu','name'=>'Hindu'],['id'=>'Confucius','name'=>'Confucius'],['id'=>'Others','name'=>'Others'],
            ];
            $ptkps = [
                ['id'=>'TK/0','name'=>'TK/0'],['id'=>'TK/1','name'=>'TK/1'],['id'=>'TK/2','name'=>'TK/2'],['id'=>'TK/3','name'=>'TK/3'],
                ['id'=>'K/0','name'=>'K/0'],['id'=>'K/1','name'=>'K/1'],['id'=>'K/2','name'=>'K/2'],['id'=>'K/3','name'=>'K/3'],
            ];
            $schedules = [['id'=>'GENERAL','name'=>'GENERAL'],['id'=>'No schedule','name'=>'No schedule'],['id'=>'standart','name'=>'standart'],['id'=>'Long shift','name'=>'Long shift']];

            return view('talenta.sandbox.employees.create', compact(
                'branches','orgs','jobs','levels','emps','genders','maritals','religs','ptkps','schedules'
            ));
        } catch (\Throwable $e) {
            return back()->withErrors(['sandbox'=>$e->getMessage()]);
        }
    }

    /* ================= CREATE / UPDATE ================= */

    private function baseEmploymentPayload(array $input, array $branch, array $org, array $job, string $jobLevelName, string $employmentStatusId): array
    {
        $genderCode = $this->mapGender($input['gender'] ?? null);
        if ($genderCode === null) throw ValidationException::withMessages(['gender' => 'Gender wajib diisi.']);

        $payload = [
            'employee_id'        => (string)($input['employee_id'] ?? ''),
            'first_name'         => (string)($input['first_name'] ?? ''),
            'last_name'          => (string)($input['last_name'] ?? ''),
            'email'              => (string)($input['email'] ?? ''),
            'date_of_birth'      => $this->normalizeDateYmd($input['date_of_birth'] ?? null),
            'gender'             => $genderCode,
            'marital_status'     => $this->mapMarital($input['marital_status'] ?? null) ?? 1,
            'religion'           => $this->mapReligion($input['religion'] ?? null) ?? 1,

            'branch'             => $branch['name'],
            'branch_id'          => is_numeric($branch['id']) ? (int)$branch['id'] : 0,
            'branch_name'        => $branch['name'],

            'organization_name'  => $org['name'],
            'job_position'       => $job['name'],
            'job_level'          => $jobLevelName,
            'employment_status'  => (string)$employmentStatusId,
            'join_date'          => $this->normalizeDateYmd($input['join_date'] ?? null),
            'schedule'           => (string)($input['schedule'] ?? ''),

            // tetap kirim PTKP
            'ptkp_status'        => $this->mapPtkp($input['ptkp_status'] ?? null) ?? 1,

            'npp_bpjs_ketenagakerjaan' => (string)($input['npp_bpjs_ketenagakerjaan'] ?? 'default') ?: 'default',
        ];

        // End date rule
        if (in_array((string)$employmentStatusId, ['2','3'], true)) {
            $end = $this->normalizeDateYmd($input['end_employment_status_date'] ?? null);
            if (!$end) throw ValidationException::withMessages(['end_employment_status_date'=>'Wajib untuk status Contract/Probation (ID=2/3).']);
            $payload['end_employment_status_date'] = $end;
        } elseif (!empty($input['end_employment_status_date'])) {
            $payload['end_employment_status_date'] = $this->normalizeDateYmd($input['end_employment_status_date']);
        }

        foreach ([
            'employee_id','first_name','email','date_of_birth','join_date','gender',
            'organization_name','job_position','job_level','employment_status','ptkp_status'
        ] as $f) {
            if (!isset($payload[$f]) || $payload[$f]==='' || $payload[$f]===null) {
                throw ValidationException::withMessages([$f => "{$f} wajib diisi."]);
            }
        }

        return $payload;
    }

    /**
     * Payroll block builder:
     * - variant0: tanpa payroll (untuk percobaan awal)
     * - variantZero: payroll lengkap + basic_salary=0
     * - variantOne: payroll lengkap + basic_salary=1 (transport), akan di-UPDATE ke 0
     */
    private function buildPayrollVariant(string $variant): array
    {
        if ($variant === 'none') return [];

        $block = [
            'tax_configuration'    => 1, // gross
            'type_salary'          => 1, // monthly
            'salary_configuration' => 1, // gross
            'jht_configuration'    => 0, // none
            'employee_tax_status'  => 1, // active
            'jp_configuration'     => 0, // none
            'overtime_status'      => 2, // no
            'bpjs_kesehatan_config'=> 1, // class 1
        ];

        if ($variant === 'zero') {
            $block['basic_salary'] = 0;
        } elseif ($variant === 'one') {
            $block['basic_salary'] = 1;
        }

        return $block;
    }

    private function tryCreate(array $payloadBase, array $branch, array $payrollBlock): array
    {
        $trials = [
            ['mode'=>'id-string','path'=>'/v2/talenta/v2/employee','query'=>['company_id'=>$this->companyId]],
            ['mode'=>'name','path'=>'/v2/talenta/v2/employee','query'=>['company_id'=>$this->companyId]],
        ];

        $errors = [];
        foreach ($trials as $t) {
            $payload = array_merge($payloadBase, $payrollBlock);

            // branch mode tweak
            if ($t['mode'] === 'id-string') {
                $payload['branch']      = (string)$branch['id']; // "0"
                $payload['branch_id']   = is_numeric($branch['id']) ? (int)$branch['id'] : 0;
            } else {
                $payload['branch']      = $branch['name'];       // "Pusat"
                $payload['branch_id']   = is_numeric($branch['id']) ? (int)$branch['id'] : 0;
            }
            $payload['branch_name'] = $branch['name'];

            Log::debug('[CREATE TRIAL]', [
                'branch_mode'=>$t['mode'],
                'has_basic_salary'=>array_key_exists('basic_salary',$payrollBlock),
                'basic_salary'=> $payrollBlock['basic_salary'] ?? null,
                'keys'=>array_keys($payload),
            ]);

            try {
                return $this->talenta->post($t['path'], $payload, $t['query']);
            } catch (\Throwable $e) {
                $errors[] = ['trial'=>$t,'error'=>$e->getMessage()];
                Log::warning('[CREATE FAIL]', end($errors));
            }
        }

        throw new \RuntimeException("Semua trial POST gagal. Detail:\n".json_encode($errors, JSON_PRETTY_PRINT));
    }

    public function store(Request $request)
    {
        // *** TIDAK DIUBAH ***
        try {
            // basic_salary TIDAK diwajibkan di form
            $request->validate([
                'employee_id'   => 'required|string',
                'first_name'    => 'required|string',
                'email'         => 'required|email',
                'date_of_birth' => 'required|date',
                'gender'        => 'required|string',
                'marital_status'=> 'required|string',
                'religion'      => 'required|string',
                'branch'        => 'required|string',
                'organization_name' => 'required|string',
                'job_position'  => 'required|string',
                'job_level'     => 'required|string',
                'employment_status' => 'required',
                'join_date'     => 'required|date',
                'ptkp_status'   => 'required|string',
                'basic_salary'  => 'nullable|integer|min:0',
            ]);

            // masters
            $branches = $this->getBranches();
            $orgs     = $this->getOrganizations();
            $jobs     = $this->getJobPositions();
            $levels   = $this->getJobLevels();
            $emps     = $this->getEmploymentStatuses();

            $branchSel = $this->mustPick('branch', (string)$request->input('branch',''), $branches);
            $orgSel    = $this->mustPick('organization_name', (string)$request->input('organization_name',''), $orgs);
            $jobSel    = $this->mustPick('job_position', (string)$request->input('job_position',''), $jobs);

            $jobLevelInput = trim((string)$request->input('job_level',''));
            if (!count($levels)) {
                if ($jobLevelInput === '') throw ValidationException::withMessages(['job_level'=>'Wajib diisi.']);
                $jobLevelName = $jobLevelInput;
            } else {
                $lev = $this->mustPick('job_level', $jobLevelInput, $levels);
                $jobLevelName = $lev['name'];
            }

            $emp = $this->mustPick('employment_status', (string)$request->input('employment_status',''), $emps);
            $employmentStatusId = $emp['id'];

            $payloadBase = $this->baseEmploymentPayload($request->all(), $branchSel, $orgSel, $jobSel, $jobLevelName, $employmentStatusId);

            // ========== 3 langkah create ==========
            // 1) tanpa payroll block
            try {
                $resp = $this->tryCreate($payloadBase, $branchSel, $this->buildPayrollVariant('none'));
            } catch (\Throwable $e1) {
                // 2) payroll + basic_salary=0
                try {
                    $resp = $this->tryCreate($payloadBase, $branchSel, $this->buildPayrollVariant('zero'));
                } catch (\Throwable $e2) {
                    $msg2 = $e2->getMessage();

                    // kalau masih ditolak "must be positive integer" atau "required", baru fallback 1 -> lalu update 0
                    $needTransport = str_contains($msg2, 'must be positive integer')
                                  || str_contains($msg2, 'basic_salary required')
                                  || str_contains($msg2, 'required');

                    if (!$needTransport) {
                        throw $e2;
                    }

                    // 3) payroll + basic_salary=1 (transport)
                    $resp = $this->tryCreate($payloadBase, $branchSel, $this->buildPayrollVariant('one'));

                    // after-create: ubah ke 0 SEGERA (biar hasil akhir tetap 0)
                    $uid = is_array($resp) ? (data_get($resp,'data.employee.user_id') ?? data_get($resp,'data.user_id')) : null;
                    if ($uid) {
                        try {
                            $updatePayload = array_merge(
                                $payloadBase,
                                $this->buildPayrollVariant('zero')
                            );
                            // tambahkan company_id via update pipeline
                            $this->sendUpdateWithFallbacks($uid, $updatePayload);
                            Log::info('[AUTO-FIX] basic_salary set to 0 after create', ['user_id'=>$uid]);
                        } catch (\Throwable $fixE) {
                            Log::error('[AUTO-FIX FAILED] set salary 0', ['err'=>$fixE->getMessage()]);
                        }
                    }
                }
            }

            $uid = is_array($resp) ? (data_get($resp,'data.employee.user_id') ?? data_get($resp,'data.user_id')) : null;

            if ($uid) {
                return redirect()->route('talenta.sandbox.employees.show', $uid)
                    ->with('success', "Created ke SANDBOX (user_id: {$uid}). Gaji saat ini 0 (default).");
            }

            return redirect()->route('talenta.sandbox.employees.index')
                ->with('success', 'Created ke SANDBOX. (user_id tidak dikembalikan API)');
        } catch (ValidationException $ve) {
            return back()->withErrors($ve->errors())->withInput();
        } catch (\Throwable $e) {
            Log::error('[SANDBOX] store error', ['e'=>$e->getMessage()]);
            return back()->withErrors(['sandbox'=>$e->getMessage()])->withInput();
        }
    }

    public function show(string $user)
    {
        try {
            $resp = $this->talenta->get("/v2/talenta/v2/employee/{$user}");
            $emp  = is_array($resp) ? data_get($resp,'data.employee') : null;
            if (!$emp) return redirect()->route('talenta.sandbox.employees.index')
                ->withErrors(['sandbox'=>'Employee tidak ditemukan']);
            return view('talenta.sandbox.employees.show', compact('emp','resp'));
        } catch (\Throwable $e) {
            return redirect()->route('talenta.sandbox.employees.index')
                ->withErrors(['sandbox'=>$e->getMessage()]);
        }
    }

    public function edit(string $user)
    {
        try {
            $resp = $this->talenta->get("/v2/talenta/v2/employee/{$user}");
            $emp  = is_array($resp) ? data_get($resp,'data.employee') : null;
            if (!$emp) {
                return redirect()->route('talenta.sandbox.employees.index')
                    ->withErrors(['sandbox' => 'Employee tidak ditemukan']);
            }

            $branches = $this->getBranches();
            $orgs     = $this->getOrganizations();
            $jobs     = $this->getJobPositions();
            $levels   = $this->getJobLevels();
            $emps     = $this->getEmploymentStatuses();

            $genders = [
                ['id'=>'Male','name'=>'Male'],
                ['id'=>'Female','name'=>'Female'],
            ];
            $maritals = [
                ['id'=>'Single','name'=>'Single'],
                ['id'=>'Married','name'=>'Married'],
                ['id'=>'Widow','name'=>'Widow'],
                ['id'=>'Widower','name'=>'Widower'],
            ];
            $religs = [
                ['id'=>'Islam','name'=>'Islam'],
                ['id'=>'Christian','name'=>'Christian'],
                ['id'=>'Catholic','name'=>'Catholic'],
                ['id'=>'Buddha','name'=>'Buddha'],
                ['id'=>'Hindu','name'=>'Hindu'],
                ['id'=>'Confucius','name'=>'Confucius'],
                ['id'=>'Others','name'=>'Others'],
            ];
            $ptkps = [
                ['id'=>'TK/0','name'=>'TK/0'],['id'=>'TK/1','name'=>'TK/1'],
                ['id'=>'TK/2','name'=>'TK/2'],['id'=>'TK/3','name'=>'TK/3'],
                ['id'=>'K/0','name'=>'K/0'],  ['id'=>'K/1','name'=>'K/1'],
                ['id'=>'K/2','name'=>'K/2'],  ['id'=>'K/3','name'=>'K/3'],
            ];
            $schedules = [
                ['id'=>'GENERAL','name'=>'GENERAL'],
                ['id'=>'No schedule','name'=>'No schedule'],
                ['id'=>'standart','name'=>'standart'],
                ['id'=>'Long shift','name'=>'Long shift'],
            ];

            return view('talenta.sandbox.employees.edit', compact(
                'user','emp','branches','orgs','jobs','levels','emps',
                'genders','maritals','religs','ptkps','schedules'
            ));
        } catch (\Throwable $e) {
            return redirect()->route('talenta.sandbox.employees.index')
                ->withErrors(['sandbox' => $e->getMessage()]);
        }
    }

    /* ===================== UPDATE: util untuk actor & current employee ===================== */

    private function getActorUserId(string $user): int
    {
        if ($this->actorUserId && $this->actorUserId > 0) return $this->actorUserId;

        try {
            $resp = $this->talenta->get("/v2/talenta/v2/employee/{$user}");
            $uid  = data_get($resp, 'data.employee.user_id');
            if (is_numeric($uid)) return (int) $uid;
        } catch (\Throwable $e) { /* ignore */ }

        if (ctype_digit($user)) return (int) $user;
        return 0; // fallback integer
    }

    private function getCurrentEmployee(string $user): array
    {
        try {
            $resp = $this->talenta->get("/v2/talenta/v2/employee/{$user}");
            $emp  = is_array($resp) ? (data_get($resp,'data.employee') ?? []) : [];
            $genderRaw  = data_get($emp, 'gender');
            $gender     = is_numeric($genderRaw) ? (int)$genderRaw : ($this->mapGender($genderRaw) ?? 1);

            $emp['__normalized'] = [
                'date_of_birth' => $this->normalizeDateYmd((string) data_get($emp, 'date_of_birth', '')),
                'join_date'     => $this->normalizeDateYmd((string) data_get($emp, 'employment.join_date', '')),
                'gender'        => $gender,
                'marital'       => (int) (data_get($emp, 'marital_status', 1)),
                'religion'      => (int) (data_get($emp, 'religion', 1)),
                'branch_id'     => (string) (data_get($emp,'employment.branch_id') ?? data_get($emp,'employment.branch.branch_id') ?? '0'),
                'org_name'      => (string) data_get($emp,'employment.organization_name',''),
                'job_position'  => (string) data_get($emp,'employment.job_position',''),
                'job_level'     => (string) data_get($emp,'employment.job_level',''),
                'emp_status'    => (string) (data_get($emp,'employment.employment_status_id') ?? data_get($emp,'employment.employment_status') ?? ''),
                'schedule'      => (string) data_get($emp,'employment.schedule',''),
                'end_status_dt' => (string) data_get($emp,'employment.end_employment_status_date',''),
                'email'         => (string) data_get($emp,'email',''),
                'first_name'    => (string) data_get($emp,'first_name',''),
                'last_name'     => (string) data_get($emp,'last_name',''),
                'employee_id'   => (string) data_get($emp,'employee_id',''),
                // Payroll
                'basic_salary'  => (string) data_get($emp,'payroll.basic_salary',''),
                'tax_conf'      => (int) (data_get($emp,'payroll.tax_configuration') ?? 1),
                'type_salary'   => (int) (data_get($emp,'payroll.type_salary') ?? 1),
                'salary_conf'   => (int) (data_get($emp,'payroll.salary_configuration') ?? 1),
                'jht_conf'      => (int) (data_get($emp,'payroll.jht_configuration') ?? 0),
                'emp_tax_stat'  => (int) (data_get($emp,'payroll.employee_tax_status') ?? 1),
                'jp_conf'       => (int) (data_get($emp,'payroll.jp_configuration') ?? 0),
                'ot_status'     => (int) (data_get($emp,'payroll.overtime_status') ?? 2),
                'bpjs_kes'      => (int) (data_get($emp,'payroll.bpjs_kesehatan_config') ?? 1),
                'ptkp_status'   => (int) (data_get($emp,'payroll.ptkp_status') ?? 1),
                'npp_bpjstk'    => (string) (data_get($emp,'npp_bpjs_ketenagakerjaan') ?? 'default'),
            ];
            return $emp;
        } catch (\Throwable $e) {
            return [];
        }
    }

    /* ===================== UPDATE: build PATCH & POST bodies ===================== */

    private function buildPatchAndPostPayloads(string $user, array $payload): array
    {
        $emp = $this->getCurrentEmployee($user);
        $cur = $emp['__normalized'] ?? [];

        // Drop schedule jika "No schedule"
        if (!empty($payload['schedule']) && strcasecmp((string)$payload['schedule'],'No schedule') === 0) {
            unset($payload['schedule']);
        }
        // Jangan kirim branch string pada PATCH
        unset($payload['branch'], $payload['branch_name']);

        // ----- PATCH diff-only -----
        $patch = [];
        foreach ($payload as $k => $v) {
            if ($v === null || $v === '') {
                if ($k === 'basic_salary' && ($v === 0 || $v === '0')) {
                    // allow set to 0
                } else {
                    continue;
                }
            }
            if (array_key_exists($k, $cur) && (string)$v === (string)$cur[$k]) continue;
            if (in_array($k, ['employee_id','email'], true)) {
                if (array_key_exists($k, $cur) && strcasecmp((string)$v, (string)$cur[$k]) === 0) continue;
            }
            if ($k === 'branch_id' && array_key_exists('branch_id', $cur) && (string)$v === (string)$cur['branch_id']) continue;
            $patch[$k] = $v;
        }
        // Wajib integer
        $patch['actor_user_id'] = $this->getActorUserId($user);

        // ----- POST fallback body lengkap (untuk update-by-id) -----
        $pick = function(string $key, $default=null) use ($payload, $cur) {
            if (array_key_exists($key, $payload) && $payload[$key] !== '' && $payload[$key] !== null) return $payload[$key];
            if (array_key_exists($key, $cur)    && $cur[$key]    !== '' && $cur[$key]    !== null)    return $cur[$key];
            return $default;
        };

        $branchId  = (string) $pick('branch_id', '0');
        $jobLevel  = (string) $pick('job_level', '');
        $empStat   = (string) $pick('employment_status', '');
        $joinDate  = (string) $pick('join_date', '');
        $orgName   = (string) $pick('organization_name', '');
        $jobPos    = (string) $pick('job_position', '');
        $email     = (string) $pick('email', '');
        $firstName = (string) $pick('first_name', '');
        $lastName  = (string) $pick('last_name', '');
        $dob       = (string) $this->normalizeDateYmd((string) $pick('date_of_birth', ''));
        $gender    = (int)    $pick('gender',  $cur['gender']   ?? 1);
        $marital   = (int)    $pick('marital_status', $cur['marital'] ?? 1);
        $religion  = (int)    $pick('religion', $cur['religion'] ?? 1);
        $nppBPJSTK = (string) $pick('npp_bpjs_ketenagakerjaan', $cur['npp_bpjstk'] ?? 'default');

        $sched = (string) $pick('schedule', '');
        if ($sched !== '' && strcasecmp($sched,'No schedule') === 0) $sched = '';

        $post = [
            'user_id'               => is_numeric(data_get($emp,'user_id')) ? (int) data_get($emp,'user_id') : $user,
            'actor_user_id'         => $this->getActorUserId($user),

            // identity (tambahan required oleh sandbox)
            'first_name'            => $firstName,
            'last_name'             => $lastName,
            'date_of_birth'         => $dob,
            'gender'                => $gender,
            'marital_status'        => $marital,
            'religion'              => $religion,
            'email'                 => $email,

            // employment (required)
            'branch'                => $branchId,           // id sebagai string
            'branch_id'             => (int) $branchId,
            'organization_name'     => $orgName,
            'job_position'          => $jobPos,
            'job_level'             => $jobLevel,
            'employment_status'     => $empStat,
            'join_date'             => $joinDate,
        ];
        if ($sched !== '') $post['schedule'] = $sched;

        $endDate = (string) $pick('end_employment_status_date', '');
        if ($endDate !== '') $post['end_employment_status_date'] = $this->normalizeDateYmd($endDate);

        // Payroll block lengkap
        $post += [
            'tax_configuration'     => (int) $pick('tax_configuration',    $cur['tax_conf']    ?? 1),
            'type_salary'           => (int) $pick('type_salary',          $cur['type_salary'] ?? 1),
            'salary_configuration'  => (int) $pick('salary_configuration', $cur['salary_conf'] ?? 1),
            'jht_configuration'     => (int) $pick('jht_configuration',    $cur['jht_conf']    ?? 0),
            'employee_tax_status'   => (int) $pick('employee_tax_status',  $cur['emp_tax_stat']?? 1),
            'jp_configuration'      => (int) $pick('jp_configuration',     $cur['jp_conf']     ?? 0),
            'overtime_status'       => (int) $pick('overtime_status',      $cur['ot_status']   ?? 2),
            'bpjs_kesehatan_config' => (int) $pick('bpjs_kesehatan_config',$cur['bpjs_kes']    ?? 1),
        ];

        $ptkp = $pick('ptkp_status', $cur['ptkp_status'] ?? null);
        if ($ptkp !== null && $ptkp !== '') $post['ptkp_status'] = (int) $ptkp;

        $post['npp_bpjs_ketenagakerjaan'] = $nppBPJSTK !== '' ? $nppBPJSTK : 'default';

        $basicSal = $pick('basic_salary', $cur['basic_salary'] ?? '');
        if ($basicSal !== '' && $basicSal !== null) {
            $post['basic_salary'] = max(0, (int) $basicSal);
        }

        return [$patch, $post];
    }

    private function sendUpdateWithFallbacks(string $user, array $payload)
    {
        $errors = [];
        [$patch, $postBody] = $this->buildPatchAndPostPayloads($user, $payload);

        // Jika patch hanya berisi actor_user_id tanpa perubahan lain, skip PATCH
        $onlyActor = (count($patch) === 1 && array_key_exists('actor_user_id', $patch));
        $q = ['company_id' => $this->companyId];

        // TRY #1: PATCH /employee/{user}
        if (!$onlyActor) {
            try {
                return $this->talenta->patch("/v2/talenta/v2/employee/{$user}", $patch, $q);
            } catch (\Throwable $e) {
                $errors[] = ['try'=>'PATCH /employee/{user}', 'error'=>$e->getMessage()];
                Log::warning('[SANDBOX UPDATE] PATCH failed', end($errors));
            }
        }

        // TRY #2: POST /employee (with user_id) — update-by-id, body lengkap
        try {
            unset($postBody['branch_name']);
            return $this->talenta->post("/v2/talenta/v2/employee", $postBody, $q);
        } catch (\Throwable $e) {
            $errors[] = ['try'=>'POST /employee (with user_id) as update', 'error'=>$e->getMessage()];
            Log::warning('[SANDBOX UPDATE] POST /employee (with user_id) failed', end($errors));
        }

        throw new \RuntimeException('Update gagal di semua fallback: ' . json_encode($errors, JSON_PRETTY_PRINT));
    }

    public function update(Request $request, string $user)
    {
        $request->validate([
            'employee_id'        => 'required|string',
            'first_name'         => 'required|string',
            'email'              => 'required|email',
            'date_of_birth'      => 'required|date',
            'gender'             => 'required|string',
            'marital_status'     => 'required|string',
            'religion'           => 'required|string',
            'branch'             => 'required|string',
            'organization_name'  => 'required|string',
            'job_position'       => 'required|string',
            'job_level'          => 'required|string',
            'employment_status'  => 'required',
            'join_date'          => 'required|date',
            'ptkp_status'        => 'required|string',
            'basic_salary'       => 'nullable|integer|min:0',
        ]);

        try {
            $branches = $this->getBranches();
            $orgs     = $this->getOrganizations();
            $jobs     = $this->getJobPositions();
            $levels   = $this->getJobLevels();
            $emps     = $this->getEmploymentStatuses();

            $branchSel = $this->mustPick('branch', (string)$request->input('branch',''), $branches);
            $orgSel    = $this->mustPick('organization_name', (string)$request->input('organization_name',''), $orgs);
            $jobSel    = $this->mustPick('job_position', (string)$request->input('job_position',''), $jobs);

            $jobLevelInput = trim((string)$request->input('job_level',''));
            if (!count($levels)) {
                if ($jobLevelInput === '') throw \Illuminate\Validation\ValidationException::withMessages(['job_level'=>'Wajib diisi.']);
                $jobLevelName = $jobLevelInput;
            } else {
                $lev = $this->mustPick('job_level', $jobLevelInput, $levels);
                $jobLevelName = $lev['name'];
            }

            $emp = $this->mustPick('employment_status', (string)$request->input('employment_status',''), $emps);
            $employmentStatusId = $emp['id'];

            // Base payload lengkap (nanti diolah menjadi PATCH diff & POST full)
            $payload = $this->baseEmploymentPayload(
                $request->all(),
                $branchSel,
                $orgSel,
                $jobSel,
                $jobLevelName,
                $employmentStatusId
            );

            // Untuk UPDATE pakai branch_id; branch string akan ditangani di builder
            $payload['branch_id'] = is_numeric($branchSel['id']) ? (int)$branchSel['id'] : 0;

            // basic_salary jika ada di form (termasuk 0)
            if ($request->has('basic_salary')) {
                $val = $request->input('basic_salary');
                if ($val !== null && $val !== '') $payload['basic_salary'] = (int) $val;
            }

            $this->sendUpdateWithFallbacks($user, $payload);

            return redirect()->route('talenta.sandbox.employees.show', $user)
                ->with('success', 'Updated di SANDBOX.');
        } catch (\Illuminate\Validation\ValidationException $ve) {
            return back()->withErrors($ve->errors())->withInput();
        } catch (\Throwable $e) {
            Log::error('[SANDBOX] update error', ['user'=>$user, 'e'=>$e->getMessage()]);
            return back()->withErrors(['sandbox'=>$e->getMessage()])->withInput();
        }
    }

    public function destroy(string $user)
    {
        try {
            $this->talenta->delete("/v2/talenta/v2/employee/{$user}");
            return redirect()->route('talenta.sandbox.employees.index')
                ->with('success', 'Employee dihapus (SANDBOX).');
        } catch (\Throwable $e) {
            Log::error('[SANDBOX] destroy error', ['user'=>$user, 'e'=>$e->getMessage()]);
            return back()->withErrors(['sandbox'=>$e->getMessage()]);
        }
    }
}