{{-- resources/views/talenta/sandbox/employees/create.blade.php --}}
@extends('layouts.app')
@section('title','Sandbox – Employees')
@section('page-title','Buat Karyawan (Sandbox)')

@section('content')
@php
    /**
     * ====== Helper: Label field yang human-friendly ======
     */
    function field_label(string $k): string {
        $map = [
            'employee_id' => 'Employee ID',
            'first_name'  => 'First Name',
            'last_name'   => 'Last Name',
            'email'       => 'Email',
            'date_of_birth' => 'Birthdate',
            'gender'      => 'Gender',
            'marital_status' => 'Marital Status',
            'religion'    => 'Religion',
            'branch'      => 'Branch',
            'organization_name' => 'Organization',
            'job_position'=> 'Job Position',
            'job_level'   => 'Job Level',
            'employment_status' => 'Employment Status',
            'join_date'   => 'Join Date',
            'end_employment_status_date' => 'End Employment Status Date',
            'schedule'    => 'Schedule',
            'ptkp_status' => 'PTKP Status',
            'npp_bpjs_ketenagakerjaan' => 'NPP BPJS Ketenagakerjaan',
            'basic_salary'=> 'Basic Salary',
            'tax_configuration' => 'Tax Configuration',
            'type_salary' => 'Type Salary',
            'salary_configuration' => 'Salary Configuration',
            'jht_configuration' => 'JHT Configuration',
            'employee_tax_status' => 'Employee Tax Status',
            'jp_configuration' => 'JP Configuration',
            'overtime_status' => 'Overtime',
            'bpjs_kesehatan_config' => 'BPJS Kesehatan Class',
        ];
        return $map[$k] ?? ucwords(str_replace(['_','-'],' ',$k));
    }

    /**
     * ====== Helper: Humanize 1 baris error ======
     * Contoh masukan:
     *  - "email tes@gmail.com has already been taken"
     *  - "branch Pusat doesn't exist"
     *  - "basic_salary must be positive integer"
     *  - "employee_id KM10001 has already been taken"
     *  - "email, branch, job_position required"
     */
    function humanize_line(string $line): string {
        $s = trim($line);

        // Remove noisy prefixes
        $s = preg_replace('~^Talenta Sandbox error:\s*~i', '', $s);

        // Required list seperti "email, branch required"
        if (preg_match('~^([a-z0-9_,\s-]+)\s+required$~i', $s, $m)) {
            $fields = array_map('trim', explode(',', $m[1]));
            $labels = array_map(fn($f)=> field_label($f), $fields);
            return 'Data wajib belum lengkap: ' . implode(', ', $labels) . '.';
        }

        // "X has already been taken"
        if (preg_match('~^(email|employee_id)\s+(.+?)\s+has already been taken$~i', $s, $m)) {
            $label = field_label(strtolower($m[1]));
            $val   = $m[2];
            if (strtolower($m[1]) === 'email') {
                return "Email \"$val\" sudah terdaftar.";
            }
            if (strtolower($m[1]) === 'employee_id') {
                return "Employee ID \"$val\" sudah digunakan.";
            }
        }

        // "branch Pusat doesn't exist"
        if (preg_match('~^(branch|organization_name|job_position|job_level)\s+(.+?)\s+doesn\'?t exist$~i', $s, $m)) {
            $label = field_label(strtolower($m[1]));
            $val   = $m[2];
            return "{$label} \"$val\" tidak ditemukan. Pastikan pilih dari daftar.";
        }

        // "must be positive integer"
        if (preg_match('~^(basic_salary|employee_tax_status)\s+must be positive integer$~i', $s, $m)) {
            $label = field_label(strtolower($m[1]));
            return "{$label} harus lebih dari 0 (angka bulat).";
        }

        // "mag: method not allowed" → hide kode teknis
        if (stripos($s, 'method not allowed') !== false) {
            return 'Aksi tidak didukung oleh server sandbox. Coba lagi dengan data yang benar.';
        }

        // Generic English
        $replacements = [
            "Data not valid" => "Data tidak valid.",
            "Missing required keys" => "Data wajib belum lengkap.",
            "required" => "wajib diisi.",
            "already been taken" => "sudah digunakan.",
            "doesn't exist" => "tidak ditemukan.",
            "must be positive integer" => "harus lebih dari 0 (angka bulat).",
        ];
        foreach ($replacements as $k=>$v) {
            if (stripos($s, $k) !== false) {
                $s = str_ireplace($k, $v, $s);
            }
        }

        // Sedikit polesan umum: awali huruf besar, akhiri titik.
        $s = rtrim($s, " \t\n\r\0\x0B.");
        $s = ucfirst($s) . '.';
        return $s;
    }

    /**
     * ====== Helper: Parse error Talenta JSON ======
     * Masukan: "Talenta Sandbox error: {json...}"
     * Keluaran: array of humanized lines
     */
    function parse_talenta_errors(string $raw): array {
        $out = [];
        if (preg_match('~Talenta Sandbox error:\s*(\{.*)$~s', $raw, $m)) {
            $json = trim($m[1]);
            $obj  = json_decode($json, true);
            if (is_array($obj)) {
                if (!empty($obj['errors']) && is_array($obj['errors'])) {
                    foreach ($obj['errors'] as $ln) {
                        $out[] = humanize_line((string)$ln);
                    }
                } elseif (!empty($obj['message'])) {
                    $out[] = humanize_line((string) $obj['message']);
                }
            }
        }
        // Fallback: kalau tak bisa parse, humanize saja seluruh raw
        if (!$out) $out[] = humanize_line($raw);
        return $out;
    }

    /**
     * ====== Kumpulkan semua error (validator + session('error')) jadi friendly ======
     */
    $friendlyErrors = [];

    // Dari validator Laravel ($errors)
    if ($errors->any()) {
        foreach ($errors->all() as $e) {
            $e = (string)$e;
            if (stripos($e, 'Talenta Sandbox error:') !== false) {
                $friendlyErrors = array_merge($friendlyErrors, parse_talenta_errors($e));
            } else {
                // Contoh validator: "employee_id wajib diisi." sudah manusiawi → tapi tetap poles sedikit
                $friendlyErrors[] = humanize_line($e);
            }
        }
    }

    // Dari session('error') (mis. error generik dari controller)
    if (session('error')) {
        $raw = (string) session('error');
        if (stripos($raw, 'Talenta Sandbox error:') !== false) {
            $friendlyErrors = array_merge($friendlyErrors, parse_talenta_errors($raw));
        } else {
            $friendlyErrors[] = humanize_line($raw);
        }
    }

    // Hilangkan duplikat & kosong
    $friendlyErrors = array_values(array_unique(array_filter(array_map('trim', $friendlyErrors))));
@endphp

<div class="max-w-6xl mx-auto">
    <div class="bg-white rounded-2xl shadow-sm border border-gray-100 px-5 py-4 mb-5 flex items-center justify-between">
        <div>
            <h2 class="text-lg font-semibold text-gray-900">Tambah Karyawan – Sandbox</h2>
            <p class="text-sm text-gray-500">
                Lengkapi informasi personal & employment. Kolom bertanda
                <span class="text-red-600">*</span> wajib diisi.
            </p>
        </div>
        <a href="{{ route('talenta.sandbox.employees.index') }}"
           class="inline-flex items-center gap-2 text-sm px-3 py-2 rounded-xl border border-gray-200 hover:bg-gray-50">
           ← Kembali
        </a>
    </div>

    {{-- Panel error (human-friendly) --}}
    @if(count($friendlyErrors))
        <div class="mb-5 rounded-xl border border-red-200 bg-red-50 text-red-700 px-4 py-3">
            <div class="font-semibold mb-1">Form belum lengkap</div>
            <ul class="list-disc ml-5 text-sm">
                @foreach($friendlyErrors as $e)
                    <li>{{ $e }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <form action="{{ route('talenta.sandbox.employees.store') }}" method="POST" class="space-y-6">
        @csrf

        {{-- ================= PERSONAL ================= --}}
        <section class="bg-white rounded-2xl shadow-sm border border-gray-100">
            <div class="px-5 py-4 border-b border-gray-100">
                <h3 class="font-semibold text-gray-900">Informasi Personal</h3>
                <p class="text-sm text-gray-500">Data dasar karyawan.</p>
            </div>

            <div class="p-5 grid grid-cols-1 md:grid-cols-2 gap-5">
                {{-- Employee ID --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Employee ID <span class="text-red-600">*</span>
                    </label>
                    <input type="text" name="employee_id" value="{{ old('employee_id') }}"
                           class="w-full rounded-xl border @error('employee_id') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2"
                           placeholder="KM10000001">
                    @error('employee_id')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- First Name --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        First Name <span class="text-red-600">*</span>
                    </label>
                    <input type="text" name="first_name" value="{{ old('first_name') }}"
                           class="w-full rounded-xl border @error('first_name') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                    @error('first_name')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- Last Name --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Last Name</label>
                    <input type="text" name="last_name" value="{{ old('last_name') }}"
                           class="w-full rounded-xl border border-gray-300 px-3 py-2">
                </div>

                {{-- Email --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Email <span class="text-red-600">*</span>
                    </label>
                    <input type="email" name="email" value="{{ old('email') }}"
                           class="w-full rounded-xl border @error('email') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2"
                           placeholder="nama@perusahaan.com">
                    @error('email')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- Birthdate --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Birthdate <span class="text-red-600">*</span>
                    </label>
                    <input type="date" name="date_of_birth" value="{{ old('date_of_birth') }}"
                           class="w-full rounded-xl border @error('date_of_birth') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                    @error('date_of_birth')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- Gender (WAJIB) --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Gender <span class="text-red-600">*</span>
                    </label>
                    <select name="gender"
                            class="w-full rounded-xl border @error('gender') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                        <option value="">-- Pilih --</option>
                        @foreach(($genders ?? []) as $opt)
                            <option value="{{ $opt['name'] }}" {{ old('gender') == $opt['name'] ? 'selected' : '' }}>
                                {{ $opt['name'] }}
                            </option>
                        @endforeach
                    </select>
                    @error('gender')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                    <p class="text-xs text-gray-500 mt-1">Wajib dipilih (Male/Female).</p>
                </div>

                {{-- Marital Status --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Marital Status <span class="text-red-600">*</span>
                    </label>
                    <select name="marital_status"
                            class="w-full rounded-xl border @error('marital_status') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                        <option value="">-- Pilih --</option>
                        @foreach(($maritals ?? []) as $opt)
                            <option value="{{ $opt['name'] }}" {{ old('marital_status') == $opt['name'] ? 'selected' : '' }}>
                                {{ $opt['name'] }}
                            </option>
                        @endforeach
                    </select>
                    @error('marital_status')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- Religion --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Religion <span class="text-red-600">*</span>
                    </label>
                    <select name="religion"
                            class="w-full rounded-xl border @error('religion') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                        <option value="">-- Pilih --</option>
                        @foreach(($religs ?? []) as $opt)
                            <option value="{{ $opt['name'] }}" {{ old('religion') == $opt['name'] ? 'selected' : '' }}>
                                {{ $opt['name'] }}
                            </option>
                        @endforeach
                    </select>
                    @error('religion')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>
            </div>
        </section>

        {{-- ================= EMPLOYMENT ================= --}}
        <section class="bg-white rounded-2xl shadow-sm border border-gray-100">
            <div class="px-5 py-4 border-b border-gray-100">
                <h3 class="font-semibold text-gray-900">Informasi Employment</h3>
                <p class="text-sm text-gray-500">Pemetaan organisasi & status kerja.</p>
            </div>

            <div class="p-5 grid grid-cols-1 md:grid-cols-2 gap-5">
                {{-- Branch --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Branch <span class="text-red-600">*</span>
                    </label>
                    <select name="branch"
                            class="w-full rounded-xl border @error('branch') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                        <option value="">-- Pilih --</option>
                        @foreach(($branches ?? []) as $opt)
                            <option value="{{ $opt['name'] }}" {{ old('branch') == $opt['name'] ? 'selected' : '' }}>
                                {{ $opt['name'] }} @if(($opt['id'] ?? '')!=='') (ID: {{ $opt['id'] }}) @endif
                            </option>
                        @endforeach
                    </select>
                    @error('branch')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                    <p class="text-xs text-gray-500 mt-1">Pilih persis “Pusat”.</p>
                </div>

                {{-- Organization --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Organization <span class="text-red-600">*</span>
                    </label>
                    <select name="organization_name"
                            class="w-full rounded-xl border @error('organization_name') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                        <option value="">-- Pilih --</option>
                        @foreach(($orgs ?? []) as $opt)
                            <option value="{{ $opt['name'] }}" {{ old('organization_name') == $opt['name'] ? 'selected' : '' }}>
                                {{ $opt['name'] }}
                            </option>
                        @endforeach
                    </select>
                    @error('organization_name')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- Job Position --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Job Position <span class="text-red-600">*</span>
                    </label>
                    <select name="job_position"
                            class="w-full rounded-xl border @error('job_position') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                        <option value="">-- Pilih --</option>
                        @foreach(($jobs ?? []) as $opt)
                            <option value="{{ $opt['name'] }}" {{ old('job_position') == $opt['name'] ? 'selected' : '' }}>
                                {{ $opt['name'] }}
                            </option>
                        @endforeach
                    </select>
                    @error('job_position')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- Job Level --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Job Level <span class="text-red-600">*</span>
                    </label>
                    @if(!empty($levels))
                        <select name="job_level"
                                class="w-full rounded-xl border @error('job_level') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                            <option value="">-- Pilih --</option>
                            @foreach($levels as $opt)
                                <option value="{{ $opt['name'] }}" {{ old('job_level') == $opt['name'] ? 'selected' : '' }}>
                                    {{ $opt['name'] }}
                                </option>
                            @endforeach
                        </select>
                    @else
                        <input type="text" name="job_level" value="{{ old('job_level') }}"
                               class="w-full rounded-xl border @error('job_level') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2"
                               placeholder='mis. "Manager" (master job level kosong)'>
                        <p class="text-xs text-gray-500 mt-1">Master Job Level kosong.</p>
                    @endif
                    @error('job_level')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- Employment Status --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Employment Status <span class="text-red-600">*</span>
                    </label>
                    <select name="employment_status" id="employment_status"
                            class="w-full rounded-xl border @error('employment_status') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                        <option value="">-- Pilih --</option>
                        @foreach(($emps ?? []) as $opt)
                            <option value="{{ $opt['id'] }}" {{ old('employment_status') == $opt['id'] ? 'selected' : '' }}>
                                {{ $opt['name'] }} (ID: {{ $opt['id'] }})
                            </option>
                        @endforeach
                    </select>
                    @error('employment_status')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                    <p class="text-xs text-gray-500 mt-1">
                        <span class="font-medium">Contract (ID=2)</span> & <span class="font-medium">Probation (ID=3)</span> wajib isi <em>End Employment Status Date</em>.
                        Permanent (ID=1) tidak perlu.
                    </p>
                </div>

                {{-- End Employment Status Date --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">End Employment Status Date</label>
                    <input type="date" name="end_employment_status_date" id="end_employment_status_date"
                           value="{{ old('end_employment_status_date') }}"
                           class="w-full rounded-xl border @error('end_employment_status_date') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                    @error('end_employment_status_date')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- Join Date --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Join Date <span class="text-red-600">*</span>
                    </label>
                    <input type="date" name="join_date" value="{{ old('join_date') }}"
                           class="w-full rounded-xl border @error('join_date') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                    @error('join_date')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- Schedule --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        Schedule <span class="text-red-600">*</span>
                    </label>
                    <select name="schedule"
                            class="w-full rounded-xl border @error('schedule') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                        <option value="">-- Pilih --</option>
                        @foreach(($schedules ?? []) as $opt)
                            <option value="{{ $opt['name'] }}" {{ old('schedule') == $opt['name'] ? 'selected' : '' }}>
                                {{ $opt['name'] }}
                            </option>
                        @endforeach
                    </select>
                    @error('schedule')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>
            </div>
        </section>

        {{-- ================= KOMPENSASI & PAJAK ================= --}}
        <section class="bg-white rounded-2xl shadow-sm border border-gray-100">
            <div class="px-5 py-4 border-b border-gray-100">
                <h3 class="font-semibold text-gray-900">Kompensasi & Pajak</h3>
                <p class="text-sm text-gray-500">
                    Saat <strong>create</strong>, kamu <strong>tidak perlu</strong> mengisi gaji pokok.
                    Sistem akan mengirim payload tanpa field gaji & blok payroll yang diwajibkan,
                    lalu memastikan gaji awal tersimpan sebagai <strong>0</strong>. Kamu bisa mengisi nominal gaji
                    kapan saja nanti melalui menu <strong>edit</strong>.
                </p>
            </div>

            <div class="p-5 grid grid-cols-1 md:grid-cols-2 gap-5">
                {{-- PTKP (wajib) --}}
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">
                        PTKP Status <span class="text-red-600">*</span>
                    </label>
                    <select name="ptkp_status"
                            class="w-full rounded-xl border @error('ptkp_status') border-red-300 ring-2 ring-red-100 @else border-gray-300 @enderror px-3 py-2">
                        <option value="">-- Pilih --</option>
                        @foreach(($ptkps ?? []) as $opt)
                            <option value="{{ $opt['name'] }}" {{ old('ptkp_status') == $opt['name'] ? 'selected' : '' }}>
                                {{ $opt['name'] }}
                            </option>
                        @endforeach
                    </select>
                    @error('ptkp_status')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>

                {{-- NPP BPJS TK (opsional) --}}
                <div class="md:col-span-2">
                    <label class="block text-sm font-medium text-gray-700 mb-1">NPP BPJS Ketenagakerjaan</label>
                    <input type="text" name="npp_bpjs_ketenagakerjaan" value="{{ old('npp_bpjs_ketenagakerjaan') }}"
                           class="w-full rounded-xl border border-gray-300 px-3 py-2"
                           placeholder='kosongkan → controller kirim "default"'>
                    @error('npp_bpjs_ketenagakerjaan')
                        <p class="text-xs text-red-600 mt-1">{{ humanize_line($message) }}</p>
                    @enderror
                </div>
            </div>
        </section>

        {{-- ================= ACTIONS ================= --}}
        <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-4 md:sticky md:bottom-4">
            <div class="flex flex-col md:flex-row items-stretch md:items-center justify-between gap-3">
                <p class="text-sm text-gray-500">Pastikan data sudah benar sebelum disimpan.</p>
                <div class="flex items-center gap-3">
                    <a href="{{ route('talenta.sandbox.employees.index') }}"
                       class="inline-flex items-center justify-center px-4 py-2 rounded-xl border border-gray-200 hover:bg-gray-50">
                        Batal
                    </a>
                    <button type="submit"
                            class="inline-flex items-center justify-center px-4 py-2 rounded-xl text-white bg-primary hover:opacity-90">
                        Simpan Karyawan
                    </button>
                </div>
            </div>
        </div>
    </form>
</div>

{{-- SweetAlert2 + dynamic rule End Date (ID 2 & 3) --}}
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
    // SweetAlert: gunakan pesan yang sudah difriendly-kan
    @php
        // Siapkan HTML list untuk Swal (jika ada error friendly)
        $swalHtml = '';
        if (count($friendlyErrors)) {
            $swalHtml = '<ul style="text-align:left;margin:0;padding-left:16px">' .
                collect($friendlyErrors)->map(fn($e)=>'<li>'.e($e).'</li>')->implode('') .
                '</ul>';
        }
    @endphp

    @if(session('success'))
        Swal.fire({
            icon: 'success',
            title: 'Berhasil',
            text: @json(session('success')),
            timer: 2000,
            timerProgressBar: true,
            showConfirmButton: false
        });
    @endif

    @if(count($friendlyErrors))
        Swal.fire({
            icon: 'error',
            title: 'Form belum lengkap',
            html: `{!! $swalHtml !!}`
        });
    @elseif(session('error'))
        // Jika ada session error tapi tidak ter-parse ke friendly list
        Swal.fire({
            icon: 'error',
            title: 'Gagal',
            text: @json(\Illuminate\Support\Str::of((string)session('error'))->limit(180))
        });
    @endif

    // End date wajib jika Employment Status = 2 (Contract) atau 3 (Probation)
    const selStatus = document.getElementById('employment_status');
    const endDate   = document.getElementById('end_employment_status_date');
    function enforceEndDateReq() {
        if (!selStatus || !endDate) return;
        const val = selStatus.value;
        if (val === '2' || val === '3') {
            endDate.setAttribute('required','required');
        } else {
            endDate.removeAttribute('required');
        }
    }
    selStatus?.addEventListener('change', enforceEndDateReq);
    // initial on load (handle old() preselect)
    enforceEndDateReq();
</script>
@endsection
