<?php

namespace App\Http\Controllers\Hirbc;

use App\Http\Controllers\Controller;
use App\Services\BusinessCentral\HirbcApiService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ManageDataController extends Controller
{
    protected $apiService;

    public function __construct(HirbcApiService $apiService)
    {
        $this->apiService = $apiService;
    }

    /**
     * Show manage data page
     */
    public function index()
    {
        try {
            // Get statistics
            $stats = $this->getDataStatistics();
            
            return view('hirbc.manage-data', compact('stats'));
            
        } catch (\Exception $e) {
            Log::error('HIRBC Manage Data Page Error: ' . $e->getMessage());
            
            return view('hirbc.manage-data', [
                'stats' => $this->getEmptyStats()
            ]);
        }
    }

    /**
     * Refresh data from Business Central API
     */
    public function refreshData(Request $request)
    {
        // Set unlimited execution time
        set_time_limit(0);
        ini_set('max_execution_time', 0);
        ini_set('memory_limit', '512M');

        try {
            Log::info('HIRBC: Starting data refresh...');
            
            $year = date('Y');
            
            // Call API service to refresh all data
            $results = $this->apiService->refreshAllData($year);
            
            // Prepare response message
            $successMonths = [];
            $failedMonths = [];
            
            foreach ($results['months'] as $month => $data) {
                if ($data['success']) {
                    $successMonths[] = ucfirst($month) . ': ' . $data['records'] . ' records';
                } else {
                    $failedMonths[] = ucfirst($month) . ': ' . ($data['error'] ?? 'Unknown error');
                }
            }
            
            $message = 'Data refresh completed!<br>';
            $message .= '<strong>Total Records:</strong> ' . $results['total_records'] . '<br>';
            
            if (!empty($successMonths)) {
                $message .= '<strong>Success:</strong><br>' . implode('<br>', $successMonths) . '<br>';
            }
            
            if ($results['ooo']['success']) {
                $message .= '<strong>OOO:</strong> ' . $results['ooo']['records'] . ' records<br>';
            }
            
            if (!empty($failedMonths)) {
                $message .= '<strong>Failed:</strong><br>' . implode('<br>', $failedMonths);
            }
            
            if ($results['success']) {
                return response()->json([
                    'success' => true,
                    'message' => $message,
                    'stats' => $this->getDataStatistics()
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $message,
                    'stats' => $this->getDataStatistics()
                ], 500);
            }
            
        } catch (\Exception $e) {
            Log::error('HIRBC Data Refresh Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get data statistics
     */
    private function getDataStatistics()
    {
        $stats = [
            'total_records' => 0,
            'ooo_records' => 0,
            'months_loaded' => 0,
            'monthly_data' => []
        ];

        try {
            $months = [
                'january', 'february', 'march', 'april', 'may', 'june',
                'july', 'august', 'september', 'october', 'november', 'december'
            ];
            
            foreach ($months as $month) {
                $tableName = $month . '_hirbc';
                
                if (DB::getSchemaBuilder()->hasTable($tableName)) {
                    $count = DB::table($tableName)->count();
                    
                    $stats['monthly_data'][$month] = [
                        'count' => $count,
                        'has_data' => $count > 0
                    ];
                    
                    $stats['total_records'] += $count;
                    
                    if ($count > 0) {
                        $stats['months_loaded']++;
                    }
                } else {
                    $stats['monthly_data'][$month] = [
                        'count' => 0,
                        'has_data' => false
                    ];
                }
            }
            
            // Count OOO records
            if (DB::getSchemaBuilder()->hasTable('ooo_hirbc')) {
                $stats['ooo_records'] = DB::table('ooo_hirbc')
                    ->where('Completed_On', '0001-01-01')
                    ->count();
            }
            
        } catch (\Exception $e) {
            Log::error('HIRBC Stats Error: ' . $e->getMessage());
        }
        
        return $stats;
    }

    /**
     * Get empty stats structure
     */
    private function getEmptyStats()
    {
        return [
            'total_records' => 0,
            'ooo_records' => 0,
            'months_loaded' => 0,
            'monthly_data' => []
        ];
    }
}