<?php

namespace App\Http\Controllers\Regent;

use App\Http\Controllers\Controller;
use App\Models\Regent\OooRegent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReportController extends Controller
{
    /**
     * Show House Use report
     */
    public function houseUse(Request $request)
    {
        $month = $request->get('month', 'all');
        
        // Get data from database
        $houseUseData = $this->getHouseUseData($month);
        
        return view('regent.reports.house-use', compact('month', 'houseUseData'));
    }

    /**
     * Show Complimentary report
     */
    public function complimentary(Request $request)
    {
        $month = $request->get('month', 'all');
        
        // Get data from database
        $complimentaryData = $this->getComplimentaryData($month);
        
        return view('regent.reports.complimentary', compact('month', 'complimentaryData'));
    }

    /**
     * Show OOO report
     */
    public function ooo(Request $request)
    {
        $month = $request->get('month', 'all');
        
        // Get OOO data where not completed
        $oooData = OooRegent::where('Completed_On', '0001-01-01')
            ->orderBy('Room_No')
            ->get();
        
        // Calculate statistics
        $roomTypes = config('regent.room_types');
        
        // Count suite rooms (all rooms in suite submenu)
        $suiteRooms = [];
        if (isset($roomTypes['suite']['submenu'])) {
            foreach ($roomTypes['suite']['submenu'] as $submenu) {
                $suiteRooms = array_merge($suiteRooms, $submenu['rooms']);
            }
        }
        
        $stats = [
            'total' => $oooData->count(),
            'standard' => $oooData->whereNotIn('Room_No', $suiteRooms)->count(),
            'suite' => $oooData->whereIn('Room_No', $suiteRooms)->count(),
        ];
        
        return view('regent.reports.ooo', compact('oooData', 'stats', 'month'));
    }

    /**
 * Show OOO Room Detail Matrix
 */
public function oooRoomDetail(Request $request, $room)
{
    try {
        // Get OOO data for this room
        $oooData = DB::table('ooo_regent')
            ->where('Room_No', $room)
            ->where('Completed_On', '0001-01-01')
            ->get();
        
        if ($oooData->isEmpty()) {
            return redirect()->route('regent.reports.ooo')
                ->with('error', "No OOO data found for Room {$room}");
        }
        
        return view('regent.reports.ooo-room-detail', compact('room', 'oooData'));
        
    } catch (\Exception $e) {
        Log::error('REGENT OOO Room Detail Error: ' . $e->getMessage());
        
        return redirect()->route('regent.reports.ooo')
            ->with('error', 'Error loading room detail');
    }
}

    /**
     * Show Revenue report
     */
    public function revenue(Request $request)
    {
        $month = $request->get('month', 'all');
        
        // Get revenue data from database
        $revenueData = $this->getRevenueData($month);
        
        return view('regent.reports.revenue', compact('month', 'revenueData'));
    }

    /**
     * Show House Use detail for specific room
     */
    public function houseUseRoomDetail(Request $request, $room)
    {
        $month = $request->get('month', 'all');
        $marketCode = 'House Use - H';
        
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];
        
        $roomData = [];
        
        if ($month === 'all') {
            // All months: get DISTINCT dates per month
            foreach ($months as $monthName) {
                $tableName = "{$monthName}_regent";
                
                try {
                    $data = DB::table($tableName)
                        ->where('Market_Code', $marketCode)
                        ->where('Room', $room)
                        ->selectRaw('DISTINCT DATE(Posting_Date) as posting_date')
                        ->get();
                    
                    $roomData[$monthName] = $data;
                    
                    Log::info("REGENT: House Use Room Detail", [
                        'room' => $room,
                        'month' => $monthName,
                        'days' => $data->count()
                    ]);
                } catch (\Exception $e) {
                    Log::error("REGENT: Error fetching house use detail: " . $e->getMessage());
                    $roomData[$monthName] = collect([]);
                }
            }
        } else {
            // Single month: get DISTINCT daily data
            $tableName = "{$month}_regent";
            
            try {
                $roomData = DB::table($tableName)
                    ->where('Market_Code', $marketCode)
                    ->where('Room', $room)
                    ->selectRaw('DISTINCT DATE(Posting_Date) as posting_date')
                    ->get();
                
                Log::info("REGENT: House Use Room Detail (Single Month)", [
                    'room' => $room,
                    'month' => $month,
                    'days' => $roomData->count()
                ]);
            } catch (\Exception $e) {
                Log::error("REGENT: Error fetching house use detail: " . $e->getMessage());
                $roomData = collect([]);
            }
        }
        
        return view('regent.reports.house-use-room-detail', compact('room', 'month', 'roomData'));
    }

    /**
     * Show Complimentary detail for specific room
     */
    public function complimentaryRoomDetail(Request $request, $room)
    {
        $month = $request->get('month', 'all');
        $marketCode = 'Complimentary - N';
        
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];
        
        $roomData = [];
        
        if ($month === 'all') {
            // All months: get DISTINCT dates per month
            foreach ($months as $monthName) {
                $tableName = "{$monthName}_regent";
                
                try {
                    $data = DB::table($tableName)
                        ->where('Market_Code', $marketCode)
                        ->where('Room', $room)
                        ->selectRaw('DISTINCT DATE(Posting_Date) as posting_date')
                        ->get();
                    
                    $roomData[$monthName] = $data;
                    
                    Log::info("REGENT: Complimentary Room Detail", [
                        'room' => $room,
                        'month' => $monthName,
                        'days' => $data->count()
                    ]);
                } catch (\Exception $e) {
                    Log::error("REGENT: Error fetching complimentary detail: " . $e->getMessage());
                    $roomData[$monthName] = collect([]);
                }
            }
        } else {
            // Single month: get DISTINCT daily data
            $tableName = "{$month}_regent";
            
            try {
                $roomData = DB::table($tableName)
                    ->where('Market_Code', $marketCode)
                    ->where('Room', $room)
                    ->selectRaw('DISTINCT DATE(Posting_Date) as posting_date')
                    ->get();
                
                Log::info("REGENT: Complimentary Room Detail (Single Month)", [
                    'room' => $room,
                    'month' => $month,
                    'days' => $roomData->count()
                ]);
            } catch (\Exception $e) {
                Log::error("REGENT: Error fetching complimentary detail: " . $e->getMessage());
                $roomData = collect([]);
            }
        }
        
        return view('regent.reports.complimentary-room-detail', compact('room', 'month', 'roomData'));
    }

    /**
     * Get House Use data from database
     */
    private function getHouseUseData($month)
    {
        $marketCode = 'House Use - H';
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];
        
        if ($month === 'all') {
            // Get data from all 12 months
            $allData = [];
            
            foreach ($months as $monthName) {
                $tableName = "{$monthName}_regent";
                
                try {
                    $data = DB::table($tableName)
                        ->where('Market_Code', $marketCode)
                        ->select('Room', 'Posting_Date', 'RoomType', 'Market_Code')
                        ->get();
                    
                    $allData[$monthName] = $data;
                } catch (\Exception $e) {
                    Log::error("REGENT: Error fetching House Use data from {$tableName}: " . $e->getMessage());
                    $allData[$monthName] = collect([]);
                }
            }
            
            return $allData;
        } else {
            // Get data from specific month
            $tableName = "{$month}_regent";
            
            try {
                return DB::table($tableName)
                    ->where('Market_Code', $marketCode)
                    ->select('Room', 'Posting_Date', 'RoomType', 'Market_Code')
                    ->get();
            } catch (\Exception $e) {
                Log::error("REGENT: Error fetching House Use data from {$tableName}: " . $e->getMessage());
                return collect([]);
            }
        }
    }

    /**
     * Get Complimentary data from database
     */
    private function getComplimentaryData($month)
    {
        $marketCode = 'Complimentary - N';
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];
        
        if ($month === 'all') {
            $allData = [];
            
            foreach ($months as $monthName) {
                $tableName = "{$monthName}_regent";
                
                try {
                    $data = DB::table($tableName)
                        ->where('Market_Code', $marketCode)
                        ->select('Room', 'Posting_Date', 'RoomType', 'Market_Code')
                        ->get();
                    
                    $allData[$monthName] = $data;
                } catch (\Exception $e) {
                    Log::error("REGENT: Error fetching Complimentary data from {$tableName}: " . $e->getMessage());
                    $allData[$monthName] = collect([]);
                }
            }
            
            return $allData;
        } else {
            $tableName = "{$month}_regent";
            
            try {
                return DB::table($tableName)
                    ->where('Market_Code', $marketCode)
                    ->select('Room', 'Posting_Date', 'RoomType', 'Market_Code')
                    ->get();
            } catch (\Exception $e) {
                Log::error("REGENT: Error fetching Complimentary data from {$tableName}: " . $e->getMessage());
                return collect([]);
            }
        }
    }

    /**
 * Get Revenue data from database
 */
private function getRevenueData($month)
{
    // ✅ HAPUS filter TRX_Code - ambil SEMUA data!
    $excludeMarketCodes = ['House Use - H', 'Complimentary - N'];
    
    $months = [
        'january', 'february', 'march', 'april', 'may', 'june',
        'july', 'august', 'september', 'october', 'november', 'december'
    ];
    
    if ($month === 'all') {
        $allData = [];
        
        foreach ($months as $monthName) {
            $tableName = "{$monthName}_regent";
            
            try {
                $data = DB::table($tableName)
                    // ✅ HAPUS: ->whereIn('TRX_Code', $revenueTrxCodes)
                    ->whereNotIn('Market_Code', $excludeMarketCodes)
                    ->where('Amount_1', '>', 0)
                    ->select('Room', 'RoomType', 'Posting_Date', 'Amount_1', 'TRX_Code', 'TRX_Desc')
                    ->get();
                
                $allData[$monthName] = $data;
            } catch (\Exception $e) {
                Log::error("REGENT: Error fetching Revenue data from {$tableName}: " . $e->getMessage());
                $allData[$monthName] = collect([]);
            }
        }
        
        return $allData;
    } else {
        $tableName = "{$month}_regent";
        
        try {
            return DB::table($tableName)
                // ✅ HAPUS: ->whereIn('TRX_Code', $revenueTrxCodes)
                ->whereNotIn('Market_Code', $excludeMarketCodes)
                ->where('Amount_1', '>', 0)
                ->select('Room', 'RoomType', 'Posting_Date', 'Amount_1', 'TRX_Code', 'TRX_Desc')
                ->get();
        } catch (\Exception $e) {
            Log::error("REGENT: Error fetching Revenue data from {$tableName}: " . $e->getMessage());
            return collect([]);
        }
    }
}
}