<?php

namespace App\Services\BusinessCentral;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class HirbndApiService extends BaseApiService
{
    protected $roomEndpoint;
    protected $oooEndpoint;
    protected $batchSizeRoom;
    protected $batchSizeOoo;

    public function __construct()
    {
        parent::__construct();
        $this->roomEndpoint = config('businesscentral.hirbnd.endpoints.room');
        $this->oooEndpoint = config('businesscentral.hirbnd.endpoints.ooo');
        $this->batchSizeRoom = config('businesscentral.batch_size.room_data');
        $this->batchSizeOoo = config('businesscentral.batch_size.ooo_data');
    }
/**
     * 🧪 TEST: Fetch sample data tanpa filter
     */
    public function testEndpoint(): array
    {
        try {
            Log::info('HIRBND TEST: Testing Room endpoint without filter...');

            // Test tanpa filter, ambil 5 data aja
            $endpoint = $this->roomEndpoint . "?\$top=5";

            Log::info('HIRBND TEST: Endpoint', ['url' => $endpoint]);

            $data = $this->get($endpoint);

            Log::info('HIRBND TEST: Success!', [
                'records' => count($data),
                'sample' => $data[0] ?? null
            ]);

            return $data;

        } catch (Exception $e) {
            Log::error('HIRBND TEST: Failed!', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }
    /**
 * Fetch room data dengan filter Posting_Date per bulan
 */
public function fetchRoomData(string $month, int $year): array
{
    try {
        Log::info("HIRBND: Starting fetch for {$month} {$year}");
        
        $monthNumber = str_pad(date('m', strtotime($month)), 2, '0', STR_PAD_LEFT);
        $startDate = "{$year}-{$monthNumber}-01";
        
        if ($monthNumber == '12') {
            $endDate = ($year + 1) . "-01-01";
        } else {
            $nextMonth = str_pad($monthNumber + 1, 2, '0', STR_PAD_LEFT);
            $endDate = "{$year}-{$nextMonth}-01";
        }

        // ✅ FORMAT YANG BERHASIL: Tanpa quote (dari test yang sukses)
        $filter = "Posting_Date ge {$startDate} and Posting_Date lt {$endDate}";
        $filterQuery = "\$filter={$filter}";
        $endpoint = $this->roomEndpoint . "?" . $filterQuery;

        Log::info("HIRBND: Fetching with filter", [
            'month' => $month,
            'year' => $year,
            'filter' => $filter,
            'date_range' => "{$startDate} to {$endDate}"
        ]);

        $startTime = microtime(true);
        
        // ✅ FIX: Get response and extract 'value' key
        $response = $this->get($endpoint);
        $data = $response['value'] ?? $response;
        
        $duration = round(microtime(true) - $startTime, 2);

        Log::info("HIRBND: ✅ Fetch completed", [
            'month' => $month,
            'year' => $year,
            'records' => count($data),
            'duration' => $duration . 's'
        ]);

        return $data;

    } catch (Exception $e) {
        Log::error("HIRBND: ❌ Fetch failed for {$month} {$year}", [
            'error' => $e->getMessage()
        ]);
        throw $e;
    }
}

    /**
     * Fetch all months data untuk 1 tahun
     */
    public function fetchAllMonthsData(int $year): array
    {
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];
        
        $allData = [];

        foreach ($months as $month) {
            try {
                $allData[$month] = $this->fetchRoomData($month, $year);
                
                // Log setiap bulan
                Log::info("HIRBND: Month summary", [
                    'month' => $month,
                    'records' => count($allData[$month])
                ]);
                
            } catch (Exception $e) {
                Log::error("HIRBND: Failed for {$month}", ['error' => $e->getMessage()]);
                $allData[$month] = [];
            }
        }

        return $allData;
    }

    /**
 * Fetch OOO data - HANYA yang Completed_On = '0001-01-01'
 */
public function fetchOOOData(): array
{
    try {
        // ✅ FORMAT YANG BERHASIL (dari test)
        $filter = "Completed_On eq 0001-01-01";
        $filterQuery = "\$filter={$filter}";
        $endpoint = $this->oooEndpoint . "?" . $filterQuery;

        Log::info('HIRBND: Fetching OOO data', [
            'filter' => $filter,
            'expected' => '~182 records'
        ]);

        $startTime = microtime(true);
        
        // ✅ FIX: Get response and extract 'value' key
        $response = $this->get($endpoint);
        $data = $response['value'] ?? $response;
        
        $duration = round(microtime(true) - $startTime, 2);

        Log::info('HIRBND: ✅ OOO data fetched', [
            'records' => count($data),
            'expected' => 182,
            'match' => abs(count($data) - 182) <= 5 ? 'GOOD' : 'CHECK',
            'duration' => $duration . 's'
        ]);

        return $data;

    } catch (Exception $e) {
        Log::error('HIRBND: ❌ Failed to fetch OOO data', [
            'error' => $e->getMessage()
        ]);
        throw $e;
    }
}

    /**
     * Remove duplicate records berdasarkan key unik
     */
    public function removeDuplicates(array $data): array
    {
        $unique = [];
        $duplicateCount = 0;

        foreach ($data as $record) {
            // Buat unique key dari kombinasi Room + Posting_Date + TRX_Code + Row_No
            $key = ($record['Room'] ?? '') . '_' . 
                   ($record['Posting_Date'] ?? '') . '_' . 
                   ($record['TRX_Code'] ?? '') . '_' . 
                   ($record['Row_No'] ?? '');

            if (!isset($unique[$key])) {
                $unique[$key] = $record;
            } else {
                $duplicateCount++;
            }
        }

        if ($duplicateCount > 0) {
            Log::warning("HIRBND: Removed {$duplicateCount} duplicate records");
        }

        return array_values($unique);
    }

    /**
 * Save room data ke monthly table
 */
public function saveRoomData(array $data, string $month): bool
{
    try {
        $tableName = config("businesscentral.hirbnd.tables.monthly.{$month}");

        if (!$tableName) {
            throw new Exception("Invalid month: {$month}");
        }

        // Remove duplicates
        $originalCount = count($data);
        $data = $this->removeDuplicates($data);
        $cleanCount = count($data);

        if ($originalCount != $cleanCount) {
            Log::info("HIRBND: Duplicate removal", [
                'month' => $month,
                'original' => $originalCount,
                'clean' => $cleanCount,
                'removed' => ($originalCount - $cleanCount)
            ]);
        }

        if (empty($data)) {
            Log::warning("HIRBND: No data to save for {$month}");
            return true;
        }

        DB::beginTransaction();

        // ✅ FIX: Use delete() instead of truncate()
        DB::table($tableName)->delete();
        Log::info("HIRBND: Table truncated", ['table' => $tableName]);

        // Insert in chunks
        $chunks = array_chunk($data, $this->batchSizeRoom);
        $totalInserted = 0;

        foreach ($chunks as $chunkIndex => $chunk) {
            $insertData = [];

            foreach ($chunk as $record) {
                $insertData[] = [
                    'odata_etag' => $record['@odata.etag'] ?? null,
                    'Amount_1' => $record['Amount_1'] ?? 0,  // ✅ FIX: Default 0 instead of null
                    'Business_Unit' => $record['Business_Unit'] ?? null,
                    'Guest_Full_Name' => $record['Guest_Full_Name'] ?? null,
                    'Interval' => $record['Interval'] ?? null,
                    'Market_Code' => $record['Market_Code'] ?? null,
                    'No_Asset' => $record['No_Asset'] ?? null,
                    'Posting_Date' => $record['Posting_Date'] ?? null,
                    'Room' => $record['Room'] ?? null,
                    'RoomType' => $record['RoomType'] ?? null,
                    'Row_No' => $record['Row_No'] ?? null,
                    'TRX_Code' => $record['TRX_Code'] ?? null,
                    'TRX_Desc' => $record['TRX_Desc'] ?? null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }

            DB::table($tableName)->insert($insertData);
            $totalInserted += count($insertData);

            Log::info("HIRBND: Chunk saved", [
                'month' => $month,
                'chunk' => $chunkIndex + 1,
                'records' => count($insertData),
                'total_so_far' => $totalInserted
            ]);
        }

        DB::commit();

        Log::info("HIRBND: ✅ Data saved successfully", [
            'month' => $month,
            'table' => $tableName,
            'total_records' => $totalInserted
        ]);

        return true;

    } catch (Exception $e) {
        if (DB::transactionLevel() > 0) {
            DB::rollBack();
        }
        
        Log::error("HIRBND: ❌ Save failed for {$month}", [
            'error' => $e->getMessage()
        ]);
        
        throw $e;
    }
}

    /**
 * Save OOO data ke table ooo_hirbnd
 * SESUAI STRUKTUR CSV YANG BENAR!
 */
public function saveOOOData(array $data): bool
{
    try {
        $tableName = config('businesscentral.hirbnd.tables.ooo');

        if (empty($data)) {
            Log::warning('HIRBND: No OOO data to save');
            return true;
        }

        // Validasi: harus 182 records!
        if (count($data) != 182) {
            Log::warning('HIRBND: OOO data count mismatch!', [
                'expected' => 182,
                'actual' => count($data)
            ]);
        }

        DB::beginTransaction();

        // ✅ FIX: Use delete() instead of truncate()
        DB::table($tableName)->delete();
        Log::info('HIRBND: OOO table truncated', ['table' => $tableName]);

        // Insert in chunks
        $chunks = array_chunk($data, $this->batchSizeOoo);
        $totalInserted = 0;

        foreach ($chunks as $chunkIndex => $chunk) {
            $insertData = [];

            foreach ($chunk as $record) {
                $insertData[] = [
                    'odata_etag' => $record['@odata.etag'] ?? null,
                    'AuxiliaryIndex1' => $record['AuxiliaryIndex1'] ?? null,
                    'Begin_Date' => $record['Begin_Date'] ?? null,
                    'Business_Unit' => $record['Business_Unit'] ?? null,
                    'Completed_On' => $record['Completed_On'] ?? null,
                    'End_Date' => $record['End_Date'] ?? null,
                    'Label' => $record['Label'] ?? null,
                    'Reason_Code' => $record['Reason_Code'] ?? null,
                    'Reason_Desc' => $record['Reason_Desc'] ?? null,
                    'Repair_Remark' => $record['Repair_Remark'] ?? null,
                    'Return_Status' => $record['Return_Status'] ?? null,
                    'Room_Class' => $record['Room_Class'] ?? null,
                    'Room_No' => $record['Room_No'] ?? null,
                    'Room_Status' => $record['Room_Status'] ?? null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }

            DB::table($tableName)->insert($insertData);
            $totalInserted += count($insertData);

            Log::info('HIRBND: OOO chunk saved', [
                'chunk' => $chunkIndex + 1,
                'records' => count($insertData),
                'total_so_far' => $totalInserted
            ]);
        }

        DB::commit();

        Log::info('HIRBND: ✅ OOO data saved successfully', [
            'table' => $tableName,
            'total_records' => $totalInserted,
            'expected' => 182
        ]);

        // Final validation
        if ($totalInserted == 182) {
            Log::info('HIRBND: ✅✅ OOO data count PERFECT! (182 records)');
        } else {
            Log::error('HIRBND: ❌ OOO data count WRONG!', [
                'expected' => 182,
                'actual' => $totalInserted
            ]);
        }

        return true;

    } catch (Exception $e) {
        if (DB::transactionLevel() > 0) {
            DB::rollBack();
        }
        
        Log::error('HIRBND: ❌ OOO save failed', [
            'error' => $e->getMessage()
        ]);
        
        throw $e;
    }
}

    /**
     * Load semua data (Room + OOO)
     */
    public function loadAllData(int $year = null): array
    {
        $year = $year ?? date('Y');
        $startTime = microtime(true);

        try {
            Log::info('HIRBND: ========================================');
            Log::info('HIRBND: 🚀 STARTING FULL DATA LOAD');
            Log::info('HIRBND: ========================================');
            Log::info('HIRBND: Year: ' . $year);

            $statistics = [
                'success' => false,
                'year' => $year,
                'room_data' => [],
                'ooo_data' => 0,
                'total_records' => 0,
                'duration' => 0,
                'errors' => [],
            ];

            // Fetch & Save Room Data per bulan
            Log::info('HIRBND: 📊 Fetching room data for all months...');
            $allMonthsData = $this->fetchAllMonthsData($year);

            foreach ($allMonthsData as $month => $data) {
                try {
                    $this->saveRoomData($data, $month);
                    $statistics['room_data'][$month] = count($data);
                    $statistics['total_records'] += count($data);
                } catch (Exception $e) {
                    $statistics['errors'][$month] = $e->getMessage();
                    Log::error("HIRBND: ❌ Failed to save {$month}", [
                        'error' => $e->getMessage()
                    ]);
                }
            }

            // Fetch & Save OOO Data
            Log::info('HIRBND: 🔴 Fetching OOO data (Completed_On = 0001-01-01)...');
            try {
                $oooData = $this->fetchOOOData();
                $this->saveOOOData($oooData);
                $statistics['ooo_data'] = count($oooData);
                $statistics['total_records'] += count($oooData);
            } catch (Exception $e) {
                $statistics['errors']['ooo'] = $e->getMessage();
                Log::error('HIRBND: ❌ Failed to save OOO data', [
                    'error' => $e->getMessage()
                ]);
            }

            $statistics['duration'] = round(microtime(true) - $startTime, 2);
            $statistics['success'] = empty($statistics['errors']);

            Log::info('HIRBND: ========================================');
            Log::info('HIRBND: ✅ LOAD COMPLETED');
            Log::info('HIRBND: ========================================');
            Log::info('HIRBND: Summary:', $statistics);

            return $statistics;

        } catch (Exception $e) {
            Log::error('HIRBND: ========================================');
            Log::error('HIRBND: ❌❌ LOAD FAILED');
            Log::error('HIRBND: ========================================');
            Log::error('HIRBND: Error: ' . $e->getMessage());
            Log::error('HIRBND: Trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }
}