@extends('layouts.regent')

@section('title', 'Manage Data')

@push('styles')
<style>
    .page-container {
        max-width: 1400px;
        margin: 0 auto;
    }

    .header-card {
        background: linear-gradient(135deg, #00897b, #26a69a);
        border-radius: 16px;
        padding: 40px;
        margin-bottom: 32px;
        color: white;
        box-shadow: 0 4px 20px rgba(0, 137, 123, 0.3);
    }

    .header-card h1 {
        font-size: 36px;
        font-weight: 700;
        margin-bottom: 8px;
    }

    .header-card p {
        font-size: 16px;
        opacity: 0.95;
    }

    .info-banner {
        background: #fff9e6;
        border-left: 4px solid #ffa726;
        padding: 20px 24px;
        border-radius: 8px;
        margin-bottom: 32px;
        display: flex;
        align-items: start;
        gap: 16px;
    }

    .info-banner i {
        font-size: 24px;
        color: #ff8f00;
        margin-top: 2px;
    }

    .info-banner-content {
        flex: 1;
    }

    .info-banner-content strong {
        display: block;
        font-size: 15px;
        margin-bottom: 8px;
        color: #e65100;
    }

    .info-banner-content p {
        font-size: 13px;
        color: #666;
        line-height: 1.6;
        margin: 0;
    }

    .status-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 24px;
        margin-bottom: 32px;
    }

    .status-card {
        background: white;
        border-radius: 12px;
        padding: 28px;
        box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
        border-left: 4px solid #00897b;
        transition: all 0.3s;
    }

    .status-card:hover {
        transform: translateY(-4px);
        box-shadow: 0 4px 20px rgba(0, 137, 123, 0.15);
    }

    .status-card-header {
        display: flex;
        align-items: center;
        gap: 16px;
        margin-bottom: 20px;
    }

    .status-icon {
        width: 56px;
        height: 56px;
        background: linear-gradient(135deg, #00897b, #26a69a);
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 24px;
    }

    .status-info h3 {
        font-size: 14px;
        color: #666;
        font-weight: 600;
        margin-bottom: 4px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .status-value {
        font-size: 32px;
        font-weight: 700;
        color: #00897b;
    }

    .action-section {
        background: white;
        border-radius: 12px;
        padding: 32px;
        box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
        margin-bottom: 32px;
    }

    .action-section h2 {
        font-size: 20px;
        font-weight: 700;
        color: #00897b;
        margin-bottom: 24px;
        display: flex;
        align-items: center;
        gap: 12px;
    }

    .action-section h2 i {
        font-size: 24px;
    }

    .btn-refresh {
        background: linear-gradient(135deg, #00897b, #26a69a);
        color: white;
        border: none;
        padding: 16px 40px;
        border-radius: 10px;
        font-size: 16px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s;
        display: inline-flex;
        align-items: center;
        gap: 12px;
        box-shadow: 0 4px 12px rgba(0, 137, 123, 0.3);
    }

    .btn-refresh:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(0, 137, 123, 0.4);
    }

    .btn-refresh:disabled {
        opacity: 0.6;
        cursor: not-allowed;
        transform: none;
    }

    .btn-refresh i {
        font-size: 18px;
    }

    .monthly-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
        gap: 20px;
        margin-top: 28px;
    }

    .month-card {
        background: #f8f9fa;
        border-radius: 10px;
        padding: 20px;
        text-align: center;
        transition: all 0.3s;
        border: 2px solid transparent;
    }

    .month-card.has-data {
        background: #e0f2f1;
        border-color: #00897b;
    }

    .month-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    }

    .month-name {
        font-size: 16px;
        font-weight: 600;
        color: #333;
        margin-bottom: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
    }

    .month-name i {
        color: #00897b;
    }

    .month-count {
        font-size: 28px;
        font-weight: 700;
        color: #00897b;
    }

    .month-card.no-data .month-count {
        color: #999;
        font-size: 18px;
    }

    .loading-overlay {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.7);
        z-index: 9999;
        align-items: center;
        justify-content: center;
    }

    .loading-overlay.active {
        display: flex;
    }

    .loading-content {
        background: white;
        border-radius: 16px;
        padding: 48px;
        text-align: center;
        max-width: 480px;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
    }

    .spinner {
        border: 4px solid #f3f3f3;
        border-top: 4px solid #00897b;
        border-radius: 50%;
        width: 64px;
        height: 64px;
        animation: spin 1s linear infinite;
        margin: 0 auto 24px;
    }

    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }

    .loading-content h3 {
        font-size: 22px;
        color: #00897b;
        margin-bottom: 12px;
        font-weight: 700;
    }

    .loading-content p {
        color: #666;
        font-size: 14px;
        margin: 0;
    }

    .alert {
        padding: 16px 20px;
        border-radius: 10px;
        margin-bottom: 24px;
        display: flex;
        align-items: center;
        gap: 12px;
        font-size: 14px;
    }

    .alert i {
        font-size: 20px;
    }

    .alert-success {
        background: #d4edda;
        color: #155724;
        border-left: 4px solid #28a745;
    }

    .alert-error {
        background: #f8d7da;
        color: #721c24;
        border-left: 4px solid #dc3545;
    }

    .data-source-info {
        background: white;
        border-radius: 12px;
        padding: 24px;
        box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
        margin-top: 32px;
    }

    .data-source-info h3 {
        font-size: 16px;
        font-weight: 700;
        color: #00897b;
        margin-bottom: 16px;
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .source-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 16px;
    }

    .source-item {
        display: flex;
        align-items: center;
        gap: 12px;
        padding: 12px;
        background: #f8f9fa;
        border-radius: 8px;
    }

    .source-item i {
        font-size: 20px;
        color: #00897b;
    }

    .source-item div {
        flex: 1;
    }

    .source-label {
        font-size: 11px;
        color: #999;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .source-value {
        font-size: 14px;
        font-weight: 600;
        color: #333;
    }
</style>
@endpush

@section('content')
<div class="page-container">
    <!-- Header -->
    <div class="header-card">
        <h1><i class="fas fa-database"></i> Manage Data</h1>
        <p>Monitor and refresh your room monitoring data from Business Central API</p>
    </div>

    <!-- Alert Container -->
    <div id="alertContainer"></div>

    <!-- Info Banner -->
    <div class="info-banner">
        <i class="fas fa-info-circle"></i>
        <div class="info-banner-content">
            <strong>⚠️ Important Information</strong>
            <p><strong>Caution:</strong> Refreshing data will fetch the latest information from Business Central API. This process may take 2-5 minutes depending on data volume. Please do not close this page during the refresh process.</p>
        </div>
    </div>

    <!-- Status Grid -->
    <div class="status-grid">
        <div class="status-card">
            <div class="status-card-header">
                <div class="status-icon">
                    <i class="fas fa-database"></i>
                </div>
                <div class="status-info">
                    <h3>Total Records</h3>
                    <div class="status-value">{{ number_format($totalRecords) }}</div>
                </div>
            </div>
        </div>

        <div class="status-card">
            <div class="status-card-header">
                <div class="status-icon">
                    <i class="fas fa-tools"></i>
                </div>
                <div class="status-info">
                    <h3>OOO Records</h3>
                    <div class="status-value">{{ number_format($oooCount) }}</div>
                </div>
            </div>
        </div>

        <div class="status-card">
            <div class="status-card-header">
                <div class="status-icon">
                    <i class="fas fa-calendar-check"></i>
                </div>
                <div class="status-info">
                    <h3>Months Loaded</h3>
                    @php
                        $monthsWithData = collect($monthlyData)->filter(fn($m) => $m['has_data'])->count();
                    @endphp
                    <div class="status-value">{{ $monthsWithData }}/12</div>
                </div>
            </div>
        </div>
    </div>

    <!-- Action Section -->
    <div class="action-section">
        <h2>
            <i class="fas fa-sync-alt"></i>
            Refresh Data from Business Central
        </h2>
        <p style="color: #666; margin-bottom: 24px; font-size: 14px;">
            Click the button below to fetch the latest room monitoring data from Microsoft Dynamics 365 Business Central.<br>
            This will update all 12 monthly tables and OOO (Out of Order) data.
        </p>
        
        <button id="refreshBtn" class="btn-refresh" onclick="refreshData()">
            <i class="fas fa-sync-alt"></i>
            <span>Refresh Data Now</span>
        </button>

        <!-- Monthly Data Details -->
        <h3 style="font-size: 16px; font-weight: 700; color: #00897b; margin-top: 40px; margin-bottom: 20px;">
            <i class="fas fa-calendar-alt"></i> Monthly Data Details
        </h3>
        
        <div class="monthly-grid">
            @foreach(['January' => 'january', 'February' => 'february', 'March' => 'march', 'April' => 'april', 'May' => 'may', 'June' => 'june', 'July' => 'july', 'August' => 'august', 'September' => 'september', 'October' => 'october', 'November' => 'november', 'December' => 'december'] as $label => $month)
                <div class="month-card {{ $monthlyData[$month]['has_data'] ? 'has-data' : 'no-data' }}">
                    <div class="month-name">
                        <i class="fas fa-calendar"></i>
                        {{ $label }}
                    </div>
                    <div class="month-count">
                        {{ $monthlyData[$month]['count'] > 0 ? number_format($monthlyData[$month]['count']) : 'No Data' }}
                    </div>
                </div>
            @endforeach
        </div>
    </div>

    <!-- Data Source Info -->
    <div class="data-source-info">
        <h3>
            <i class="fas fa-server"></i>
            Data Source Information
        </h3>
        <div class="source-grid">
            <div class="source-item">
                <i class="fas fa-building"></i>
                <div>
                    <div class="source-label">Business Unit</div>
                    <div class="source-value">RBC</div>
                </div>
            </div>
            <div class="source-item">
                <i class="fas fa-table"></i>
                <div>
                    <div class="source-label">Monthly Tables</div>
                    <div class="source-value">12 Tables</div>
                </div>
            </div>
            <div class="source-item">
                <i class="fas fa-door-open"></i>
                <div>
                    <div class="source-label">Room Types</div>
                    <div class="source-value">5 Categories</div>
                </div>
            </div>
            <div class="source-item">
                <i class="fas fa-cloud"></i>
                <div>
                    <div class="source-label">Data Source</div>
                    <div class="source-value">BC API</div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Loading Overlay -->
<div id="loadingOverlay" class="loading-overlay">
    <div class="loading-content">
        <div class="spinner"></div>
        <h3>Refreshing Data...</h3>
        <p>Please wait while we fetch data from Business Central API</p>
        <p style="color: #999; font-size: 12px; margin-top: 12px;">This may take 2-5 minutes</p>
    </div>
</div>
@endsection

@push('scripts')
<script>
    function refreshData() {
        const btn = document.getElementById('refreshBtn');
        const loading = document.getElementById('loadingOverlay');
        const alertContainer = document.getElementById('alertContainer');
        
        // Disable button and show loading
        btn.disabled = true;
        loading.classList.add('active');
        alertContainer.innerHTML = '';
        
        // Make AJAX request
        fetch('{{ route("regent.manage-data.refresh") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            }
        })
        .then(response => response.json())
        .then(data => {
            loading.classList.remove('active');
            btn.disabled = false;
            
            if (data.success) {
                alertContainer.innerHTML = `
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle"></i>
                        <div>
                            <strong>${data.message}</strong><br>
                            <small>Total records: ${data.total_records.toLocaleString()}</small>
                        </div>
                    </div>
                `;
                
                // Reload page after 2 seconds
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            } else {
                alertContainer.innerHTML = `
                    <div class="alert alert-error">
                        <i class="fas fa-exclamation-circle"></i>
                        <div><strong>${data.message}</strong></div>
                    </div>
                `;
            }
        })
        .catch(error => {
            loading.classList.remove('active');
            btn.disabled = false;
            
            alertContainer.innerHTML = `
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <div><strong>Failed to refresh data. Please try again.</strong></div>
                </div>
            `;
        });
    }
</script>
@endpush