<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use GuzzleHttp\Promise;
use GuzzleHttp\Promise\Create;
use GuzzleHttp\Promise\Utils;
use Illuminate\Support\Str;
use GuzzleHttp\Exception\RequestException;

class BusinessCentralService
{
    public function getAccessToken()
    {
        $client = new Client();
        $response = $client->post("https://login.microsoftonline.com/".env('AZURE_TENANT_ID')."/oauth2/v2.0/token", [
            'form_params' => [
                'grant_type' => 'client_credentials',
                'client_id' => env('AZURE_CLIENT_ID'),
                'client_secret' => env('AZURE_CLIENT_SECRET'),
                'scope' => 'https://api.businesscentral.dynamics.com/.default'
            ]
        ]);
        $body = json_decode((string)$response->getBody(), true);

        return $body['access_token'];
    }

    public function getCompanyId($token)
    {
        $client = new Client();
        $response = $client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies", [
            'headers' => [
                'Authorization' => "Bearer {$token}",
                'Accept'        => 'application/json'
            ]
        ]);
        $companies = json_decode((string)$response->getBody(), true);
        $role = session('user')['role'] ?? null; 
        $companyId1 = $companies['value'][0]['id'];
        $companyId2 = $companies['value'][1]['id'];

        return [
            'companyId1' => $companyId1,
            'companyId2' => $companyId2
        ];
    }

    public function __construct()
    {
     $this->token = $this->getAccessToken();
     $companyIds = $this->getCompanyId($this->token);
     $this->companyRegent = $companyIds['companyId1'];
     $this->companyHIN    = $companyIds['companyId2'];
     
     $activeCompany = session('current_company_name') ?? session('user')['role'];
    if ($activeCompany === 'Regent' or $activeCompany === 'SUPER') {
        $this->companyId = $this->companyRegent;
    } elseif ($activeCompany === 'HIN' ) {
        $this->companyId = $this->companyHIN;
    }
     $this->client = new Client();
    }

    function getMetadata() {

        $customersResponse = $this->client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/\$metadata", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ]);
        $xml = simplexml_load_string((string)$customersResponse->getBody());
        $xml->registerXPathNamespace('edmx', 'http://docs.oasis-open.org/odata/ns/edmx'); 
        $xml->registerXPathNamespace('edm', 'http://docs.oasis-open.org/odata/ns/edm');
        $schemas = $xml->xpath('//edm:Schema');
        
        $entities = [];

        foreach ($schemas as $schema) {
            foreach ($schema->EntityType as $entity) {
                $entityName = (string) $entity['Name'];
                $properties = [];

                foreach ($entity->Property as $prop) {
                    $properties[] = [
                        'name' => (string) $prop['Name'],
                        'type' => (string) $prop['Type'],
                        'nullable' => isset($prop['Nullable']) ? (string) $prop['Nullable'] : 'true',
                    ];
                }

                $entities[$entityName] = $properties;
            }
        }
        
        echo "<pre>";
        echo "ðŸ“¦ Entity: purchaseOrderLine\n\n";

        foreach ($entities['purchaseOrderLine'] as $field) {
            echo " â–¸ {$field['name']} ({$field['type']})";
            echo $field['nullable'] === 'false' ? " [NOT NULL]" : "";
            echo "\n";
        }
        echo "</pre>";
        exit;
    }

    public function getAllPurchaseQtys()
    {
        $start = microtime(true);
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Purchase_Lines_Excel?\$filter=Outstanding_Quantity gt 0 and (Status eq 'Released' or Status eq 'Pending Approval' or Status eq 'Pending Prepayment')  and (Location_Code eq 'CI.1010') &\$select=No,Location_Code,Outstanding_Quantity";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json'
            ]
        ];

        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);
            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }
            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);

        $result = [];
        foreach ($allData as $line) {
            $key = $line['No'] . '_' . $line['Location_Code'];
            $result[$key] = ($result[$key] ?? 0) + ($line['Outstanding_Quantity'] ?? 0);
        }
        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Purchaseqty: {$elapsed} seconds");
        

        return $result;
    }

    public function getAllTransferQtys()
    {
        $start = microtime(true);
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Transfer_Order_Line_Excel?\$filter=Status eq 'Released' and Quantity_Shipped lt Quantity and (Transfer_From_Code eq 'CI.1010'
            ) &\$select=Document_No,Item_No,Transfer_From_Code,OutstandingShipped";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=1000'
            ]
        ];

        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);
            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }
            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);

        $result = [];
        foreach ($allData as $line) {
            $key = $line['Item_No'] . '_' . $line['Transfer_from_Code'];
            $result[$key] = ($result[$key] ?? 0) + ($line['OutstandingShipped'] ?? 0);
        }

        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Transfer Qty: {$elapsed} seconds");
        return $result;
    }

    public function getTransferLinesFromBC()
    {
        $start = microtime(true);
        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Transfer_Order_Line_Excel?\$filter=Status eq 'Released' and Transfer_from_Code eq 'CI.1010'";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=5000',
            ]
        ];

        $allData = [];
        $url = $baseUrl;

        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);

            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }

            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);
        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Transfer Qty: {$elapsed} seconds");
        return $allData;
    }


    function getPurchaseLineFromBC() {
    $url = "https://api.businesscentral.dynamics.com/v2.0/"
         . env('AZURE_TENANT_ID') . "/"
         . env('BC_ENVIRONMENT')
         . "/api/v2.0/companies(" . $this->companyId . ")/purchaseOrders(83ea2b56-d008-ee11-8f70-00224857ecae)/purchaseOrderLines";

    try {
        $resp = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=1000',
            ],
            // Optional: let 4xx/5xx come back without throwing
            // 'http_errors' => false,
        ]);

        return [
            'ok'      => true,
            'status'  => $resp->getStatusCode(),
            'reason'  => $resp->getReasonPhrase(),
            'headers' => $resp->getHeaders(),
            'json'    => json_decode((string)$resp->getBody(), true),
            'raw'     => (string)$resp->getBody(),
        ];

    } catch (RequestException $e) {
        $res     = $e->getResponse();
        $status  = $res ? $res->getStatusCode() : null;
        $reason  = $res ? $res->getReasonPhrase() : null;
        $headers = $res ? $res->getHeaders() : [];
        $raw     = $res ? (string)$res->getBody() : $e->getMessage();
        $json    = json_decode($raw, true);

        // Log everything (no truncation in logs)
        Log::error('BC GET purchaseOrderLines failed', [
            'url'      => $url,
            'status'   => $status,
            'reason'   => $reason,
            'headers'  => $headers,
            'raw'      => $raw,
            'exception'=> $e->getMessage(),
        ]);

        // Return a rich error object to caller
        return [
            'ok'       => false,
            'status'   => $status,
            'reason'   => $reason,
            'headers'  => $headers,
            'errorRaw' => $raw,
            'error'    => $json ?? $raw,  // parsed JSON if possible
            'message'  => $e->getMessage(),
        ];
    } catch (\Throwable $e) {
        Log::error('Unexpected error calling BC', [
            'url'      => $url,
            'exception'=> $e->getMessage(),
        ]);

        return [
            'ok'      => false,
            'status'  => 500,
            'reason'  => 'Unexpected Error',
            'error'   => $e->getMessage(),
        ];
    }
}

    function getStockkeepingFromBC()
    {
        $filter = urlencode("(Location_Code eq 'CI.1010')");
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/StockkeepingUnit?\$filter={$filter}";
        
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ];
        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);

            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }

            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);
        return ['value' => $allData];
    }

    function itemUoM()
    {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Item_UOM";
        
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ];
        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);

            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }

            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);

        dd($allData);
        return ['value' => $allData];
    }


    public function getAllVendorsMap()
    {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/vendors?\$select=number,displayName";
        $vendors = [];
        do {
            $response = $this->client->get($url, [
                'headers' => [
                    'Authorization' => "Bearer {$this->token}",
                    'Accept' => 'application/json',
                    'Prefer' => 'odata.maxpagesize=1000'
                ]
            ]);
            $data = json_decode($response->getBody(), true);

            foreach ($data['value'] as $vendor) {
                $vendors[$vendor['number']] = $vendor['displayName'];
            }

            $url = $data['@odata.nextLink'] ?? null; 

        } while ($url);

        return $vendors;
    }


    public function getVendor() {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/vendors";

        $response = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json'
            ]
        ]);
        return json_decode($response->getBody(), true);
    }

    
    
    function CreatePurchaseOrder() {
        $poPayload = [
            "VendorNumber" => "VFB0011", 
            "orderDate" => date('Y-m-d')
            ];

        $response = $this->$client->post("https://api.businesscentral.dynamics.com/v2.0/".env('AZURE_TENANT_ID')."/".env('BC_ENVIRONMENT')."/api/v2.0/companies({$this->companyId})/purchaseOrders", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode($poPayload)
        ]);

        $data = json_decode((string)$response->getBody(), true);
        return $data;
    }

    function getPOFromBC() {

        $response = $this->client->get(
            "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/purchaseOrders?\$filter=number eq '30858'",
            [
                'headers' => [
                    'Authorization' => "Bearer {$this->token}",
                    'Accept'        => 'application/json'
                ]
            ]
        );

        return json_decode((string)$response->getBody(), true);
    }

    public function getPoSuggestions($userName)
        {
            $start = microtime(true);
            $stockkeeping = $purchaseRaw = $transferRaw = $priceRaw = $vendorRaw = [];
            $baseOdata = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/ODataV4/Company('" . $this->companyId . "')";
            $baseApi = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})";
            $activeCompany = session('current_company_name') ?? session('user')['role'];
            if ($activeCompany === 'Regent' or $activeCompany === 'SUPER') {
                $transferLocationCode = "(Transfer_from_Code eq 'CI.1010' or Transfer_from_Code eq 'CI.1020')";
                $locationCode = "(Location_Code eq 'CI.1010' or Location_Code eq 'CI.1020')";
            } else if ($activeCompany === 'HIN') {
                $transferLocationCode = "(Transfer_from_Code eq 'HIN.1000' or Transfer_from_Code eq 'HIN.1200' or Transfer_from_Code eq 'HIN.1300' or Transfer_from_Code eq 'HIN.3000')";
                $locationCode = "(Location_Code eq 'HIN.1000' or Location_Code eq 'HIN.1200' or Location_Code eq 'HIN.1300' or Location_Code eq 'HIN.3000')";
            }
            
            
            $headers = [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=50000'
            ];
            
            $prefixes = ['FO', 'BV', 'HO', 'MT', 'BG'];
            $itemInvenLocPromises = [];
            
            foreach ($prefixes as $prefix) {
                $itemInvenLocPromises[$prefix] = $this->client->getAsync(
                    "$baseOdata/ItemInvenLoc?\$filter=" . urlencode("startswith(ItemNo,'$prefix')"),
                    ['headers' => $headers]
                );
            }

            $promises = [
                'stockkeeping' => $this->client->getAsync(
                    "$baseOdata/APIStockkeeping?\$filter=" . urlencode($locationCode),
                    ['headers' => $headers]
                ),

                'purchaseQtys' => $this->client->getAsync("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
                        "/ODataV4/Company('" . $this->companyId . "')/Purchase_Lines_Excel?\$filter=Outstanding_Quantity gt 0 and (Status eq 'Released' or Status eq 'Pending Approval' or Status eq 'Pending Prepayment') 
                        and $locationCode &\$select=No,Document_No,Approved_Date,Status,Location_Code,Outstanding_Quantity,Unit_of_Measure_Code,Buy_from_Vendor_Name",
                        ['headers' => $headers]
                    ),
                'openPurchase' => $this->client->getAsync("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
                        "/ODataV4/Company('" . $this->companyId . "')/Purchase_Lines_Excel?\$filter=Outstanding_Quantity gt 0 and (Status eq 'Open') 
                        and $locationCode &\$select=No,Document_No,Approved_Date,Status,Location_Code,Outstanding_Quantity,Unit_of_Measure_Code",
                        ['headers' => $headers]
                    ),
                'transferQtys' => $this->client->getAsync(
                    "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
                    "/ODataV4/Company('" . $this->companyId . "')/Transfer_Order_Line_Excel?" .
                    "\$filter=" . urlencode("Status eq 'Released' and Quantity_Shipped lt Quantity and (Transfer_from_Code eq 'CI.1010' or Transfer_from_Code eq 'CI.1020')") .
                    "&\$select=Document_No,Item_No,Transfer_From_Code,OutstandingShipped,Shipment_Date,Description,Unit_of_Measure_Code",
                    ['headers' => $headers]
                ),
                'priceLists' => $this->client->getAsync(
                    "$baseOdata/Price_List_Lines?" .
                    '$filter=' . urlencode("SourceNo ne null and StartingDate le " . now()->toDateString() . " and EndingDate ge " . now()->toDateString()) .
                    '&$select=Asset_No,Product_No,Unit_of_Measure_Code,DirectUnitCost,SourceNo',
                    ['headers' => $headers]
                ),
                'vendors' => $this->client->getAsync(
                    "$baseApi/vendors?\$select=number,displayName",
                    ['headers' => $headers]
                ),
                'itemUoM' => $this->client->getAsync(
                    "$baseOdata/Item_UOM?\$select=Item_No,Code,Qty_per_Unit_of_Measure",
                    ['headers' => $headers]
                )
            ];
            
            $promises = array_merge($promises, $itemInvenLocPromises);
            $responses = Utils::settle($promises)->wait();
            $elapsed = microtime(true) - $start;
            Log::debug('Fetch Runtime:', ['seconds' => $elapsed]);
            if ($responses['stockkeeping']['state'] === 'fulfilled') {
                $stockkeeping = json_decode($responses['stockkeeping']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            if ($responses['purchaseQtys']['state'] === 'fulfilled') {
                $purchaseRaw = json_decode($responses['purchaseQtys']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            if ($responses['openPurchase']['state'] === 'fulfilled') {
                $openRaw = json_decode($responses['openPurchase']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            if ($responses['transferQtys']['state'] === 'fulfilled') {
                $transferRaw = json_decode($responses['transferQtys']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            if ($responses['priceLists']['state'] === 'fulfilled') {
                $priceRaw = json_decode($responses['priceLists']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            $itemUoMMap = [];
            if ($responses['itemUoM']['state'] === 'fulfilled') {
                $itemUoMRaw = json_decode($responses['itemUoM']['value']->getBody()->getContents(), true)['value'] ?? [];
                foreach ($itemUoMRaw as $uom) {
                    $itemNo = $uom['Item_No'];
                    $uomCode = $uom['Code'];
                    $qtyPerUOM = $uom['Qty_per_Unit_of_Measure'] ?? 1;
                    $itemUoMMap[$itemNo][$uomCode] = $qtyPerUOM;
                }
            }
            $ItemInvenLoc = [];
            foreach (['FO', 'BV', 'HO', 'MT', 'BG'] as $prefix) {
                if (isset($responses[$prefix]['state']) && $responses[$prefix]['state'] === 'fulfilled') {
                    $body = json_decode($responses[$prefix]['value']->getBody()->getContents(), true);
                    $ItemInvenLoc = array_merge($ItemInvenLoc, $body['value']);
                }
            }
            if ($responses['vendors']['state'] === 'fulfilled') {
                $vendorRaw = json_decode($responses['vendors']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            $purchaseQtys = [];
            $purchaseDetailsMap = [];   
            foreach ($purchaseRaw as $line) {
                if (strpos($line['Document_No'], 'PRO') !== false) {
                    continue;
                }
                $uomCode = $line['Unit_of_Measure_Code'];
                $multiplier = $itemUoMMap[$line['No']][$uomCode] ?? 1;
                $qtyBaseUoM = $line['Outstanding_Quantity'] * $multiplier;
                $key = $line['No'] . '_' . $line['Location_Code'];
                $purchaseQtys[$key] = ($purchaseQtys[$key] ?? 0) + $qtyBaseUoM;

                $purchaseDetailsMap[$key][] = [
                    'document_no' => $line['Document_No'],
                    'approved_date' => isset($line['Approved_Date']) ? date('d-m-Y', strtotime($line['Approved_Date'])) : null,
                    'status' => $line['Status'],
                    'outstanding_qty' => $qtyBaseUoM, 
                    'uom'             => $uomCode,
                    'multiplier'      => $multiplier,
                    'vendor' => $line['Buy_from_Vendor_Name']
                ];
            }
            $openPurchaseMap = [];
            foreach ($openRaw as $line) {
                if (strpos($line['Document_No'], 'PRO') !== false || strpos($line['Document_No'], 'PI') !== false || strpos($line['Document_No'], 'PPPRE') !== false || strpos($line['Document_No'], 'PPPCM') !== false) {
                    continue;
                }
                $uomCode = $line['Unit_of_Measure_Code'];
                $multiplier = $itemUoMMap[$line['No']][$uomCode] ?? 1;

                $qtyBaseUoM = $line['Outstanding_Quantity'] * $multiplier;


                $key = $line['No'] . '_' . $line['Location_Code'];

                $openPurchaseMap[$key][] = [
                    'document_no' => $line['Document_No'],
                    'approved_date' => isset($line['Approved_Date']) ? date('d-m-Y', strtotime($line['Approved_Date'])) : null,
                    'status' => $line['Status'],
                    'outstanding_qty' => $qtyBaseUoM, 
                    'uom'             => $uomCode,
                    'multiplier'      => $multiplier
                ];
            }
            $transferQtys = [];
            $transferDates = [];
            $transferDetailsMap = [];
            foreach ($transferRaw as $line) {
                $itemNo = $line['Item_No'];
                $location = $line['Transfer_from_Code'];
                if (!in_array($location, ['CI.1010', 'CI.1020', 'HIN.1000', 'HIN.1200', 'HIN.1300', 'HIN.3000'])) {
                    continue;
                }
                $key = $line['Item_No'] . '_' . $line['Transfer_from_Code'];
                $outstanding = $line['OutstandingShipped'] ?? 0;
                if ($activeCompany === 'Regent' or $activeCompany === 'SUPER') {
                    $outstandingBaseUoM = ($line['OutstandingShipped'] ?? 0) * $multiplier;
                } if ($activeCompany === 'HIN') {
                    $outstandingBaseUoM = ($line['OutstandingShipped'] ?? 0) / $multiplier;
                }

                if ($outstanding > 0) {
                    $transferQtys[$key] = ($transferQtys[$key] ?? 0) + $outstandingBaseUoM;
                    $transferDetailsMap[$itemNo . '_' . $location][] = [
                        'document_no' => $line['Document_No'],
                        'shipment_date' => $line['Shipment_Date'],
                        'quantity'      => $outstandingBaseUoM,
                        'uom'           => $uomCode,
                        'multiplier'    => $multiplier
                    ];
                    if (!empty($line['Shipment_Date'])) {
                        $date = $line['Shipment_Date'];
                        if (!isset($transferDates[$key]) || $date < $transferDates[$key]) {
                            $transferDates[$key] = $date;
                        }
                    }
                }
            }
            $priceMap = [];
            $unitCostsByItem = []; 
            foreach ($priceRaw as $line) {
                $itemNo   = $line['Asset_No'] ?? null;
                $vendorNo = $line['SourceNo'] ?? null;
                $uomCode  = $line['Unit_of_Measure_Code'] ?? null;
                $unitCost = $line['DirectUnitCost'] ?? null;

                if (!empty($itemNo) && !empty($vendorNo)) {
                    $priceMap[$itemNo][$vendorNo] = true;
                    if ($unitCost !== null) {
                        $multiplier = $uomMap[$itemNo][$uomCode] ?? 1; 
                        $costPerBaseUoM = $unitCost / $multiplier;
                        $unitCostsByItem[$itemNo][$vendorNo] = $costPerBaseUoM;
                    }
                }
            }
            foreach ($priceMap as $itemNo => $vendorList) {
                $priceMap[$itemNo] = array_keys($vendorList);
            }

            $vendorMap = [];
            foreach ($vendorRaw as $v) {
                $vendorMap[$v['number']] = $v['displayName'];
            }
            $skuKeys = [];
            foreach ($stockkeeping as $sku) {
                $skuKeys[$sku['Item_No'] . '_' . $sku['Location_Code']] = true;
            }
            $itemInvenKey = [];
            foreach ($ItemInvenLoc as $i => $item) {
                $itemInvenKey[$item['ItemNo'] . '_' . $item['Location']] = $i;
            }
            $results = [];
            foreach ($stockkeeping as $sku) {
                $itemNo = $sku['Item_No'];
                $location = $sku['Location_Code'];
                $stock = $sku['Inventory'];
                $minQty = $sku['MinInven'];
                $maxQty = $sku['MaxInven'];
                $description = $sku['Description'];
                $active = $sku['Active'];
                $comment = $sku['Comment'];
                $dbStatus = $sku['Status'];
                $key = $itemNo . '_' . $location;
                $normalizedDescription = strtoupper(preg_replace('/[^A-Z0-9]/', '', $description));

                if ($location === 'CI.1010' && str_contains($normalizedDescription, 'FRUITVEGETABLE')) {
                    continue;
                }
                $onPO = $purchaseQtys[$key] ?? 0;
                $inTransfer = $transferQtys[$key] ?? 0;
                $qtyToOrder = max(0, ceil(($maxQty + $inTransfer) - ($stock + $onPO)));
                $vendorNos = $priceMap[$itemNo] ?? [];
                $transferDate = $transferDates[$key] ?? null;
                
                if ($qtyToOrder <= 0) {
                    $status = "No Need Order";
                }
                if ($qtyToOrder > 0) {
                    if ( ($stock + $onPO) - ($inTransfer) < $minQty ) {
                        $status = "Need Order";
                        if ($inTransfer > 0) {
                            $status = "Need Order For TO";
                        }
                    } else
                    {
                        $status = "Order For Stock";
                    }
                }
                if ($onPO > 0 && $stock < ($minQty+$inTransfer) && ($stock+$onPO) >= ($minQty+$inTransfer)){
                    $status = "Follow Up PO";
                }

                if (empty($vendorNos)) {
                $results[] = [  
                    'item_no'       => $itemNo,
                    'description'   => $description,
                    'stock'         => $stock,
                    'min_qty'       => $minQty,
                    'max_qty'       => $maxQty,
                    'on_po'         => $onPO,
                    'in_transfer'   => $inTransfer,
                    'qty_to_order'  => $qtyToOrder,
                    'vendor_name'   => 'No Vendor',
                    'vendor_no'     => '0',
                    'location_code' => $location,
                    'unit_cost' => '0',
                    'status'        => $status,
                    'transfer_lines'=> $transferDetailsMap[$key] ?? [],
                    'po_lines' => $purchaseDetailsMap[$key] ?? [],
                    'openPo' => $openPurchaseMap[$key] ?? [],
                    'Active' => $active,
                    'Comment' => $comment,
                    'Status' => $dbStatus,
                    'need_shipment' => ($stock > 0 && $inTransfer > 0)
                ];
            } else {
                foreach ($vendorNos as $vendorNo) {
                    $results[] = [
                        'item_no'       => $itemNo,
                        'description'   => $description,
                        'stock'         => $stock,
                        'min_qty'       => $minQty,
                        'max_qty'       => $maxQty,
                        'on_po'         => $onPO,
                        'in_transfer'   => $inTransfer,
                        'qty_to_order'  => $qtyToOrder,
                        'vendor_name'   => $vendorMap[$vendorNo] ?? null,
                        'vendor_no'     => $vendorNo,
                        'location_code' => $location,
                        'unit_cost' => $unitCostsByItem[$itemNo][$vendorNo],
                        'status'        => $status,
                        'transfer_lines'=> $transferDetailsMap[$key] ?? [],
                        'po_lines' => $purchaseDetailsMap[$key] ?? [],
                        'openPo' => $openPurchaseMap[$key] ?? [],
                        'Active' => $active,
                        'Comment' => $comment,
                        'Status' => $dbStatus,
                        'need_shipment' => ($stock > 0 && $inTransfer > 0)
                    ];
                }
                }

            }

            foreach ($transferRaw as $line) {
                $itemNo = $line['Item_No'];
                $location = $line['Transfer_from_Code'];
                $key = $itemNo . '_' . $location;
                if (!in_array($location, ['CI.1010', 'CI.1020', 'HIN.1000', 'HIN.1200', 'HIN.1300', 'HIN.3000'])) {
                    continue;
                }
                if (isset($skuKeys[$key])) {
                    continue;
                }
                $onPO = $purchaseQtys[$key] ?? 0;
                $inTransfer = $transferQtys[$key] ?? 0;
                $transferDate = $line['Shipment_Date'] ?? null;

                $description = $line['Description'] ?? '';
                $normalizedDescription = strtoupper(preg_replace('/[^A-Z0-9]/', '', $description));
                if ($location === 'CI.1010' && str_contains($normalizedDescription, 'FRUITVEGETABLE')) {
                    continue;
                }
                
                $stock = isset($itemInvenKey[$key]) ? $ItemInvenLoc[$itemInvenKey[$key]]['Inventory'] : 0;
                $active = isset($itemInvenKey[$key]) ? $ItemInvenLoc[$itemInvenKey[$key]]['Active'] : true;
                $Comment = isset($itemInvenKey[$key]) ? $ItemInvenLoc[$itemInvenKey[$key]]['Comment'] : '';
                $dbStatus = isset($itemInvenKey[$key]) ? $ItemInvenLoc[$itemInvenKey[$key]]['Status'] : '__';
                $minQty = 0;
                $maxQty = 0;
                $qtyToOrder = max(0, ceil($maxQty + $inTransfer - ($stock + $onPO)));

                if ($qtyToOrder <= 0) {
                    $status = "No Need Order";
                } else {
                    if (($stock + $onPO) - $inTransfer < $minQty) {
                        $status = "Need Order";
                        if ($inTransfer > 0) {
                                $status = "Need Order For TO";
                        }
                    } else {
                        $status = "Order For Stock";
                    }
                }
                if ($onPO > 0 && $stock < ($minQty+$inTransfer) && ($stock+$onPO) > ($minQty+$inTransfer)){
                    $status = "Follow Up PO";
                }

                $vendorNos = $priceMap[$itemNo] ?? [];

                if (empty($vendorNos)) {
                    $results[] = [
                        'item_no'       => $itemNo,
                        'description'   => $description,
                        'stock'         => $stock,
                        'min_qty'       => $minQty,
                        'max_qty'       => $maxQty,
                        'on_po'         => $onPO,
                        'in_transfer'   => $inTransfer,
                        'qty_to_order'  => $qtyToOrder,
                        'vendor_name'   => 'No Vendor',
                        'vendor_no'     => '0',
                        'location_code' => $location,
                        'unit_cost'     => '0',
                        'status'        => $status,
                        'transfer_lines' => $transferDetailsMap[$key] ?? [],
                        'po_lines' => $purchaseDetailsMap[$key] ?? [],
                        'openPo' => $openPurchaseMap[$key] ?? [],
                        'Active' => $active,
                        'Comment' => $Comment,
                        'Status' => $dbStatus,
                        'need_shipment' => ($stock > 0 && $inTransfer > 0)
                        
                    ];
                } else {
                    foreach ($vendorNos as $vendorNo) {
                        $results[] = [
                            'item_no'       => $itemNo,
                            'description'   => $description,
                            'stock'         => $stock,
                            'min_qty'       => $minQty,
                            'max_qty'       => $maxQty,
                            'on_po'         => $onPO,
                            'in_transfer'   => $inTransfer,
                            'qty_to_order'  => $qtyToOrder,
                            'vendor_name'   => $vendorMap[$vendorNo] ?? null,
                            'vendor_no'     => $vendorNo,
                            'location_code' => $location,
                            'unit_cost'     => $unitCostsByItem[$itemNo][$vendorNo] ?? 0,
                            'status'        => $status,
                            'transfer_lines' => $transferDetailsMap[$key] ?? [],
                            'po_lines' => $purchaseDetailsMap[$key] ?? [],
                            'openPo' => $openPurchaseMap[$key] ?? [],
                            'Active' => $active,
                            'Comment' => $Comment,
                            'Status' => $dbStatus,
                            'need_shipment' => ($stock > 0 && $inTransfer > 0)
                        ];
                    }
                }
            }

            $results = array_filter($results, function ($item) {
                $desc = strtoupper(preg_replace('/[^A-Z0-9]/', '', $item['description'] ?? ''));
                return !(($item['location_code'] === 'CI.1010') && str_contains($desc, 'FRUITVEGETABLE'));
            });
            return [
                'items' => $results,
                'vendors' => $vendorMap
            ];
        }


    public function getItemIdByNo(string $itemNo)
    {
        $filter = urlencode("number eq '$itemNo'");
        $response = $this->client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
       "/api/v2.0/companies({$this->companyId})/items?\$filter=$filter", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ]
        ]);

        $items = json_decode($response->getBody(), true);
        return $items['value'][0]['id'] ?? null;
    }


    public function createPurchaseOrderForVendor($vendorNo, $userName, $location)
    {
        if ($location == 'CI.1010') {
            $purchaser = 'TN01';
        } 
        else if ($location == 'CI.1020'){
            $purchaser = 'UT01';
        } else if ($location == 'HIN.1200' || $location == 'HIN.1300') {
            $purchaser = 'PE01';
        } else if ($location == 'HIN.1000') {
            $purchaser = 'PW01';
        } else if ($location == 'HIN.3000') {
            $purchaser = 'FB01';
        }

        $payload = [
            'vendorNumber' => $vendorNo,
            'orderDate' => now()->toDateString(),
            'purchaser' => $purchaser,
            'shortcutDimension2Code' => "100000",
            'requestedReceiptDate' => today()->toDateString()
        ];

        $response = $this->client->post("https://api.businesscentral.dynamics.com/v2.0/".env('AZURE_TENANT_ID')."/".env('BC_ENVIRONMENT')."/api/v2.0/companies({$this->companyId})/purchaseOrders", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ],
            'body' => json_encode($payload)
        ]);
        
        $body = json_decode($response->getBody(), true);
        return json_decode($response->getBody(), true);
    }

    public function getLocationIdByCode($locationCode)
    {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/locations?\$filter=code eq '{$locationCode}'";

        $response = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json'
            ]
        ]);

        $data = json_decode((string)$response->getBody(), true);
        
        return $data['value'][0]['id'] ?? null;
    }


    // public function addPurchaseLineToPO($poId, $itemNo, $quantity, $cost, $location, $userName)
    // {
    //     $start = microtime(true);
    
    //     $itemId = $this->getItemIdByNo($itemNo);
    //     $locationCode = $location;
    //     if (str_contains($itemNo, 'BV') && $cost > 1000000) $locationCode = 'CI.1051';
    //     $locationId = $this->getLocationIdByCode($locationCode);
    
    //     $payload = json_encode([
    //         '@odata.type'       => '#Microsoft.NAV.purchaseOrderLine',
    //         'lineType'  => 'Item',
    //         'itemId'     => $itemId,
    //         'quantity'   => (float)$quantity
    //     ]);
    
    //     $url = "https://api.businesscentral.dynamics.com/v2.0/"
    //          . env('AZURE_TENANT_ID') . "/"
    //          . env('BC_ENVIRONMENT')
    //          . "/api/v2.0/companies({$this->companyId})/purchaseOrders({$poId})/purchaseOrderLines";

    //     try {
    //         $response = $this->client->post($url, [
    //             'headers' => [
    //                 'Authorization' => "Bearer {$this->token}",
    //                 'Accept'        => 'application/json',
    //                 'Content-Type'  => 'application/json',
    //                 'OData-Version'     => '4.0',
    //                 'OData-MaxVersion'  => '4.0',
    //             ],
    //             'body' => $payload,
    //         ]);
    
    //         Log::info("Create POLine runtime: " . (microtime(true) - $start) . " seconds");
    //         return json_decode((string)$response->getBody(), true);
    
    //     } catch (RequestException $e) {
    //         $res     = $e->getResponse();
    //         $status  = $res?->getStatusCode();
    //         $reason  = $res?->getReasonPhrase();
    //         $headers = $res?->getHeaders() ?? [];
    //         $raw     = $res ? (string)$res->getBody() : $e->getMessage();
    
    //         // Optional telemetry IDs (if present)
    //         $corrId = $headers['x-correlation-id'][0] ?? ($headers['MS-CorrelationId'][0] ?? null);
    //         $reqId  = $headers['request-id'][0] ?? null;
    
    //         Log::error('BC addPurchaseLineToPO failed', [
    //             'url'           => $url,
    //             'status'        => $status,
    //             'reason'        => $reason,
    //             'headers'       => $headers,
    //             'payload_json'  => json_decode($payload, true),
    //             'error_raw'     => $raw,      // ← full upstream body (no truncation)
    //             'requestId'     => $reqId,
    //             'correlationId' => $corrId,
    //             'exception'     => $e->getMessage(),
    //         ]);
    //         throw $e; // or return a normalized error if you prefer
    //     }
    // }
    
    
    public function getAsyncAddPurchaseLinePromise($poId, $itemNo, $quantity, $userName)
    {   
        $start = microtime(true);
        $itemId     = $this->getItemIdByNo($itemNo);
        $locationId = $this->getLocationIdByCode(in_array($userName, [
                'titania@citbi.onmicrosoft.com',
                'adminbc@citbi.onmicrosoft.com'
            ]) ? 'CI.1010' : 'CI.1020');

        $payload = [
            'itemId'     => $itemId,
            'quantity'   => (float) $quantity,
            'locationId' => $locationId,
        ];
        $elapsed = microtime(true) - $start;
        Log::info("Create POLine runtime: {$elapsed} seconds");
        return $this->client->postAsync(
            "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})/purchaseOrders({$poId})/purchaseOrderLines",
            [
                'headers' => [
                    'Authorization' => "Bearer {$this->token}",
                    'Accept'        => 'application/json',
                    'Content-Type'  => 'application/json',
                ],
                'body' => json_encode($payload),
            ]
        );
    }

    public function addPurchaseLineToPO(string $poNo, string $itemNo, float $quantity, $cost, string $locationCode = '', $userName, string $poId)
    {
        $base = "https://api.businesscentral.dynamics.com/v2.0/"
          . env('AZURE_TENANT_ID') . "/"
          . env('BC_ENVIRONMENT');

        $root = $base . "/api/citbi/sku/v1.0";
        $linesUrl = "{$root}/companies({$this->companyId})/purchaseOrderLinesCustom"; // <-- exact casing!

        $headersJson = [
            'Authorization'    => "Bearer {$this->token}",
            'Accept'           => 'application/json',
            'Content-Type'     => 'application/json',
        ];

        try {
            $flt = urlencode("documentType eq 'Order' and documentNo eq '{$poNo}'");
            $lastUrl = $linesUrl . "?\$filter={$flt}&\$orderby=lineNo desc&\$top=1&\$select=lineNo";
            $lastResp = $this->client->get($lastUrl, ['headers' => $headersJson]);
            $lastJson = json_decode((string)$lastResp->getBody(), true);
            $lastLineNo = (int)($lastJson['value'][0]['lineNo'] ?? 0);
            $nextLineNo = max(10000, $lastLineNo + 10000);

            $payload = [
                'documentType' => 'Order',
                'documentNo'   => (string)$poNo,
                'lineNo'       => $nextLineNo,
                'itemNo'       => $itemNo,
                'type'         => 'Item',
                'quantity'     => (float)$quantity
            ];
            if ($locationCode !== '') {
                $payload['locationCode'] = $locationCode;
            }
            $resp = $this->client->post($linesUrl, [ 'headers' => $headersJson, 'body' => json_encode($payload, JSON_UNESCAPED_SLASHES), ]); 
            $data = json_decode((string)$resp->getBody(), true); 
            $etag  = $data['@odata.etag'] ?? '*';
            $idUrl = $data['@odata.id']   ?? null;
            \Log::info('API line created (doc keys)', [ 'url' => $linesUrl, 'payload' => $payload, 'response' => $data ]);

            return $data;

        } catch (\GuzzleHttp\Exception\RequestException $e) {
            $res = $e->getResponse();
            $raw = $res ? (string)$res->getBody() : $e->getMessage();
            \Log::error('Custom API purchase line POST failed', [
                'url'       => $linesUrl,
                'status'    => $res?->getStatusCode(),
                'reason'    => $res?->getReasonPhrase(),
                'headers'   => $res?->getHeaders(),
                'payload'   => $payload ?? null,
                'error_raw' => $raw,
                'requestId' => $res?->getHeader('request-id')[0] ?? null,
            ]);
            throw $e;
        }
    }

    public function getAllPurchasePrices()
    {
        $start = microtime(true);
        $today = now()->toDateString(); // e.g. "2025-06-17"
        $filter = "SourceNo ne null and StartingDate le $today and EndingDate ge $today";
        $encodedFilter = urlencode($filter);

        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Price_List_Lines?\$filter={$encodedFilter}";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=1000'
            ]
        ];

        $allData = [];

        try {
            do {
                $response = $this->client->get($url, $headers);
                $decoded = json_decode((string)$response->getBody(), true);

                if (isset($decoded['value'])) {
                    $allData = array_merge($allData, $decoded['value']);
                }

                $url = $decoded['@odata.nextLink'] ?? null;
            } while ($url);

            return collect($allData)->groupBy('Asset_No');
        } catch (\Exception $e) {
            Log::error("Failed to fetch Price List Lines: " . $e->getMessage());
            return collect(); 
        }
        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Priccss: {$elapsed} seconds");
        
    }
}
