<?php

namespace App\Http\Controllers;

use App\Models\{ItemRequest, ItemRequestLine, ApprovalFlowTemplate};
use App\Models\{Company, User, ApprovalAction};
use App\Models\VendorRequestLine;
use App\Models\RequestsLog;
use App\Models\UserRequests;
use App\Models\OpnameSchedule;
use App\Models\ItemBefore;
use App\Services\ApprovalEngine;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use App\Services\BusinessCentralService;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use App\Models\ItemUserInput;
use App\Models\ActivityLog;
class ApprovalWebController extends Controller
{
    protected function resolveCurrentUser(): User
    {
        $sess = session('user') ?? [];
        $email = $sess['email'] ?? null;
        $name  = $sess['name'] ?? ($sess['displayName'] ?? 'MS User');
        abort_unless($email, 401, 'Not authenticated');
        return User::firstOrCreate(['email'=>$email], ['name'=>$name]);
    }

    protected function activeCompanyId(): ?int
    {
        $key = session('current_company_name') ?? data_get(session('user'), 'role');

        $map = [
            'Regent' => 'CI',   
            'HIN'    => 'TBI',  
        ];

        $code = $map[$key] ?? null;

        if (!$code) {
            return null; // SUPER with no selection => show everything
        }

        return Company::where('code', $code)->value('id');
    }

   
    public function show($id)
    {
        $me     = $this->resolveCurrentUser();
        $myrole = $me->role;

        // =========================
        // REQUEST UTAMA
        // =========================
        $req = ItemRequest::with([
            'requester',
            'lines',
            'flow.steps',
            'instance.actions.actor',
        ])->findOrFail($id);

        if ($req->userRequests()->exists()) {
            $req->load(['userRequests.user:id,name,role_id,email']);
        }

        $status = $req->status;

        // =========================
        // ITEM USER INPUTS
        // =========================
        $itemQuery = ItemUserInput::where('item_request_id', $id);

        $itemPreview = $itemQuery
            ->orderBy('id')
            ->limit(10)
            ->get();

        $itemCount = $itemQuery->count();

        // =========================
        // VENDOR
        // =========================
        $vendor = VendorRequestLine::where('item_request_id', $id)->get();

        // =========================
        // LOG
        // =========================
        $log = ActivityLog::where('url', 'like', "%/{$id}%")
            ->latest()
            ->get();

        $requestlog = RequestsLog::first() ?? new RequestsLog();

        // =====================================================
        // 🔥 STEP 2 STATUS (LOGIC BARU - 1 LOKASI SAJA)
        // =====================================================
        $step2status = null;

        $businessUnit = $me->business_unit;           // RBC / HIC / CI
        $validUnits   = ['RBC','HIC','CI'];

        if (in_array($businessUnit, $validUnits)) {

            // 1️⃣ Ambil jadwal opname aktif untuk unit ini
            $opname = OpnameSchedule::where('location', $businessUnit)
                ->whereIn('status', ['Pending','Start','Pause'])
                ->latest()
                ->first();

            if ($opname) {

                // 2️⃣ Lokasi spesifik dari item request (contoh: RBC.2100)
                $requestLocation = $req->location;

                // 3️⃣ Cek apakah data step 2 untuk lokasi ini SUDAH ADA
                $existsStep2 = \App\Models\ItemBeforeS2::where('periode', $opname->periode)
                    ->where('location', $requestLocation)
                    ->exists();

                // 4️⃣ Status untuk UI
                $step2status = [
                    'opname'          => $opname,
                    'location'        => $requestLocation,
                    'is_full'         => $existsStep2,   // ✅ LOGIC BARU
                    'can_start'       => !$existsStep2 && $opname->status === 'Start',
                    'can_rollback'    => $existsStep2,
                ];
            }
        }

        // =========================
        // RENDER VIEW
        // =========================
        return view('approvals.show', compact(
            'req',
            'vendor',
            'log',
            'requestlog',
            'myrole',
            'me',
            'status',
            'itemPreview',
            'itemCount',
            'step2status'
        ));
    }


    
   public function search(Request $request)
{
    $me     = auth()->user(); // atau ubah ke user aktifmu
    $roleId = $me->role_id ?? null;
    $cid    = $this->activeCompanyId();

    $q      = $request->query('q');
    $status = $request->query('status');
    $from   = $request->query('from');
    $to     = $request->query('to');

    // ===========================
    // 🔍 Query utama
    // ===========================
    $items = ItemRequest::query()
        ->when($roleId == 6, function ($q) use ($me) {
            // Jika role ID = 6, tampilkan hanya request yang pernah user ikuti (via user_requests)
            $q->whereExists(function ($sub) use ($me) {
                $sub->select(DB::raw(1))
                    ->from('user_requests')
                    ->whereRaw('user_requests.item_request_id = item_requests.id')
                    ->where('user_requests.user_id', $me->id);
            });
        }, function ($q) use ($me) {
            // Default: tampilkan request yang dibuat oleh user itu sendiri
            $q->where('requester_id', $me->id);
        })
        ->when($cid, fn($q) => $q->where('company_id', $cid))
        // Exclude request di mana user menjadi PIC
        ->whereNotExists(function ($sub) use ($me) {
            $sub->select(DB::raw(1))
                ->from('user_requests')
                ->whereRaw('user_requests.item_request_id = item_requests.id')
                ->where('user_requests.pic_id', $me->id);
        })
        // Eager load relasi penting
        ->with([
            'requester:id,name,email',
            'flow:id,name,version',
            'lines',
            'lines_vendor',
            'userRequests.user:id,name,email',
            'userRequests.picUser:id,name,email'
        ])
        ->when($q, function ($query, $q) {
            $query->where('id', $q)
                  ->orWhere('vendor_name', 'like', "%$q%")
                  ->orWhereHas('requester', fn($qr) => 
                      $qr->where('name', 'like', "%$q%")
                         ->orWhere('email', 'like', "%$q%")
                  )
                  ->orWhereHas('flow', fn($qf) =>
                      $qf->where('name', 'like', "%$q%")
                  );
        })
        ->when($status, fn($query) => $query->where('status', $status))
        ->when($from, fn($query) => $query->whereDate('posting_date', '>=', $from))
        ->when($to, fn($query) => $query->whereDate('posting_date', '<=', $to))
        ->latest()
        ->limit(20)
        ->get();

    // ===========================
    // 🔹 Generate HTML tabel
    // ===========================
    $html = '<div class="card card-rounded shadow-sm"><div class="table-responsive"><table class="table align-middle">';
    $html .= '<thead class="table-light">
        <tr>
          <th style="width:70px">#</th>
          <th>Vendor</th>
          <th>Requester</th>
          <th>From User</th>
          <th>Flow</th>
          <th>Status</th>
          <th style="width:80px">Step</th>
          <th style="width:80px">Lines</th>
          <th style="width:140px">Posting Date</th>
          <th style="width:160px">Updated</th>
          <th style="width:90px">Open</th>
        </tr>
      </thead><tbody>';

    if ($items->count()) {
        foreach ($items as $r) {
            $cls = match ($r->status) {
                'Approved' => 'bg-success',
                'Rejected' => 'bg-danger',
                'InReview' => 'bg-warning text-dark',
                default    => 'bg-secondary',
            };

            $linesCount = $r->flow_template_id == 6
                ? $r->lines_vendor()->count()
                : $r->lines()->count();

            // Ambil nama user pertama di userRequests (jika ada)
            $fromUser = $r->userRequests->first()?->user?->name ?? '—';

            $html .= '<tr>
                <td><span class="badge bg-secondary">#' . $r->id . '</span></td>
                <td>' . e($r->vendor_name) . '</td>
                <td>' . e($r->requester->name ?? '—');
            if ($r->requester?->email)
                $html .= '<div class="small text-muted">' . e($r->requester->email) . '</div>';
            $html .= '</td>
                <td><span class="badge bg-info text-dark mb-1">' . e($fromUser) . '</span></td>
                <td>' . e($r->flow->name ?? '—');
            if ($r->flow?->version)
                $html .= ' <span class="text-muted small">v' . e($r->flow->version) . '</span>';
            $html .= '</td>
                <td><span class="badge ' . $cls . '">' . e($r->status) . '</span></td>
                <td>' . e($r->current_step_no) . '</td>
                <td><span class="badge bg-info">' . $linesCount . '</span></td>
                <td>' . optional($r->posting_date)->format('Y-m-d') . '</td>
                <td>' . optional($r->updated_at)->format('Y-m-d H:i') . '</td>
                <td><a href="' . route('approvals.show', $r->id) . '" class="btn btn-outline-primary btn-sm">
                    <i class="fa fa-eye me-1"></i> View</a></td>
            </tr>';
        }
    } else {
        $html .= '<tr><td colspan="11" class="text-muted">No requests found.</td></tr>';
    }

    $html .= '</tbody></table></div></div>';

    return response()->json(['html' => $html]);
}




public function all(\Illuminate\Http\Request $request)
{
    $cid     = $this->activeCompanyId();

    $items = \App\Models\ItemRequest::query()
        ->with([
            'requester:id,name,email',
            'flow:id,name,version',
            'instance:id,item_request_id,status,started_at',
            'userRequests.user:id,name'
        ])
        ->when($cid, fn($q) => $q->where('company_id', $cid))
        ->withCount('lines')
        ->withCount('lines_vendor')
        ->orderByDesc('id')
        ->get(); // load all data

    return view('approvals.all', compact('items'));
}

public function submit($id, ApprovalEngine $eng)
{
    $me  = $this->resolveCurrentUser();
    $req = ItemRequest::with('flow.steps','instance')->findOrFail($id);

    // =========================
    // STATUS GUARD
    // =========================
    abort_unless(
        in_array($req->status, ['Draft','Rejected','Cancelled'], true),
        422,
        'Only Draft/Rejected/Cancelled can be submitted.'
    );

    // =========================
    // PERMISSION GUARD
    // =========================
    $isOwner = (int) $req->requester_id === (int) $me->id;
    $isAdmin = method_exists($me, 'isAdmin')
        ? (bool) $me->isAdmin()
        : (optional($me->role)->code === 'ADMIN');

    abort_unless($isOwner || $isAdmin, 403, 'You do not have permission to submit this request.');

    // =========================
    // DATA EXISTENCE GUARD
    // =========================
    if ((int) $req->flow_template_id === 6) {

        // Vendor flow
        $vendorCount = VendorRequestLine::where('item_request_id', $id)->count();
        abort_if($vendorCount === 0, 422, 'Add at least one vendor before submitting.');

    } elseif ((int) $req->flow_template_id === 1 /* contoh: Opname / Excel */) {

        // Opname / Excel flow → cek item_user_inputs
        $inputCount = \App\Models\ItemUserInput::where('item_request_id', $id)->count();
        abort_if($inputCount === 0, 422, 'Add at least one item before submitting.');

    } else {

        // Legacy flow → cek request lines
        abort_if(!$req->lines()->exists(), 422, 'Add at least one line before submitting.');
    }

    // =========================
    // APPROVAL STATE GUARD
    // =========================
    abort_if(
        optional($req->instance)->status === 'InProgress',
        422,
        'An approval is already in progress.'
    );

    abort_if(
        $req->status === 'Approved',
        422,
        'This request is already approved.'
    );

    // =========================
    // SUBMIT
    // =========================
    $eng->submit($req, $me);

    return redirect()
        ->route('approvals.show', $id)
        ->with('success', 'Submitted for approval');
}

    // public function submit($id, ApprovalEngine $eng)
    // {
    //     $me  = $this->resolveCurrentUser();
    //     $req = ItemRequest::with('flow.steps','instance')->findOrFail($id);
    //     $vendor = VendorRequestLine::where('item_request_id', $id)->get();
    //     // Only these may be (re)submitted
    //     abort_unless(in_array($req->status, ['Draft','Rejected','Cancelled'], true), 422,
    //         'Only Draft/Rejected/Cancelled can be submitted.');
    
    //     // Owner or admin only
    //     $isOwner = (int)$req->requester_id === (int)$me->id;
    //     $isAdmin = method_exists($me, 'isAdmin')
    //         ? (bool)$me->isAdmin()
    //         : (optional($me->role)->code === 'ADMIN');
    //     abort_unless($isOwner || $isAdmin, 403, 'You do not have permission to submit this request.');
    
    //     // abort_if(!$req->lines()->exists(), 422, 'Add at least one line before submitting.');
    //     if ($req->flow_template_id == 6) {
    //         abort_if($vendor->isEmpty(), 422, 'Add at least one vendor before submitting.');
    //     } else {
    //         abort_if(!$req->lines()->exists(), 422, 'Add at least one line before submitting.');
    //     }
    //     // Dont allow submit while an approval instance is in progress
    //     abort_if(optional($req->instance)->status === 'InProgress', 422, 'An approval is already in progress.');
    
    //     // If its already fully approved, block resubmit explicitly (defensive)
    //     abort_if($req->status === 'Approved', 422, 'This request is already approved.');
    
    //     // Submit via engine
    //     $eng->submit($req, $me);
    
    //     // Notify current approvers (engine should have advanced to step 1)
    //     // \App\Support\ApprovalNotifier::notifyCurrentApprovers($req->fresh(), 'requested', $me->name ?? $me->email);
    
    //     return redirect()->route('approvals.show', $id)
    //         ->with('success', 'Submitted for approval');
    // }

    public function approve($id, Request $r, ApprovalEngine $eng)
    {
        $me = $this->resolveCurrentUser();
        $req = ItemRequest::with('flow.steps','instance')->findOrFail($id);
        $result = $eng->act($req, $me->id, 'Approve', $r->input('comment'));

        $flowFinished = false;
        if (is_array($result) && array_key_exists('finished', $result)) {
            $flowFinished = (bool) $result['finished'];
        } elseif (is_bool($result)) {
            $flowFinished = $result;
        } else {
            $flowFinished =
                $req->status === 'Approved' ||
                optional($req->instance)->status === 'Approved' ||
                !is_null(optional($req->instance)->completed_at);
        }

        if ($flowFinished) {
        $requester = $req->requester;
            if ($requester) {
                \App\Support\ApprovalNotifier::notifyUser($requester, $req, 'approved', 'Your request has been fully approved', $me->name ?? $me->email);
            }
        } else {
            \App\Support\ApprovalNotifier::notifyCurrentApprovers($req, 'requested', $me->name ?? $me->email);
        }
        
        return back()->with('success','Approved.');
    }
    
    public function reject($id, Request $r, ApprovalEngine $eng)
    {
        $me = $this->resolveCurrentUser();
        $req = ItemRequest::with('flow.steps','instance')->findOrFail($id);
        $eng->act($req, $me->id, 'Reject', $r->input('comment'));
        $requester = $req->requester;
        // if ($requester) {
        //      \App\Support\ApprovalNotifier::notifyUser($requester, $req, 'rejected', 'Your request was rejected', $me->name ?? $me->email);
        // }
 
        return back()->with('success','Rejected.');
    }
    

public function inbox(Request $request)
{
    $me  = $this->resolveCurrentUser();
    $cid = $this->activeCompanyId();

    $base = ItemRequest::query()
        ->with(['requester','flow','instance'])
        ->when($cid, fn($q) => $q->where('company_id', $cid))
        ->where('status', 'InReview')
        ->whereExists(function ($q) {
            $q->from('approval_instances as ai')
              ->whereColumn('ai.item_request_id','item_requests.id')
              ->where('ai.status','InProgress');
        })

        // 🔥 HITUNG LINE DARI item_user_inputs
        ->withCount([
            'itemUserInputs as lines_count' => function ($q) {
                $q->select(DB::raw('count(*)'));
            }
        ]);

    if (!$me->isAdmin()) {
        $base->whereExists(function ($q) use ($me) {
            $q->from('approval_step_templates as ast')
              ->join('approval_step_approvers as asa', 'asa.step_template_id', '=', 'ast.id')
              ->whereColumn('ast.flow_template_id','item_requests.flow_template_id')
              ->whereColumn('ast.step_no','item_requests.current_step_no')
              ->where(function ($qq) use ($me) {
                  $qq->where('asa.approver_user_id', $me->id)
                     ->orWhere('asa.approver_role_id', $me->role_id);
              });
        });
    }

    $items = $base
        ->orderByDesc('id')
        ->paginate(15);

    $stepsCount = ApprovalFlowTemplate::withCount('steps')
        ->whereIn('id', $items->pluck('flow_template_id'))
        ->get()
        ->pluck('steps_count','id');

    return view('approvals.inbox', compact('items','stepsCount'));
}


public function myForms(\Illuminate\Http\Request $request)
{
    $me  = $this->resolveCurrentUser();
    $cid = $this->activeCompanyId();
    $userId = (string) $me->id;
    $roleId = (string) $me->role_id;

    // =========================
    // MY REQUESTS
    // =========================
    $items = ItemRequest::query()
        ->when($roleId == 6, function ($q) use ($me) {
            $q->whereExists(function ($sub) use ($me) {
                $sub->select(DB::raw(1))
                    ->from('user_requests')
                    ->whereRaw('user_requests.item_request_id = item_requests.id')
                    ->where('user_requests.user_id', $me->id);
            });
        }, function ($q) use ($me) {
            $q->where('requester_id', $me->id);
        })

        ->when($cid, fn($q) => $q->where('company_id', $cid))

        // EXCLUDE request yg user jadi PIC
        ->whereNotExists(function ($sub) use ($me) {
            $sub->select(DB::raw(1))
                ->from('user_requests')
                ->whereRaw('user_requests.item_request_id = item_requests.id')
                ->where('user_requests.pic_id', $me->id);
        })

        // 🔥 GANTI COUNT KE item_user_inputs
        ->withCount([
            'itemUserInputs as lines_count',
            'lines_vendor'
        ])

        // RELATIONS
        ->with('userRequests:id,item_request_id,user_id,pic_id')
        ->with('userRequests.user:id,name,email')
        ->with('userRequests.picUser:id,name,email')

        ->orderByDesc('id')
        ->get();

    // =========================
    // REQUESTS WHERE I AM PIC
    // =========================
    $itemsByPic = ItemRequest::query()
        ->whereExists(function ($sub) use ($me) {
            $sub->select(DB::raw(1))
                ->from('user_requests')
                ->whereRaw('user_requests.item_request_id = item_requests.id')
                ->where('user_requests.pic_id', $me->id);
        })
        ->when($cid, fn($q) => $q->where('company_id', $cid))

        // 🔥 GANTI COUNT KE item_user_inputs
        ->withCount([
            'itemUserInputs as lines_count',
            'lines_vendor'
        ])

        ->with([
            'userRequests:id,item_request_id,user_id,pic_id',
            'userRequests.user:id,name,email',
        ])

        ->latest()
        ->paginate(12)
        ->withQueryString();

    // 🔥 Dedup user requests dengan nama yang sama
    $itemsByPic->getCollection()->transform(function ($item) {
        $item->userRequests = $item->userRequests
            ->unique(fn($ur) => optional($ur->user)->name);
        return $item;
    });

    // =========================
    // PENDING FOR ME
    // =========================
    $pendingForMe = ItemRequest::query()
        ->with(['requester','flow','instance'])
        ->when($cid, fn($q) => $q->where('company_id', $cid))
        ->where('status', 'InReview')
        ->whereExists(function ($q) {
            $q->from('approval_instances as ai')
              ->whereColumn('ai.item_request_id','item_requests.id')
              ->where('ai.status','InProgress');
        })
        ->whereExists(function ($q) use ($userId, $roleId) {
            $q->from('approval_step_templates as ast')
              ->join('approval_step_approvers as asa', 'asa.step_template_id', '=', 'ast.id')
              ->whereColumn('ast.flow_template_id','item_requests.flow_template_id')
              ->whereColumn('ast.step_no','item_requests.current_step_no')
              ->where(function ($qq) use ($userId, $roleId) {
                  $qq->where('asa.approver_user_id', $userId)
                     ->orWhere('asa.approver_role_id', $roleId);
              });
        })
        ->orderByDesc('id')
        ->paginate(10)
        ->withQueryString();

    $stepsCount = \App\Models\ApprovalFlowTemplate::withCount('steps')
        ->whereIn('id', $pendingForMe->pluck('flow_template_id'))
        ->get()
        ->pluck('steps_count','id');

    return view('approvals.my_forms', compact(
        'items',
        'pendingForMe',
        'stepsCount',
        'roleId',
        'itemsByPic'
    ));
}


    public function edit($id, ApprovalEngine $eng)
    {
        $me  = $this->resolveCurrentUser();
        $req = ItemRequest::with('lines')->findOrFail($id);

        abort_unless(
            in_array($req->status, ['Draft','Rejected','Approved','Cancelled','InReview']),
            422,
            'This request cannot be edited.'
        );

        $isOwner = ((int)$req->requester_id === (int)$me->id);
        $isAdmin = method_exists($me, 'isAdmin')
            ? (bool)$me->isAdmin()
            : (optional($me->role)->code === 'ADMIN');

        $canEdit = false;
        if (in_array($req->status, ['Draft','Rejected','InReview','Cancelled'], true)) {
            $canEdit = $isOwner || $isAdmin;
        } elseif ($req->status === 'Approved') {
            $canEdit = $isAdmin || $eng->isLastApprover($req, $me);
        }

        abort_unless($canEdit, 403, 'You do not have permission to edit this request.');

        $flows = ApprovalFlowTemplate::where('object_type','NewItem')
            ->where('is_active',1)
            ->orderByDesc('version')
            ->get();

        $companies = $me->isAdmin()
            ? Company::orderBy('name')->get()
            : Company::whereIn('id', $me->allowedCompanyIds())->orderBy('name')->get();

        // 🔥 Ambil existing excel data
        $existingItems = ItemUserInput::where('item_request_id', $id)->get();

        return view('approvals.edit', [
            'req'           => $req,
            'flows'         => $flows,
            'companies'     => $companies,
            'existingItems' => $existingItems, // 👈 penting
        ]);
    }
    

    public function updateExcel(Request $request, $id)
    {
        $request->validate([
            'excel_payload' => 'required|string'
        ]);

        $rows = json_decode($request->excel_payload, true);
        if (!is_array($rows)) {
            return back()->withErrors('Invalid Excel payload.');
        }

        DB::transaction(function () use ($rows, $id) {

            // 🔥 HAPUS DATA LAMA
            ItemUserInput::where('item_request_id', $id)->delete();

            // 🔥 INSERT ULANG DARI EXCEL
            $now = now();
            $insert = [];

            foreach ($rows as $r) {
                $insert[] = [
                    'item_request_id'     => $id,
                    'storage_location'    => $r['storage_location'] ?? null,
                    'code'                => $r['code'] ?? null,
                    'material_description'=> $r['material_description'] ?? null,
                    'actual'              => $r['actual'] ?? 0,
                    'qty_mbc'             => $r['qty_mbc'] ?? 0,
                    'selisih'             => $r['selisih'] ?? 0,
                    'base_unit_of_measure'=> $r['base_unit_of_measure'] ?? null,
                    'note'                => $r['note'] ?? null,
                    'created_at'          => $now,
                    'updated_at'          => $now,
                ];
            }

            foreach (array_chunk($insert, 1000) as $chunk) {
                DB::table('item_user_inputs')->insert($chunk);
            }
        });

        return redirect()
            ->route('approvals.show', $id)
            ->with('success', 'Excel data updated successfully.');
    }

public function destroyDraft($id)
{
    $me = auth()->user(); // ambil user login
    $req = ItemRequest::with(['lines','attachments','instance.actions','userRequests.user'])->findOrFail($id);

    // ambil ID owner dinamis
    $ownerId = $req->userRequests->isNotEmpty()
        ? optional($req->userRequests->first()->user)->id
        : $req->requester_id;

    // cek hak akses
    $isOwner = $me && ((int) $me->id === (int) $ownerId);
    $isAdmin = optional($me->role)->code === 'ADMIN';

    abort_if($req->status !== 'Draft', 422, 'Only draft requests can be deleted.');
    abort_unless($isOwner || $isAdmin, 403, 'You do not have permission to delete this request.');

    DB::transaction(function () use ($req) {
        // hapus instance actions & instance
        if ($req->instance) {
            $req->instance->actions()->delete();
            $req->instance()->delete();
        }

        // hapus attachments dari storage & database
        $req->attachments->each(function ($att) {
            if ($att->path) {
                \Storage::disk('public')->delete($att->path);
            }
            $att->delete();
        });

        // hapus lines
        $req->lines()->delete();

        // hapus request itu sendiri
        $req->delete();
    });

    return redirect()->route('approvals.mine')->with('success', 'Draft deleted successfully.');
}

    public function cancel($id, Request $r)
    {
        $me  = $this->resolveCurrentUser();
        $req = ItemRequest::with(['instance'])->findOrFail($id);

        abort_unless($req->status === 'InReview', 422, 'Only in-review requests can be cancelled.');
        $isOwner = ((int) $req->requester_id === (int) $me->id);
        $isAdmin = method_exists($me, 'isAdmin')
            ? (bool) $me->isAdmin()
            : (optional($me->role)->code === 'ADMIN');
    
        abort_unless($isOwner || $isAdmin, 403, 'You do not have permission to edit this request.');


        $ins = $req->instance;
        abort_if(!$ins, 422, 'No approval instance found.');
        abort_unless($ins->status === 'InProgress', 422, 'Instance is not in progress.');

        DB::transaction(function () use ($req, $ins, $me, $r) {
            ApprovalAction::create([
                'approval_instance_id' => $ins->id,
                'step_no'              => $req->current_step_no ?? 0,
                'actor_user_id'        => $me->id,
                'action'               => 'Cancel',            
                'comment'              => $r->input('comment'),
                'acted_at'             => now(),
            ]);

            // close the instance
            $ins->status       = 'Cancelled';
            $ins->ended_at = now();
            $ins->save();

            $req->status = 'Cancelled';
            $req->save();
        });

        return redirect()->route('approvals.show', $req->id)
            ->with('success', 'Request has been cancelled.');
    }

    public function downloadAttachment(\App\Models\ItemRequest $req, \App\Models\ItemRequestAttachment $att)
    {
        if ((int)$att->item_request_id !== (int)$req->id) {
            Log::warning('DL abort: attachment not owned by request', [
                'req_id' => $req->id, 'att_id' => $att->id,
            ]);
            abort(404);
        }

        return \Storage::disk('public')->download($att->path, $att->original_name);
    }
    
    public function previewAttachment(\App\Models\ItemRequest $req, \App\Models\ItemRequestAttachment $att)
    {
        if ((int) $att->item_request_id !== (int) $req->id) {
            \Log::warning('Preview abort: attachment not owned by request', [
                'req_id' => $req->id,
                'att_id' => $att->id,
            ]);
            abort(404);
        }
    
        $path = \Storage::disk('public')->path($att->path);
    
        abort_unless(file_exists($path), 404);
    
        return response()->file($path, [
            'Content-Type'        => $att->mime,
            'Content-Disposition' => 'inline; filename="'.$att->original_name.'"',
        ]);
    }

    public function duplicate($id, \Illuminate\Http\Request $r)
    {
        $me  = $this->resolveCurrentUser();

        $src = ItemRequest::with(['lines','attachments'])->findOrFail($id);

        abort_unless($src->status === 'Rejected', 422, 'Only rejected requests can be duplicated.');
        $isOwner = (int)$src->requester_id === (int)$me->id;
        $isAdmin = method_exists($me, 'isAdmin') ? $me->isAdmin() : (optional($me->role)->code === 'ADMIN');
        
        abort_unless($isOwner || $isAdmin, 403, 'You are not allowed to duplicate this request.');

        $copyAttachments = (bool) $r->boolean('copy_attachments'); 

        $new = DB::transaction(function () use ($src, $me, $copyAttachments) {

            $new = ItemRequest::create([
                'company_id'       => (int) $src->company_id,
                'requester_id'     => $me->id,
                'flow_template_id' => $src->flow_template_id,
                'posting_date'     => $src->posting_date,        
                'remarks'          => $src->remarks,
                'vendor_name'      => $src->vendor_name,
                'status'           => 'Draft',
                'current_step_no'  => 0,
            ]);

            $payloads = [];
            foreach ($src->lines as $ln) {
                $payloads[] = [
                    'article_name'                   => $ln->article_name,
                    'type'                           => $ln->type,
                    'base_unit_code'                 => $ln->base_unit_code,
                    'gl_account_no'                  => $ln->gl_account_no,
                    'inventory_posting_group_code'   => $ln->inventory_posting_group_code,
                    'gen_prod_posting_group_code'    => $ln->gen_prod_posting_group_code,
                    'l1'                             => $ln->l1,
                    'l2'                             => $ln->l2,
                    'article_no'                     => $ln->article_no,
                    'line_remarks'                   => $ln->line_remarks,
                    'vendor_quotes'                  => $ln->vendor_quotes,
                ];
            }
            if (!empty($payloads)) {
                $new->lines()->createMany($payloads);
            }

            // (Optional) copy attachments
            if ($copyAttachments && $src->attachments?->count()) {
                foreach ($src->attachments as $att) {
                    $from = $att->path; // e.g. requests/{oldId}/file.pdf
                    $to   = 'requests/'.$new->id.'/'.basename($att->path);
                    // if stored on 'public' disk:
                    if (\Storage::disk('public')->exists($from)) {
                        \Storage::disk('public')->copy($from, $to);
                        $new->attachments()->create([
                            'uploader_user_id' => $me->id,
                            'original_name'    => $att->original_name,
                            'path'             => $to,
                            'mime'             => $att->mime,
                            'size'             => $att->size,
                        ]);
                    }
                }
            }

            return $new;
        });

        return redirect()->route('approvals.edit', $new->id)
            ->with('success', "Draft #{$new->id} created from rejected request #{$src->id}.");
    }
    public function print($id)
{
    $req = \App\Models\ItemRequest::with([
        'requester','company','flow.steps.approverUser',
        'instance.actions.actor','lines'
    ])->findOrFail($id);

    $vendorHeader = collect($req->lines)
        ->flatMap(fn($l) => collect($l->vendor_quotes ?? [])->pluck('name')->filter())
        ->countBy()->sortDesc()->keys()->take(3)->values()->all();

    // If there’s a selected vendor anywhere, make it the first header column
    $firstSelected = collect($req->lines)
        ->flatMap(fn($l) => collect($l->vendor_quotes ?? []))
        ->firstWhere('selected', true);

    if ($firstSelected) {
        $name = $firstSelected['name'] ?? null;
        if ($name) {
            $vendorHeader = collect([$name])
                ->merge(collect($vendorHeader)->reject(fn($v) => $v === $name))
                ->take(3)->values()->all();
        }
    }

    $vendor = VendorRequestLine::where('item_request_id', $id)->first();

    // ============================================================
    // ➕ TAMBAHAN: Hitung jumlah approver dari ApprovalStepTemplate
    // ============================================================
    $flowId = $req->flow_template_id;

    $approverTemplates = \App\Models\ApprovalStepTemplate::where('flow_template_id', $flowId)
        ->orderBy('step_no')
        ->get();

    // total approver diperlukan
    $totalApproversNeeded = $approverTemplates->count();

    // kirim juga template approvers (punya actor_id)
    $approverSlots = $approverTemplates;

    // ============================================================

    return view('approvals.print', compact(
        'req','vendorHeader','vendor',
        'totalApproversNeeded','approverSlots'
    ));
}

public function sendNotify($id, Request $request)
{
    $me = $this->resolveCurrentUser();
    $req = ItemRequest::with('flow.steps','instance')->findOrFail($id);

    // Tentukan penerima — contoh: requester
    $requester = $req->requester;

    if ($requester) {
        // Kalau requester tidak ada, kirim ke approver aktif
        \App\Support\ApprovalNotifier::notifyCurrentApprovers(
            $req,
            'requested',
            $me->name ?? $me->email
        );
    } else {
            \App\Support\ApprovalNotifier::notifyUser(
            $requester,
            $req,
            'info',
            $request->message ?? 'There is an update regarding your request.',
            $me->name ?? $me->email
        );
    }

    return back()->with('success', 'Notifikasi berhasil dikirim.');
}


}
