<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Redirect;
use App\Services\AzureAuthService;
use Laravel\Socialite\Facades\Socialite;
use Illuminate\Support\Str;
use SocialiteProviders\Microsoft\Provider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use App\Models\User; 

class AzureAuthController extends Controller
{
    protected AzureAuthService $authService;

    public function __construct(AzureAuthService $authService)
    {
        $this->authService = $authService;
    }


    public function login(Request $request)
    {
        $request->validate([
            'email'    => 'required|email',
            'password' => 'required'
        ]);

        $credentials = $request->only('email', 'password');

        if (Auth::attempt($credentials, true)) {
            $request->session()->regenerate();

            $user = Auth::user();
            $email = strtolower($user->email);
            $bu = strtoupper($user->business_unit ?? '');
            $role = 'Regent'; // default

            // 🔥 1) SUPER USER BERDASARKAN EMAIL
            if (
                str_contains($email, 'admin') ||
                str_contains($email, 'michael') ||
                str_contains($email, 'kristina') ||
                str_contains($email, 'costingho') ||
                str_contains($email, 'erlina') ||
                str_contains($email, 'teguh')
            ) {
                $role = 'SUPER';
            }
            // 🔥 2) ROLE BERDASARKAN BUSINESS UNIT
            elseif ($bu !== '') {
                if (str_contains($bu, 'RBC') || str_contains($bu, 'CI')) {
                    $role = 'Regent';
                } elseif (str_contains($bu, 'HIC')) {
                    $role = 'HIN';
                }
            }
            // 🔥 3) fallback kosong tetap Regent (default di atas)

            // ⭐ SIMPAN SESSION AGAR MIDDLEWARE BISA BACA
            session([
                'user' => [
                    'id'    => $user->id,
                    'name'  => $user->name,
                    'email' => $email,
                    'role'  => $role
                ]
            ]);

            return redirect()->route('dashboard');
        }

        return back()->withErrors(['email' => 'Email atau password salah.']);
    }





    public function store(Request $request)
    {
        $request->validate([
            'name'     => 'required|min:3|max:50',
            'email'    => 'required|email|unique:users,email',
            'password' => [
                'required','min:8','max:20',
                'regex:/[A-Z]/','regex:/[a-z]/','regex:/[0-9]/'
            ]
        ]);

        $user = User::create([
            'name'     => $request->name,
            'email'    => strtolower($request->email),
            'password' => Hash::make($request->password),
            'role_id'  => 6, 
        ]);
        Log::error($user);
        Auth::login($user);
        return redirect()->route('dashboard');
    }

    public function redirectToProvider()
    {
        return Socialite::driver('microsoft')
        ->scopes(['openid', 'offline_access', 'User.Read', 'Files.Read'])
        ->redirect();
    }

    public function handleProviderCallback(Request $request)
    {
        try {
            $user = Socialite::driver('microsoft')->stateless()->user(); 
            $email = strtolower($user->getEmail());
            $name  = $user->getName();
            $role = null;
        
            if (
                str_contains($email, 'adnyani') ||
                str_contains($email, 'titania')
            ) {
                $role = 'Regent';
            } elseif (
                str_contains($email, 'pebri') ||
                str_contains($email, 'pratiwi') ||
                str_contains($email, 'feybe')
            ) {
                $role = 'HIN';
            } elseif (
                str_contains($email, 'admin') ||
                str_contains($email, 'michael') ||
                str_contains($email, 'kristina') ||
                str_contains($email, 'costingho') ||
                str_contains($email, 'erlina') ||
                str_contains($email, 'teguh')
            ) {
                $role = 'SUPER';
            } else {
                $role = 'Regent';
            }

            $local = User::firstOrCreate(['email' => $email], ['name' => $name]);
            Auth::login($local, true);         
            $request->session()->regenerate(); 
                    
            session([
                'user' => [
                    'name' => $user->getName(),
                    'email' => $user->getEmail(),
                    'role' => $role
                ],
                'graph' => [
                    'access_token'  => $user->token,
                    'refresh_token' => $user->refreshToken ?? null, // may be null if offline_access wasn’t granted earlier
                    'expires_at'    => now()->addSeconds((int)($user->expiresIn ?? 3600))->toIso8601String(),
                ],
            ]);
           return redirect()->route('dashboard');

        } catch (\Exception $e) {
            \Log::error('Socialite error: ' . $e->getMessage());
        }
    }


    public function logout(Request $request)
    {
        // 🔥 Logout Laravel Auth (login biasa)
        auth()->logout();

        // 🔥 Hapus session khusus Microsoft & login manual
        session()->forget(['user', 'graph']);
        session()->flush();

        // 🔐 Amankan session
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login');
    }

}