<?php

namespace App\Http\Controllers;

use App\Models\UserRequest;
use App\Models\ItemRequest;
use App\Models\ItemRequestLine;
use App\Models\ItemRequestAttachment;
use App\Models\User;
use App\Models\{ApprovalFlowTemplate, Company};
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
class UserRequestController extends Controller
{
      protected function resolveCurrentUser(): User
    {
        $sess  = session('user') ?? [];
        $email = $sess['email'] ?? null;
        $name  = $sess['name'] ?? ($sess['displayName'] ?? 'MS User');

        abort_unless($email, 401, 'Not authenticated');

        return User::firstOrCreate(['email' => $email], ['name' => $name]);
    }
    public function create()
    {
        $me = $this->resolveCurrentUser();
        $user = auth()->user();
        $pics = User::where('role_id', '=', 4 )
                ->orderBy('name')
                ->get(['id', 'name']);
        $flows = ApprovalFlowTemplate::where('object_type','NewItem')
            ->where('is_active',1)->orderByDesc('version')->get();
        
        $companies = $me->isAdmin()
            ? Company::orderBy('name')->get()
            : Company::whereIn('id', $me->allowedCompanyIds())->orderBy('name')->get();
        $defaultCompanyId = $me->defaultCompanyId() ?? optional($companies->first())->id;
        return view('approvals.user_request', compact('user','pics','flows','companies','defaultCompanyId'));
    }

    public function createallinone()
    {
        $me = $this->resolveCurrentUser();
        $user = auth()->user();
        $pics = User::where('role_id', '=', 4 )
                ->orderBy('name')
                ->get(['id', 'name']);
        $flows = ApprovalFlowTemplate::where('object_type','NewItem')
            ->where('is_active',1)->orderByDesc('version')->get();
        
        $companies = $me->isAdmin()
            ? Company::orderBy('name')->get()
            : Company::whereIn('id', $me->allowedCompanyIds())->orderBy('name')->get();
        $defaultCompanyId = $me->defaultCompanyId() ?? optional($companies->first())->id;
        return view('approvals.allone_user_requests', compact('user','pics','flows','companies','defaultCompanyId'));
    }

public function store(Request $r)
{
    $user = auth()->user();

    $data = $r->validate([
        'company_id'       => 'required|integer|exists:companies,id',
        'flow_template_id' => 'required|exists:approval_flow_templates,id',
        'posting_date'     => 'nullable|date',
        'items'            => 'required|array|min:1',
        'items.*.item_name' => 'required|string|max:255',
        'items.*.pic_id'   => 'nullable|integer|exists:users,id',
        'items.*.remarks'  => 'nullable|string',
        'items.*.gen_prod_posting_group_code'  => 'nullable|string|max:50',
        'items.*.inventory_posting_group_code' => 'nullable|string|max:50',
        'items.*.attachments'   => ['array'],
        'items.*.attachments.*' => ['file','max:20120'],
    ]);

    $createdRequests = [];

    DB::transaction(function () use ($data, $user, &$createdRequests) {
        foreach ($data['items'] as $item) {
            $picId = $item['pic_id'] ?? $user->id;

            // === 1️⃣ Buat ItemRequest (satu per item)
            $itemRequest = ItemRequest::create([
                'company_id'       => $data['company_id'],
                'requester_id'     => $picId,
                'flow_template_id' => $data['flow_template_id'],
                'posting_date'     => $data['posting_date'] ?? now(),
                'remarks'          => $item['remarks'] ?? null,
                'vendor_name'      => null,
                'status'           => 'Draft',
                'current_step_no'  => 0,
            ]);

            // === 2️⃣ Buat UserRequest
            UserRequest::create([
                'company_id'       => $data['company_id'],
                'user_id'          => $user->id,
                'pic_id'           => $picId,
                'item_request_id'  => $itemRequest->id,
                'item_name'        => $item['item_name'],
                'status'           => 'Draft',
            ]);

            // === 3️⃣ Buat ItemRequestLine
            ItemRequestLine::create([
                'item_request_id'              => $itemRequest->id,
                'article_name'                 => $item['item_name'],
                'type'                         => null,
                'base_unit_code'               => null,
                'gl_account_no'                => null,
                'inventory_posting_group_code' => $item['inventory_posting_group_code'] ?? null,
                'gen_prod_posting_group_code'  => $item['gen_prod_posting_group_code'] ?? null,
                'l1'                            => null,
                'l2'                            => null,
                'article_no'                    => null,
                'line_remarks'                  => $item['remarks'] ?? null,
                'vendor_quotes'                 => null,
            ]);

            // === 4️⃣ Simpan attachment (jika ada)
            $files = $item['attachments'] ?? [];
            if (is_array($files)) {
                foreach ($files as $file) {
                    $path = $file->store("requests/{$itemRequest->id}", 'public');
                    ItemRequestAttachment::create([
                        'item_request_id'  => $itemRequest->id,
                        'uploader_user_id' => $user->id,
                        'original_name'    => $file->getClientOriginalName(),
                        'path'             => $path,
                        'mime'             => $file->getClientMimeType(),
                        'size'             => $file->getSize(),
                    ]);
                }
            }

            // simpan id tiap itemRequest untuk redirect nanti
            $createdRequests[] = $itemRequest->id;
        }
    });

    return response()->json([
        'success' => true,
        'message' => 'Semua draft berhasil dibuat',
        'redirect_urls' => collect($createdRequests)->map(fn($id) => route('approvals.show', $id)),
    ]);
}

public function storeAllInOne(Request $r)
{
    $user = auth()->user();

    // ================================
    // 1. Validate data
    // ================================
    $data = $r->validate([
        'company_id'       => 'required|integer|exists:companies,id',
        'flow_template_id' => 'required|exists:approval_flow_templates,id',
        'posting_date'     => 'nullable|date',

        'items'                 => 'required|array|min:1',
        'items.*.item_name'     => 'required|string|max:255',
        'items.*.pic_id'        => 'nullable|integer|exists:users,id',
        'items.*.remarks'       => 'nullable|string',
        'items.*.gen_prod_posting_group_code'  => 'nullable|string|max:50',
        'items.*.inventory_posting_group_code' => 'nullable|string|max:50',
        'items.*.attachments'   => ['array'],
        'items.*.attachments.*' => ['file','max:20120'],
    ]);

    $itemRequestId = null;
    $createdLines   = [];

    DB::transaction(function () use ($data, $user, &$itemRequestId, &$createdLines) {

        // =====================================================
        // 2️⃣ BUAT 1 SAJA ItemRequest (untuk SEMUA item)
        // =====================================================
        $firstItem = $data['items'][0];
        $picId     = $firstItem['pic_id'] ?? $user->id;

        $itemRequest = ItemRequest::create([
            'company_id'       => $data['company_id'],
            'requester_id'     => $picId,
            'flow_template_id' => $data['flow_template_id'],
            'posting_date'     => $data['posting_date'] ?? now(),
            'remarks'          => $firstItem['remarks'] ?? null, // optional — bisa pakai gabungan remarks
            'vendor_name'      => null,
            'status'           => 'Draft',
            'current_step_no'  => 0,
        ]);

        $itemRequestId = $itemRequest->id;

        // =====================================================
        // 3️⃣ LOOP setiap item → Buat UserRequest & ItemRequestLine
        // =====================================================
        foreach ($data['items'] as $item) {

            $picId = $item['pic_id'] ?? $user->id;

            // ==== Buat UserRequest (per item)
            UserRequest::create([
                'company_id'       => $data['company_id'],
                'user_id'          => $user->id,
                'pic_id'           => $picId,
                'item_request_id'  => $itemRequestId,
                'item_name'        => $item['item_name'],
                'status'           => 'Draft',
            ]);

            // ==== Buat ItemRequestLine (per item — tetap 1 per 1)
            $line = ItemRequestLine::create([
                'item_request_id'              => $itemRequestId,
                'article_name'                 => $item['item_name'],
                'type'                         => null,
                'base_unit_code'               => null,
                'gl_account_no'                => null,
                'inventory_posting_group_code' => $item['inventory_posting_group_code'] ?? null,
                'gen_prod_posting_group_code'  => $item['gen_prod_posting_group_code'] ?? null,
                'l1'                           => null,
                'l2'                           => null,
                'article_no'                   => null,
                'line_remarks'                 => $item['remarks'] ?? null,
                'vendor_quotes'                => null,
            ]);

            $createdLines[] = $line->id;

            // =====================================================
            // 4️⃣ Attachment — tetap per-item tetapi 1 ItemRequest
            // =====================================================
            $files = $item['attachments'] ?? [];

            if (is_array($files) && count($files) > 0) {
                foreach ($files as $file) {
                    $path = $file->store("requests/{$itemRequestId}", 'public');

                    ItemRequestAttachment::create([
                        'item_request_id'  => $itemRequestId,
                        'uploader_user_id' => $user->id,
                        'original_name'    => $file->getClientOriginalName(),
                        'path'             => $path,
                        'mime'             => $file->getClientMimeType(),
                        'size'             => $file->getSize(),
                    ]);
                }
            }
        }
    });

    // =====================================================
    // 5️⃣ Response → hanya 1 redirect (1 ItemRequest)
    // =====================================================
    return response()->json([
        'success' => true,
        'message' => 'Semua item berhasil disimpan dalam satu ItemRequest',
        'redirect_url' => route('approvals.show', $itemRequestId),
    ]);
}

}
