<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ItemRequest;
use App\Models\VendorRequestLine;
use App\Models\ApprovalFlowTemplate;
use App\Models\Company;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Arr;
class VendorController extends Controller

{
    /**
     * Tampilkan form input vendor baru
     */
    public function create()
    {
        $companies = Company::all();
        $flows = ApprovalFlowTemplate::where('object_type', 'NewItem')
                                     ->where('is_active', 1)
                                     ->orderByDesc('version')
                                     ->get();

        return view('approvals.new_vendor', compact('companies', 'flows'));
    }

    /**
     * Simpan vendor baru beserta request & attachments
     */
public function store(Request $request)
{
    $isAutosave = $request->boolean('is_autosave');
    $user       = Auth::user();

    /*
    |--------------------------------------------------------------------------
    | 1) VALIDASI
    |--------------------------------------------------------------------------
    | Autosave → validasi longgar (tanpa required)
    | Normal   → validasi penuh
    */
    if ($isAutosave) {

        $validated = $request->validate([
            'vendor_name'      => 'nullable|string|max:255',
            'remarks'          => 'nullable|string|max:1000',
            'address'          => 'nullable|string|max:500',
            'npwp'             => 'nullable|string|max:50',
            'phone'            => 'nullable|string|max:20',
            'company_id'       => 'nullable|exists:companies,id',
            'flow_template_id' => 'nullable|exists:approval_flow_templates,id',

            'banks'                      => 'nullable|array',
            'banks.*.bank_name'          => 'nullable|string|max:100',
            'banks.*.account_holder'     => 'nullable|string|max:100',
            'banks.*.account_number'     => 'nullable|string|max:50',

            'lines'                           => 'nullable|array',
            'lines.*.article_name'            => 'nullable|string',
            'lines.*.type'                    => 'nullable|string',
            'lines.*.base_unit_code'          => 'nullable|string',
            'lines.*.gl_account_no'           => 'nullable|string',
            'lines.*.inventory_posting_group_code' => 'nullable|string',
            'lines.*.gen_prod_posting_group_code'  => 'nullable|string',
            'lines.*.l1'                      => 'nullable|string',
            'lines.*.l2'                      => 'nullable|string',
            'lines.*.article_no'              => 'nullable|string',
            'lines.*.line_remarks'            => 'nullable|string',
            'lines.*.vendor_quotes'           => 'nullable|array',
            'lines.*.vendor_quotes.*.name'    => 'nullable|string|max:120',
            'lines.*.vendor_quotes.*.price'   => 'nullable|numeric|min:0',
            'lines.*.vendor_quotes.*.currency'=> 'nullable|string|max:10',
            'lines.*.vendor_selected'         => 'nullable|integer',

            'attachments.*' => 'nullable|file|max:5120',
        ]);

    } else {

        $validated = $request->validate([
            'vendor_name'      => 'required|string|max:255',
            'remarks'          => 'nullable|string|max:1000',
            'address'          => 'required|string|max:500',
            'npwp'             => 'nullable|string|max:50',
            'phone'            => 'nullable|string|max:20',
            'company_id'       => 'required|exists:companies,id',
            'flow_template_id' => 'required|exists:approval_flow_templates,id',

            'banks'                      => 'required|array',
            'banks.*.bank_name'          => 'required|string|max:100',
            'banks.*.account_holder'     => 'required|string|max:100',
            'banks.*.account_number'     => 'required|string|max:50',

            'lines'                           => 'required|array|min:1',
            'lines.*.article_name'            => 'required|string',
            'lines.*.type'                    => 'required|in:Inventory,Service,NonInventory',
            'lines.*.base_unit_code'          => 'required|string',
            'lines.*.gl_account_no'           => 'nullable|string',
            'lines.*.inventory_posting_group_code' => 'nullable|string',
            'lines.*.gen_prod_posting_group_code'  => 'nullable|string',
            'lines.*.l1'                      => 'nullable|string',
            'lines.*.l2'                      => 'nullable|string',
            'lines.*.article_no'              => 'nullable|string',
            'lines.*.line_remarks'            => 'nullable|string',
            'lines.*.vendor_quotes'           => 'nullable|array',
            'lines.*.vendor_quotes.*.name'    => 'nullable|string|max:120',
            'lines.*.vendor_quotes.*.price'   => 'nullable|numeric|min:0',
            'lines.*.vendor_quotes.*.currency'=> 'nullable|string|max:10',
            'lines.*.vendor_selected'         => 'nullable|integer',

            'attachments.*' => 'nullable|file|max:5120',
        ]);
    }


    /*
    |--------------------------------------------------------------------------
    | 2) CREATE HEADER + BANKS + LINES + ATTACHMENTS
    |--------------------------------------------------------------------------
    */
    $files = $request->file('attachments', []);

    DB::transaction(function () use ($validated, $request, $user, $files, &$itemRequest) {

        // HEADER
        $itemRequest = ItemRequest::create([
            'company_id'       => $validated['company_id'] ?? null,
            'requester_id'     => $user->id,
            'flow_template_id' => $validated['flow_template_id'] ?? null,
            'posting_date'     => now(),
            'remarks'          => $validated['remarks'] ?? null,
            'vendor_name'      => $validated['vendor_name'] ?? null,
            'status'           => 'Draft',
            'current_step_no'  => 0,
        ]);

        // BANK LINES
        if (!empty($validated['banks'])) {
            foreach ($validated['banks'] as $bank) {
                VendorRequestLine::create([
                    'item_request_id' => $itemRequest->id,
                    'company_id'      => $validated['company_id'] ?? null,
                    'vendor_name'     => $validated['vendor_name'] ?? null,
                    'npwp'            => $validated['npwp'] ?? null,
                    'phone'           => $validated['phone'] ?? null,
                    'address'         => $validated['address'] ?? null,
                    'remarks'         => $validated['remarks'] ?? null,
                    'bank_name'       => $bank['bank_name'] ?? null,
                    'account_holder'  => $bank['account_holder'] ?? null,
                    'account_number'  => $bank['account_number'] ?? null,
                ]);
            }
        }

        // ITEM REQUEST LINES
        if (!empty($validated['lines'])) {
            $payloads = [];

            foreach ($validated['lines'] as $ln) {

                $quotesRaw   = $ln['vendor_quotes'] ?? [];
                $selectedIdx = $ln['vendor_selected'] ?? null;

                // quotes
                $quotes = [];
                foreach ($quotesRaw as $idx => $q) {
                    if (blank($q['name'] ?? null) && blank($q['price'] ?? null)) continue;

                    $quotes[] = [
                        'name'     => $q['name'] ?? '',
                        'price'    => $q['price'] ?? null,
                        'currency' => $q['currency'] ?? 'IDR',
                        'selected' => (string)$idx === (string)$selectedIdx,
                    ];
                }

                $clean = Arr::only($ln, [
                    'article_name','type','base_unit_code','gl_account_no',
                    'inventory_posting_group_code','gen_prod_posting_group_code',
                    'l1','l2','article_no','line_remarks',
                ]);

                $clean['vendor_quotes']   = $quotes ?: null;
                $clean['item_request_id'] = $itemRequest->id;

                $payloads[] = $clean;
            }

            $itemRequest->lines()->createMany($payloads);
        }

        // ATTACHMENTS
        DB::afterCommit(function () use ($itemRequest, $files) {
            foreach ($files as $file) {
                $path = $file->store("requests/{$itemRequest->id}", 'public');
                $itemRequest->attachments()->create([
                    'uploader_user_id' => auth()->id(),
                    'original_name'    => $file->getClientOriginalName(),
                    'path'             => $path,
                    'mime'             => $file->getClientMimeType(),
                    'size'             => $file->getSize(),
                ]);
            }
        });
    });


    /*
    |--------------------------------------------------------------------------
    | 3) RETURN → CONDITIONAL
    |--------------------------------------------------------------------------
    */
    if ($isAutosave) {
        return response()->json([
            'autosaved'   => true,
            'new_id'      => $itemRequest->id,
            'redirect_to' => route('vendors.edit', $itemRequest->id),
        ]);
    }
    Log::error($isAutosave);
    return redirect()
        ->route('vendors.create')
        ->with('success', 'Vendor + Lines berhasil dibuat dengan Request ID: ' . $itemRequest->id);
}



 public function edit($id)
    {
        $req = ItemRequest::findOrFail($id);
        $companies = Company::all();
        $flows = ApprovalFlowTemplate::all();
        $vendor = VendorRequestLine::where('item_request_id',$id)->get();
  
        return view('approvals.edit_vendor', compact('req', 'companies', 'flows', 'vendor'));
    }

    public function update(Request $request, $id)
{
    Log::info('Vendor Request check:', $request->all());

    $request->validate([
        'vendor_name' => 'required|string|max:255',
        'remarks'     => 'nullable|string|max:1000',
        'address'     => 'required|string|max:500',
        'npwp'        => 'nullable|string|max:50',
        'phone'       => 'nullable|string|max:20',
        'company_id'  => 'required|exists:companies,id',
        'flow_template_id' => 'required|exists:approval_flow_templates,id',
        'banks'       => 'required|array',
        'banks.*.bank_name'       => 'required|string|max:100',
        'banks.*.account_holder'  => 'required|string|max:100',
        'banks.*.account_number'  => 'required|string|max:50',
        'lines'       => 'required|array|min:1',
        'lines.*.article_name'    => 'required|string',
        'lines.*.type'            => 'required|in:Inventory,Service,NonInventory',
        'lines.*.base_unit_code'  => 'required|string',
        'lines.*.gl_account_no'           => 'nullable|string',
        'lines.*.inventory_posting_group_code' => 'nullable|string',
        'lines.*.gen_prod_posting_group_code'  => 'nullable|string',
        'lines.*.l1'              => 'nullable|string',
        'lines.*.l2'              => 'nullable|string',
        'lines.*.article_no'      => 'nullable|string',
        'lines.*.line_remarks'    => 'nullable|string',
        'lines.*.vendor_quotes'   => 'nullable|array',
        'lines.*.vendor_quotes.*.name'     => 'nullable|string|max:120',
        'lines.*.vendor_quotes.*.price'    => 'nullable|numeric|min:0',
        'lines.*.vendor_quotes.*.currency' => 'nullable|string|max:10',
        'lines.*.vendor_selected' => 'nullable|integer',
        'attachments.*'           => 'nullable|file|max:5120',
    ]);

    $itemRequest = ItemRequest::findOrFail($id);
    $files = $request->file('attachments', []);

    DB::transaction(function () use ($request, $itemRequest, $files) {
        // 1. Update header
        $itemRequest->update([
            'company_id'       => (int)$request->company_id,
            'flow_template_id' => $request->flow_template_id,
            'vendor_name'      => $request->vendor_name,
            'remarks'          => $request->remarks ?? null,
            'npwp'             => $request->npwp,
            'phone'            => $request->phone,
            'address'          => $request->address,
        ]);

        // 2. Replace vendor lines
        VendorRequestLine::where('item_request_id', $itemRequest->id)->delete();
        foreach ($request->banks as $bank) {
            VendorRequestLine::create([
                'item_request_id'  => $itemRequest->id,
                'company_id'       => $request->company_id,
                'vendor_name'      => $request->vendor_name,
                'npwp'             => $request->npwp,
                'phone'            => $request->phone,
                'address'          => $request->address,
                'remarks'          => $request->remarks ?? null,
                'bank_name'        => $bank['bank_name'],
                'account_holder'   => $bank['account_holder'],
                'account_number'   => $bank['account_number'],
            ]);
        }

        // 3. Replace item request lines
        $itemRequest->lines()->delete(); // hapus dulu semua line lama
        $payloads = [];
        foreach ($request->lines as $ln) {
            $quotesRaw   = $ln['vendor_quotes'] ?? [];
            $selectedIdx = $ln['vendor_selected'] ?? null;

            $quotes = [];
            foreach ($quotesRaw as $idx => $q) {
                if (blank($q['name'] ?? null) && blank($q['price'] ?? null)) continue;
                $quotes[] = [
                    'name'     => trim((string)($q['name'] ?? '')),
                    'price'    => is_numeric($q['price'] ?? null) ? (float)$q['price'] : null,
                    'currency' => $q['currency'] ?? 'IDR',
                    'selected' => (string)$idx === (string)$selectedIdx,
                ];
            }

            $clean = Arr::only($ln, [
                'article_name','type','base_unit_code','gl_account_no',
                'inventory_posting_group_code','gen_prod_posting_group_code',
                'l1','l2','article_no','line_remarks',
            ]);
            $clean['vendor_quotes']   = $quotes ?: null;
            $clean['item_request_id'] = $itemRequest->id;

            $payloads[] = $clean;
        }

        $itemRequest->lines()->createMany($payloads);

        // 4. Upload attachment baru
        DB::afterCommit(function () use ($itemRequest, $files) {
            foreach ($files as $file) {
                $path = $file->store("requests/{$itemRequest->id}", 'public');
                $itemRequest->attachments()->create([
                    'uploader_user_id' => auth()->id(),
                    'original_name'    => $file->getClientOriginalName(),
                    'path'             => $path,
                    'mime'             => $file->getClientMimeType(),
                    'size'             => $file->getSize(),
                ]);
            }
        });
    });

    return redirect()->route('approvals.show', $itemRequest->id)
                     ->with('success', 'Vendor + Lines berhasil diperbarui.');
}


}
