<?php

// namespace App\Http\Middleware;

// use Closure;
// use Illuminate\Http\Request;
// use App\Models\ActivityLog;
// use Illuminate\Support\Facades\Auth;

// class TrackUserActivity
// {
//     public function handle(Request $request, Closure $next)
//     {
//         $response = $next($request);

//         try {
//             $user = Auth::user();

//             $logData = [
//                 'user_id'    => $user?->id,
//                 'user_name'  => $user?->name ?? 'Guest',
//                 'method'     => $request->method(),
//                 'url'        => $request->fullUrl(),
//                 'ip_address' => $request->ip(),
//                 'action'     => $this->detectAction($request),
//                 'payload'    => $this->getPayload($request),
//             ];

//             ActivityLog::create($logData);
//         } catch (\Throwable $e) {
//             \Log::error("Activity log failed: " . $e->getMessage());
//         }

//         return $response;
//     }

//     private function detectAction(Request $request): string
//     {
//         $method = $request->method();

//         return match ($method) {
//             'POST'   => 'Create Data',
//             'PUT', 
//             'PATCH'  => 'Update Data',
//             'DELETE' => 'Delete Data',
//             default  => 'View Page',
//         };
//     }

//     private function getPayload(Request $request): ?array
//     {
//         if (in_array($request->method(), ['POST', 'PUT', 'PATCH', 'DELETE'])) {
//             return $request->except(['_token', '_method']);
//         }
//         return null;
//     }
// }


namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\ActivityLog;

class TrackUserActivity
{
    public function handle(Request $request, Closure $next)
    {
        // Jalankan request terlebih dahulu
        $response = $next($request);

        try {
            // Dapatkan nama route saat ini
            $route = $request->route();
            $routeName = optional($route?->getAction())['as'] ?? null;

            /**
             * ✅ Batasi logging hanya untuk halaman approvals.show
             *    Jadi middleware tetap aktif global, tapi hanya mencatat route tertentu.
             */
            if ($routeName !== 'approvals.show') {
                return $response;
            }

            $user   = Auth::user();
            if (!$user) {
                return $response; // Tidak perlu log kalau belum login
            }

            $method = $request->method();
            $actionName = $routeName ?? $route?->uri();
            $ip     = $request->ip();
            $url    = $request->fullUrl();

            // Tentukan jenis aksi berdasarkan method
            $action = match ($method) {
                'POST'        => 'Create Data',
                'PUT', 'PATCH'=> 'Update Data',
                'DELETE'      => 'Delete Data',
                default       => 'View Page',
            };

            // Ambil perubahan dari form (jika ada)
            $changes = $this->getFormChanges($request);

            // Simpan log ke database
            ActivityLog::create([
                'user_id'     => $user->id,
                'user_name'   => $user->name,
                'action'      => $action,
                'model_type'  => null,
                'model_id'    => null,
                'description' => $actionName,
                'changes'     => $changes,
                'ip_address'  => $ip,
                'url'         => $url,
                'method'      => $method,
            ]);

        } catch (\Throwable $e) {
            // Catat error ke log laravel tapi jangan hentikan aplikasi
            Log::error('Failed to log user activity: ' . $e->getMessage());
        }

        return $response;
    }

    /**
     * 🔹 Ambil perubahan dari form
     *    - Bisa menangani nested array
     *    - Bisa mencatat file upload
     */
    private function getFormChanges(Request $request): ?string
    {
        $newData = $request->except(['_token', '_method', '_old_data']);
        $oldData = json_decode($request->input('_old_data', '{}'), true);

        $changesText = [];

        $compare = function ($old, $new, $prefix = '') use (&$compare, &$changesText) {
            foreach ($new as $key => $value) {
                $oldValue = $old[$key] ?? null;
                $currentKey = $prefix ? "{$prefix}.{$key}" : $key;

                if (is_array($value)) {
                    $compare(is_array($oldValue) ? $oldValue : [], $value, $currentKey);
                } elseif ($value instanceof \Illuminate\Http\UploadedFile) {
                    $changesText[] = "'{$currentKey}' di-upload file '{$value->getClientOriginalName()}'";
                } else {
                    $oldStr = is_null($oldValue) ? '' : (string)$oldValue;
                    $newStr = is_null($value) ? '' : (string)$value;

                    if ($oldStr !== '' && $oldStr !== $newStr) {
                        $changesText[] = "'{$currentKey}' dari '{$oldStr}' menjadi '{$newStr}'";
                    }
                }
            }
        };

        $compare($oldData, $newData);

        if (empty($changesText)) {
            return null;
        }

        return "Perubahan data: " . implode("; ", $changesText);
    }
}
