@extends('layouts.app')
@section('title', 'New Opname Request')

@push('styles')
    <style>
        .card-rounded {
            border-radius: 1rem
        }

        .badge-soft {
            border: 1px solid rgba(0, 0, 0, .05);
            background: rgba(13, 110, 253, .06);
            color: #0d6efd
        }

        .hint-box {
            background: #f8f9fa;
            border: 1px dashed #dee2e6;
            border-radius: .5rem;
            padding: .75rem;
            font-size: .875rem;
        }
    </style>
@endpush

@section('content')
    <div class="container my-4" x-data="newForm({
        company_id: '{{ old('company_id') ?? ($companies[0]->id ?? '') }}',
        flow_template_id: '{{ old('flow_template_id') ?? ($flows[0]->id ?? '') }}',
        posting_date: '{{ old('posting_date') ?? now()->toDateString() }}',
        remarks: @js(old('remarks') ?? '')
    })">

        <div class="d-flex justify-content-between align-items-center mb-3">
            <h1 class="h4 mb-0">Create Opname Request</h1>
            <a href="{{ route('approvals.mine') }}" class="btn btn-outline-secondary">
                <i class="fa fa-inbox me-1"></i> My Forms
            </a>
        </div>

        <form method="POST" action="{{ route('approvals.store') }}" enctype="multipart/form-data" @submit="beforeSubmit"
            novalidate>
            @csrf

            {{-- ================= HEADER ================= --}}
            <div class="card card-rounded shadow-sm mb-3">
                <div class="card-header bg-light fw-semibold d-flex justify-content-between">
                    <span><i class="fa fa-circle-info me-2"></i>Header</span>
                    <div class="d-flex gap-2">
                        <a href="{{ route('approvals.mine') }}" class="btn btn-outline-secondary btn-sm">Cancel</a>
                        <button type="submit" class="btn btn-success btn-sm" :disabled="!canSubmit">
                            <i class="fa fa-save me-1"></i> Save
                        </button>
                    </div>
                </div>

                <div class="card-body row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Company</label>
                        <select name="company_id" class="form-select" x-model="header.company_id" required>
                            @foreach ($companies as $c)
                                <option value="{{ $c->id }}">{{ $c->code }} — {{ $c->name }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-4">
                        <label class="form-label">Flow Template</label>
                        <select name="flow_template_id" class="form-select" x-model="header.flow_template_id" required>
                            @foreach ($flows as $f)
                                <option value="{{ $f->id }}">{{ $f->name }} (v{{ $f->version }})</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-3">
                        <label class="form-label">Posting Date</label>
                        <input type="date" name="posting_date" class="form-control" x-model="header.posting_date">
                    </div>

                    <div class="col-md-4">
                        <label class="form-label mb-2">Location</label>
                        <select name="location" class="form-control" required>
                            <option value="">-- Pilih Lokasi --</option>
                            @foreach ($locationCodes as $code)
                                <option value="{{ $code }}">{{ $code }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-12">
                        <label class="form-label">Remarks</label>
                        <textarea name="remarks" rows="2" class="form-control" x-model="header.remarks"></textarea>
                    </div>
                </div>
            </div>

            {{-- ================= ATTACHMENT ================= --}}
            <div class="card card-rounded shadow-sm">
                <div class="card-header bg-light fw-semibold d-flex justify-content-between align-items-center">
                    <div>
                        <i class="fa fa-paperclip me-2"></i>Excel Upload
                        <span class="badge badge-soft ms-2">Required</span>
                    </div>

                    <a href="{{ asset('template_stockopname.xlsx') }}" class="btn btn-outline-primary btn-sm"
                        target="_blank">
                        <i class="fa fa-download me-1"></i> Download Template
                    </a>
                </div>

                <div class="card-body" x-data="excelUpload($root)">

                    <div class="hint-box mb-3">
                        <strong>Upload Excel Template</strong>
                        <ul class="mb-1 ps-3">
                            <li>Format: <code>.xlsx</code> / <code>.xls</code></li>
                            <li>Header must match template</li>
                            <li>File is validated before saving</li>
                        </ul>
                    </div>

                    <input type="file" class="form-control" accept=".xlsx,.xls" x-ref="fileInput" @change="handleUpload">

                    {{-- FILE PREVIEW --}}
                    <template x-if="file">
                        <div class="mt-3 border rounded p-2 bg-light">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <div class="fw-semibold" x-text="file.name"></div>
                                    <div class="small text-muted">
                                        Size: <span x-text="(file.size/1024/1024).toFixed(2)+' MB'"></span>
                                    </div>
                                </div>
                                <button type="button" class="btn btn-outline-danger btn-sm" @click="remove">
                                    <i class="fa fa-trash me-1"></i> Remove
                                </button>
                            </div>
                        </div>
                    </template>

                    {{-- STATUS --}}
                    <template x-if="status === 'success'">
                        <div class="alert alert-success mt-3">
                            <i class="fa fa-check-circle me-1"></i>
                            Excel parsed successfully (<strong x-text="rows.length"></strong> rows)
                        </div>
                    </template>

                    <template x-if="status === 'error'">
                        <div class="alert alert-danger mt-3">
                            <i class="fa fa-triangle-exclamation me-1"></i>
                            <span x-text="error"></span>
                        </div>
                    </template>

                    {{-- PREVIEW TABLE --}}
                    <template x-if="rows.length">
                        <div class="table-responsive mt-3">
                            <table class="table table-sm table-bordered">
                                <thead class="table-light">
                                    <tr>
                                        <template x-for="h in headers" :key="h">
                                            <th x-text="h"></th>
                                        </template>
                                    </tr>
                                </thead>
                                <tbody>
                                    <template x-for="(r,i) in rows.slice(0,10)" :key="i">
                                        <tr>
                                            <template x-for="h in headers" :key="h">
                                                <td x-text="r[h]"></td>
                                            </template>
                                        </tr>
                                    </template>
                                </tbody>
                            </table>
                            <div class="small text-muted">Showing first 10 rows</div>
                        </div>
                    </template>

                    <input type="hidden" name="excel_payload" :value="jsonPayload">

                </div>
            </div>

        </form>
    </div>
@endsection

@push('scripts')
    <script src="https://cdn.jsdelivr.net/npm/xlsx/dist/xlsx.full.min.js"></script>

    <script>
        document.addEventListener('alpine:init', () => {

            Alpine.data('newForm', (initial) => ({
                header: initial,
                canSubmit: false,
                beforeSubmit(e) {
                    if (!this.canSubmit) {
                        e.preventDefault();
                        alert('Please upload and validate Excel before saving.');
                    }
                }
            }));

            Alpine.data('excelUpload', (root) => ({
                file: null,
                status: null,
                error: null,
                rows: [],
                headers: [],

                get jsonPayload() {
                    return JSON.stringify(this.rows);
                },

                handleUpload(e) {
                    this.reset();
                    const f = e.target.files[0];
                    if (!f) return;

                    this.file = f;

                    const reader = new FileReader();
                    reader.onload = (evt) => {
                        try {
                            const wb = XLSX.read(new Uint8Array(evt.target.result), {
                                type: 'array'
                            });
                            const sheet = wb.Sheets[wb.SheetNames[0]];
                            const raw = XLSX.utils.sheet_to_json(sheet, {
                                defval: '',
                                raw: false
                            });

                            if (!raw.length) {
                                this.error = 'Excel file is empty.';
                                this.status = 'error';
                                return;
                            }

                            const normalized = raw.map((row, idx) => {
                                const out = {};
                                Object.keys(row).forEach(k => {
                                    out[k.toLowerCase().trim().replace(/\s+/g,
                                        '_')] =
                                        String(row[k]).trim();
                                });
                                out.__row = idx + 2;
                                return out;
                            });

                            const required = [
                                'storage_location',
                                'code',
                                'material_description',
                                'actual',
                                'base_unit_of_measure'
                            ];

                            for (const col of required) {
                                if (!(col in normalized[0])) {
                                    this.error = `Missing required column: ${col}`;
                                    this.status = 'error';
                                    return;
                                }
                            }

                            for (const r of normalized) {
                                for (const col of required) {
                                    if (!r[col]) {
                                        this.error =
                                            `Data kosong pada baris ${r.__row}, kolom "${col}"`;
                                        this.status = 'error';
                                        Alpine.$data(root).canSubmit = false;
                                        return;
                                    }
                                }
                            }

                            this.rows = normalized;
                            this.headers = Object.keys(normalized[0]).filter(h => h !== '__row');
                            this.status = 'success';
                            Alpine.$data(root).canSubmit = true;

                        } catch (err) {
                            this.error = 'Failed to read Excel file.';
                            this.status = 'error';
                        }
                    };

                    reader.readAsArrayBuffer(f);
                },

                remove() {
                    this.reset();
                    this.$refs.fileInput.value = '';
                    Alpine.$data(root).canSubmit = false;
                },

                reset() {
                    this.file = null;
                    this.status = null;
                    this.error = null;
                    this.rows = [];
                    this.headers = [];
                }
            }));

        });
    </script>
@endpush
