<?php
// Set page title
$page_title = "Buyer Management";

// Include header
include '../template_it/header.php';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        // Create new buyer
        $nama_buyer = trim($_POST['nama_buyer']);
        $kode_buyer = strtoupper(trim($_POST['kode_buyer']));
        
        // Check if kode_buyer already exists
        $check_query = "SELECT id FROM buyers WHERE kode_buyer = ?";
        $check_stmt = $conn->prepare($check_query);
        $check_stmt->bind_param("s", $kode_buyer);
        $check_stmt->execute();
        $result = $check_stmt->get_result();
        
        if ($result->num_rows > 0) {
            $_SESSION['error_message'] = "Kode Buyer '{$kode_buyer}' sudah ada!";
        } else {
            $insert_query = "INSERT INTO buyers (nama_buyer, kode_buyer, created_at, updated_at) 
                             VALUES (?, ?, NOW(), NOW())";
            $stmt = $conn->prepare($insert_query);
            $stmt->bind_param("ss", $nama_buyer, $kode_buyer);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Buyer '{$nama_buyer}' berhasil ditambahkan!";
                
                // Log audit
                $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                                VALUES ('CREATE_BUYER', ?, 'it', ?, NOW(), ?)";
                $audit_stmt = $conn->prepare($audit_query);
                $audit_details = "Created buyer: {$nama_buyer} ({$kode_buyer})";
                $ip_address = $_SERVER['REMOTE_ADDR'];
                $audit_stmt->bind_param("iss", $_SESSION['user_id'], $audit_details, $ip_address);
                $audit_stmt->execute();
                $audit_stmt->close();
            } else {
                $_SESSION['error_message'] = "Error: " . $stmt->error;
            }
            $stmt->close();
        }
        $check_stmt->close();
        
        header("Location: buyer_collection.php");
        exit();
        
    } elseif ($action === 'update') {
        // Update existing buyer
        $buyer_id = intval($_POST['buyer_id']);
        $nama_buyer = trim($_POST['nama_buyer']);
        $kode_buyer = strtoupper(trim($_POST['kode_buyer']));
        
        // Check if kode_buyer already exists (exclude current buyer)
        $check_query = "SELECT id FROM buyers WHERE kode_buyer = ? AND id != ?";
        $check_stmt = $conn->prepare($check_query);
        $check_stmt->bind_param("si", $kode_buyer, $buyer_id);
        $check_stmt->execute();
        $result = $check_stmt->get_result();
        
        if ($result->num_rows > 0) {
            $_SESSION['error_message'] = "Kode Buyer '{$kode_buyer}' sudah digunakan buyer lain!";
        } else {
            $update_query = "UPDATE buyers 
                             SET nama_buyer = ?, kode_buyer = ?, updated_at = NOW() 
                             WHERE id = ?";
            $stmt = $conn->prepare($update_query);
            $stmt->bind_param("ssi", $nama_buyer, $kode_buyer, $buyer_id);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Buyer '{$nama_buyer}' berhasil diupdate!";
                
                // Log audit
                $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                                VALUES ('UPDATE_BUYER', ?, 'it', ?, NOW(), ?)";
                $audit_stmt = $conn->prepare($audit_query);
                $audit_details = "Updated buyer: {$nama_buyer} ({$kode_buyer})";
                $ip_address = $_SERVER['REMOTE_ADDR'];
                $audit_stmt->bind_param("iss", $_SESSION['user_id'], $audit_details, $ip_address);
                $audit_stmt->execute();
                $audit_stmt->close();
            } else {
                $_SESSION['error_message'] = "Error: " . $stmt->error;
            }
            $stmt->close();
        }
        $check_stmt->close();
        
        header("Location: buyer_collection.php");
        exit();
        
    } elseif ($action === 'delete') {
        // Delete buyer
        $buyer_id = intval($_POST['buyer_id']);
        
        // Check if buyer has associated colors
        $check_colors_query = "SELECT COUNT(*) as total FROM master_color WHERE buyer_id = ?";
        $check_stmt = $conn->prepare($check_colors_query);
        $check_stmt->bind_param("i", $buyer_id);
        $check_stmt->execute();
        $result = $check_stmt->get_result();
        $row = $result->fetch_assoc();
        
        if ($row['total'] > 0) {
            $_SESSION['error_message'] = "Buyer tidak dapat dihapus karena masih memiliki {$row['total']} data warna!";
        } else {
            // Get buyer name for audit log
            $get_buyer_query = "SELECT nama_buyer, kode_buyer FROM buyers WHERE id = ?";
            $get_stmt = $conn->prepare($get_buyer_query);
            $get_stmt->bind_param("i", $buyer_id);
            $get_stmt->execute();
            $buyer_result = $get_stmt->get_result();
            $buyer_data = $buyer_result->fetch_assoc();
            $get_stmt->close();
            
            $delete_query = "DELETE FROM buyers WHERE id = ?";
            $stmt = $conn->prepare($delete_query);
            $stmt->bind_param("i", $buyer_id);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Buyer '{$buyer_data['nama_buyer']}' berhasil dihapus!";
                
                // Log audit
                $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                                VALUES ('DELETE_BUYER', ?, 'it', ?, NOW(), ?)";
                $audit_stmt = $conn->prepare($audit_query);
                $audit_details = "Deleted buyer: {$buyer_data['nama_buyer']} ({$buyer_data['kode_buyer']})";
                $ip_address = $_SERVER['REMOTE_ADDR'];
                $audit_stmt->bind_param("iss", $_SESSION['user_id'], $audit_details, $ip_address);
                $audit_stmt->execute();
                $audit_stmt->close();
            } else {
                $_SESSION['error_message'] = "Error: " . $stmt->error;
            }
            $stmt->close();
        }
        $check_stmt->close();
        
        header("Location: buyer_collection.php");
        exit();
    }
}

// Get all buyers
$buyers_query = "SELECT b.*, 
                 (SELECT COUNT(*) FROM master_color WHERE buyer_id = b.id) as total_colors,
                 (SELECT COUNT(*) FROM collections WHERE buyer_id = b.id) as total_collections
                 FROM buyers b 
                 ORDER BY b.nama_buyer ASC";
$buyers_result = $conn->query($buyers_query);
?>

<!-- Buyer Management Content -->
<div class="container mx-auto max-w-7xl">
    
    <!-- Alert Messages -->
    <?php if (isset($_SESSION['success_message'])): ?>
    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative mb-4" role="alert">
        <strong class="font-bold">Berhasil!</strong>
        <span class="block sm:inline"><?php echo htmlspecialchars($_SESSION['success_message']); ?></span>
        <button onclick="this.parentElement.remove()" class="absolute top-0 bottom-0 right-0 px-4 py-3">
            <i class="fas fa-times"></i>
        </button>
    </div>
    <?php unset($_SESSION['success_message']); endif; ?>

    <?php if (isset($_SESSION['error_message'])): ?>
    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-4" role="alert">
        <strong class="font-bold">Error!</strong>
        <span class="block sm:inline"><?php echo htmlspecialchars($_SESSION['error_message']); ?></span>
        <button onclick="this.parentElement.remove()" class="absolute top-0 bottom-0 right-0 px-4 py-3">
            <i class="fas fa-times"></i>
        </button>
    </div>
    <?php unset($_SESSION['error_message']); endif; ?>

    <!-- Page Header -->
    <div class="bg-white rounded-lg shadow-md p-6 mb-6">
        <div class="flex items-center justify-between">
            <div class="flex items-center">
                <div class="bg-blue-500 rounded-full p-3 mr-4">
                    <i class="fas fa-building text-white text-2xl"></i>
                </div>
                <div>
                    <h1 class="text-2xl font-bold text-gray-800">Buyer Management</h1>
                    <p class="text-gray-600">Kelola data buyer dan informasi terkait</p>
                </div>
            </div>
            <button onclick="openCreateModal()" 
                    class="bg-blue-500 hover:bg-blue-600 text-white px-6 py-3 rounded-lg transition-colors duration-200 font-semibold">
                <i class="fas fa-plus mr-2"></i>Tambah Buyer Baru
            </button>
        </div>
    </div>

    <!-- Buyers Table -->
    <div class="bg-white rounded-lg shadow-md overflow-hidden">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-800">
                <i class="fas fa-list mr-2 text-blue-500"></i>Daftar Buyer
            </h2>
        </div>
        
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">No</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Kode Buyer</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nama Buyer</th>
                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Total Colors</th>
                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Total Collections</th>
                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php 
                    $no = 1;
                    if ($buyers_result->num_rows > 0):
                        while($buyer = $buyers_result->fetch_assoc()): 
                    ?>
                    <tr class="hover:bg-gray-50 transition-colors">
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo $no++; ?></td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="px-3 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 text-blue-800">
                                <?php echo htmlspecialchars($buyer['kode_buyer']); ?>
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($buyer['nama_buyer']); ?></div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-center">
                            <span class="px-3 py-1 text-sm font-semibold text-green-800 bg-green-100 rounded-full">
                                <?php echo $buyer['total_colors']; ?>
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-center">
                            <span class="px-3 py-1 text-sm font-semibold text-purple-800 bg-purple-100 rounded-full">
                                <?php echo $buyer['total_collections']; ?>
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-center text-sm font-medium">
                            <button onclick='openEditModal(<?php echo json_encode($buyer); ?>)' 
                                    class="text-blue-600 hover:text-blue-900 mr-3" title="Edit">
                                <i class="fas fa-edit"></i> Edit
                            </button>
                            <button onclick="confirmDelete(<?php echo $buyer['id']; ?>, '<?php echo htmlspecialchars($buyer['nama_buyer'], ENT_QUOTES); ?>', <?php echo $buyer['total_colors']; ?>)" 
                                    class="text-red-600 hover:text-red-900" title="Delete">
                                <i class="fas fa-trash"></i> Delete
                            </button>
                        </td>
                    </tr>
                    <?php 
                        endwhile;
                    else:
                    ?>
                    <tr>
                        <td colspan="6" class="px-6 py-8 text-center text-gray-500">
                            <i class="fas fa-inbox text-4xl mb-2"></i>
                            <p>Belum ada data buyer</p>
                        </td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Create Buyer Modal -->
<div id="createModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-lg shadow-lg rounded-md bg-white">
        <div class="flex items-center justify-between mb-4 pb-3 border-b">
            <h3 class="text-xl font-bold text-gray-800">
                <i class="fas fa-plus-circle text-blue-500 mr-2"></i>Tambah Buyer Baru
            </h3>
            <button onclick="closeCreateModal()" class="text-gray-400 hover:text-gray-600">
                <i class="fas fa-times text-2xl"></i>
            </button>
        </div>
        
        <form method="POST" action="" onsubmit="return validateForm('create')">
            <input type="hidden" name="action" value="create">
            
            <div class="space-y-4 mb-4">
                <!-- Nama Buyer -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Nama Buyer <span class="text-red-500">*</span>
                    </label>
                    <input type="text" name="nama_buyer" required
                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                           placeholder="Contoh: Ethan Allen">
                </div>
                
                <!-- Kode Buyer -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Kode Buyer <span class="text-red-500">*</span>
                    </label>
                    <input type="text" name="kode_buyer" required maxlength="10"
                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent uppercase"
                           placeholder="Contoh: EA" style="text-transform: uppercase;">
                    <p class="text-xs text-gray-500 mt-1">Max 10 karakter, otomatis UPPERCASE</p>
                </div>
            </div>
            
            <div class="flex justify-end space-x-3 pt-3 border-t">
                <button type="button" onclick="closeCreateModal()"
                        class="px-6 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600 transition-colors">
                    <i class="fas fa-times mr-2"></i>Cancel
                </button>
                <button type="submit"
                        class="px-6 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors">
                    <i class="fas fa-save mr-2"></i>Simpan
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Buyer Modal -->
<div id="editModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-lg shadow-lg rounded-md bg-white">
        <div class="flex items-center justify-between mb-4 pb-3 border-b">
            <h3 class="text-xl font-bold text-gray-800">
                <i class="fas fa-edit text-blue-500 mr-2"></i>Edit Buyer
            </h3>
            <button onclick="closeEditModal()" class="text-gray-400 hover:text-gray-600">
                <i class="fas fa-times text-2xl"></i>
            </button>
        </div>
        
        <form method="POST" action="" onsubmit="return validateForm('edit')">
            <input type="hidden" name="action" value="update">
            <input type="hidden" name="buyer_id" id="edit_buyer_id">
            
            <div class="space-y-4 mb-4">
                <!-- Nama Buyer -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Nama Buyer <span class="text-red-500">*</span>
                    </label>
                    <input type="text" name="nama_buyer" id="edit_nama_buyer" required
                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                </div>
                
                <!-- Kode Buyer -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Kode Buyer <span class="text-red-500">*</span>
                    </label>
                    <input type="text" name="kode_buyer" id="edit_kode_buyer" required maxlength="10"
                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent uppercase"
                           style="text-transform: uppercase;">
                </div>
            </div>
            
            <div class="flex justify-end space-x-3 pt-3 border-t">
                <button type="button" onclick="closeEditModal()"
                        class="px-6 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600 transition-colors">
                    <i class="fas fa-times mr-2"></i>Cancel
                </button>
                <button type="submit"
                        class="px-6 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors">
                    <i class="fas fa-save mr-2"></i>Update
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Delete Form (Hidden) -->
<form id="deleteForm" method="POST" action="">
    <input type="hidden" name="action" value="delete">
    <input type="hidden" name="buyer_id" id="delete_buyer_id">
</form>

<script>
// Open Create Modal
function openCreateModal() {
    document.getElementById('createModal').classList.remove('hidden');
}

// Close Create Modal
function closeCreateModal() {
    document.getElementById('createModal').classList.add('hidden');
}

// Open Edit Modal
function openEditModal(buyer) {
    document.getElementById('edit_buyer_id').value = buyer.id;
    document.getElementById('edit_nama_buyer').value = buyer.nama_buyer;
    document.getElementById('edit_kode_buyer').value = buyer.kode_buyer;
    
    document.getElementById('editModal').classList.remove('hidden');
}

// Close Edit Modal
function closeEditModal() {
    document.getElementById('editModal').classList.add('hidden');
}

// Confirm Delete
function confirmDelete(buyerId, buyerName, totalColors) {
    if (totalColors > 0) {
        alert(`Buyer "${buyerName}" tidak dapat dihapus karena masih memiliki ${totalColors} data warna!\n\nHapus data warna terlebih dahulu.`);
        return false;
    }
    
    if (confirm(`Apakah Anda yakin ingin menghapus buyer "${buyerName}"?\n\nTindakan ini tidak dapat dibatalkan!`)) {
        document.getElementById('delete_buyer_id').value = buyerId;
        document.getElementById('deleteForm').submit();
    }
}

// Validate Form
function validateForm(type) {
    const prefix = type === 'edit' ? 'edit_' : '';
    const namaBuyer = document.querySelector(`input[name="nama_buyer"]${type === 'edit' ? '#edit_nama_buyer' : ''}`).value.trim();
    const kodeBuyer = document.querySelector(`input[name="kode_buyer"]${type === 'edit' ? '#edit_kode_buyer' : ''}`).value.trim();
    
    if (!namaBuyer) {
        alert('Nama Buyer harus diisi!');
        return false;
    }
    
    if (!kodeBuyer) {
        alert('Kode Buyer harus diisi!');
        return false;
    }
    
    if (kodeBuyer.length > 10) {
        alert('Kode Buyer maksimal 10 karakter!');
        return false;
    }
    
    return true;
}

// Auto uppercase kode buyer
document.querySelectorAll('input[name="kode_buyer"]').forEach(input => {
    input.addEventListener('input', function() {
        this.value = this.value.toUpperCase();
    });
});

// Close modal when clicking outside
document.getElementById('createModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeCreateModal();
    }
});

document.getElementById('editModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeEditModal();
    }
});

// Auto hide alerts after 5 seconds
setTimeout(function() {
    const alerts = document.querySelectorAll('[role="alert"]');
    alerts.forEach(function(alert) {
        alert.style.transition = 'opacity 0.5s';
        alert.style.opacity = '0';
        setTimeout(function() {
            alert.remove();
        }, 500);
    });
}, 5000);
</script>

<?php
include '../template_it/footer.php';
?>