<?php
// Check if user is logged in BEFORE including header
session_start();
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'borrower') {
    header("Location: ../login.php");
    exit();
}

$page_title = "History Peminjaman";
include '../template_borrower/header.php';

$borrower_id = $_SESSION['user_id'];

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

// Filter status
$status_filter = isset($_GET['status']) ? $_GET['status'] : '';

// Query untuk menghitung total records
$count_query = "SELECT COUNT(DISTINCT st.id) as total 
                FROM stock_transactions st 
                WHERE st.borrower_id = ?";
$count_params = [$borrower_id];
$count_types = "i";

if ($status_filter) {
    $count_query .= " AND st.status = ?";
    $count_params[] = $status_filter;
    $count_types .= "s";
}

$count_stmt = $conn->prepare($count_query);
$count_stmt->bind_param($count_types, ...$count_params);
$count_stmt->execute();
$total_records = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_records / $limit);

// Query utama untuk mengambil data
$query = "SELECT st.*, 
                 bu.nama_borrower,
                 COUNT(ti.id) as total_items,
                 GROUP_CONCAT(DISTINCT ti.code_color SEPARATOR ', ') as code_colors,
                 GROUP_CONCAT(DISTINCT CONCAT(ti.code_color, ':', mc.color_name) SEPARATOR '|') as items_detail
          FROM stock_transactions st
          LEFT JOIN borrower_users bu ON st.borrower_id = bu.id
          LEFT JOIN transaction_items ti ON st.id = ti.transaction_id
          LEFT JOIN color_stock_items csi ON ti.stock_item_id = csi.id
          LEFT JOIN master_color mc ON csi.master_color_id = mc.id
          WHERE st.borrower_id = ?";

$params = [$borrower_id];
$types = "i";

if ($status_filter) {
    $query .= " AND st.status = ?";
    $params[] = $status_filter;
    $types .= "s";
}

$query .= " GROUP BY st.id ORDER BY st.created_at DESC LIMIT ? OFFSET ?";
$params[] = $limit;
$params[] = $offset;
$types .= "ii";

$stmt = $conn->prepare($query);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
?>

<!-- Load Html5-QRCode Library -->
<script src="https://unpkg.com/html5-qrcode@2.3.8/html5-qrcode.min.js"></script>

<div class="w-full max-w-7xl mx-auto px-2 sm:px-4">
    <!-- Header Section -->
    <div class="bg-white rounded-lg shadow-md p-4 sm:p-6 mb-4 sm:mb-6">
        <div class="flex items-center mb-2">
            <i class="fas fa-history text-purple-500 text-2xl mr-3"></i>
            <h2 class="text-xl sm:text-2xl font-bold text-gray-800">History Peminjaman</h2>
        </div>
        <p class="text-sm text-gray-600">Riwayat semua peminjaman yang pernah Anda lakukan</p>
    </div>

    <!-- Filter Section -->
    <div class="bg-white rounded-lg shadow-md p-4 sm:p-6 mb-4 sm:mb-6">
        <div class="flex flex-col sm:flex-row gap-4 items-center">
            <label class="text-sm font-medium text-gray-700">Filter Status:</label>
            <select id="statusFilter" onchange="filterByStatus()" 
                    class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500">
                <option value="">Semua Status</option>
                <option value="approved" <?php echo $status_filter === 'approved' ? 'selected' : ''; ?>>Approved</option>
                <option value="returned" <?php echo $status_filter === 'returned' ? 'selected' : ''; ?>>Returned</option>
                <option value="overdue" <?php echo $status_filter === 'overdue' ? 'selected' : ''; ?>>Overdue</option>
            </select>
            
            <?php if ($status_filter): ?>
            <a href="history_peminjaman.php" 
               class="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600 transition-colors text-sm">
                <i class="fas fa-times mr-1"></i>Clear Filter
            </a>
            <?php endif; ?>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <?php
        // Get statistics
        $stats_query = "SELECT 
                          COUNT(*) as total_transactions,
                          SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_count,
                          SUM(CASE WHEN status = 'returned' THEN 1 ELSE 0 END) as returned_count,
                          SUM(CASE WHEN status = 'overdue' THEN 1 ELSE 0 END) as overdue_count
                        FROM stock_transactions 
                        WHERE borrower_id = ?";
        $stats_stmt = $conn->prepare($stats_query);
        $stats_stmt->bind_param("i", $borrower_id);
        $stats_stmt->execute();
        $stats = $stats_stmt->get_result()->fetch_assoc();
        ?>
        
        <div class="bg-white rounded-lg shadow-md p-4">
            <div class="flex items-center">
                <div class="bg-blue-100 p-3 rounded-full">
                    <i class="fas fa-list text-blue-600 text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-gray-600 text-sm">Total</p>
                    <p class="text-2xl font-bold text-gray-800"><?php echo $stats['total_transactions']; ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-md p-4">
            <div class="flex items-center">
                <div class="bg-green-100 p-3 rounded-full">
                    <i class="fas fa-check-circle text-green-600 text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-gray-600 text-sm">Approved</p>
                    <p class="text-2xl font-bold text-gray-800"><?php echo $stats['approved_count']; ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-md p-4">
            <div class="flex items-center">
                <div class="bg-blue-100 p-3 rounded-full">
                    <i class="fas fa-undo text-blue-600 text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-gray-600 text-sm">Returned</p>
                    <p class="text-2xl font-bold text-gray-800"><?php echo $stats['returned_count']; ?></p>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-md p-4">
            <div class="flex items-center">
                <div class="bg-red-100 p-3 rounded-full">
                    <i class="fas fa-exclamation-triangle text-red-600 text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-gray-600 text-sm">Overdue</p>
                    <p class="text-2xl font-bold text-gray-800"><?php echo $stats['overdue_count']; ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- History Table -->
    <div class="bg-white rounded-lg shadow-md overflow-hidden">
        <div class="p-4 sm:p-6 border-b border-gray-200">
            <h3 class="text-lg font-semibold text-gray-800">
                <i class="fas fa-table mr-2"></i>Data Peminjaman
            </h3>
        </div>

        <?php if ($result->num_rows > 0): ?>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Transaksi
                        </th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Items
                        </th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Tanggal
                        </th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Status
                        </th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Aksi
                        </th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php while ($row = $result->fetch_assoc()): ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-4 py-4 whitespace-nowrap">
                            <div>
                                <div class="text-sm font-medium text-gray-900">
                                    <?php echo htmlspecialchars($row['transaction_number']); ?>
                                </div>
                                <div class="text-sm text-gray-500">
                                    <?php echo $row['total_items']; ?> item(s)
                                </div>
                            </div>
                        </td>
                        <td class="px-4 py-4">
                            <div class="text-sm text-gray-900">
                                <?php 
                                $items_detail = explode('|', $row['items_detail']);
                                $display_items = array_slice($items_detail, 0, 3);
                                foreach ($display_items as $item) {
                                    if (strpos($item, ':') !== false) {
                                        list($code, $name) = explode(':', $item, 2);
                                        echo '<span class="inline-block bg-purple-100 text-purple-800 text-xs px-2 py-1 rounded mr-1 mb-1">' . htmlspecialchars($code) . '</span>';
                                    }
                                }
                                if (count($items_detail) > 3) {
                                    echo '<span class="text-xs text-gray-500">+' . (count($items_detail) - 3) . ' more</span>';
                                }
                                ?>
                            </div>
                        </td>
                        <td class="px-4 py-4 whitespace-nowrap">
                            <div class="text-sm text-gray-900">
                                <?php echo date('d M Y', strtotime($row['transaction_date'])); ?>
                            </div>
                            <div class="text-sm text-gray-500">
                                Return: <?php echo date('d M Y', strtotime($row['expected_return_date'])); ?>
                            </div>
                        </td>
                        <td class="px-4 py-4 whitespace-nowrap">
                            <?php
                            $status_class = '';
                            $status_text = '';
                            switch($row['status']) {
                                case 'approved':
                                    $status_class = 'bg-green-100 text-green-800';
                                    $status_text = 'Approved';
                                    break;
                                case 'returned':
                                    $status_class = 'bg-blue-100 text-blue-800';
                                    $status_text = 'Returned';
                                    break;
                                case 'overdue':
                                    $status_class = 'bg-red-100 text-red-800';
                                    $status_text = 'Overdue';
                                    break;
                                default:
                                    $status_class = 'bg-gray-100 text-gray-800';
                                    $status_text = ucfirst($row['status']);
                            }
                            ?>
                            <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $status_class; ?>">
                                <?php echo $status_text; ?>
                            </span>
                        </td>
                        <td class="px-4 py-4 whitespace-nowrap text-sm">
                            <button onclick="viewDetail(<?php echo $row['id']; ?>)" 
                                    class="bg-blue-500 hover:bg-blue-600 text-white px-3 py-1 rounded text-xs">
                                <i class="fas fa-eye mr-1"></i>Detail
                            </button>
                        </td>
                    </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
        <div class="bg-white px-4 py-3 border-t border-gray-200 sm:px-6">
            <div class="flex items-center justify-between">
                <div class="flex-1 flex justify-between sm:hidden">
                    <?php if ($page > 1): ?>
                    <a href="?page=<?php echo $page-1; ?><?php echo $status_filter ? '&status='.$status_filter : ''; ?>" 
                       class="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                        Previous
                    </a>
                    <?php endif; ?>
                    
                    <?php if ($page < $total_pages): ?>
                    <a href="?page=<?php echo $page+1; ?><?php echo $status_filter ? '&status='.$status_filter : ''; ?>" 
                       class="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                        Next
                    </a>
                    <?php endif; ?>
                </div>
                
                <div class="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                    <div>
                        <p class="text-sm text-gray-700">
                            Showing <span class="font-medium"><?php echo $offset + 1; ?></span> to 
                            <span class="font-medium"><?php echo min($offset + $limit, $total_records); ?></span> of 
                            <span class="font-medium"><?php echo $total_records; ?></span> results
                        </p>
                    </div>
                    <div>
                        <nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px">
                            <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <a href="?page=<?php echo $i; ?><?php echo $status_filter ? '&status='.$status_filter : ''; ?>" 
                               class="<?php echo $i == $page ? 'bg-purple-50 border-purple-500 text-purple-600' : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50'; ?> 
                                      relative inline-flex items-center px-4 py-2 border text-sm font-medium">
                                <?php echo $i; ?>
                            </a>
                            <?php endfor; ?>
                        </nav>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php else: ?>
        <div class="text-center py-12">
            <i class="fas fa-inbox text-gray-300 text-6xl mb-4"></i>
            <p class="text-gray-500 text-lg mb-2">Belum Ada History</p>
            <p class="text-gray-400 text-sm">Anda belum pernah melakukan peminjaman</p>
            <a href="ajukan_peminjaman.php" 
               class="mt-4 inline-flex items-center px-4 py-2 bg-purple-500 text-white rounded-lg hover:bg-purple-600 transition-colors">
                <i class="fas fa-plus mr-2"></i>Mulai Peminjaman
            </a>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Detail Modal -->
<div id="detailModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 max-w-4xl shadow-lg rounded-md bg-white">
        <div class="flex items-center justify-between mb-4 border-b pb-3">
            <h3 class="text-xl font-bold text-gray-800">
                <i class="fas fa-info-circle text-purple-600 mr-2"></i>
                Detail Peminjaman
            </h3>
            <button onclick="closeDetailModal()" class="text-gray-400 hover:text-gray-600">
                <i class="fas fa-times text-2xl"></i>
            </button>
        </div>
        
        <div id="detailContent">
            <!-- Content will be loaded here -->
        </div>
    </div>
</div>

<!-- Return Scan Modal -->
<div id="returnScanModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 max-w-2xl shadow-lg rounded-md bg-white">
        <div class="flex items-center justify-between mb-4 border-b pb-3">
            <h3 class="text-xl font-bold text-gray-800">
                <i class="fas fa-qrcode text-green-600 mr-2"></i>
                Scan Barcode Pengembalian
            </h3>
            <button onclick="closeReturnScanModal()" class="text-gray-400 hover:text-gray-600">
                <i class="fas fa-times text-2xl"></i>
            </button>
        </div>
        
        <div class="space-y-4">
            <!-- Item Info -->
            <div id="returnItemInfo" class="bg-purple-50 border border-purple-200 rounded-lg p-4">
                <!-- Will be populated by JS -->
            </div>

            <!-- Scanner -->
            <div class="bg-white rounded-lg">
                <div id="returnReader" class="rounded-lg overflow-hidden border-2 border-gray-300" style="display:none;"></div>

                <div id="returnScanStatus" class="bg-gray-50 border-2 border-dashed border-gray-300 rounded-lg p-4 text-center">
                    <i class="fas fa-camera text-gray-400 text-3xl mb-2"></i>
                    <p class="text-gray-600 text-sm font-medium">Klik tombol untuk mulai scan</p>
                </div>

                <button type="button" 
                        id="btnStartReturnScan"
                        class="w-full mt-4 bg-green-500 hover:bg-green-600 text-white font-semibold py-3 px-4 rounded-lg transition duration-200 flex items-center justify-center shadow-md">
                    <i class="fas fa-camera mr-2"></i>
                    Mulai Scan Barcode
                </button>
            </div>

            <div class="bg-blue-50 border border-blue-200 rounded-lg p-3">
                <h4 class="text-sm font-semibold text-blue-800 mb-2 flex items-center">
                    <i class="fas fa-info-circle mr-2"></i>
                    Petunjuk Pengembalian
                </h4>
                <ul class="text-xs text-blue-700 space-y-1">
                    <li class="flex items-start">
                        <i class="fas fa-check text-blue-500 mr-2 mt-0.5"></i>
                        <span>Pastikan Anda mengarahkan kamera ke barcode yang BENAR</span>
                    </li>
                    <li class="flex items-start">
                        <i class="fas fa-check text-blue-500 mr-2 mt-0.5"></i>
                        <span>Jika barcode tidak sesuai, pengembalian akan ditolak</span>
                    </li>
                    <li class="flex items-start">
                        <i class="fas fa-check text-blue-500 mr-2 mt-0.5"></i>
                        <span>Setelah berhasil, status akan otomatis berubah menjadi "Finished"</span>
                    </li>
                </ul>
            </div>
        </div>
    </div>
</div>

<!-- Toast Notification -->
<div id="toast" class="fixed top-4 right-4 z-[60] hidden">
    <div class="bg-white rounded-lg shadow-lg p-4 min-w-[300px] border-l-4" id="toastContent">
        <div class="flex items-center">
            <i id="toastIcon" class="text-2xl mr-3"></i>
            <div class="flex-1">
                <p id="toastMessage" class="font-medium text-sm"></p>
            </div>
        </div>
    </div>
</div>

<style>
/* Custom styling untuk QR Code reader */
#returnReader {
    width: 100% !important;
    border: none !important;
}

#returnReader__dashboard_section {
    display: none !important;
}

#returnReader__scan_region {
    border-radius: 8px !important;
}

#returnReader video {
    border-radius: 8px !important;
}

#returnReader__camera_selection {
    display: none !important;
}
</style>

<script>
let returnHtml5QrCode = null;
let isReturnScanning = false;
let isReturnProcessing = false;
let lastReturnScanTime = 0;
let returnScanCooldown = 1000;
let expectedCodeColor = '';
let currentTransactionItemId = 0;

// Toast notification function
function showToast(message, type = 'success') {
    const toast = document.getElementById('toast');
    const toastContent = document.getElementById('toastContent');
    const toastIcon = document.getElementById('toastIcon');
    const toastMessage = document.getElementById('toastMessage');
    
    if (type === 'success') {
        toastContent.classList.remove('border-red-500', 'border-yellow-500');
        toastContent.classList.add('border-green-500');
        toastIcon.className = 'fas fa-check-circle text-green-500 text-2xl mr-3';
    } else if (type === 'warning') {
        toastContent.classList.remove('border-red-500', 'border-green-500');
        toastContent.classList.add('border-yellow-500');
        toastIcon.className = 'fas fa-exclamation-triangle text-yellow-500 text-2xl mr-3';
    } else {
        toastContent.classList.remove('border-green-500', 'border-yellow-500');
        toastContent.classList.add('border-red-500');
        toastIcon.className = 'fas fa-exclamation-circle text-red-500 text-2xl mr-3';
    }
    
    toastMessage.textContent = message;
    toast.classList.remove('hidden');
    
    setTimeout(() => {
        toast.classList.add('hidden');
    }, 3000);
}

// Update return scan status
function updateReturnScanStatus(message, icon = 'fa-camera', color = 'gray') {
    const status = document.getElementById('returnScanStatus');
    status.innerHTML = `
        <i class="fas ${icon} text-${color}-500 text-3xl mb-2"></i>
        <p class="text-${color}-600 text-sm font-medium">${message}</p>
    `;
}

// Open return scan modal
function openReturnScanModal(transactionItemId, codeColor, colorName) {
    expectedCodeColor = codeColor;
    currentTransactionItemId = transactionItemId;
    
    const itemInfo = document.getElementById('returnItemInfo');
    itemInfo.innerHTML = `
        <h4 class="font-semibold text-purple-800 mb-2">Item yang akan dikembalikan:</h4>
        <div class="grid grid-cols-1 sm:grid-cols-2 gap-2 text-sm">
            <div>
                <span class="font-medium text-purple-600">Code Color:</span>
                <p class="text-purple-800 font-bold font-mono">${codeColor}</p>
            </div>
            <div>
                <span class="font-medium text-purple-600">Color Name:</span>
                <p class="text-purple-800 font-bold">${colorName}</p>
            </div>
        </div>
        <p class="text-xs text-purple-600 mt-2">
            <i class="fas fa-exclamation-circle mr-1"></i>
            Pastikan Anda scan barcode dengan code: <strong>${codeColor}</strong>
        </p>
    `;
    
    document.getElementById('returnScanModal').classList.remove('hidden');
    updateReturnScanStatus('Klik tombol untuk mulai scan', 'fa-camera', 'gray');
}

// Close return scan modal
function closeReturnScanModal() {
    stopReturnScanning();
    document.getElementById('returnScanModal').classList.add('hidden');
    expectedCodeColor = '';
    currentTransactionItemId = 0;
}

// Process return barcode
function processReturnBarcode(decodedText) {
    const currentTime = Date.now();
    
    if (currentTime - lastReturnScanTime < returnScanCooldown) {
        return;
    }
    
    if (!isReturnScanning || isReturnProcessing) {
        return;
    }
    
    const cleanedText = decodedText.trim().replace(/[^a-zA-Z0-9]/g, '');
    
    if (cleanedText.length < 3 || cleanedText.length > 20) {
        return;
    }
    
    lastReturnScanTime = currentTime;
    isReturnProcessing = true;
    
    updateReturnScanStatus('Memproses kode...', 'fa-spinner fa-spin', 'purple');
    
    // Check if scanned code matches expected code
    if (cleanedText !== expectedCodeColor) {
        showToast(`Barcode tidak sesuai! Harap scan code: ${expectedCodeColor}`, 'error');
        isReturnProcessing = false;
        updateReturnScanStatus('Barcode salah! Arahkan ke barcode yang benar...', 'fa-exclamation-circle', 'red');
        
        setTimeout(() => {
            if (isReturnScanning) {
                updateReturnScanStatus('Arahkan kamera ke barcode...', 'fa-camera', 'purple');
            }
        }, 2000);
        return;
    }
    
    // Process return
    const formData = new FormData();
    formData.append('ajax_return', '1');
    formData.append('transaction_item_id', currentTransactionItemId);
    formData.append('code_color', cleanedText);
    
    fetch('process_return.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast(data.message, 'success');
            stopReturnScanning();
            updateReturnScanStatus('Pengembalian berhasil!', 'fa-check-circle', 'green');
            
            setTimeout(() => {
                closeReturnScanModal();
                window.location.reload();
            }, 2000);
        } else {
            showToast(data.message, 'error');
            isReturnProcessing = false;
            updateReturnScanStatus('Pengembalian gagal...', 'fa-exclamation-circle', 'red');
            
            setTimeout(() => {
                if (isReturnScanning) {
                    updateReturnScanStatus('Arahkan kamera ke barcode...', 'fa-camera', 'purple');
                }
            }, 2000);
        }
    })
    .catch(error => {
        console.error('Fetch Error:', error);
        showToast('Terjadi kesalahan sistem', 'error');
        isReturnProcessing = false;
        updateReturnScanStatus('Error sistem. Coba lagi...', 'fa-exclamation-triangle', 'red');
        
        setTimeout(() => {
            if (isReturnScanning) {
                updateReturnScanStatus('Arahkan kamera ke barcode...', 'fa-camera', 'purple');
            }
        }, 2000);
    });
}

// Start return scanning
function startReturnScanning() {
    const readerDiv = document.getElementById('returnReader');
    
    if (!returnHtml5QrCode) {
        returnHtml5QrCode = new Html5Qrcode("returnReader");
    }
    
    updateReturnScanStatus('Memulai kamera...', 'fa-spinner fa-spin', 'purple');
    document.getElementById('btnStartReturnScan').innerHTML = '<i class="fas fa-stop mr-2"></i>Stop Scan';
    document.getElementById('btnStartReturnScan').className = 'w-full mt-4 bg-red-500 hover:bg-red-600 text-white font-semibold py-3 px-4 rounded-lg transition duration-200 flex items-center justify-center shadow-md';
    readerDiv.style.display = 'block';
    
    const config = {
        fps: 10,
        qrbox: { width: 250, height: 250 },
        aspectRatio: 1.0,
        formatsToSupport: [
            Html5QrcodeSupportedFormats.CODE_128,
            Html5QrcodeSupportedFormats.CODE_39,
            Html5QrcodeSupportedFormats.EAN_13,
            Html5QrcodeSupportedFormats.EAN_8,
            Html5QrcodeSupportedFormats.UPC_A,
            Html5QrcodeSupportedFormats.UPC_E,
            Html5QrcodeSupportedFormats.QR_CODE
        ]
    };
    
    returnHtml5QrCode.start(
        { facingMode: "environment" },
        config,
        (decodedText, decodedResult) => {
            processReturnBarcode(decodedText);
        },
        (errorMessage) => {
            // Ignore scanning errors
        }
    ).then(() => {
        isReturnScanning = true;
        isReturnProcessing = false;
        lastReturnScanTime = 0;
        updateReturnScanStatus(`Arahkan kamera ke barcode: ${expectedCodeColor}`, 'fa-camera', 'purple');
    }).catch(err => {
        console.error('Unable to start scanning:', err);
        showToast('Gagal mengakses kamera', 'error');
        updateReturnScanStatus('Gagal mengakses kamera', 'fa-exclamation-triangle', 'red');
        document.getElementById('btnStartReturnScan').innerHTML = '<i class="fas fa-camera mr-2"></i>Mulai Scan Barcode';
        document.getElementById('btnStartReturnScan').className = 'w-full mt-4 bg-green-500 hover:bg-green-600 text-white font-semibold py-3 px-4 rounded-lg transition duration-200 flex items-center justify-center shadow-md';
        readerDiv.style.display = 'none';
    });
}

// Stop return scanning
function stopReturnScanning() {
    const readerDiv = document.getElementById('returnReader');
    
    if (returnHtml5QrCode && isReturnScanning) {
        returnHtml5QrCode.stop().then(() => {
            isReturnScanning = false;
            isReturnProcessing = false;
            readerDiv.style.display = 'none';
            document.getElementById('btnStartReturnScan').innerHTML = '<i class="fas fa-camera mr-2"></i>Mulai Scan Barcode';
            document.getElementById('btnStartReturnScan').className = 'w-full mt-4 bg-green-500 hover:bg-green-600 text-white font-semibold py-3 px-4 rounded-lg transition duration-200 flex items-center justify-center shadow-md';
        }).catch(err => {
            console.error('Error stopping scanner:', err);
        });
    }
}

// Button click handler
document.getElementById('btnStartReturnScan').addEventListener('click', function() {
    if (!isReturnScanning) {
        startReturnScanning();
    } else {
        stopReturnScanning();
        updateReturnScanStatus('Klik tombol untuk mulai scan', 'fa-camera', 'gray');
    }
});

function filterByStatus() {
    const status = document.getElementById('statusFilter').value;
    const url = new URL(window.location);
    if (status) {
        url.searchParams.set('status', status);
    } else {
        url.searchParams.delete('status');
    }
    url.searchParams.delete('page');
    window.location.href = url;
}

function viewDetail(transactionId) {
    document.getElementById('detailModal').classList.remove('hidden');
    document.getElementById('detailContent').innerHTML = '<div class="text-center py-4"><i class="fas fa-spinner fa-spin text-2xl text-gray-400"></i></div>';
    
    fetch(`get_transaction_detail.php?id=${transactionId}`)
        .then(response => response.text())
        .then(data => {
            document.getElementById('detailContent').innerHTML = data;
        })
        .catch(error => {
            document.getElementById('detailContent').innerHTML = '<div class="text-center py-4 text-red-500">Error loading detail</div>';
        });
}

function closeDetailModal() {
    document.getElementById('detailModal').classList.add('hidden');
}

// Close modals when clicking outside
window.onclick = function(event) {
    const detailModal = document.getElementById('detailModal');
    const returnModal = document.getElementById('returnScanModal');
    
    if (event.target === detailModal) {
        closeDetailModal();
    }
    if (event.target === returnModal) {
        closeReturnScanModal();
    }
}

// Clean up on page unload
window.addEventListener('beforeunload', () => {
    if (returnHtml5QrCode && isReturnScanning) {
        returnHtml5QrCode.stop();
    }
});
</script>

<?php include '../template_borrower/footer.php'; ?>