<?php
// Start session
session_start();

// Include database connection
require_once 'config/connection.php';

// Redirect if already logged in
if (isset($_SESSION['user_id']) && isset($_SESSION['user_type'])) {
    switch ($_SESSION['user_type']) {
        case 'it':
            header("Location: it/dashboard.php");
            exit();
        case 'admin':
            header("Location: admin/dashboard.php");
            exit();
        case 'approval':
            header("Location: approval/dashboard.php");
            exit();
        case 'borrower':
            header("Location: borrower/dashboard.php");
            exit();
    }
}

// Initialize variables
$error_message = '';
$success_message = '';

// Process login form
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Get and sanitize inputs
    $username = trim($_POST['username']);
    $password = $_POST['password'];
    
    // Validate inputs
    if (empty($username) || empty($password)) {
        $error_message = "Username dan password harus diisi!";
    } else {
        // Array of user types and their tables with redirect paths
        $user_tables = [
            'it' => ['table' => 'it_users', 'redirect' => 'it/dashboard.php'],
            'admin' => ['table' => 'admin_users', 'redirect' => 'admin/dashboard.php'],
            'approval' => ['table' => 'approval_users', 'redirect' => 'approval/dashboard.php'],
            'borrower' => ['table' => 'borrower_users', 'redirect' => 'borrower/dashboard.php']
        ];
        
        $user_found = false;
        $user_data = null;
        $found_user_type = '';
        $redirect_path = '';
        
        // Check each user table
        foreach ($user_tables as $user_type => $config) {
            $table = $config['table'];
            
            // Query user from database
            $query = "SELECT * FROM {$table} WHERE username = ? AND status = 'active'";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("s", $username);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows == 1) {
                $user_data = $result->fetch_assoc();
                $found_user_type = $user_type;
                $redirect_path = $config['redirect'];
                $user_found = true;
                $stmt->close();
                break; // Stop searching once user is found
            }
            
            $stmt->close();
        }
        
        if ($user_found) {
            // Verify password
            if (password_verify($password, $user_data['password'])) {
                // Set session variables
                $_SESSION['user_id'] = $user_data['id'];
                $_SESSION['username'] = $user_data['username'];
                $_SESSION['user_type'] = $found_user_type;
                
                // Set additional session data based on user type
                if ($found_user_type == 'approval') {
                    $_SESSION['nama_lengkap'] = $user_data['nama_lengkap'];
                    $_SESSION['jabatan'] = $user_data['jabatan'];
                    $_SESSION['nik'] = $user_data['nik'];
                } elseif ($found_user_type == 'borrower') {
                    $_SESSION['nama_borrower'] = $user_data['nama_borrower'];
                    $_SESSION['department'] = $user_data['department'];
                    $_SESSION['nik'] = $user_data['nik'];
                }
                
                // Update last login
                $table = $user_tables[$found_user_type]['table'];
                $update_query = "UPDATE {$table} SET last_login = NOW() WHERE id = ?";
                $update_stmt = $conn->prepare($update_query);
                $update_stmt->bind_param("i", $user_data['id']);
                $update_stmt->execute();
                $update_stmt->close();
                
                // Log audit
                $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                                VALUES ('LOGIN', ?, ?, ?, NOW(), ?)";
                $audit_stmt = $conn->prepare($audit_query);
                $audit_details = "User {$username} logged in as {$found_user_type}";
                $ip_address = $_SERVER['REMOTE_ADDR'];
                $audit_stmt->bind_param("isss", $user_data['id'], $found_user_type, $audit_details, $ip_address);
                $audit_stmt->execute();
                $audit_stmt->close();
                
                // Redirect to dashboard
                header("Location: " . $redirect_path);
                exit();
            } else {
                $error_message = "Password salah!";
            }
        } else {
            $error_message = "Username tidak ditemukan atau akun tidak aktif!";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=no">
    <title>Login | Color Access System</title>
    <link rel="icon" href="images/logo.png" type="image/png">
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        'poppins': ['Poppins', 'sans-serif'],
                    },
                    backgroundImage: {
                        'gradient-primary': 'linear-gradient(135deg, #2C5364 0%, #203A43 50%, #0F2027 100%)',
                    },
                    colors: {
                        primary: '#203A43',
                        secondary: '#2C5364',
                        tertiary: '#0F2027',
                    }
                }
            }
        }
    </script>
    <style>
        html, body {
            height: 100%;
            overflow: hidden;
        }
        
        body {
            background: linear-gradient(135deg, #2C5364 0%, #203A43 50%, #0F2027 100%);
            font-family: 'Poppins', sans-serif;
        }

        .main-container {
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 1rem;
        }

        /* Mobile Portrait */
        @media (max-width: 480px) and (orientation: portrait) {
            .main-container {
                padding: 0.75rem;
            }
            
            .logo-section {
                margin-bottom: 1rem;
            }
            
            .logo-image {
                height: 4rem;
                width: 4rem;
            }
            
            .title-text {
                font-size: 1.5rem;
                margin-bottom: 0.25rem;
            }
            
            .subtitle-text {
                font-size: 0.75rem;
            }
            
            .login-card {
                padding: 1.25rem;
                margin-top: 0.75rem;
            }
            
            .card-title {
                font-size: 1.25rem;
                margin-bottom: 1rem;
            }
            
            .form-spacing {
                gap: 0.75rem;
            }
            
            .input-field {
                padding: 0.75rem 1rem 0.75rem 2.5rem;
                font-size: 0.875rem;
            }
            
            .submit-btn {
                padding: 0.75rem;
                font-size: 0.875rem;
            }
            
            .footer-text {
                font-size: 0.625rem;
                margin-top: 0.75rem;
            }
        }

        /* Mobile Landscape */
        @media (max-width: 812px) and (orientation: landscape) {
            .main-container {
                padding: 0.5rem;
            }
            
            .content-wrapper {
                display: flex;
                align-items: center;
                gap: 2rem;
                max-width: 100%;
            }
            
            .logo-section {
                flex-shrink: 0;
                margin-bottom: 0;
                text-align: center;
            }
            
            .logo-image {
                height: 3rem;
                width: 3rem;
            }
            
            .title-text {
                font-size: 1.25rem;
                margin-bottom: 0.25rem;
            }
            
            .subtitle-text {
                font-size: 0.75rem;
            }
            
            .login-card {
                flex: 1;
                max-width: 20rem;
                padding: 1.25rem;
                margin-top: 0;
            }
            
            .card-title {
                font-size: 1.125rem;
                margin-bottom: 0.75rem;
            }
            
            .form-spacing {
                gap: 0.75rem;
            }
            
            .footer-text {
                display: none;
            }
        }

        /* Tablet */
        @media (min-width: 481px) and (max-width: 1024px) and (orientation: portrait) {
            .logo-image {
                height: 5rem;
                width: 5rem;
            }
            
            .login-card {
                max-width: 24rem;
                padding: 2rem;
            }
        }

        /* Desktop */
        @media (min-width: 1025px) {
            .logo-image {
                height: 6rem;
                width: 6rem;
            }
            
            .login-card {
                max-width: 28rem;
                padding: 2rem;
            }
        }

        /* Focus and interaction styles */
        .input-field:focus {
            outline: none;
            border-color: #3B82F6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }

        .fade-in {
            animation: fadeIn 0.6s ease-out;
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <div class="w-full max-w-md mx-auto">
            <!-- For landscape mobile, use flex layout -->
            <div class="content-wrapper lg:block">
                <!-- Logo and Title -->
                <div class="logo-section text-center fade-in">
                    <div class="flex justify-center mb-2 sm:mb-3">
                        <div class="rounded-full p-1 bg-white ring-2 sm:ring-4 ring-white/50">
                            <img src="images/logo.png" alt="Logo" class="logo-image rounded-full object-cover">
                        </div>
                    </div>
                    <h2 class="title-text text-2xl sm:text-3xl font-bold text-white">Color Access System</h2>
                    <p class="subtitle-text text-sm text-gray-300">Sistem Peminjaman Warna Furniture</p>
                </div>

                <!-- Login Card -->
                <div class="login-card bg-white rounded-xl shadow-2xl fade-in">
                    <h3 class="card-title text-xl font-bold text-gray-800 text-center">Login</h3>

                    <!-- Error Message -->
                    <?php if (!empty($error_message)): ?>
                    <div class="bg-red-100 border border-red-400 text-red-700 px-3 py-2 rounded-lg mb-3 text-sm" role="alert">
                        <div class="flex items-center">
                            <i class="fas fa-exclamation-circle mr-2 flex-shrink-0"></i>
                            <span><?php echo htmlspecialchars($error_message); ?></span>
                        </div>
                    </div>
                    <?php endif; ?>

                    <!-- Success Message -->
                    <?php if (!empty($success_message)): ?>
                    <div class="bg-green-100 border border-green-400 text-green-700 px-3 py-2 rounded-lg mb-3 text-sm" role="alert">
                        <div class="flex items-center">
                            <i class="fas fa-check-circle mr-2 flex-shrink-0"></i>
                            <span><?php echo htmlspecialchars($success_message); ?></span>
                        </div>
                    </div>
                    <?php endif; ?>

                    <!-- Login Form -->
                    <form method="POST" action="" class="form-spacing flex flex-col">
                        <!-- Username -->
                        <div>
                            <label for="username" class="block text-sm font-medium text-gray-700 mb-1.5">
                                Username <span class="text-red-500">*</span>
                            </label>
                            <div class="relative">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <i class="fas fa-user text-gray-400"></i>
                                </div>
                                <input type="text" 
                                       id="username" 
                                       name="username" 
                                       required
                                       class="input-field w-full pl-10 pr-4 py-2.5 border border-gray-300 rounded-lg transition-all"
                                       placeholder="Masukkan username"
                                       autocomplete="username"
                                       autocapitalize="none"
                                       autocorrect="off"
                                       value="<?php echo isset($_POST['username']) ? htmlspecialchars($_POST['username']) : ''; ?>">
                            </div>
                        </div>

                        <!-- Password -->
                        <div>
                            <label for="password" class="block text-sm font-medium text-gray-700 mb-1.5">
                                Password <span class="text-red-500">*</span>
                            </label>
                            <div class="relative">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <i class="fas fa-lock text-gray-400"></i>
                                </div>
                                <input type="password" 
                                       id="password" 
                                       name="password" 
                                       required
                                       class="input-field w-full pl-10 pr-12 py-2.5 border border-gray-300 rounded-lg transition-all"
                                       placeholder="Masukkan password"
                                       autocomplete="current-password">
                                <button type="button" 
                                        onclick="togglePassword()" 
                                        class="absolute inset-y-0 right-0 pr-3 flex items-center"
                                        aria-label="Toggle password visibility">
                                    <i id="toggleIcon" class="fas fa-eye text-gray-400 hover:text-gray-600"></i>
                                </button>
                            </div>
                        </div>

                        <!-- Submit Button -->
                        <button type="submit"
                                class="submit-btn w-full bg-gradient-primary text-white py-2.5 rounded-lg font-semibold hover:opacity-90 transition-all duration-200 shadow-lg hover:shadow-xl transform hover:-translate-y-0.5 mt-2">
                            <i class="fas fa-sign-in-alt mr-2"></i>Login
                        </button>
                    </form>

                    <!-- Help Text -->
                    <div class="text-center mt-3">
                        <p class="text-xs text-gray-500">
                            Lupa password? Hubungi IT Administrator
                        </p>
                    </div>
                </div>
            </div>

            <!-- Footer -->
            <div class="footer-text text-center mt-4">
                <p class="text-xs text-gray-300">
                    &copy; <?php echo date('Y'); ?> Color Access System. All rights reserved.
                </p>
            </div>
        </div>
    </div>

    <script>
        // Toggle password visibility
        function togglePassword() {
            const passwordInput = document.getElementById('password');
            const toggleIcon = document.getElementById('toggleIcon');
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                toggleIcon.classList.remove('fa-eye');
                toggleIcon.classList.add('fa-eye-slash');
            } else {
                passwordInput.type = 'password';
                toggleIcon.classList.remove('fa-eye-slash');
                toggleIcon.classList.add('fa-eye');
            }
        }

        // Auto hide alerts after 5 seconds
        setTimeout(function() {
            const alerts = document.querySelectorAll('[role="alert"]');
            alerts.forEach(function(alert) {
                alert.style.transition = 'opacity 0.5s';
                alert.style.opacity = '0';
                setTimeout(function() {
                    alert.remove();
                }, 500);
            });
        }, 5000);

        // Auto focus on username field (desktop only)
        document.addEventListener('DOMContentLoaded', function() {
            if (window.innerWidth > 768) {
                document.getElementById('username').focus();
            }
        });

        // Prevent zoom on iOS
        if (/iPad|iPhone|iPod/.test(navigator.userAgent)) {
            const inputs = document.querySelectorAll('input');
            inputs.forEach(input => {
                input.style.fontSize = '16px';
            });
        }
    </script>
</body>
</html>